/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define XITK_WIDGET_C

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>
#include <unistd.h>

#include <xine/sorted_array.h>

#include "_xitk.h"
#include "tips.h"
#include "menu.h"
#include "backend.h"

#define XITK_MAX_GROUP_LEVELS 32

#if 0
void dump_widget_type(xitk_widget_t *w) {
  if(w->type & WIDGET_GROUP) {
    printf("WIDGET_TYPE_GROUP: ");
    if(w->type & WIDGET_GROUP_MEMBER)
      printf("[THE_WIDGET] ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_BROWSER)
      printf("WIDGET_TYPE_BROWSER | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_FILEBROWSER)
      printf("WIDGET_TYPE_FILEBROWSER | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_MRLBROWSER)
      printf("WIDGET_TYPE_MRLBROWSER | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_COMBO)
      printf("WIDGET_TYPE_COMBO | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_TABS)
      printf("WIDGET_TYPE_TABS | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_INTBOX)
      printf("WIDGET_TYPE_INTBOX | ");
    if((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_MENU)
      printf("WIDGET_TYPE_MENU | ");
  }

  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_BUTTON)      printf("WIDGET_TYPE_BUTTON ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON) printf("WIDGET_TYPE_LABELBUTTON ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_SLIDER)      printf("WIDGET_TYPE_SLIDER ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABEL)       printf("WIDGET_TYPE_LABEL ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_CHECKBOX)    printf("WIDGET_TYPE_CHECKBOX ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_IMAGE)       printf("WIDGET_TYPE_IMAGE ");
  if((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INPUTTEXT)   printf("WIDGET_TYPE_INPUTTEXT ");
  printf("\n");
}
#endif

/* NOTE: at least for WIDGET_EVENT_{KEY|DESTROY|CLICK|CHANGE_SKIN}, xitk_widget_t.event ()
 * may modify the widget list, including deleting widgets -- with or without the help of
 * application callbacks.
 * Practical example: unclick a skin browser button -> change to skin with less browser
 * buttons -> delete that very button.
 * Remember to re-examine widget pointers after such call. */

static void xitk_widget_rel_init (xitk_widget_rel_t *r, const char *type) {
  if (xitk_init_NULL ()) {
    xitk_dnode_init (&r->node);
    r->group = NULL;
  }
  xitk_dlist_init (&r->list);
  memcpy (r->t.type, type, sizeof (r->t.type));
}

static int xitk_widget_rel_join (xitk_widget_rel_t *r, xitk_widget_rel_t *group) {
  if (r->group == group)
    return 0;
  if (r->group) {
    xitk_dnode_remove (&r->node);
    r->group = NULL;
  }
  if (group) {
    xitk_widget_rel_t *p = group;
    /* loop protection. */
    do {
      if (p == r)
        return 1;
    } while ((p = p->group));
    xitk_dlist_add_tail (&group->list, &r->node);
    r->group = group;
  }
  return 0;
}

static void xitk_widget_rel_deinit (xitk_widget_rel_t *r) {
  if (r->group) {
    xitk_dnode_remove (&r->node);
    r->group = NULL;
  }
  while (1) {
    xitk_widget_rel_t *s;

    xitk_container (s, r->list.tail.prev, node);
    if (!s->node.prev)
      break;
    xitk_dnode_remove (&s->node);
    s->group = NULL;
  }
}

/*
 * Alloc a memory area of size_t size.
 */
void *xitk_xmalloc(size_t size) {
  void *ptrmalloc;

  /* prevent xitk_xmalloc(0) of possibly returning NULL */
  if(!size)
    size++;

  if((ptrmalloc = calloc(1, size)) == NULL) {
    XITK_DIE("%s(): calloc() failed: %s\n", __FUNCTION__, strerror(errno));
  }

  return ptrmalloc;
}

/*
 * Call the notify_change_skin function, if exist, of each widget in
 * the specifier widget list.
 */
void xitk_widget_list_change_skin (xitk_widget_list_t *wl, xitk_skin_config_t *skonfig) {
  xitk_widget_t *w;
  widget_event_t event;
  int level = 0, n = 0;

  event.type = WIDGET_EVENT_CHANGE_SKIN;
  event.skonfig = skonfig;
  xitk_container (w, wl->list.list.head.next, parent.node);
  while (1) {
    if (w->parent.node.next) {
      if (w->parent.list.head.next != &w->parent.list.tail) {
        xitk_container (w, w->parent.list.head.next, parent.node);
        level++;
        continue;
      }
    } else {
      if (--level < 0)
        break;
      xitk_container (w, &w->parent.node, parent.list.tail);
      /* do the group when it already can see its members with new skin. */
    }
    w->event (w, &event);
    n++;
    xitk_container (w, w->parent.node.next, parent.node);
  }
  wl->flags |= XITK_WL_CYCLE_CHANGED;
  wl->widget_under_mouse = NULL;
  if (wl->xitk->verbosity >= 2)
    printf ("xitk.window.skin.change (%s) = %d.\n", wl->name, n);
}

/*
 * (re)paint the specified widget list.
 */

#define XITK_WIDGET_STATE_PAINT (XITK_WIDGET_STATE_UNSET | XITK_WIDGET_STATE_ENABLE |\
  XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_CLICK | XITK_WIDGET_STATE_ON |\
  XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS)

static int _xitk_widget_paint (xitk_widget_t *w, widget_event_t *e) {
  XITK_HV_INIT;
  int r;

  e->type = WIDGET_EVENT_PAINT;
  e->x = XITK_HV_H (w->pos);
  e->y = XITK_HV_V (w->pos);
  e->width = XITK_HV_H (w->size);
  e->height = XITK_HV_V (w->size);
  r = w->event (w, e);
  w->shown_state = w->state;
  return r;
}

int xitk_partial_paint_widget_list (xitk_widget_list_t *wl, xitk_hull_t *hull) {
  xitk_widget_t *w;
  int level = 0;
  XITK_HV_INIT;
  xitk_hv_t h1, h2;
  int n = 0;
  widget_event_t event = { .type = WIDGET_EVENT_PAINT };

  if (!wl)
    return 0;
  if (!wl->xwin)
    return 0;

  if (hull)
    wl->flags |= XITK_WL_EXPOSED;

  /* optimize for "paint all". */
  if (!hull || !(hull->x1 | (hull->x2 ^ wl->xwin->width) | hull->y1 | (hull->y2 ^ wl->xwin->height))) {
    xitk_container (w, wl->list.list.head.next, parent.node);
    while (1) {
      if (w->parent.node.next) {
        event.x = XITK_HV_H (w->pos);
        event.y = XITK_HV_V (w->pos);
        event.width = XITK_HV_H (w->size);
        event.height = XITK_HV_V (w->size);
        w->event (w, &event);
        w->shown_state = w->state;
        n++;
        if (w->parent.list.head.next != &w->parent.list.tail) {
          xitk_container (w, w->parent.list.head.next, parent.node);
          level++;
          continue;
        }
      } else {
        if (--level < 0)
          break;
        xitk_container (w, &w->parent.node, parent.list.tail);
      }
      xitk_container (w, w->parent.node.next, parent.node);
    }
    if (wl->xitk->verbosity >= 2) {
      if (hull)
        printf ("xitk.window.repaint (%s, %d x %d) = %d.\n", wl->name, hull->x2, hull->y2, n);
      else
        printf ("xitk.window.repaint (user, %s) = %d.\n", wl->name, n);
    }
    return n;
  }

  XITK_HV_H (h1) = hull->x1;
  XITK_HV_V (h1) = hull->y1;
  XITK_HV_H (h2) = hull->x2;
  XITK_HV_V (h2) = hull->y2;

  xitk_container (w, wl->list.list.head.next, parent.node);
  while (1) {
    if (w->parent.node.next) {
      xitk_hv_t wh1, wh2;
      wh1.w = xitk_hv_max (w->pos.w, h1.w);
      wh2.w = xitk_hv_min (w->pos.w + w->size.w, h2.w);
      wh2.w -= wh1.w;
      if (XITK_HV_IS_RECT (wh2)) {
        if (w->type & WIDGET_PARTIAL_PAINTABLE) {
          event.x = XITK_HV_H (wh1);
          event.width = XITK_HV_H (wh2);
          event.y = XITK_HV_V (wh1);
          event.height = XITK_HV_V (wh2);
        } else {
          event.x = XITK_HV_H (w->pos);
          event.width = XITK_HV_H (w->size);
          event.y = XITK_HV_V (w->pos);
          event.height = XITK_HV_V (w->size);
        }
        event.type = WIDGET_EVENT_PAINT;
        w->event (w, &event);
        w->shown_state = w->state;
        n += 1;
      }
      if (w->parent.list.head.next != &w->parent.list.tail) {
        xitk_container (w, w->parent.list.head.next, parent.node);
        level++;
        continue;
      }
    } else {
      if (--level < 0)
        break;
      xitk_container (w, &w->parent.node, parent.list.tail);
    }
    xitk_container (w, w->parent.node.next, parent.node);
  }
  if (wl->xitk->verbosity >= 2)
    printf ("xitk.window.repaint (%s, x %d..%d, y %d..%d) = %d.\n", wl->name,
      hull->x1, hull->x2, hull->y1, hull->y2, n);
  return n;
}

void xitk_widget_set_window_focusable (xitk_widget_t *w) {
  if (!w)
    return;
  w->type &= ~(WIDGET_FOCUSABLE | WIDGET_CLICKABLE | WIDGET_TABABLE);
  w->type |= WIDGET_WINDOW_FOCUSABLE;
  if (w->tips_string && !w->tips_string[0] && w->wl && w->wl->xwin) {
    const char *title = xitk_window_set_window_title (w->wl->xwin, NULL);
    xitk_set_widget_tips (w, title);
  }
}

void xitk_widget_list_window_focus (xitk_widget_list_t *wl, int focus) {
  xitk_widget_t *w;
  widget_event_t event;
  uint32_t new_state = focus ? XITK_WIDGET_STATE_FOCUS : 0;

  if (!wl)
    return;
  if (!wl->xwin)
    return;

  w = wl->widget_focused;
  if (w) {
    if ((w->type & (WIDGET_GROUP_MEMBER | WIDGET_GROUP_MENU | WIDGET_KEEP_FOCUS))
      == (WIDGET_GROUP_MEMBER | WIDGET_GROUP_MENU | WIDGET_KEEP_FOCUS)) {
      /* nasty menu kludge. */
      w->state |= XITK_WIDGET_STATE_FOCUS;
    } else {
      w->state = (w->state & ~XITK_WIDGET_STATE_FOCUS) | new_state;
    }
    _xitk_widget_paint (w, &event);
  }

  /* window focus indicators shall be top level only. */
  xitk_container (w, wl->list.list.head.next, parent.node);
  while (w->parent.node.next) {
    if ((w->type & WIDGET_WINDOW_FOCUSABLE) && ((w->state & XITK_WIDGET_STATE_FOCUS) != new_state)) {
      w->state = (w->state & ~XITK_WIDGET_STATE_FOCUS) | new_state;
      _xitk_widget_paint (w, &event);
    }
    xitk_container (w, w->parent.node.next, parent.node);
  }
}

/*
 * Return 1 if mouse pointer is in widget area.
 */
#ifdef YET_UNUSED
int xitk_is_inside_widget (xitk_widget_t *widget, int x, int y) {
  if(!widget) {
    XITK_WARNING("widget was NULL.\n");
    return 0;
  }

  if(((x >= widget->x) && (x < (widget->x + widget->width))) &&
     ((y >= widget->y) && (y < (widget->y + widget->height)))) {
    widget_event_t        event;

    event.type = WIDGET_EVENT_INSIDE;
    event.x    = x;
    event.y    = y;
    return widget->event (widget, &event) != 2;
  }

  return 0;
}
#endif

/* return widget present at specified coords. note that disabled widgets are
 * accepted as well to show tips for them. */
static xitk_widget_t *xitk_get_widget_at (xitk_widget_list_t *wl, int x, int y) {
  xitk_widget_t *w;
  int level = 0;
  widget_event_t event;
  XITK_HV_INIT;
  xitk_hv_t here;

  if (!wl)
    return NULL;

  /* trap overflow (we dont have windows that large ;-) */
  if (((uint32_t)(x + 32768) | (uint32_t)(y + 32768)) & 0xffff0000)
    return NULL;

  event.type = WIDGET_EVENT_INSIDE;
  XITK_HV_H (here) = x;
  XITK_HV_V (here) = y;
  /* Overlapping widgets paint in forward order. Hit test in reverse then. */
  xitk_container (w, wl->list.list.tail.prev, parent.node);
  while (1) {
    if (w->parent.node.prev) {
      if ((w->state & XITK_WIDGET_STATE_VISIBLE) && xitk_hv_inside (here.w, w->pos.w, w->size.w)) {
        if (w->parent.list.tail.prev != &w->parent.list.head) {
          xitk_container (w, w->parent.list.tail.prev, parent.node);
          level++;
          continue;
        }
        event.x = x;
        event.y = y;
        if (w->event (w, &event) != 2)
          return w;
      }
    } else {
      if (--level < 0)
        break;
      xitk_container (w, &w->parent.node, parent.list.head);
    }
    xitk_container (w, w->parent.node.prev, parent.node);
  }
  return NULL;
}

/* return 0 (no widget), 1 (no redirection), 2 (redirection found) */
static int xitk_widget_apply_focus_redirect (xitk_widget_t **w) {
  xitk_widget_t *fr = *w;
  if (!fr)
    return 0;
  if (!fr->focus_redirect.group)
    return 1;
  do {
    xitk_container (fr, fr->focus_redirect.group, focus_redirect);
  } while (fr->focus_redirect.group);
  *w = fr;
  return 2;
}

/* optimize: if mouse still is over the same widget (or over one of its focus_redirect
 * sources), avoid the full widget tree scan. */
static xitk_widget_t *xitk_widget_test_focus_redirect (xitk_widget_t *w, int x, int y) {
  int level = 0;
  XITK_HV_INIT;
  xitk_hv_t here;

  if (!w)
    return NULL;
  /* trap overflow (we dont have windows that large ;-). */
  if (((uint32_t)(x + 32768) | (uint32_t)(y + 32768)) & 0xffff0000)
    return NULL;
  XITK_HV_H (here) = x;
  XITK_HV_V (here) = y;
  /* test the widget itself first. */
  if (((w->state & (XITK_WIDGET_STATE_GROUP | XITK_WIDGET_STATE_VISIBLE)) == XITK_WIDGET_STATE_VISIBLE)
    && xitk_hv_inside (here.w, w->pos.w, w->size.w))
    return w;
  xitk_container (w, w->focus_redirect.list.head.next, focus_redirect.node);
  while (1) {
    if (w->focus_redirect.node.next) {
      if (((w->state & (XITK_WIDGET_STATE_GROUP | XITK_WIDGET_STATE_VISIBLE)) == XITK_WIDGET_STATE_VISIBLE)
        && xitk_hv_inside (here.w, w->pos.w, w->size.w))
        return w;
      if (w->focus_redirect.list.head.next != &w->focus_redirect.list.tail) {
        xitk_container (w, w->focus_redirect.list.head.next, focus_redirect.node);
        level++;
        continue;
      }
    } else {
      if (--level < 0)
        return NULL;
      xitk_container (w, &w->focus_redirect.node, focus_redirect.list.tail);
    }
    xitk_container (w, w->focus_redirect.node.next, focus_redirect.node);
  }
}

/*
 * Call notify_focus (with FOCUS_MOUSE_[IN|OUT] as focus state),
 * function in right widget (the one who get, and the one who lose focus).
 */
void xitk_motion_notify_widget_list (xitk_widget_list_t *wl, int x, int y, unsigned int state) {
  xitk_widget_t *w;
  widget_event_t event;

  if(!wl) {
    XITK_WARNING("widget list was NULL.\n");
    return;
  }

  wl->mouse.x = x;
  wl->mouse.y = y;
  wl->qual    = state;

  w = wl->widget_pressed;
  if (w && (state & MODIFIER_BUTTON1)) {
    if (!(w->type & WIDGET_FOCUSABLE)) /* window focus image? */
      return;
    /* While holding a widget, it is the only one relevant. */
    if ((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_SLIDER) {
      /* Convenience: while holding the slider, user need not stay within its
       * graphical bounds. Just move to closest possible. */
      wl->widget_focused   = w;
      event.type           = WIDGET_EVENT_CLICK;
      event.x              = x;
      event.y              = y;
      event.pressed        = 1;
      event.button         = 1;
      event.modifier       = state;
      w->event (w, &event);
    } else {
      /* For others, just (un)focus as needed. */
      uint32_t new_state = w->state & ~(XITK_WIDGET_STATE_CLICK | XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
      if (xitk_widget_test_focus_redirect (w, x, y)) {
        new_state |= XITK_WIDGET_STATE_CLICK | XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS;
        wl->widget_under_mouse = w;
      } else {
        wl->widget_under_mouse = NULL;
      }
      if (new_state != w->state) {
        w->state = new_state;
        _xitk_widget_paint (w, &event);
      }
    }
    return;
  }

  if (!(state & MODIFIER_MOUSE_OUTSIDE)) {
    w = xitk_widget_test_focus_redirect (wl->widget_under_mouse, x, y);
    if (w) {
      event.type = WIDGET_EVENT_INSIDE;
      event.x    = x;
      event.y    = y;
      if (w->event (w, &event) == 2)
        w = NULL;
    }
    if (!w) {
      w = xitk_get_widget_at (wl, x, y);
      if (!w) {
        wl->flags &= ~XITK_WL_NO_MOUSE_FOCUS;
      } else if (wl->flags & XITK_WL_NO_MOUSE_FOCUS) {
        /* see _xitk_widget_able () */
        return;
      }
    }
    xitk_widget_apply_focus_redirect (&w);
  } else {
    w = NULL;
    wl->flags &= ~XITK_WL_NO_MOUSE_FOCUS;
  }

  do {
    if (!wl->widget_under_mouse) {
      /* maybe take over old focus? */
      if (!wl->widget_focused)
        break;
      /* HACK: menu sub branch opens with first entry focused. over it anyway to show sub sub. */
      if ((wl->widget_focused->type & WIDGET_GROUP_MENU) && (w == wl->widget_focused)) {
        if (state & MODIFIER_MOUSE_MOVE)
          break;
        return;
      }
      wl->widget_under_mouse = wl->widget_focused;
    }
    if (w == wl->widget_under_mouse) {
      /* might be WIDGET_WINDOW_FOCUSABLE. */
      if (!(wl->widget_under_mouse->type & WIDGET_FOCUSABLE))
        return;
      /* inputtext uses this to set mouse pointer. */
      wl->widget_under_mouse->state |= XITK_WIDGET_STATE_MOUSE;
      if (wl->widget_under_mouse->state != wl->widget_under_mouse->shown_state)
        _xitk_widget_paint (wl->widget_under_mouse, &event);
      return;
    }
    if (!(wl->widget_under_mouse->type & WIDGET_FOCUSABLE))
      break;
#if 0
    /* entering a new widget always means leaving the previous one,
     * even if the new one has no visible response. */
    if (w && !(w->type & WIDGET_FOCUSABLE))
      break;
#endif
    if (wl->widget_under_mouse == wl->widget_focused) {
      if (wl->widget_under_mouse->type & WIDGET_KEEP_FOCUS) {
        if ((wl->widget_under_mouse->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INPUTTEXT) {
          wl->widget_under_mouse->state &= ~XITK_WIDGET_STATE_MOUSE;
          if (wl->widget_under_mouse->state != wl->widget_under_mouse->shown_state)
            _xitk_widget_paint (wl->widget_under_mouse, &event);
          break;
        }
        if ((wl->widget_under_mouse->type & WIDGET_GROUP_COMBO) || !w)
          break;
      }
      wl->widget_focused = NULL;
      wl->widget_under_mouse->state &= ~XITK_WIDGET_STATE_FOCUS;
    }
    wl->widget_under_mouse->state &= ~XITK_WIDGET_STATE_MOUSE;
    if (wl->widget_under_mouse->state != wl->widget_under_mouse->shown_state)
      _xitk_widget_paint (wl->widget_under_mouse, &event);
  } while (0);

  wl->widget_under_mouse = w;
  /* Only give focus and paint when tips are accepted, otherwise associated window is invisible.
   * This call may occur from MotionNotify directly after iconifying window.
   * Also, xitk_tips_show_widget_tips (tips, NULL) now saves us xitk_tips_hide_tips (tips). */
  if (!xitk_tips_show_widget_tips (wl->xitk->tips, w))
    return;

  if (w) {
    /* disabled widgets: now that tips are shown, we are done here. */
    if (!(w->state & XITK_WIDGET_STATE_ENABLE))
      return;
    if (w->type & WIDGET_FOCUSABLE) {
#if 0
      dump_widget_type (w);
#endif
      if (w->type & WIDGET_GROUP_MENU) {
        /* for menu, the latest of keyboard and mouse focuses shall take effect. */
        wl->widget_focused = w;
        w->state |= XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS;
        _xitk_widget_paint (w, &event);
        xitk_menu_auto_pop (w);
      } else {
        /* If widget still marked pressed or focus received, it shall receive the focus again. */
        w->state |= (w == wl->widget_pressed) || (w == wl->widget_focused)
          ? (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS) : XITK_WIDGET_STATE_MOUSE;
        _xitk_widget_paint (w, &event);
      }
    }
  }
}

/*
 * Call notify_focus (with FOCUS_[RECEIVED|LOST] as focus state),
 * then call notify_click function, if exist, of widget at coords x/y.
 */
int xitk_click_notify_widget_list (xitk_widget_list_t *wl, int x, int y, int button, int bUp, int modifier) {
  int res = 0;
  xitk_widget_t *w;
  widget_event_t e1, e2;

  if(!wl) {
    XITK_WARNING("widget list was NULL.\n");
    return 0;
  }

  wl->mouse.x = e1.x = x;
  wl->mouse.y = e1.y = y;
  wl->qual    = e1.modifier = modifier;
  e1.type     = WIDGET_EVENT_CLICK;
  e1.button   = button;

  if ((modifier & MODIFIER_BUTTON1) && (button != 1)
    && wl->widget_pressed && ((wl->widget_pressed->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_SLIDER)) {
    /* keep held slider */
    w = wl->widget_focused = wl->widget_pressed;
  } else {
    w = xitk_get_widget_at (wl, x, y);
    if (xitk_widget_apply_focus_redirect (&w) == 2) {
      XITK_HV_INIT;
      xitk_hv_t here;
      XITK_HV_H (here) = x;
      XITK_HV_V (here) = y;
      here.w = xitk_hv_max (here.w, w->pos.w);
      here.w = xitk_hv_min (here.w, w->pos.w + w->size.w - 0x00010001);
      e1.x = XITK_HV_H (here);
      e1.y = XITK_HV_V (here);
    }
  }

  if (w != wl->widget_under_mouse) {
    if (wl->widget_under_mouse) {
      /* xitk.c already did xitk_tips_hide_tips (wl->xitk->tips). */
      wl->widget_under_mouse->state &= ~XITK_WIDGET_STATE_MOUSE;
      if (wl->widget_under_mouse->state != wl->widget_under_mouse->shown_state)
        _xitk_widget_paint (wl->widget_under_mouse, &e2);
    }
    wl->widget_under_mouse = w;
  }

  if (w) {
    /* non clickable widgets are just part of the window drag handle, ... */
    if (!(w->type & WIDGET_CLICKABLE))
      return 0;
    /* ... while disabled ones are not. */
    if (!(w->state & XITK_WIDGET_STATE_ENABLE))
      return 1;
  }

  if (!bUp) {
    uint32_t state = XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS;

    if (w != wl->widget_focused) {
      state = XITK_WIDGET_STATE_MOUSE;
      if (wl->widget_focused) {
        if ((wl->widget_focused->type & WIDGET_FOCUSABLE) &&
          (wl->widget_focused->state & XITK_WIDGET_STATE_ENABLE)) {
          wl->widget_focused->state &= ~(XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
          if (wl->widget_focused->state != wl->widget_focused->shown_state)
            _xitk_widget_paint (wl->widget_focused, &e2);
        }
        wl->widget_focused = NULL;
      }
      if (button <= 3) {
        state = XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS;
        wl->widget_focused = w;
      }
    }

    if (button == 1)
      wl->widget_pressed = w;
    if (!w)
      return 0;

    if (w->type & WIDGET_FOCUSABLE)
      w->state |= state;

    e1.pressed = 1;
    res = w->event (w, &e1);
    if (((w->state & (XITK_WIDGET_STATE_TOGGLE | XITK_WIDGET_STATE_IMMEDIATE)) == XITK_WIDGET_STATE_IMMEDIATE)
      && (button == 1))
      xitk_tips_repeat (wl->xitk->tips, w);

  } else { /* bUp */

    if (wl->widget_pressed) {
      if ((wl->widget_pressed->type & WIDGET_CLICKABLE) &&
        (wl->widget_pressed->state & XITK_WIDGET_STATE_ENABLE)) {
        e1.pressed = 0;
        res = wl->widget_pressed->event (wl->widget_pressed, &e1);
        /* user may still hold a slider (see xitk_motion_notify_widget_list ()).
         * ungrab it here. */
        if (wl->widget_pressed && (w != wl->widget_pressed) && (button == 1)) {
          if (wl->widget_pressed->type & WIDGET_FOCUSABLE) {
            wl->widget_pressed->state &= ~(XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
            wl->widget_focused = NULL;
            _xitk_widget_paint (wl->widget_pressed, &e2);
          }
        }
        if (wl->widget_pressed &&
          ((wl->widget_pressed->state & (XITK_WIDGET_STATE_TOGGLE | XITK_WIDGET_STATE_IMMEDIATE)) == XITK_WIDGET_STATE_IMMEDIATE))
          xitk_tips_repeat (wl->xitk->tips, NULL);
      }

      if (button == 1)
        wl->widget_pressed = NULL;
    }
  }

  return res;
}

/*
 * Find the first focusable widget in wl, according to direction
 */

static int xitk_widget_pos_cmp (void *a, void *b) {
  xitk_widget_t *d = (xitk_widget_t *)a;
  xitk_widget_t *e = (xitk_widget_t *)b;
  xitk_widget_rel_t *path[2 * XITK_MAX_GROUP_LEVELS], *rel;
  xitk_widget_rel_t **level_d = path + XITK_MAX_GROUP_LEVELS - 1, **level_e = path + 2 * XITK_MAX_GROUP_LEVELS - 1;
  int gd, ge;
  if (d == e)
    return 0;
  /* sort widgets by center top down then left right. keep groups together. */
  for (*level_d = rel = &d->parent; rel->group; *--level_d = rel = rel->group) ;
  for (*level_e = rel = &e->parent; rel->group; *--level_e = rel = rel->group) ;
  while (*level_d == *level_e)
    level_d++, level_e++;
  xitk_container (d, *level_d, parent);
  xitk_container (e, *level_e, parent);
  gd = d->pos.w + ((d->size.w >> 1) & 0x7fff7fff);
  ge = e->pos.w + ((e->size.w >> 1) & 0x7fff7fff);
  return gd - ge;
}

static int _xitk_widget_focus (xitk_widget_t *w) {
  widget_event_t event;
  xitk_widget_list_t *wl;

  if (!w)
    return 0;
  wl = w->wl;
  if (w == w->wl->widget_focused)
    return 2;

  if (wl->widget_under_mouse && (wl->widget_under_mouse != w)) {
    /* we navigate by keyboard now. dont confuse user with leftover mouse focus.
     * switch back by simply moving the mouse again. */
    wl->widget_under_mouse->state &= ~(XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
    if (wl->widget_under_mouse != wl->widget_focused) {
      if (wl->widget_under_mouse->state != wl->widget_under_mouse->shown_state)
        _xitk_widget_paint (wl->widget_under_mouse, &event);
    }
    wl->widget_under_mouse = NULL;
  }
  if (wl->widget_focused) {
    if ((wl->widget_focused->state & XITK_WIDGET_STATE_ENABLE)) {
      wl->widget_focused->state &= ~(XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
      _xitk_widget_paint (wl->widget_focused, &event);
    }
  }
  wl->widget_focused = w;
  xitk_tips_hide_tips (wl->xitk->tips);
  /* NOTE: this may redirect focus to a group member. rereaad wl->widget_focused. */
  wl->widget_focused->state |= XITK_WIDGET_STATE_FOCUS;
  _xitk_widget_paint (wl->widget_focused, &event);
  return 1;
}

int xitk_set_focus_to_next_widget (xitk_widget_list_t *wl, int backward, int modifier) {
  xitk_widget_t *w;
  int i, n;
  XITK_HV_INIT;

  (void)modifier;
  if (!wl)
    return 0;
  if (!wl->xwin)
    return 0;

  if (!wl->cycle_list) {
    wl->cycle_list = xine_sarray_new (128, xitk_widget_pos_cmp);
    if (!wl->cycle_list)
      return 0;
    wl->flags |= XITK_WL_CYCLE_CHANGED;
  } else if (wl->cycle_arrow) {
    wl->flags |= XITK_WL_CYCLE_CHANGED;
  }
  if (wl->flags & XITK_WL_CYCLE_CHANGED) {
    int level = 0;

    wl->flags &= ~XITK_WL_CYCLE_CHANGED;
    wl->cycle_arrow = NULL;
    xine_sarray_clear (wl->cycle_list);
    n = 0;
    xitk_container (w, wl->list.list.head.next, parent.node);
    while (1) {
      if (w->parent.node.next) {
        if (w->type & WIDGET_TABABLE) {
          if (((w->state & (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE))
            == (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE))
            && XITK_HV_H (w->size) && XITK_HV_V (w->size)) {
            xine_sarray_add (wl->cycle_list, w);
            n += 1;
          }
          /* no need to scan the members of a tabable widget. */
        } else if (w->parent.list.head.next != &w->parent.list.tail) {
          level++;
          xitk_container (w, w->parent.list.head.next, parent.node);
          continue;
        }
      } else {
        if (--level < 0)
          break;
        xitk_container (w, &w->parent.node, parent.list.tail);
      }
      xitk_container (w, w->parent.node.next, parent.node);
    }
    if (wl->xitk->verbosity >= 2)
      printf ("xitk.widget_list.cycle.tab.rebuild (%s, %d).\n", wl->name, (int)n);
  } else {
    n = xine_sarray_size (wl->cycle_list);
  }
  if (!n)
    return 0;

  i = backward ? 0 : n - 1;
  w = wl->widget_focused;
  if (w) {
    int j;
    if (!(w->type & WIDGET_TABABLE)) {
      do {
        xitk_widget_t *nw;
        xitk_container (nw, w->parent.group, parent);
        if (memcmp (nw->parent.t.type, XITK_WG_MAGIC, sizeof (nw->parent.t.type)))
          break;
        w = nw;
      } while (!(w->type & WIDGET_TABABLE));
      wl->widget_focused = w;
    }
    j = xine_sarray_binary_search (wl->cycle_list, w);
    if (j >= 0)
      i = j;
  }

  if (backward) {
    i = i > 0 ? i - 1 : n - 1;
  } else {
    i = i < n - 1 ? i + 1 : 0;
  }
  w = xine_sarray_get (wl->cycle_list, i);

  return _xitk_widget_focus (w);
}

int xitk_widgets_cycle (xitk_widget_t * const *list, uint32_t n, uint32_t xitk_key_arrow) {
  static const int8_t _dir[XITK_KEY_LASTCODE] = {
    [XITK_KEY_UP]    = -1,
    [XITK_KEY_LEFT]  = -1,
    [XITK_KEY_DOWN]  = 1,
    [XITK_KEY_RIGHT] = 1
  };
  xitk_widget_list_t *wl = NULL;
  xitk_widget_t *w;
  uint32_t u, have;
  int dir, i;
  XITK_HV_INIT;

  if (xitk_key_arrow >= XITK_KEY_LASTCODE)
    return 0;
  dir = _dir[xitk_key_arrow];
  if (!dir)
    return 0;
  /* find widget list. */
  for (have = u = 0; u < n; u++) {
    w = list[u];
    if (w && w->wl)
      break;
  }
  if (u >= n)
    return 0;
  wl = w->wl;
  /* new list? */
  if (!wl->cycle_list) {
    wl->cycle_list = xine_sarray_new (128, xitk_widget_pos_cmp);
    if (!wl->cycle_list)
      return 0;
    wl->flags |= XITK_WL_CYCLE_CHANGED;
  } else if (wl->cycle_arrow != list) {
    wl->flags |= XITK_WL_CYCLE_CHANGED;
  }
  if (wl->flags & XITK_WL_CYCLE_CHANGED) {
    /* rebuild. */
    wl->flags &= ~XITK_WL_CYCLE_CHANGED;
    wl->cycle_arrow = list;
    xine_sarray_clear (wl->cycle_list);
    for (have = 0; u < n; u++) {
      xitk_widget_rel_t *p;
      w = list[u];
      if (!w)
        continue;
      if (w->wl != wl)
        continue;
      /* paranois: w (still) in list? */
      for (p = &w->parent; p && (p != &wl->list); p = p->group) ;
      if (!p)
        continue;
      if ((w->type & WIDGET_FOCUSABLE) &&
        ((w->state & (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE))
        == (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE))
        && XITK_HV_H (w->size) && XITK_HV_V (w->size)) {
        have++;
        xine_sarray_add (wl->cycle_list, w);
      }
    }
    if (wl->xitk->verbosity >= 2)
      printf ("xitk.widget_list.cycle.arrow_keys.rebuild (%s, %d).\n", wl->name, (int)have);
  } else {
    have = xine_sarray_size (wl->cycle_list);
  }
  if (!have)
    return 0;

  w = wl->widget_focused;
  if (w && ((i = xine_sarray_binary_search (wl->cycle_list, w)) >= 0)) {
    u = i + have + dir;
    i = u % have;
  } else {
    i = dir < 0 ? have - 1 : 0;
  }
  w = xine_sarray_get (wl->cycle_list, i);

  return _xitk_widget_focus (w);
}

static void _xitk_widget_find_tabable (xitk_widget_t **_w) {
  xitk_widget_t *w;

  xitk_widget_apply_focus_redirect (_w);
  w = *_w;
  if (!(w->type & WIDGET_TABABLE) && (w->parent.list.head.next != &w->parent.list.tail)) {
    xitk_widget_t *stack[XITK_MAX_GROUP_LEVELS];
    int level = 0;
    /* some group widgets (intbox etc.) are not tabable themselves, but contain
     * multiple user focusable members. just get the first one then. */
    xitk_container (w, w->parent.list.head.next, parent.node);
    while (1) {
      if (!w->parent.node.next) {
        if (--level < 0)
          break;
        xitk_container (w, stack[level]->parent.node.next, parent.node);
        continue;
      }
      if ((w->type & WIDGET_TABABLE) &&
        ((w->state & (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE))
          == (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE)))
        break;
      if ((w->parent.list.head.next == &w->parent.list.tail) || (level >= XITK_MAX_GROUP_LEVELS)) {
        xitk_container (w, w->parent.node.next, parent.node);
        continue;
      }
      stack[level++] = w;
      xitk_container (w, w->parent.list.head.next, parent.node);
    }
    *_w = (w->parent.node.next) ? w : NULL;
  }
}

/*
 *
 */
int xitk_set_focus_to_widget (xitk_widget_t *w) {
  xitk_widget_list_t *wl;
  xitk_widget_rel_t *p;

  if (!w)
    return 0;
  wl = w->wl;
  if (!wl) {
    XITK_WARNING("widget list is NULL.\n");
    return 0;
  }
  if (!wl->xwin)
    return 0;
  if (!(w->state & XITK_WIDGET_STATE_ENABLE))
    return 0;

  /* paranois: w (still) in list? */
  for (p = &w->parent; p && (p != &wl->list); p = p->group) ;
  if (p) {
    _xitk_widget_find_tabable (&w);
    return _xitk_widget_focus (w);
  }

  XITK_WARNING ("widget not found in list.\n");
  return 0;
}

/*
 * Return the focused widget.
 */
xitk_widget_t *xitk_get_focused_widget (xitk_widget_list_t *wl) {
  return wl ? wl->widget_focused : NULL;
}

#ifdef YET_UNUSED
/*
 * Return the pressed widget.
 */
xitk_widget_t *xitk_get_pressed_widget (xitk_widget_list_t *wl) {
  return wl ? wl->widget_pressed : NULL;
}
#endif

/*
 * Return widget width.
 */
int xitk_get_widget_width (xitk_widget_t *w) {
  XITK_HV_INIT;
  return w ? XITK_HV_H (w->size) : 0;
}

/*
 * Return widget height.
 */
int xitk_get_widget_height (xitk_widget_t *w) {
  XITK_HV_INIT;
  return w ? XITK_HV_V (w->size) : 0;
}

/*
 * Set position of a widget.
 */
int xitk_set_widget_pos (xitk_widget_t *w, int x, int y) {
  XITK_HV_INIT;
  xitk_hv_t here;
  /* do not warn here as this may be NULL intentionally,
   * eg with optional sub widgets in an intbox. */
  if (!w)
    return 0;
  XITK_HV_H (here) = x;
  XITK_HV_V (here) = y;
  if (w->pos.w == here.w)
    return 1;
  w->pos.w = here.w;
  if (w->wl) {
    w->wl->flags |= XITK_WL_CYCLE_CHANGED;
    if (w->wl->widget_under_mouse == w)
      w->wl->widget_under_mouse = NULL;
  }
  return 1;
}

/* Get screen relative position of a widget. */
int xitk_widget_get_pos (xitk_widget_t *w, xitk_rect_t *r) {
  XITK_HV_INIT;

  if (!w || !r)
    return 0;
  if (!w->wl)
    return 0;
  xitk_window_get_window_position (w->wl->xwin, r);
  r->x += XITK_HV_H (w->pos);
  r->y += XITK_HV_V (w->pos);
  r->width = XITK_HV_H (w->size);
  r->height = XITK_HV_V (w->size);
  return 1;
}

uint32_t xitk_get_widget_type (xitk_widget_t *w) {
  return w ? w->type : 0;
}

/*
 * Destroy widgets.
 */
void xitk_widgets_delete (xitk_widget_t **w, unsigned int n) {
  if (w) {
    w += n;
    for (; n; n--) {
      xitk_widget_t *_w = *--w;

      if (_w) {
        widget_event_t event;
        /* recursion safety (_w.delete). */
        *w = NULL;

        if (_w->wl) {
          _w->wl->flags |= XITK_WL_CYCLE_CHANGED;
          if (_w == _w->wl->widget_focused)
            _w->wl->widget_focused = NULL;
          if (_w == _w->wl->widget_under_mouse) {
            xitk_tips_hide_tips (_w->wl->xitk->tips);
            _w->wl->widget_under_mouse = NULL;
          }
          if (_w == _w->wl->widget_pressed)
            _w->wl->widget_pressed = NULL;
        }

        xitk_clipboard_unregister_widget (_w);
        _w->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
        if (_w->type & WIDGET_GROUP) {
          event.type = WIDGET_EVENT_ENABLE;
          _w->event (_w, &event);
          _w->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
        }
        if ((_w->state ^ _w->shown_state) & XITK_WIDGET_STATE_PAINT)
          _xitk_widget_paint (_w, &event);

        event.type = WIDGET_EVENT_DESTROY;
        _w->event (_w, &event);

        xitk_widget_rel_deinit (&_w->parent);
        xitk_widget_rel_deinit (&_w->focus_redirect);

        xitk_ref_string_unref (&_w->tips_string);

        /* spread nails in debug mode. */
        if (_w->wl && _w->wl->xitk && (_w->wl->xitk->verbosity >= 2)) {
          uint32_t *q = (uint32_t *)(void *)_w, n = _w->bytesize >> 2;
          while (n--)
            *q++ = 0xaa55aa55;
        }

        free (_w);
      }
    }
  }
}

/*
 * Destroy widgets from widget list.
 */
void xitk_destroy_widgets (xitk_widget_list_t *wl) {
  xitk_widget_t *wr, *stack[XITK_MAX_GROUP_LEVELS];
  int level = 0, verbosity = 0;
  widget_event_t event;

  if(!wl) {
    XITK_WARNING("widget list was NULL.\n");
    return;
  }

  wl->flags |= XITK_WL_CYCLE_CHANGED;
  wl->widget_focused = NULL;
  xitk_tips_hide_tips (wl->xitk->tips);
  wl->widget_under_mouse = NULL;
  wl->widget_pressed = NULL;

  if (wl->xitk)
    verbosity = wl->xitk->verbosity;

  /* go backwards to reduce memory fragmentation. */
  xitk_container (wr, wl->list.list.tail.prev, parent.node);
  while (1) {
    xitk_widget_t *w;
    /* level up? */
    if (!wr->parent.node.prev) {
      if (--level < 0)
        break;
      wr = stack[level];
      w = wr;
    } else {
      w = wr;
      xitk_clipboard_unregister_widget (w);
      /* hide/desable. */
      w->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      if (w->type & WIDGET_GROUP) {
        event.type = WIDGET_EVENT_ENABLE;
        w->event (w, &event);
        w->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      }
      if ((w->state ^ w->shown_state) & XITK_WIDGET_STATE_PAINT)
        _xitk_widget_paint (w, &event);
      /* send destroy message. */
      event.type = WIDGET_EVENT_DESTROY;
      w->event (w, &event);
      xitk_widget_rel_deinit (&w->focus_redirect);
      /* do possible remaining members. */
      if ((wr->parent.list.tail.prev != &wr->parent.list.head) && (level < XITK_MAX_GROUP_LEVELS)) {
        stack[level++] = wr;
        xitk_container (wr, wr->parent.list.tail.prev, parent.node);
        continue;
      }
    }
    /* unlink. */
    xitk_container (wr, wr->parent.node.prev, parent.node);
    xitk_widget_rel_deinit (&w->parent);
    xitk_ref_string_unref (&w->tips_string);
    /* spread nails in debug mode. */
    if (verbosity >= 2) {
      uint32_t *q = (uint32_t *)(void *)w, n = w->bytesize >> 2;
      while (n--)
        *q++ = 0xaa55aa55;
    }
    free (w);
  }
}

/* hide widgets while the window is unmapped or iconified.
 * this is not very efficient because
 * a) without a desktop compositor, invisible paints have no effect anyway, and
 * b) with a desktop compositor, we actually want to update the currently
 *    invisible window background layer because it will re-show later without
 *    new expose events.
 * instead, reset known animated label widgets at application level on
 * XITK_EV_{HIDE|SHOW} events to cut the big thing.
 */
#ifdef YET_UNUSED
/* Show widgets from widget list. */
void xitk_show_widgets (xitk_widget_list_t *wl, int draw) {
  xitk_widget_t *w, *stack[XITK_MAX_GROUP_LEVELS];
  int level = 0;

  if (!wl) {
    XITK_WARNING ("widget list was NULL.\n");
    return;
  }

  xitk_container (w, wl->list.list.head.next, parent.node);
#if 1
  /* this is currently used with draw == 0 only, after un-iconifying a window.
   * paint will happen later by expose event. */
  (void)draw;
#else
  if (draw) {
    widget_event_t event;

    while (1) {
      if (!w->parent.node.next) {
        if (--level < 0)
          break;
        xitk_container (w, stack[level]->parent.node.next, parent.node);
        continue;
      }

      w->state = (w->state & ~XITK_WIDGET_STATE_VISIBLE) | (w->saved_state & XITK_WIDGET_STATE_VISIBLE);
      if ((w->state ^ w->shown_state) & XITK_WIDGET_STATE_PAINT)
        _xitk_widget_paint (w, &event);

      if ((w->parent.list.head.next == &w->parent.list.tail) || (level >= XITK_MAX_GROUP_LEVELS)) {
        xitk_container (w, w->parent.node.next, parent.node);
        continue;
      }
      stack[level++] = w;
      xitk_container (w, w->parent.list.head.next, parent.node);
    }
  } else
#endif
  {
    while (1) {
      if (!w->parent.node.next) {
        if (--level < 0)
          break;
        xitk_container (w, stack[level]->parent.node.next, parent.node);
        continue;
      }

      w->state = (w->state & ~XITK_WIDGET_STATE_VISIBLE) | (w->saved_state & XITK_WIDGET_STATE_VISIBLE);

      if ((w->parent.list.head.next == &w->parent.list.tail) || (level >= XITK_MAX_GROUP_LEVELS)) {
        xitk_container (w, w->parent.node.next, parent.node);
        continue;
      }
      stack[level++] = w;
      xitk_container (w, w->parent.list.head.next, parent.node);
    }
  }
}

/* Hide widgets from widget list, and suppress useless paints after iconifying a window. */
void xitk_hide_widgets (xitk_widget_list_t *wl) {
  xitk_widget_t *w, *stack[XITK_MAX_GROUP_LEVELS];
  int level = 0;
  widget_event_t event;

  if (!wl) {
    XITK_WARNING ("widget list was NULL.\n");
    return;
  }

  if (wl->widget_under_mouse)
    xitk_tips_hide_tips (wl->xitk->tips);

  xitk_container (w, wl->list.list.head.next, parent.node);
  while (1) {
    if (!w->parent.node.next) {
      if (--level < 0)
        break;
      xitk_container (w, stack[level]->parent.node.next, parent.node);
      continue;
    }

    w->saved_state = w->state;
    w->state &= ~XITK_WIDGET_STATE_VISIBLE;
    if ((w->state ^ w->shown_state) & XITK_WIDGET_STATE_PAINT)
      _xitk_widget_paint (w, &event);

    if ((w->parent.list.head.next == &w->parent.list.tail) || (level >= XITK_MAX_GROUP_LEVELS)) {
      xitk_container (w, w->parent.node.next, parent.node);
      continue;
    }
    stack[level++] = w;
    xitk_container (w, w->parent.list.head.next, parent.node);
  }
}
#endif

int xitk_widget_list_anim (xitk_widget_list_t *wl, int start) {
  xitk_widget_t *w, *stack[XITK_MAX_GROUP_LEVELS];
  int level = 0;
  uint32_t want = (start & 1) * XITK_WIDGET_STATE_VISIBLE, n;
  widget_event_t event;

  if (!wl)
    return 0;

  n = 0;
  xitk_container (w, wl->list.list.head.next, parent.node);
  while (1) {
    if (!w->parent.node.next) {
      if (--level < 0)
        break;
      xitk_container (w, stack[level]->parent.node.next, parent.node);
      continue;
    }
    if (((w->state ^ want) & (XITK_WIDGET_STATE_WANIM | XITK_WIDGET_STATE_VISIBLE))
      == (XITK_WIDGET_STATE_WANIM | XITK_WIDGET_STATE_VISIBLE)) {
      w->state ^= XITK_WIDGET_STATE_VISIBLE;
      n += _xitk_widget_paint (w, &event);
    }

    if ((w->parent.list.head.next == &w->parent.list.tail) || (level >= XITK_MAX_GROUP_LEVELS)) {
      xitk_container (w, w->parent.node.next, parent.node);
      continue;
    }
    stack[level++] = w;
    xitk_container (w, w->parent.list.head.next, parent.node);
  }
  return n;
}

static uint32_t _xitk_widget_able (xitk_widget_t *w) {
  XITK_HV_INIT;
  widget_event_t event;
  uint32_t d;

  w->state &= xitk_bitmove (w->state, XITK_WIDGET_STATE_VISIBLE, XITK_WIDGET_STATE_MOUSE) | ~XITK_WIDGET_STATE_MOUSE;

  d = w->state ^ w->shown_state;

  if (d & (XITK_WIDGET_STATE_UNSET | XITK_WIDGET_STATE_ENABLE)) {
    if (w->state & XITK_WIDGET_STATE_ENABLE) {
      /* enabling. */
      if (!(w->wl->flags & XITK_WL_EXPOSED)) {
        /* window not yet mapped. honour early mouse over protection globally. */
        if (!(w->state & XITK_WIDGET_STATE_RECHECK_MOUSE))
          w->wl->flags |= XITK_WL_NO_MOUSE_FOCUS;
      } else if (!w->wl->widget_under_mouse) {
        xitk_hv_t here;
        XITK_HV_H (here) = w->wl->mouse.x;
        XITK_HV_V (here) = w->wl->mouse.y;
        if (xitk_hv_inside (here.w, w->pos.w, w->size.w)) {
          event.type = WIDGET_EVENT_INSIDE;
          event.x = w->wl->mouse.x;
          event.y = w->wl->mouse.y;
          if (w->event (w, &event) != 2) {
            /* enabling the widget under mouse. this may happen when opening menu,
             * or rolling a list. grab mouse focus only if the caller wants it
             * (tabs, browser). otherwise, block mouse focus until mouse has left
             * all widgets, and prevent accidental modification (eg combo). */
            if (w->state & XITK_WIDGET_STATE_RECHECK_MOUSE) {
              w->state |= XITK_WIDGET_STATE_MOUSE;
              d = w->state ^ w->shown_state;
              w->wl->flags &= ~XITK_WL_NO_MOUSE_FOCUS;
            } else {
              w->wl->flags |= XITK_WL_NO_MOUSE_FOCUS;
            }
          }
        }
      }
      w->state &= ~XITK_WIDGET_STATE_RECHECK_MOUSE;
    } else {
      /* disabling implies full unfocus, and gfx unmouse (tips still allowed). */
      if (w->type & WIDGET_FOCUSABLE) {
        d |= XITK_WIDGET_STATE_FOCUS;
        w->state &= ~(XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_MOUSE);
      }
    }
    if (w->type & WIDGET_GROUP) {
      event.type = WIDGET_EVENT_ENABLE;
      w->event (w, &event);
    }
  }

  if (d & (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE))
    w->wl->flags |= XITK_WL_CYCLE_CHANGED;

  if (d & (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS)) {
    if (d & XITK_WIDGET_STATE_MOUSE) {
      if (w->state & XITK_WIDGET_STATE_MOUSE)
        w->wl->widget_under_mouse = w;
      else if (w == w->wl->widget_under_mouse)
        w->wl->widget_under_mouse = NULL;
    }
    if (d & XITK_WIDGET_STATE_FOCUS) {
      if (w->state & XITK_WIDGET_STATE_FOCUS) {
        /* wl->widget_focused is for tabable widgets only. */
        if (w->wl->widget_focused && (w != w->wl->widget_focused) && (w->type & WIDGET_TABABLE)) {
          w->wl->widget_focused->state &= ~XITK_WIDGET_STATE_FOCUS;
          _xitk_widget_paint (w->wl->widget_focused, &event);
        }
        w->wl->widget_focused = w;
      } else if (w == w->wl->widget_focused) {
        w->wl->widget_focused = NULL;
      }
    }
  }
  /* NOTE: group widgets use this to reposition their members. otherwise, skip the useless paint. */
  if ((d & XITK_WIDGET_STATE_PAINT) && ((w->wl->flags & XITK_WL_EXPOSED) || (w->type & WIDGET_GROUP)))
    _xitk_widget_paint (w, &event);
  return d;
}

unsigned int xitk_widgets_state (xitk_widget_t * const *w, unsigned int n, unsigned int mask, unsigned int state) {
  uint32_t _new = 0;

  if (!w)
    return 0;

  mask &= XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON | XITK_WIDGET_STATE_IMMEDIATE |
    XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_RECHECK_MOUSE;
  if (!mask) {
    int i = n;
    while (--i >= 0) {
      if (w[i])
        return w[i]->state;
    }
    return 0;
  }
  state &= mask;
  mask = ~mask;

  for (; n; n--) {
    xitk_widget_t *_w = *w++;

    if (!_w)
      continue;
    _w->state &= mask;
    _w->state |= state;
    _new = _w->state;
    if (!_w->wl)
      continue;
    _xitk_widget_able (_w);
    _new = _w->state;
  }
  return _new;
}

/*
 *
 */
xitk_image_t *xitk_get_widget_foreground_skin (xitk_widget_t *w) {
  widget_event_t event;
  xitk_image_t  *image = NULL;

  if (!w)
    return NULL;

  event.type = WIDGET_EVENT_GET_SKIN;
  event.skin_layer = FOREGROUND_SKIN;
  event.image = &image;

  return w->event (w, &event) ? image : NULL;
}

/*
 *
 */
#ifdef YET_UNUSED
xitk_image_t *xitk_get_widget_background_skin(xitk_widget_t *w) {
  widget_event_t event;
  xitk_image_t  *image = NULL;

  if (!w)
    return NULL;

  event.type = WIDGET_EVENT_GET_SKIN;
  event.skin_layer = BACKGROUND_SKIN;
  event.image = &image;

  return w->event (w, &event) ? image : NULL;
}
#endif

void xitk_set_widget_tips_and_timeout (xitk_widget_t *w, const char *str, unsigned int timeout) {
  if (w) {
    if (w->wl && (w == w->wl->widget_under_mouse))
      xitk_tips_hide_tips (w->wl->xitk->tips);
    if (str != XITK_TIPS_STRING_KEEP) {
      xitk_ref_string_unref (&w->tips_string);
      if (str)
        w->tips_string = xitk_ref_string_ref (str, 0);
    }
    w->tips_timeout = timeout;
    if (w->type & (WIDGET_GROUP | WIDGET_GROUP_MEMBER)) {
      widget_event_t  event;

      event.type         = WIDGET_EVENT_TIPS_TIMEOUT;
      event.tips_timeout = timeout;
      w->event (w, &event);
    }
  }
}

int xitk_widget_mode (xitk_widget_t *w, int mask, int mode) {
  if (!w)
    return 0;

  mask &= WIDGET_TABABLE | WIDGET_FOCUSABLE | WIDGET_CLICKABLE | WIDGET_KEEP_FOCUS | WIDGET_KEYABLE
        | WIDGET_GROUP_MEMBER | WIDGET_GROUP_MASK;
  w->type = (w->type & ~mask) | (mode & mask);
  return w->type & (WIDGET_TABABLE | WIDGET_FOCUSABLE | WIDGET_CLICKABLE | WIDGET_KEEP_FOCUS | WIDGET_KEYABLE
                    | WIDGET_GROUP_MEMBER | WIDGET_GROUP_MASK);
}

int xitk_widget_key_event (xitk_widget_t *w, const xitk_be_event_t *bee, int mode) {
  widget_event_t event;

  if (!w || !bee)
    return 0;
  if ((bee->type != XITK_EV_KEY_UP) && (bee->type != XITK_EV_KEY_DOWN))
    return 0;

  event.type = WIDGET_EVENT_KEY;
  if (mode != 1) {
    event.modifier = bee->qual;
    event.string = bee->utf8;
  } else {
    if (!(w->state & XITK_WIDGET_STATE_ENABLE))
      return 0;
    event.modifier = bee->qual & ~(MODIFIER_SHIFT | MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5);
    event.string = " ";
  }
  event.pressed = bee->type == XITK_EV_KEY_DOWN;
  event.button = bee->more;
  if (mode == 2) {
    _xitk_widget_find_tabable (&w);
    if (!w)
      return 0;
    if (event.pressed) {
      /* already focused may mean this event was rejected by this widget before.
       * under mouse cares for mouse buttons translated to keys. */
      if ((w == w->wl->widget_focused) && (w == w->wl->widget_under_mouse))
        return 0;
      _xitk_widget_focus (w);
    } else {
      /* dont send key ups without the downs before. */
      if (w != w->wl->widget_focused)
        return 0;
    }
  }

  do {
    if (w->type & WIDGET_KEYABLE) {
      int handled;
      /* closing a combo window by XITK_KEY_ESC down usually sends the up to the parent window.
       * dont close that one as well by hotkey, and set focus on down only. */
      if (!(w->state & XITK_WIDGET_STATE_FOCUS) && (mode == 1) && event.pressed)
        _xitk_widget_focus (w);
      handled = w->event (w, &event);
      if (handled)
        return handled;
    }
    xitk_container (w, w->parent.group, parent);
  } while (!memcmp (w->parent.t.type, XITK_WG_MAGIC, sizeof (w->parent.t.type)));

  return 0;
}


xitk_widget_t *xitk_widget_new (const xitk_new_widget_t *nw, size_t size) {
  xitk_widget_t *w;
  xitk_widget_list_t *wl = NULL;
  void *userdata = NULL;
  const char *skin_element_name = NULL;

  if (nw) {
    userdata = nw->userdata;
    skin_element_name = nw->skin_element_name;
    wl = nw->wl;
    if (wl && !((uintptr_t)wl & 3) && !memcmp (&wl->list.t.type, XITK_WL_MAGIC, 4)) {
      ;
    } else {
      XITK_DIE ("%s(%d): xitk_new_widget_t invalid!\n", __FUNCTION__, __LINE__);
    }
  }

  if (size < sizeof (*w))
    size = sizeof (*w);
  w = xitk_xmalloc (size);
  if (!w)
    return NULL;

  w->bytesize = size;
  xitk_widget_rel_init (&w->parent, XITK_WG_MAGIC);
  w->wl = wl;
  w->userdata = userdata;
  xitk_widget_rel_init (&w->focus_redirect, XITK_WG_MAGIC);

  if (xitk_init_NULL ()) {
    w->event = NULL;
    w->tips_string = NULL;
  }
#if 0
  w->pos.w = w->size.w = 0;
  w->type = 0;
  w->tips_timeout = 0;
#endif

  w->state =
  w->saved_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
  w->shown_state = XITK_WIDGET_STATE_UNSET;

  if (skin_element_name && skin_element_name[0])
    strlcpy (w->skin_element_name, skin_element_name, sizeof (w->skin_element_name));
  else
    memcpy (w->skin_element_name, "-", 2);

  return w;
};

int xitk_widget_set_parent (xitk_widget_t *w, xitk_widget_t *parent) {
  if (!w)
    return 2;
  if (w->wl) {
    w->wl->flags |= XITK_WL_CYCLE_CHANGED;
    if (!(parent && parent->parent.node.next)) {
      /* w removed from list, or moved to a hidden container. */
      if (w == w->wl->widget_focused) {
        w->state &= ~XITK_WIDGET_STATE_FOCUS;
        w->wl->widget_focused = NULL;
      }
      if (w == w->wl->widget_under_mouse) {
        w->state &= ~XITK_WIDGET_STATE_MOUSE;
        w->wl->widget_under_mouse = NULL;
      }
    }
  }
  return xitk_widget_rel_join (&w->parent, parent ? &parent->parent : NULL);
}

int xitk_widget_set_focus_redirect (xitk_widget_t *w, xitk_widget_t *focus_redirect) {
  if (!w)
    return 2;
  return xitk_widget_rel_join (&w->focus_redirect,
    focus_redirect && (focus_redirect->wl == w->wl) ? &focus_redirect->focus_redirect : NULL);
}

void xitk_widget_register_win_pos (xitk_widget_t *w, int set) {
  if (w && w->wl) {
    if (set) {
      w->wl->widget_win_pos = w;
    } else {
      if (w->wl->widget_win_pos == w)
        w->wl->widget_win_pos = NULL;
    }
  }
}

int xitk_widget_select (xitk_widget_t *w, int index) {
  widget_event_t e;
  if (!w)
    return 0;
  e.type = WIDGET_EVENT_SELECT;
  e.button = index;
  return w->event (w, &e);
}

xitk_widget_t *_xitk_new_widget_apply (const xitk_new_widget_t *nw, xitk_widget_t *w) {
  widget_event_t  event;
  /* simplify hit tests. */
  w->saved_state = w->state = (w->state & ~XITK_WIDGET_STATE_GROUP) | (w->type & WIDGET_GROUP);

  w->user_id = nw->user_id;

  if (nw->add_state) {
    xitk_widget_list_t *wl = w->wl;
    xitk_widget_rel_t *group = nw->group ? &nw->group->parent : &wl->list;
    /* simplified xitk_add_widget () */
    xitk_widget_rel_join (&w->parent, group);
    if (nw->add_state != XITK_WIDGET_STATE_KEEP) {
#define _ADD_MASK (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | \
  XITK_WIDGET_STATE_IMMEDIATE | XITK_WIDGET_STATE_ON | XITK_WIDGET_STATE_RECHECK_MOUSE)
      if (!(nw->add_state & XITK_WIDGET_STATE_KEEP))
        w->state &= ~_ADD_MASK;
      w->state |= nw->add_state & _ADD_MASK;
      _xitk_widget_able (w);
    }
  }
  if (nw->mode_mask) {
    const uint32_t mask = nw->mode_mask & (WIDGET_TABABLE | WIDGET_FOCUSABLE | WIDGET_CLICKABLE |
      WIDGET_KEEP_FOCUS | WIDGET_KEYABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_MASK);
    w->type &= ~mask;
    w->type |= nw->mode_value & mask;
  }
  if (nw->tips) {
    w->tips_timeout = XITK_TIPS_TIMEOUT_AUTO;
    w->tips_string = xitk_ref_string_ref (nw->tips, 0);
    if (w->type & (WIDGET_GROUP | WIDGET_GROUP_MEMBER)) {
      event.type         = WIDGET_EVENT_TIPS_TIMEOUT;
      event.tips_timeout = XITK_TIPS_TIMEOUT_AUTO;
      w->event (w, &event);
    }
  }
  return w;
}

int xitk_widget_user_id (xitk_widget_t *w) {
  return w ? w->user_id : 0;
}
