/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

// defines an XML utility interface

#ifndef __CIMXMLUTILS_H
#define __CIMXMLUTILS_H

#include <string>
#include <map>
#include <vector>
#include "XMLTypes.h"
#include <openwsman/cpp/Exception.h>

using namespace std;
using namespace CimXMLTypesNamespace;
using namespace WsmanExceptionNamespace;

namespace CimXMLUtilsNamespace
{
	// function declarations
	// this function should be calles once at the beginning of the program
	void InitXMLLibrary();
	// this function should be called once at the end of the program
	void TerminateXMLLibrary();

	// Exception class, encapsulates exceptions thrown by the XML lib
	// used to implement the XML utility interface
	class CIMXMLException : public GeneralWsmanException
	{
	public:
		CIMXMLException(const char* message)
			:GeneralWsmanException(message, WSMAN_XML_ERROR){}
	};

	// forward class declaration
	class XMLElementImpl;

	// XMLElement represents an XML element node
	class XMLElement
	{
		friend class XMLDocImpl;
		friend class XMLElementImpl;
	private:
		XMLElementImpl* impl;
		XMLElement(XMLElementImpl* i);

	public:
		XMLElement(const XMLElement& other);

		~XMLElement();

		XMLElement& operator=(const XMLElement &other );

		XMLElement CreateChildNode(const string& nodeName,
			const string& ns,
			const string& prefix,
			const string* text = NULL);

		void CreateLeafNode(const string& nodeName,
			const string& ns,
			const string& prefix,
			const string& nodeValue);

		void AddText(const string& nodeValue);

		bool HasNextSibling() const;

		XMLElement GetNextSibling() const;

		bool HasChildren() const;

		bool IsLeafNode() const;

		XMLElement GetFirstChild() const;

		string GetNodeName() const;

		string GetTextValue() const;

		string ToString(bool incRoot = false) const;

		string GetNSUri() const;

		void GetAttributes(map<string, string>& attribs) const;

		string GetAttribValue(const string& name) const;

		void AddAttribValue(const string& name, const string& value);

		void AddNSDefinition(const string& ns, const string* prefix = NULL);

		template <class T>
		void AppendLeaf(const string& name,
			const T& t,
			const string& ns = "",
			const string& prefix = "")
		{
			CreateLeafNode(name, ns, prefix,
				CimXMLTypesNamespace::XMLTypeConverter::TypeToString(t));
		}

		template <class T>
		void AppendLeafArray(const string& name,
			const vector<T>& vec,
			const string& ns = "",
			const string& prefix = "")
		{
			for(unsigned int i = 0 ; i < vec.size(); i++)
			{
				AppendLeaf(name, vec[i], ns, prefix);
			}
		}

		template <class T>
		void AppendNode(const string& name,
			const T& t,
			const string& ns = "",
			const string& prefix = "")
		{
		  XMLElement e = CreateChildNode(name, ns, prefix);
			t.SerializeMembers(e);
		}

		template <class T>
		void AppendNodeArray(const string& name,
			const vector<T>& vec,
			const string& ns = "",
			const string& prefix = "")
		{
			for(unsigned int i = 0 ; i < vec.size(); i++)
			{
				AppendNode(name, vec[i], ns, prefix);
			}
		}

		template <class T>
		void SetValue(T& t) const
		{
			CimXMLTypesNamespace::
				XMLTypeConverter::
				StringToType(GetTextValue(),t);
		}

		template <class T>
		void AddValue(vector<T>& vec) const
		{
			T t;
			CimXMLTypesNamespace::
				XMLTypeConverter::
				StringToType(GetTextValue(),t);
			vec.push_back(t);
		}

		template <class T>
		void SetComplexType(T& t) const
		{
			t.Deserialize(*this);
		}

		template <class T>
		void AddComplexType(vector<T>& vec) const
		{
			T t;
			t.Deserialize(*this);
			vec.push_back(t);
		}

	};

	class XMLDocImpl;

	// XML Doc represents an XML document
	class XMLDoc
	{
	private:
		XMLDocImpl* impl;

		XMLDoc(const XMLDoc& xml);

		XMLDoc& operator=(const XMLDoc &other );

	public:
		XMLDoc(const string& xml);

		XMLDoc(const char* rootName = "root",
			const char* uri = NULL,
			const char* prefix = NULL);

		~XMLDoc();

		XMLElement GetRootNode();

		string ToString(bool incVersionStr = false);
	};

} //CimXMLUtilsNamespace
#endif
