#ifndef CAPSTONE_SYSTEMZ_H
#define CAPSTONE_SYSTEMZ_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2014-2015 */

#ifdef __cplusplus
extern "C" {
#endif

#include "platform.h"

#ifdef _MSC_VER
#pragma warning(disable:4201)
#endif

/// Enums corresponding to SystemZ condition codes
typedef enum sysz_cc {
	SYSZ_CC_INVALID = 0,	///< invalid CC (default)

	SYSZ_CC_O,
	SYSZ_CC_H,
	SYSZ_CC_NLE,
	SYSZ_CC_L,
	SYSZ_CC_NHE,
	SYSZ_CC_LH,
	SYSZ_CC_NE,
	SYSZ_CC_E,
	SYSZ_CC_NLH,
	SYSZ_CC_HE,
	SYSZ_CC_NL,
	SYSZ_CC_LE,
	SYSZ_CC_NH,
	SYSZ_CC_NO,
} sysz_cc;

/// Operand type for instruction's operands
typedef enum sysz_op_type {
	SYSZ_OP_INVALID = 0, ///< = CS_OP_INVALID (Uninitialized).
	SYSZ_OP_REG, ///< = CS_OP_REG (Register operand).
	SYSZ_OP_IMM, ///< = CS_OP_IMM (Immediate operand).
	SYSZ_OP_MEM, ///< = CS_OP_MEM (Memory operand).
	SYSZ_OP_ACREG = 64,	///< Access register operand.
} sysz_op_type;

/// SystemZ registers
typedef enum sysz_reg {
	SYSZ_REG_INVALID = 0,

	SYSZ_REG_0,
	SYSZ_REG_1,
	SYSZ_REG_2,
	SYSZ_REG_3,
	SYSZ_REG_4,
	SYSZ_REG_5,
	SYSZ_REG_6,
	SYSZ_REG_7,
	SYSZ_REG_8,
	SYSZ_REG_9,
	SYSZ_REG_10,
	SYSZ_REG_11,
	SYSZ_REG_12,
	SYSZ_REG_13,
	SYSZ_REG_14,
	SYSZ_REG_15,
	SYSZ_REG_CC,
	SYSZ_REG_F0,
	SYSZ_REG_F1,
	SYSZ_REG_F2,
	SYSZ_REG_F3,
	SYSZ_REG_F4,
	SYSZ_REG_F5,
	SYSZ_REG_F6,
	SYSZ_REG_F7,
	SYSZ_REG_F8,
	SYSZ_REG_F9,
	SYSZ_REG_F10,
	SYSZ_REG_F11,
	SYSZ_REG_F12,
	SYSZ_REG_F13,
	SYSZ_REG_F14,
	SYSZ_REG_F15,

	SYSZ_REG_R0L,

	SYSZ_REG_A0,
	SYSZ_REG_A1,
	SYSZ_REG_A2,
	SYSZ_REG_A3,
	SYSZ_REG_A4,
	SYSZ_REG_A5,
	SYSZ_REG_A6,
	SYSZ_REG_A7,
	SYSZ_REG_A8,
	SYSZ_REG_A9,
	SYSZ_REG_A10,
	SYSZ_REG_A11,
	SYSZ_REG_A12,
	SYSZ_REG_A13,
	SYSZ_REG_A14,
	SYSZ_REG_A15,
	SYSZ_REG_C0,
	SYSZ_REG_C1,
	SYSZ_REG_C2,
	SYSZ_REG_C3,
	SYSZ_REG_C4,
	SYSZ_REG_C5,
	SYSZ_REG_C6,
	SYSZ_REG_C7,
	SYSZ_REG_C8,
	SYSZ_REG_C9,
	SYSZ_REG_C10,
	SYSZ_REG_C11,
	SYSZ_REG_C12,
	SYSZ_REG_C13,
	SYSZ_REG_C14,
	SYSZ_REG_C15,
	SYSZ_REG_V0,
	SYSZ_REG_V1,
	SYSZ_REG_V2,
	SYSZ_REG_V3,
	SYSZ_REG_V4,
	SYSZ_REG_V5,
	SYSZ_REG_V6,
	SYSZ_REG_V7,
	SYSZ_REG_V8,
	SYSZ_REG_V9,
	SYSZ_REG_V10,
	SYSZ_REG_V11,
	SYSZ_REG_V12,
	SYSZ_REG_V13,
	SYSZ_REG_V14,
	SYSZ_REG_V15,
	SYSZ_REG_V16,
	SYSZ_REG_V17,
	SYSZ_REG_V18,
	SYSZ_REG_V19,
	SYSZ_REG_V20,
	SYSZ_REG_V21,
	SYSZ_REG_V22,
	SYSZ_REG_V23,
	SYSZ_REG_V24,
	SYSZ_REG_V25,
	SYSZ_REG_V26,
	SYSZ_REG_V27,
	SYSZ_REG_V28,
	SYSZ_REG_V29,
	SYSZ_REG_V30,
	SYSZ_REG_V31,
	SYSZ_REG_F16,
	SYSZ_REG_F17,
	SYSZ_REG_F18,
	SYSZ_REG_F19,
	SYSZ_REG_F20,
	SYSZ_REG_F21,
	SYSZ_REG_F22,
	SYSZ_REG_F23,
	SYSZ_REG_F24,
	SYSZ_REG_F25,
	SYSZ_REG_F26,
	SYSZ_REG_F27,
	SYSZ_REG_F28,
	SYSZ_REG_F29,
	SYSZ_REG_F30,
	SYSZ_REG_F31,
	SYSZ_REG_F0Q,
	SYSZ_REG_F4Q,

	SYSZ_REG_ENDING,
} sysz_reg;

/// Instruction's operand referring to memory
/// This is associated with SYSZ_OP_MEM operand type above
typedef struct sysz_op_mem {
	uint8_t base;		///< base register, can be safely interpreted as
				///< a value of type `sysz_reg`, but it is only
				///< one byte wide
	uint8_t index;		///< index register, same conditions apply here
	uint64_t length;	///< BDLAddr operand
	int64_t disp;	///< displacement/offset value
} sysz_op_mem;

/// Instruction operand
typedef struct cs_sysz_op {
	sysz_op_type type;	///< operand type
	union {
		sysz_reg reg;		///< register value for REG operand
		int64_t imm;		///< immediate value for IMM operand
		sysz_op_mem mem;	///< base/disp value for MEM operand
	};
} cs_sysz_op;

// Instruction structure
typedef struct cs_sysz {
	sysz_cc cc;		///< Code condition
	/// Number of operands of this instruction,
	/// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_sysz_op operands[6]; ///< operands for this instruction.
} cs_sysz;

/// SystemZ instruction
typedef enum sysz_insn {
	SYSZ_INS_INVALID = 0,

	SYSZ_INS_A,
	SYSZ_INS_ADB,
	SYSZ_INS_ADBR,
	SYSZ_INS_AEB,
	SYSZ_INS_AEBR,
	SYSZ_INS_AFI,
	SYSZ_INS_AG,
	SYSZ_INS_AGF,
	SYSZ_INS_AGFI,
	SYSZ_INS_AGFR,
	SYSZ_INS_AGHI,
	SYSZ_INS_AGHIK,
	SYSZ_INS_AGR,
	SYSZ_INS_AGRK,
	SYSZ_INS_AGSI,
	SYSZ_INS_AH,
	SYSZ_INS_AHI,
	SYSZ_INS_AHIK,
	SYSZ_INS_AHY,
	SYSZ_INS_AIH,
	SYSZ_INS_AL,
	SYSZ_INS_ALC,
	SYSZ_INS_ALCG,
	SYSZ_INS_ALCGR,
	SYSZ_INS_ALCR,
	SYSZ_INS_ALFI,
	SYSZ_INS_ALG,
	SYSZ_INS_ALGF,
	SYSZ_INS_ALGFI,
	SYSZ_INS_ALGFR,
	SYSZ_INS_ALGHSIK,
	SYSZ_INS_ALGR,
	SYSZ_INS_ALGRK,
	SYSZ_INS_ALHSIK,
	SYSZ_INS_ALR,
	SYSZ_INS_ALRK,
	SYSZ_INS_ALY,
	SYSZ_INS_AR,
	SYSZ_INS_ARK,
	SYSZ_INS_ASI,
	SYSZ_INS_AXBR,
	SYSZ_INS_AY,
	SYSZ_INS_BCR,
	SYSZ_INS_BRC,
	SYSZ_INS_BRCL,
	SYSZ_INS_CGIJ,
	SYSZ_INS_CGRJ,
	SYSZ_INS_CIJ,
	SYSZ_INS_CLGIJ,
	SYSZ_INS_CLGRJ,
	SYSZ_INS_CLIJ,
	SYSZ_INS_CLRJ,
	SYSZ_INS_CRJ,
	SYSZ_INS_BER,
	SYSZ_INS_JE,
	SYSZ_INS_JGE,
	SYSZ_INS_LOCE,
	SYSZ_INS_LOCGE,
	SYSZ_INS_LOCGRE,
	SYSZ_INS_LOCRE,
	SYSZ_INS_STOCE,
	SYSZ_INS_STOCGE,
	SYSZ_INS_BHR,
	SYSZ_INS_BHER,
	SYSZ_INS_JHE,
	SYSZ_INS_JGHE,
	SYSZ_INS_LOCHE,
	SYSZ_INS_LOCGHE,
	SYSZ_INS_LOCGRHE,
	SYSZ_INS_LOCRHE,
	SYSZ_INS_STOCHE,
	SYSZ_INS_STOCGHE,
	SYSZ_INS_JH,
	SYSZ_INS_JGH,
	SYSZ_INS_LOCH,
	SYSZ_INS_LOCGH,
	SYSZ_INS_LOCGRH,
	SYSZ_INS_LOCRH,
	SYSZ_INS_STOCH,
	SYSZ_INS_STOCGH,
	SYSZ_INS_CGIJNLH,
	SYSZ_INS_CGRJNLH,
	SYSZ_INS_CIJNLH,
	SYSZ_INS_CLGIJNLH,
	SYSZ_INS_CLGRJNLH,
	SYSZ_INS_CLIJNLH,
	SYSZ_INS_CLRJNLH,
	SYSZ_INS_CRJNLH,
	SYSZ_INS_CGIJE,
	SYSZ_INS_CGRJE,
	SYSZ_INS_CIJE,
	SYSZ_INS_CLGIJE,
	SYSZ_INS_CLGRJE,
	SYSZ_INS_CLIJE,
	SYSZ_INS_CLRJE,
	SYSZ_INS_CRJE,
	SYSZ_INS_CGIJNLE,
	SYSZ_INS_CGRJNLE,
	SYSZ_INS_CIJNLE,
	SYSZ_INS_CLGIJNLE,
	SYSZ_INS_CLGRJNLE,
	SYSZ_INS_CLIJNLE,
	SYSZ_INS_CLRJNLE,
	SYSZ_INS_CRJNLE,
	SYSZ_INS_CGIJH,
	SYSZ_INS_CGRJH,
	SYSZ_INS_CIJH,
	SYSZ_INS_CLGIJH,
	SYSZ_INS_CLGRJH,
	SYSZ_INS_CLIJH,
	SYSZ_INS_CLRJH,
	SYSZ_INS_CRJH,
	SYSZ_INS_CGIJNL,
	SYSZ_INS_CGRJNL,
	SYSZ_INS_CIJNL,
	SYSZ_INS_CLGIJNL,
	SYSZ_INS_CLGRJNL,
	SYSZ_INS_CLIJNL,
	SYSZ_INS_CLRJNL,
	SYSZ_INS_CRJNL,
	SYSZ_INS_CGIJHE,
	SYSZ_INS_CGRJHE,
	SYSZ_INS_CIJHE,
	SYSZ_INS_CLGIJHE,
	SYSZ_INS_CLGRJHE,
	SYSZ_INS_CLIJHE,
	SYSZ_INS_CLRJHE,
	SYSZ_INS_CRJHE,
	SYSZ_INS_CGIJNHE,
	SYSZ_INS_CGRJNHE,
	SYSZ_INS_CIJNHE,
	SYSZ_INS_CLGIJNHE,
	SYSZ_INS_CLGRJNHE,
	SYSZ_INS_CLIJNHE,
	SYSZ_INS_CLRJNHE,
	SYSZ_INS_CRJNHE,
	SYSZ_INS_CGIJL,
	SYSZ_INS_CGRJL,
	SYSZ_INS_CIJL,
	SYSZ_INS_CLGIJL,
	SYSZ_INS_CLGRJL,
	SYSZ_INS_CLIJL,
	SYSZ_INS_CLRJL,
	SYSZ_INS_CRJL,
	SYSZ_INS_CGIJNH,
	SYSZ_INS_CGRJNH,
	SYSZ_INS_CIJNH,
	SYSZ_INS_CLGIJNH,
	SYSZ_INS_CLGRJNH,
	SYSZ_INS_CLIJNH,
	SYSZ_INS_CLRJNH,
	SYSZ_INS_CRJNH,
	SYSZ_INS_CGIJLE,
	SYSZ_INS_CGRJLE,
	SYSZ_INS_CIJLE,
	SYSZ_INS_CLGIJLE,
	SYSZ_INS_CLGRJLE,
	SYSZ_INS_CLIJLE,
	SYSZ_INS_CLRJLE,
	SYSZ_INS_CRJLE,
	SYSZ_INS_CGIJNE,
	SYSZ_INS_CGRJNE,
	SYSZ_INS_CIJNE,
	SYSZ_INS_CLGIJNE,
	SYSZ_INS_CLGRJNE,
	SYSZ_INS_CLIJNE,
	SYSZ_INS_CLRJNE,
	SYSZ_INS_CRJNE,
	SYSZ_INS_CGIJLH,
	SYSZ_INS_CGRJLH,
	SYSZ_INS_CIJLH,
	SYSZ_INS_CLGIJLH,
	SYSZ_INS_CLGRJLH,
	SYSZ_INS_CLIJLH,
	SYSZ_INS_CLRJLH,
	SYSZ_INS_CRJLH,
	SYSZ_INS_BLR,
	SYSZ_INS_BLER,
	SYSZ_INS_JLE,
	SYSZ_INS_JGLE,
	SYSZ_INS_LOCLE,
	SYSZ_INS_LOCGLE,
	SYSZ_INS_LOCGRLE,
	SYSZ_INS_LOCRLE,
	SYSZ_INS_STOCLE,
	SYSZ_INS_STOCGLE,
	SYSZ_INS_BLHR,
	SYSZ_INS_JLH,
	SYSZ_INS_JGLH,
	SYSZ_INS_LOCLH,
	SYSZ_INS_LOCGLH,
	SYSZ_INS_LOCGRLH,
	SYSZ_INS_LOCRLH,
	SYSZ_INS_STOCLH,
	SYSZ_INS_STOCGLH,
	SYSZ_INS_JL,
	SYSZ_INS_JGL,
	SYSZ_INS_LOCL,
	SYSZ_INS_LOCGL,
	SYSZ_INS_LOCGRL,
	SYSZ_INS_LOCRL,
	SYSZ_INS_LOC,
	SYSZ_INS_LOCG,
	SYSZ_INS_LOCGR,
	SYSZ_INS_LOCR,
	SYSZ_INS_STOCL,
	SYSZ_INS_STOCGL,
	SYSZ_INS_BNER,
	SYSZ_INS_JNE,
	SYSZ_INS_JGNE,
	SYSZ_INS_LOCNE,
	SYSZ_INS_LOCGNE,
	SYSZ_INS_LOCGRNE,
	SYSZ_INS_LOCRNE,
	SYSZ_INS_STOCNE,
	SYSZ_INS_STOCGNE,
	SYSZ_INS_BNHR,
	SYSZ_INS_BNHER,
	SYSZ_INS_JNHE,
	SYSZ_INS_JGNHE,
	SYSZ_INS_LOCNHE,
	SYSZ_INS_LOCGNHE,
	SYSZ_INS_LOCGRNHE,
	SYSZ_INS_LOCRNHE,
	SYSZ_INS_STOCNHE,
	SYSZ_INS_STOCGNHE,
	SYSZ_INS_JNH,
	SYSZ_INS_JGNH,
	SYSZ_INS_LOCNH,
	SYSZ_INS_LOCGNH,
	SYSZ_INS_LOCGRNH,
	SYSZ_INS_LOCRNH,
	SYSZ_INS_STOCNH,
	SYSZ_INS_STOCGNH,
	SYSZ_INS_BNLR,
	SYSZ_INS_BNLER,
	SYSZ_INS_JNLE,
	SYSZ_INS_JGNLE,
	SYSZ_INS_LOCNLE,
	SYSZ_INS_LOCGNLE,
	SYSZ_INS_LOCGRNLE,
	SYSZ_INS_LOCRNLE,
	SYSZ_INS_STOCNLE,
	SYSZ_INS_STOCGNLE,
	SYSZ_INS_BNLHR,
	SYSZ_INS_JNLH,
	SYSZ_INS_JGNLH,
	SYSZ_INS_LOCNLH,
	SYSZ_INS_LOCGNLH,
	SYSZ_INS_LOCGRNLH,
	SYSZ_INS_LOCRNLH,
	SYSZ_INS_STOCNLH,
	SYSZ_INS_STOCGNLH,
	SYSZ_INS_JNL,
	SYSZ_INS_JGNL,
	SYSZ_INS_LOCNL,
	SYSZ_INS_LOCGNL,
	SYSZ_INS_LOCGRNL,
	SYSZ_INS_LOCRNL,
	SYSZ_INS_STOCNL,
	SYSZ_INS_STOCGNL,
	SYSZ_INS_BNOR,
	SYSZ_INS_JNO,
	SYSZ_INS_JGNO,
	SYSZ_INS_LOCNO,
	SYSZ_INS_LOCGNO,
	SYSZ_INS_LOCGRNO,
	SYSZ_INS_LOCRNO,
	SYSZ_INS_STOCNO,
	SYSZ_INS_STOCGNO,
	SYSZ_INS_BOR,
	SYSZ_INS_JO,
	SYSZ_INS_JGO,
	SYSZ_INS_LOCO,
	SYSZ_INS_LOCGO,
	SYSZ_INS_LOCGRO,
	SYSZ_INS_LOCRO,
	SYSZ_INS_STOCO,
	SYSZ_INS_STOCGO,
	SYSZ_INS_STOC,
	SYSZ_INS_STOCG,
	SYSZ_INS_BASR,
	SYSZ_INS_BR,
	SYSZ_INS_BRAS,
	SYSZ_INS_BRASL,
	SYSZ_INS_J,
	SYSZ_INS_JG,
	SYSZ_INS_BRCT,
	SYSZ_INS_BRCTG,
	SYSZ_INS_C,
	SYSZ_INS_CDB,
	SYSZ_INS_CDBR,
	SYSZ_INS_CDFBR,
	SYSZ_INS_CDGBR,
	SYSZ_INS_CDLFBR,
	SYSZ_INS_CDLGBR,
	SYSZ_INS_CEB,
	SYSZ_INS_CEBR,
	SYSZ_INS_CEFBR,
	SYSZ_INS_CEGBR,
	SYSZ_INS_CELFBR,
	SYSZ_INS_CELGBR,
	SYSZ_INS_CFDBR,
	SYSZ_INS_CFEBR,
	SYSZ_INS_CFI,
	SYSZ_INS_CFXBR,
	SYSZ_INS_CG,
	SYSZ_INS_CGDBR,
	SYSZ_INS_CGEBR,
	SYSZ_INS_CGF,
	SYSZ_INS_CGFI,
	SYSZ_INS_CGFR,
	SYSZ_INS_CGFRL,
	SYSZ_INS_CGH,
	SYSZ_INS_CGHI,
	SYSZ_INS_CGHRL,
	SYSZ_INS_CGHSI,
	SYSZ_INS_CGR,
	SYSZ_INS_CGRL,
	SYSZ_INS_CGXBR,
	SYSZ_INS_CH,
	SYSZ_INS_CHF,
	SYSZ_INS_CHHSI,
	SYSZ_INS_CHI,
	SYSZ_INS_CHRL,
	SYSZ_INS_CHSI,
	SYSZ_INS_CHY,
	SYSZ_INS_CIH,
	SYSZ_INS_CL,
	SYSZ_INS_CLC,
	SYSZ_INS_CLFDBR,
	SYSZ_INS_CLFEBR,
	SYSZ_INS_CLFHSI,
	SYSZ_INS_CLFI,
	SYSZ_INS_CLFXBR,
	SYSZ_INS_CLG,
	SYSZ_INS_CLGDBR,
	SYSZ_INS_CLGEBR,
	SYSZ_INS_CLGF,
	SYSZ_INS_CLGFI,
	SYSZ_INS_CLGFR,
	SYSZ_INS_CLGFRL,
	SYSZ_INS_CLGHRL,
	SYSZ_INS_CLGHSI,
	SYSZ_INS_CLGR,
	SYSZ_INS_CLGRL,
	SYSZ_INS_CLGXBR,
	SYSZ_INS_CLHF,
	SYSZ_INS_CLHHSI,
	SYSZ_INS_CLHRL,
	SYSZ_INS_CLI,
	SYSZ_INS_CLIH,
	SYSZ_INS_CLIY,
	SYSZ_INS_CLR,
	SYSZ_INS_CLRL,
	SYSZ_INS_CLST,
	SYSZ_INS_CLY,
	SYSZ_INS_CPSDR,
	SYSZ_INS_CR,
	SYSZ_INS_CRL,
	SYSZ_INS_CS,
	SYSZ_INS_CSG,
	SYSZ_INS_CSY,
	SYSZ_INS_CXBR,
	SYSZ_INS_CXFBR,
	SYSZ_INS_CXGBR,
	SYSZ_INS_CXLFBR,
	SYSZ_INS_CXLGBR,
	SYSZ_INS_CY,
	SYSZ_INS_DDB,
	SYSZ_INS_DDBR,
	SYSZ_INS_DEB,
	SYSZ_INS_DEBR,
	SYSZ_INS_DL,
	SYSZ_INS_DLG,
	SYSZ_INS_DLGR,
	SYSZ_INS_DLR,
	SYSZ_INS_DSG,
	SYSZ_INS_DSGF,
	SYSZ_INS_DSGFR,
	SYSZ_INS_DSGR,
	SYSZ_INS_DXBR,
	SYSZ_INS_EAR,
	SYSZ_INS_FIDBR,
	SYSZ_INS_FIDBRA,
	SYSZ_INS_FIEBR,
	SYSZ_INS_FIEBRA,
	SYSZ_INS_FIXBR,
	SYSZ_INS_FIXBRA,
	SYSZ_INS_FLOGR,
	SYSZ_INS_IC,
	SYSZ_INS_ICY,
	SYSZ_INS_IIHF,
	SYSZ_INS_IIHH,
	SYSZ_INS_IIHL,
	SYSZ_INS_IILF,
	SYSZ_INS_IILH,
	SYSZ_INS_IILL,
	SYSZ_INS_IPM,
	SYSZ_INS_L,
	SYSZ_INS_LA,
	SYSZ_INS_LAA,
	SYSZ_INS_LAAG,
	SYSZ_INS_LAAL,
	SYSZ_INS_LAALG,
	SYSZ_INS_LAN,
	SYSZ_INS_LANG,
	SYSZ_INS_LAO,
	SYSZ_INS_LAOG,
	SYSZ_INS_LARL,
	SYSZ_INS_LAX,
	SYSZ_INS_LAXG,
	SYSZ_INS_LAY,
	SYSZ_INS_LB,
	SYSZ_INS_LBH,
	SYSZ_INS_LBR,
	SYSZ_INS_LCDBR,
	SYSZ_INS_LCEBR,
	SYSZ_INS_LCGFR,
	SYSZ_INS_LCGR,
	SYSZ_INS_LCR,
	SYSZ_INS_LCXBR,
	SYSZ_INS_LD,
	SYSZ_INS_LDEB,
	SYSZ_INS_LDEBR,
	SYSZ_INS_LDGR,
	SYSZ_INS_LDR,
	SYSZ_INS_LDXBR,
	SYSZ_INS_LDXBRA,
	SYSZ_INS_LDY,
	SYSZ_INS_LE,
	SYSZ_INS_LEDBR,
	SYSZ_INS_LEDBRA,
	SYSZ_INS_LER,
	SYSZ_INS_LEXBR,
	SYSZ_INS_LEXBRA,
	SYSZ_INS_LEY,
	SYSZ_INS_LFH,
	SYSZ_INS_LG,
	SYSZ_INS_LGB,
	SYSZ_INS_LGBR,
	SYSZ_INS_LGDR,
	SYSZ_INS_LGF,
	SYSZ_INS_LGFI,
	SYSZ_INS_LGFR,
	SYSZ_INS_LGFRL,
	SYSZ_INS_LGH,
	SYSZ_INS_LGHI,
	SYSZ_INS_LGHR,
	SYSZ_INS_LGHRL,
	SYSZ_INS_LGR,
	SYSZ_INS_LGRL,
	SYSZ_INS_LH,
	SYSZ_INS_LHH,
	SYSZ_INS_LHI,
	SYSZ_INS_LHR,
	SYSZ_INS_LHRL,
	SYSZ_INS_LHY,
	SYSZ_INS_LLC,
	SYSZ_INS_LLCH,
	SYSZ_INS_LLCR,
	SYSZ_INS_LLGC,
	SYSZ_INS_LLGCR,
	SYSZ_INS_LLGF,
	SYSZ_INS_LLGFR,
	SYSZ_INS_LLGFRL,
	SYSZ_INS_LLGH,
	SYSZ_INS_LLGHR,
	SYSZ_INS_LLGHRL,
	SYSZ_INS_LLH,
	SYSZ_INS_LLHH,
	SYSZ_INS_LLHR,
	SYSZ_INS_LLHRL,
	SYSZ_INS_LLIHF,
	SYSZ_INS_LLIHH,
	SYSZ_INS_LLIHL,
	SYSZ_INS_LLILF,
	SYSZ_INS_LLILH,
	SYSZ_INS_LLILL,
	SYSZ_INS_LMG,
	SYSZ_INS_LNDBR,
	SYSZ_INS_LNEBR,
	SYSZ_INS_LNGFR,
	SYSZ_INS_LNGR,
	SYSZ_INS_LNR,
	SYSZ_INS_LNXBR,
	SYSZ_INS_LPDBR,
	SYSZ_INS_LPEBR,
	SYSZ_INS_LPGFR,
	SYSZ_INS_LPGR,
	SYSZ_INS_LPR,
	SYSZ_INS_LPXBR,
	SYSZ_INS_LR,
	SYSZ_INS_LRL,
	SYSZ_INS_LRV,
	SYSZ_INS_LRVG,
	SYSZ_INS_LRVGR,
	SYSZ_INS_LRVR,
	SYSZ_INS_LT,
	SYSZ_INS_LTDBR,
	SYSZ_INS_LTEBR,
	SYSZ_INS_LTG,
	SYSZ_INS_LTGF,
	SYSZ_INS_LTGFR,
	SYSZ_INS_LTGR,
	SYSZ_INS_LTR,
	SYSZ_INS_LTXBR,
	SYSZ_INS_LXDB,
	SYSZ_INS_LXDBR,
	SYSZ_INS_LXEB,
	SYSZ_INS_LXEBR,
	SYSZ_INS_LXR,
	SYSZ_INS_LY,
	SYSZ_INS_LZDR,
	SYSZ_INS_LZER,
	SYSZ_INS_LZXR,
	SYSZ_INS_MADB,
	SYSZ_INS_MADBR,
	SYSZ_INS_MAEB,
	SYSZ_INS_MAEBR,
	SYSZ_INS_MDB,
	SYSZ_INS_MDBR,
	SYSZ_INS_MDEB,
	SYSZ_INS_MDEBR,
	SYSZ_INS_MEEB,
	SYSZ_INS_MEEBR,
	SYSZ_INS_MGHI,
	SYSZ_INS_MH,
	SYSZ_INS_MHI,
	SYSZ_INS_MHY,
	SYSZ_INS_MLG,
	SYSZ_INS_MLGR,
	SYSZ_INS_MS,
	SYSZ_INS_MSDB,
	SYSZ_INS_MSDBR,
	SYSZ_INS_MSEB,
	SYSZ_INS_MSEBR,
	SYSZ_INS_MSFI,
	SYSZ_INS_MSG,
	SYSZ_INS_MSGF,
	SYSZ_INS_MSGFI,
	SYSZ_INS_MSGFR,
	SYSZ_INS_MSGR,
	SYSZ_INS_MSR,
	SYSZ_INS_MSY,
	SYSZ_INS_MVC,
	SYSZ_INS_MVGHI,
	SYSZ_INS_MVHHI,
	SYSZ_INS_MVHI,
	SYSZ_INS_MVI,
	SYSZ_INS_MVIY,
	SYSZ_INS_MVST,
	SYSZ_INS_MXBR,
	SYSZ_INS_MXDB,
	SYSZ_INS_MXDBR,
	SYSZ_INS_N,
	SYSZ_INS_NC,
	SYSZ_INS_NG,
	SYSZ_INS_NGR,
	SYSZ_INS_NGRK,
	SYSZ_INS_NI,
	SYSZ_INS_NIHF,
	SYSZ_INS_NIHH,
	SYSZ_INS_NIHL,
	SYSZ_INS_NILF,
	SYSZ_INS_NILH,
	SYSZ_INS_NILL,
	SYSZ_INS_NIY,
	SYSZ_INS_NR,
	SYSZ_INS_NRK,
	SYSZ_INS_NY,
	SYSZ_INS_O,
	SYSZ_INS_OC,
	SYSZ_INS_OG,
	SYSZ_INS_OGR,
	SYSZ_INS_OGRK,
	SYSZ_INS_OI,
	SYSZ_INS_OIHF,
	SYSZ_INS_OIHH,
	SYSZ_INS_OIHL,
	SYSZ_INS_OILF,
	SYSZ_INS_OILH,
	SYSZ_INS_OILL,
	SYSZ_INS_OIY,
	SYSZ_INS_OR,
	SYSZ_INS_ORK,
	SYSZ_INS_OY,
	SYSZ_INS_PFD,
	SYSZ_INS_PFDRL,
	SYSZ_INS_RISBG,
	SYSZ_INS_RISBHG,
	SYSZ_INS_RISBLG,
	SYSZ_INS_RLL,
	SYSZ_INS_RLLG,
	SYSZ_INS_RNSBG,
	SYSZ_INS_ROSBG,
	SYSZ_INS_RXSBG,
	SYSZ_INS_S,
	SYSZ_INS_SDB,
	SYSZ_INS_SDBR,
	SYSZ_INS_SEB,
	SYSZ_INS_SEBR,
	SYSZ_INS_SG,
	SYSZ_INS_SGF,
	SYSZ_INS_SGFR,
	SYSZ_INS_SGR,
	SYSZ_INS_SGRK,
	SYSZ_INS_SH,
	SYSZ_INS_SHY,
	SYSZ_INS_SL,
	SYSZ_INS_SLB,
	SYSZ_INS_SLBG,
	SYSZ_INS_SLBR,
	SYSZ_INS_SLFI,
	SYSZ_INS_SLG,
	SYSZ_INS_SLBGR,
	SYSZ_INS_SLGF,
	SYSZ_INS_SLGFI,
	SYSZ_INS_SLGFR,
	SYSZ_INS_SLGR,
	SYSZ_INS_SLGRK,
	SYSZ_INS_SLL,
	SYSZ_INS_SLLG,
	SYSZ_INS_SLLK,
	SYSZ_INS_SLR,
	SYSZ_INS_SLRK,
	SYSZ_INS_SLY,
	SYSZ_INS_SQDB,
	SYSZ_INS_SQDBR,
	SYSZ_INS_SQEB,
	SYSZ_INS_SQEBR,
	SYSZ_INS_SQXBR,
	SYSZ_INS_SR,
	SYSZ_INS_SRA,
	SYSZ_INS_SRAG,
	SYSZ_INS_SRAK,
	SYSZ_INS_SRK,
	SYSZ_INS_SRL,
	SYSZ_INS_SRLG,
	SYSZ_INS_SRLK,
	SYSZ_INS_SRST,
	SYSZ_INS_ST,
	SYSZ_INS_STC,
	SYSZ_INS_STCH,
	SYSZ_INS_STCY,
	SYSZ_INS_STD,
	SYSZ_INS_STDY,
	SYSZ_INS_STE,
	SYSZ_INS_STEY,
	SYSZ_INS_STFH,
	SYSZ_INS_STG,
	SYSZ_INS_STGRL,
	SYSZ_INS_STH,
	SYSZ_INS_STHH,
	SYSZ_INS_STHRL,
	SYSZ_INS_STHY,
	SYSZ_INS_STMG,
	SYSZ_INS_STRL,
	SYSZ_INS_STRV,
	SYSZ_INS_STRVG,
	SYSZ_INS_STY,
	SYSZ_INS_SXBR,
	SYSZ_INS_SY,
	SYSZ_INS_TM,
	SYSZ_INS_TMHH,
	SYSZ_INS_TMHL,
	SYSZ_INS_TMLH,
	SYSZ_INS_TMLL,
	SYSZ_INS_TMY,
	SYSZ_INS_X,
	SYSZ_INS_XC,
	SYSZ_INS_XG,
	SYSZ_INS_XGR,
	SYSZ_INS_XGRK,
	SYSZ_INS_XI,
	SYSZ_INS_XIHF,
	SYSZ_INS_XILF,
	SYSZ_INS_XIY,
	SYSZ_INS_XR,
	SYSZ_INS_XRK,
	SYSZ_INS_XY,
	SYSZ_INS_AD,
	SYSZ_INS_ADR,
	SYSZ_INS_ADTR,
	SYSZ_INS_ADTRA,
	SYSZ_INS_AE,
	SYSZ_INS_AER,
	SYSZ_INS_AGH,
	SYSZ_INS_AHHHR,
	SYSZ_INS_AHHLR,
	SYSZ_INS_ALGSI,
	SYSZ_INS_ALHHHR,
	SYSZ_INS_ALHHLR,
	SYSZ_INS_ALSI,
	SYSZ_INS_ALSIH,
	SYSZ_INS_ALSIHN,
	SYSZ_INS_AP,
	SYSZ_INS_AU,
	SYSZ_INS_AUR,
	SYSZ_INS_AW,
	SYSZ_INS_AWR,
	SYSZ_INS_AXR,
	SYSZ_INS_AXTR,
	SYSZ_INS_AXTRA,
	SYSZ_INS_B,
	SYSZ_INS_BAKR,
	SYSZ_INS_BAL,
	SYSZ_INS_BALR,
	SYSZ_INS_BAS,
	SYSZ_INS_BASSM,
	SYSZ_INS_BC,
	SYSZ_INS_BCT,
	SYSZ_INS_BCTG,
	SYSZ_INS_BCTGR,
	SYSZ_INS_BCTR,
	SYSZ_INS_BE,
	SYSZ_INS_BH,
	SYSZ_INS_BHE,
	SYSZ_INS_BI,
	SYSZ_INS_BIC,
	SYSZ_INS_BIE,
	SYSZ_INS_BIH,
	SYSZ_INS_BIHE,
	SYSZ_INS_BIL,
	SYSZ_INS_BILE,
	SYSZ_INS_BILH,
	SYSZ_INS_BIM,
	SYSZ_INS_BINE,
	SYSZ_INS_BINH,
	SYSZ_INS_BINHE,
	SYSZ_INS_BINL,
	SYSZ_INS_BINLE,
	SYSZ_INS_BINLH,
	SYSZ_INS_BINM,
	SYSZ_INS_BINO,
	SYSZ_INS_BINP,
	SYSZ_INS_BINZ,
	SYSZ_INS_BIO,
	SYSZ_INS_BIP,
	SYSZ_INS_BIZ,
	SYSZ_INS_BL,
	SYSZ_INS_BLE,
	SYSZ_INS_BLH,
	SYSZ_INS_BM,
	SYSZ_INS_BMR,
	SYSZ_INS_BNE,
	SYSZ_INS_BNH,
	SYSZ_INS_BNHE,
	SYSZ_INS_BNL,
	SYSZ_INS_BNLE,
	SYSZ_INS_BNLH,
	SYSZ_INS_BNM,
	SYSZ_INS_BNMR,
	SYSZ_INS_BNO,
	SYSZ_INS_BNP,
	SYSZ_INS_BNPR,
	SYSZ_INS_BNZ,
	SYSZ_INS_BNZR,
	SYSZ_INS_BO,
	SYSZ_INS_BP,
	SYSZ_INS_BPP,
	SYSZ_INS_BPR,
	SYSZ_INS_BPRP,
	SYSZ_INS_BRCTH,
	SYSZ_INS_BRXH,
	SYSZ_INS_BRXHG,
	SYSZ_INS_BRXLE,
	SYSZ_INS_BRXLG,
	SYSZ_INS_BSA,
	SYSZ_INS_BSG,
	SYSZ_INS_BSM,
	SYSZ_INS_BXH,
	SYSZ_INS_BXHG,
	SYSZ_INS_BXLE,
	SYSZ_INS_BXLEG,
	SYSZ_INS_BZ,
	SYSZ_INS_BZR,
	SYSZ_INS_CD,
	SYSZ_INS_CDFBRA,
	SYSZ_INS_CDFR,
	SYSZ_INS_CDFTR,
	SYSZ_INS_CDGBRA,
	SYSZ_INS_CDGR,
	SYSZ_INS_CDGTR,
	SYSZ_INS_CDGTRA,
	SYSZ_INS_CDLFTR,
	SYSZ_INS_CDLGTR,
	SYSZ_INS_CDPT,
	SYSZ_INS_CDR,
	SYSZ_INS_CDS,
	SYSZ_INS_CDSG,
	SYSZ_INS_CDSTR,
	SYSZ_INS_CDSY,
	SYSZ_INS_CDTR,
	SYSZ_INS_CDUTR,
	SYSZ_INS_CDZT,
	SYSZ_INS_CE,
	SYSZ_INS_CEDTR,
	SYSZ_INS_CEFBRA,
	SYSZ_INS_CEFR,
	SYSZ_INS_CEGBRA,
	SYSZ_INS_CEGR,
	SYSZ_INS_CER,
	SYSZ_INS_CEXTR,
	SYSZ_INS_CFC,
	SYSZ_INS_CFDBRA,
	SYSZ_INS_CFDR,
	SYSZ_INS_CFDTR,
	SYSZ_INS_CFEBRA,
	SYSZ_INS_CFER,
	SYSZ_INS_CFXBRA,
	SYSZ_INS_CFXR,
	SYSZ_INS_CFXTR,
	SYSZ_INS_CGDBRA,
	SYSZ_INS_CGDR,
	SYSZ_INS_CGDTR,
	SYSZ_INS_CGDTRA,
	SYSZ_INS_CGEBRA,
	SYSZ_INS_CGER,
	SYSZ_INS_CGIB,
	SYSZ_INS_CGIBE,
	SYSZ_INS_CGIBH,
	SYSZ_INS_CGIBHE,
	SYSZ_INS_CGIBL,
	SYSZ_INS_CGIBLE,
	SYSZ_INS_CGIBLH,
	SYSZ_INS_CGIBNE,
	SYSZ_INS_CGIBNH,
	SYSZ_INS_CGIBNHE,
	SYSZ_INS_CGIBNL,
	SYSZ_INS_CGIBNLE,
	SYSZ_INS_CGIBNLH,
	SYSZ_INS_CGIT,
	SYSZ_INS_CGITE,
	SYSZ_INS_CGITH,
	SYSZ_INS_CGITHE,
	SYSZ_INS_CGITL,
	SYSZ_INS_CGITLE,
	SYSZ_INS_CGITLH,
	SYSZ_INS_CGITNE,
	SYSZ_INS_CGITNH,
	SYSZ_INS_CGITNHE,
	SYSZ_INS_CGITNL,
	SYSZ_INS_CGITNLE,
	SYSZ_INS_CGITNLH,
	SYSZ_INS_CGRB,
	SYSZ_INS_CGRBE,
	SYSZ_INS_CGRBH,
	SYSZ_INS_CGRBHE,
	SYSZ_INS_CGRBL,
	SYSZ_INS_CGRBLE,
	SYSZ_INS_CGRBLH,
	SYSZ_INS_CGRBNE,
	SYSZ_INS_CGRBNH,
	SYSZ_INS_CGRBNHE,
	SYSZ_INS_CGRBNL,
	SYSZ_INS_CGRBNLE,
	SYSZ_INS_CGRBNLH,
	SYSZ_INS_CGRT,
	SYSZ_INS_CGRTE,
	SYSZ_INS_CGRTH,
	SYSZ_INS_CGRTHE,
	SYSZ_INS_CGRTL,
	SYSZ_INS_CGRTLE,
	SYSZ_INS_CGRTLH,
	SYSZ_INS_CGRTNE,
	SYSZ_INS_CGRTNH,
	SYSZ_INS_CGRTNHE,
	SYSZ_INS_CGRTNL,
	SYSZ_INS_CGRTNLE,
	SYSZ_INS_CGRTNLH,
	SYSZ_INS_CGXBRA,
	SYSZ_INS_CGXR,
	SYSZ_INS_CGXTR,
	SYSZ_INS_CGXTRA,
	SYSZ_INS_CHHR,
	SYSZ_INS_CHLR,
	SYSZ_INS_CIB,
	SYSZ_INS_CIBE,
	SYSZ_INS_CIBH,
	SYSZ_INS_CIBHE,
	SYSZ_INS_CIBL,
	SYSZ_INS_CIBLE,
	SYSZ_INS_CIBLH,
	SYSZ_INS_CIBNE,
	SYSZ_INS_CIBNH,
	SYSZ_INS_CIBNHE,
	SYSZ_INS_CIBNL,
	SYSZ_INS_CIBNLE,
	SYSZ_INS_CIBNLH,
	SYSZ_INS_CIT,
	SYSZ_INS_CITE,
	SYSZ_INS_CITH,
	SYSZ_INS_CITHE,
	SYSZ_INS_CITL,
	SYSZ_INS_CITLE,
	SYSZ_INS_CITLH,
	SYSZ_INS_CITNE,
	SYSZ_INS_CITNH,
	SYSZ_INS_CITNHE,
	SYSZ_INS_CITNL,
	SYSZ_INS_CITNLE,
	SYSZ_INS_CITNLH,
	SYSZ_INS_CKSM,
	SYSZ_INS_CLCL,
	SYSZ_INS_CLCLE,
	SYSZ_INS_CLCLU,
	SYSZ_INS_CLFDTR,
	SYSZ_INS_CLFIT,
	SYSZ_INS_CLFITE,
	SYSZ_INS_CLFITH,
	SYSZ_INS_CLFITHE,
	SYSZ_INS_CLFITL,
	SYSZ_INS_CLFITLE,
	SYSZ_INS_CLFITLH,
	SYSZ_INS_CLFITNE,
	SYSZ_INS_CLFITNH,
	SYSZ_INS_CLFITNHE,
	SYSZ_INS_CLFITNL,
	SYSZ_INS_CLFITNLE,
	SYSZ_INS_CLFITNLH,
	SYSZ_INS_CLFXTR,
	SYSZ_INS_CLGDTR,
	SYSZ_INS_CLGIB,
	SYSZ_INS_CLGIBE,
	SYSZ_INS_CLGIBH,
	SYSZ_INS_CLGIBHE,
	SYSZ_INS_CLGIBL,
	SYSZ_INS_CLGIBLE,
	SYSZ_INS_CLGIBLH,
	SYSZ_INS_CLGIBNE,
	SYSZ_INS_CLGIBNH,
	SYSZ_INS_CLGIBNHE,
	SYSZ_INS_CLGIBNL,
	SYSZ_INS_CLGIBNLE,
	SYSZ_INS_CLGIBNLH,
	SYSZ_INS_CLGIT,
	SYSZ_INS_CLGITE,
	SYSZ_INS_CLGITH,
	SYSZ_INS_CLGITHE,
	SYSZ_INS_CLGITL,
	SYSZ_INS_CLGITLE,
	SYSZ_INS_CLGITLH,
	SYSZ_INS_CLGITNE,
	SYSZ_INS_CLGITNH,
	SYSZ_INS_CLGITNHE,
	SYSZ_INS_CLGITNL,
	SYSZ_INS_CLGITNLE,
	SYSZ_INS_CLGITNLH,
	SYSZ_INS_CLGRB,
	SYSZ_INS_CLGRBE,
	SYSZ_INS_CLGRBH,
	SYSZ_INS_CLGRBHE,
	SYSZ_INS_CLGRBL,
	SYSZ_INS_CLGRBLE,
	SYSZ_INS_CLGRBLH,
	SYSZ_INS_CLGRBNE,
	SYSZ_INS_CLGRBNH,
	SYSZ_INS_CLGRBNHE,
	SYSZ_INS_CLGRBNL,
	SYSZ_INS_CLGRBNLE,
	SYSZ_INS_CLGRBNLH,
	SYSZ_INS_CLGRT,
	SYSZ_INS_CLGRTE,
	SYSZ_INS_CLGRTH,
	SYSZ_INS_CLGRTHE,
	SYSZ_INS_CLGRTL,
	SYSZ_INS_CLGRTLE,
	SYSZ_INS_CLGRTLH,
	SYSZ_INS_CLGRTNE,
	SYSZ_INS_CLGRTNH,
	SYSZ_INS_CLGRTNHE,
	SYSZ_INS_CLGRTNL,
	SYSZ_INS_CLGRTNLE,
	SYSZ_INS_CLGRTNLH,
	SYSZ_INS_CLGT,
	SYSZ_INS_CLGTE,
	SYSZ_INS_CLGTH,
	SYSZ_INS_CLGTHE,
	SYSZ_INS_CLGTL,
	SYSZ_INS_CLGTLE,
	SYSZ_INS_CLGTLH,
	SYSZ_INS_CLGTNE,
	SYSZ_INS_CLGTNH,
	SYSZ_INS_CLGTNHE,
	SYSZ_INS_CLGTNL,
	SYSZ_INS_CLGTNLE,
	SYSZ_INS_CLGTNLH,
	SYSZ_INS_CLGXTR,
	SYSZ_INS_CLHHR,
	SYSZ_INS_CLHLR,
	SYSZ_INS_CLIB,
	SYSZ_INS_CLIBE,
	SYSZ_INS_CLIBH,
	SYSZ_INS_CLIBHE,
	SYSZ_INS_CLIBL,
	SYSZ_INS_CLIBLE,
	SYSZ_INS_CLIBLH,
	SYSZ_INS_CLIBNE,
	SYSZ_INS_CLIBNH,
	SYSZ_INS_CLIBNHE,
	SYSZ_INS_CLIBNL,
	SYSZ_INS_CLIBNLE,
	SYSZ_INS_CLIBNLH,
	SYSZ_INS_CLM,
	SYSZ_INS_CLMH,
	SYSZ_INS_CLMY,
	SYSZ_INS_CLRB,
	SYSZ_INS_CLRBE,
	SYSZ_INS_CLRBH,
	SYSZ_INS_CLRBHE,
	SYSZ_INS_CLRBL,
	SYSZ_INS_CLRBLE,
	SYSZ_INS_CLRBLH,
	SYSZ_INS_CLRBNE,
	SYSZ_INS_CLRBNH,
	SYSZ_INS_CLRBNHE,
	SYSZ_INS_CLRBNL,
	SYSZ_INS_CLRBNLE,
	SYSZ_INS_CLRBNLH,
	SYSZ_INS_CLRT,
	SYSZ_INS_CLRTE,
	SYSZ_INS_CLRTH,
	SYSZ_INS_CLRTHE,
	SYSZ_INS_CLRTL,
	SYSZ_INS_CLRTLE,
	SYSZ_INS_CLRTLH,
	SYSZ_INS_CLRTNE,
	SYSZ_INS_CLRTNH,
	SYSZ_INS_CLRTNHE,
	SYSZ_INS_CLRTNL,
	SYSZ_INS_CLRTNLE,
	SYSZ_INS_CLRTNLH,
	SYSZ_INS_CLT,
	SYSZ_INS_CLTE,
	SYSZ_INS_CLTH,
	SYSZ_INS_CLTHE,
	SYSZ_INS_CLTL,
	SYSZ_INS_CLTLE,
	SYSZ_INS_CLTLH,
	SYSZ_INS_CLTNE,
	SYSZ_INS_CLTNH,
	SYSZ_INS_CLTNHE,
	SYSZ_INS_CLTNL,
	SYSZ_INS_CLTNLE,
	SYSZ_INS_CLTNLH,
	SYSZ_INS_CMPSC,
	SYSZ_INS_CP,
	SYSZ_INS_CPDT,
	SYSZ_INS_CPXT,
	SYSZ_INS_CPYA,
	SYSZ_INS_CRB,
	SYSZ_INS_CRBE,
	SYSZ_INS_CRBH,
	SYSZ_INS_CRBHE,
	SYSZ_INS_CRBL,
	SYSZ_INS_CRBLE,
	SYSZ_INS_CRBLH,
	SYSZ_INS_CRBNE,
	SYSZ_INS_CRBNH,
	SYSZ_INS_CRBNHE,
	SYSZ_INS_CRBNL,
	SYSZ_INS_CRBNLE,
	SYSZ_INS_CRBNLH,
	SYSZ_INS_CRDTE,
	SYSZ_INS_CRT,
	SYSZ_INS_CRTE,
	SYSZ_INS_CRTH,
	SYSZ_INS_CRTHE,
	SYSZ_INS_CRTL,
	SYSZ_INS_CRTLE,
	SYSZ_INS_CRTLH,
	SYSZ_INS_CRTNE,
	SYSZ_INS_CRTNH,
	SYSZ_INS_CRTNHE,
	SYSZ_INS_CRTNL,
	SYSZ_INS_CRTNLE,
	SYSZ_INS_CRTNLH,
	SYSZ_INS_CSCH,
	SYSZ_INS_CSDTR,
	SYSZ_INS_CSP,
	SYSZ_INS_CSPG,
	SYSZ_INS_CSST,
	SYSZ_INS_CSXTR,
	SYSZ_INS_CU12,
	SYSZ_INS_CU14,
	SYSZ_INS_CU21,
	SYSZ_INS_CU24,
	SYSZ_INS_CU41,
	SYSZ_INS_CU42,
	SYSZ_INS_CUDTR,
	SYSZ_INS_CUSE,
	SYSZ_INS_CUTFU,
	SYSZ_INS_CUUTF,
	SYSZ_INS_CUXTR,
	SYSZ_INS_CVB,
	SYSZ_INS_CVBG,
	SYSZ_INS_CVBY,
	SYSZ_INS_CVD,
	SYSZ_INS_CVDG,
	SYSZ_INS_CVDY,
	SYSZ_INS_CXFBRA,
	SYSZ_INS_CXFR,
	SYSZ_INS_CXFTR,
	SYSZ_INS_CXGBRA,
	SYSZ_INS_CXGR,
	SYSZ_INS_CXGTR,
	SYSZ_INS_CXGTRA,
	SYSZ_INS_CXLFTR,
	SYSZ_INS_CXLGTR,
	SYSZ_INS_CXPT,
	SYSZ_INS_CXR,
	SYSZ_INS_CXSTR,
	SYSZ_INS_CXTR,
	SYSZ_INS_CXUTR,
	SYSZ_INS_CXZT,
	SYSZ_INS_CZDT,
	SYSZ_INS_CZXT,
	SYSZ_INS_D,
	SYSZ_INS_DD,
	SYSZ_INS_DDR,
	SYSZ_INS_DDTR,
	SYSZ_INS_DDTRA,
	SYSZ_INS_DE,
	SYSZ_INS_DER,
	SYSZ_INS_DIAG,
	SYSZ_INS_DIDBR,
	SYSZ_INS_DIEBR,
	SYSZ_INS_DP,
	SYSZ_INS_DR,
	SYSZ_INS_DXR,
	SYSZ_INS_DXTR,
	SYSZ_INS_DXTRA,
	SYSZ_INS_ECAG,
	SYSZ_INS_ECCTR,
	SYSZ_INS_ECPGA,
	SYSZ_INS_ECTG,
	SYSZ_INS_ED,
	SYSZ_INS_EDMK,
	SYSZ_INS_EEDTR,
	SYSZ_INS_EEXTR,
	SYSZ_INS_EFPC,
	SYSZ_INS_EPAIR,
	SYSZ_INS_EPAR,
	SYSZ_INS_EPCTR,
	SYSZ_INS_EPSW,
	SYSZ_INS_EREG,
	SYSZ_INS_EREGG,
	SYSZ_INS_ESAIR,
	SYSZ_INS_ESAR,
	SYSZ_INS_ESDTR,
	SYSZ_INS_ESEA,
	SYSZ_INS_ESTA,
	SYSZ_INS_ESXTR,
	SYSZ_INS_ETND,
	SYSZ_INS_EX,
	SYSZ_INS_EXRL,
	SYSZ_INS_FIDR,
	SYSZ_INS_FIDTR,
	SYSZ_INS_FIER,
	SYSZ_INS_FIXR,
	SYSZ_INS_FIXTR,
	SYSZ_INS_HDR,
	SYSZ_INS_HER,
	SYSZ_INS_HSCH,
	SYSZ_INS_IAC,
	SYSZ_INS_ICM,
	SYSZ_INS_ICMH,
	SYSZ_INS_ICMY,
	SYSZ_INS_IDTE,
	SYSZ_INS_IEDTR,
	SYSZ_INS_IEXTR,
	SYSZ_INS_IPK,
	SYSZ_INS_IPTE,
	SYSZ_INS_IRBM,
	SYSZ_INS_ISKE,
	SYSZ_INS_IVSK,
	SYSZ_INS_JGM,
	SYSZ_INS_JGNM,
	SYSZ_INS_JGNP,
	SYSZ_INS_JGNZ,
	SYSZ_INS_JGP,
	SYSZ_INS_JGZ,
	SYSZ_INS_JM,
	SYSZ_INS_JNM,
	SYSZ_INS_JNP,
	SYSZ_INS_JNZ,
	SYSZ_INS_JP,
	SYSZ_INS_JZ,
	SYSZ_INS_KDB,
	SYSZ_INS_KDBR,
	SYSZ_INS_KDTR,
	SYSZ_INS_KEB,
	SYSZ_INS_KEBR,
	SYSZ_INS_KIMD,
	SYSZ_INS_KLMD,
	SYSZ_INS_KM,
	SYSZ_INS_KMA,
	SYSZ_INS_KMAC,
	SYSZ_INS_KMC,
	SYSZ_INS_KMCTR,
	SYSZ_INS_KMF,
	SYSZ_INS_KMO,
	SYSZ_INS_KXBR,
	SYSZ_INS_KXTR,
	SYSZ_INS_LAE,
	SYSZ_INS_LAEY,
	SYSZ_INS_LAM,
	SYSZ_INS_LAMY,
	SYSZ_INS_LASP,
	SYSZ_INS_LAT,
	SYSZ_INS_LCBB,
	SYSZ_INS_LCCTL,
	SYSZ_INS_LCDFR,
	SYSZ_INS_LCDR,
	SYSZ_INS_LCER,
	SYSZ_INS_LCTL,
	SYSZ_INS_LCTLG,
	SYSZ_INS_LCXR,
	SYSZ_INS_LDE,
	SYSZ_INS_LDER,
	SYSZ_INS_LDETR,
	SYSZ_INS_LDXR,
	SYSZ_INS_LDXTR,
	SYSZ_INS_LEDR,
	SYSZ_INS_LEDTR,
	SYSZ_INS_LEXR,
	SYSZ_INS_LFAS,
	SYSZ_INS_LFHAT,
	SYSZ_INS_LFPC,
	SYSZ_INS_LGAT,
	SYSZ_INS_LGG,
	SYSZ_INS_LGSC,
	SYSZ_INS_LLGFAT,
	SYSZ_INS_LLGFSG,
	SYSZ_INS_LLGT,
	SYSZ_INS_LLGTAT,
	SYSZ_INS_LLGTR,
	SYSZ_INS_LLZRGF,
	SYSZ_INS_LM,
	SYSZ_INS_LMD,
	SYSZ_INS_LMH,
	SYSZ_INS_LMY,
	SYSZ_INS_LNDFR,
	SYSZ_INS_LNDR,
	SYSZ_INS_LNER,
	SYSZ_INS_LNXR,
	SYSZ_INS_LOCFH,
	SYSZ_INS_LOCFHE,
	SYSZ_INS_LOCFHH,
	SYSZ_INS_LOCFHHE,
	SYSZ_INS_LOCFHL,
	SYSZ_INS_LOCFHLE,
	SYSZ_INS_LOCFHLH,
	SYSZ_INS_LOCFHM,
	SYSZ_INS_LOCFHNE,
	SYSZ_INS_LOCFHNH,
	SYSZ_INS_LOCFHNHE,
	SYSZ_INS_LOCFHNL,
	SYSZ_INS_LOCFHNLE,
	SYSZ_INS_LOCFHNLH,
	SYSZ_INS_LOCFHNM,
	SYSZ_INS_LOCFHNO,
	SYSZ_INS_LOCFHNP,
	SYSZ_INS_LOCFHNZ,
	SYSZ_INS_LOCFHO,
	SYSZ_INS_LOCFHP,
	SYSZ_INS_LOCFHR,
	SYSZ_INS_LOCFHRE,
	SYSZ_INS_LOCFHRH,
	SYSZ_INS_LOCFHRHE,
	SYSZ_INS_LOCFHRL,
	SYSZ_INS_LOCFHRLE,
	SYSZ_INS_LOCFHRLH,
	SYSZ_INS_LOCFHRM,
	SYSZ_INS_LOCFHRNE,
	SYSZ_INS_LOCFHRNH,
	SYSZ_INS_LOCFHRNHE,
	SYSZ_INS_LOCFHRNL,
	SYSZ_INS_LOCFHRNLE,
	SYSZ_INS_LOCFHRNLH,
	SYSZ_INS_LOCFHRNM,
	SYSZ_INS_LOCFHRNO,
	SYSZ_INS_LOCFHRNP,
	SYSZ_INS_LOCFHRNZ,
	SYSZ_INS_LOCFHRO,
	SYSZ_INS_LOCFHRP,
	SYSZ_INS_LOCFHRZ,
	SYSZ_INS_LOCFHZ,
	SYSZ_INS_LOCGHI,
	SYSZ_INS_LOCGHIE,
	SYSZ_INS_LOCGHIH,
	SYSZ_INS_LOCGHIHE,
	SYSZ_INS_LOCGHIL,
	SYSZ_INS_LOCGHILE,
	SYSZ_INS_LOCGHILH,
	SYSZ_INS_LOCGHIM,
	SYSZ_INS_LOCGHINE,
	SYSZ_INS_LOCGHINH,
	SYSZ_INS_LOCGHINHE,
	SYSZ_INS_LOCGHINL,
	SYSZ_INS_LOCGHINLE,
	SYSZ_INS_LOCGHINLH,
	SYSZ_INS_LOCGHINM,
	SYSZ_INS_LOCGHINO,
	SYSZ_INS_LOCGHINP,
	SYSZ_INS_LOCGHINZ,
	SYSZ_INS_LOCGHIO,
	SYSZ_INS_LOCGHIP,
	SYSZ_INS_LOCGHIZ,
	SYSZ_INS_LOCGM,
	SYSZ_INS_LOCGNM,
	SYSZ_INS_LOCGNP,
	SYSZ_INS_LOCGNZ,
	SYSZ_INS_LOCGP,
	SYSZ_INS_LOCGRM,
	SYSZ_INS_LOCGRNM,
	SYSZ_INS_LOCGRNP,
	SYSZ_INS_LOCGRNZ,
	SYSZ_INS_LOCGRP,
	SYSZ_INS_LOCGRZ,
	SYSZ_INS_LOCGZ,
	SYSZ_INS_LOCHHI,
	SYSZ_INS_LOCHHIE,
	SYSZ_INS_LOCHHIH,
	SYSZ_INS_LOCHHIHE,
	SYSZ_INS_LOCHHIL,
	SYSZ_INS_LOCHHILE,
	SYSZ_INS_LOCHHILH,
	SYSZ_INS_LOCHHIM,
	SYSZ_INS_LOCHHINE,
	SYSZ_INS_LOCHHINH,
	SYSZ_INS_LOCHHINHE,
	SYSZ_INS_LOCHHINL,
	SYSZ_INS_LOCHHINLE,
	SYSZ_INS_LOCHHINLH,
	SYSZ_INS_LOCHHINM,
	SYSZ_INS_LOCHHINO,
	SYSZ_INS_LOCHHINP,
	SYSZ_INS_LOCHHINZ,
	SYSZ_INS_LOCHHIO,
	SYSZ_INS_LOCHHIP,
	SYSZ_INS_LOCHHIZ,
	SYSZ_INS_LOCHI,
	SYSZ_INS_LOCHIE,
	SYSZ_INS_LOCHIH,
	SYSZ_INS_LOCHIHE,
	SYSZ_INS_LOCHIL,
	SYSZ_INS_LOCHILE,
	SYSZ_INS_LOCHILH,
	SYSZ_INS_LOCHIM,
	SYSZ_INS_LOCHINE,
	SYSZ_INS_LOCHINH,
	SYSZ_INS_LOCHINHE,
	SYSZ_INS_LOCHINL,
	SYSZ_INS_LOCHINLE,
	SYSZ_INS_LOCHINLH,
	SYSZ_INS_LOCHINM,
	SYSZ_INS_LOCHINO,
	SYSZ_INS_LOCHINP,
	SYSZ_INS_LOCHINZ,
	SYSZ_INS_LOCHIO,
	SYSZ_INS_LOCHIP,
	SYSZ_INS_LOCHIZ,
	SYSZ_INS_LOCM,
	SYSZ_INS_LOCNM,
	SYSZ_INS_LOCNP,
	SYSZ_INS_LOCNZ,
	SYSZ_INS_LOCP,
	SYSZ_INS_LOCRM,
	SYSZ_INS_LOCRNM,
	SYSZ_INS_LOCRNP,
	SYSZ_INS_LOCRNZ,
	SYSZ_INS_LOCRP,
	SYSZ_INS_LOCRZ,
	SYSZ_INS_LOCZ,
	SYSZ_INS_LPCTL,
	SYSZ_INS_LPD,
	SYSZ_INS_LPDFR,
	SYSZ_INS_LPDG,
	SYSZ_INS_LPDR,
	SYSZ_INS_LPER,
	SYSZ_INS_LPP,
	SYSZ_INS_LPQ,
	SYSZ_INS_LPSW,
	SYSZ_INS_LPSWE,
	SYSZ_INS_LPTEA,
	SYSZ_INS_LPXR,
	SYSZ_INS_LRA,
	SYSZ_INS_LRAG,
	SYSZ_INS_LRAY,
	SYSZ_INS_LRDR,
	SYSZ_INS_LRER,
	SYSZ_INS_LRVH,
	SYSZ_INS_LSCTL,
	SYSZ_INS_LTDR,
	SYSZ_INS_LTDTR,
	SYSZ_INS_LTER,
	SYSZ_INS_LTXR,
	SYSZ_INS_LTXTR,
	SYSZ_INS_LURA,
	SYSZ_INS_LURAG,
	SYSZ_INS_LXD,
	SYSZ_INS_LXDR,
	SYSZ_INS_LXDTR,
	SYSZ_INS_LXE,
	SYSZ_INS_LXER,
	SYSZ_INS_LZRF,
	SYSZ_INS_LZRG,
	SYSZ_INS_M,
	SYSZ_INS_MAD,
	SYSZ_INS_MADR,
	SYSZ_INS_MAE,
	SYSZ_INS_MAER,
	SYSZ_INS_MAY,
	SYSZ_INS_MAYH,
	SYSZ_INS_MAYHR,
	SYSZ_INS_MAYL,
	SYSZ_INS_MAYLR,
	SYSZ_INS_MAYR,
	SYSZ_INS_MC,
	SYSZ_INS_MD,
	SYSZ_INS_MDE,
	SYSZ_INS_MDER,
	SYSZ_INS_MDR,
	SYSZ_INS_MDTR,
	SYSZ_INS_MDTRA,
	SYSZ_INS_ME,
	SYSZ_INS_MEE,
	SYSZ_INS_MEER,
	SYSZ_INS_MER,
	SYSZ_INS_MFY,
	SYSZ_INS_MG,
	SYSZ_INS_MGH,
	SYSZ_INS_MGRK,
	SYSZ_INS_ML,
	SYSZ_INS_MLR,
	SYSZ_INS_MP,
	SYSZ_INS_MR,
	SYSZ_INS_MSC,
	SYSZ_INS_MSCH,
	SYSZ_INS_MSD,
	SYSZ_INS_MSDR,
	SYSZ_INS_MSE,
	SYSZ_INS_MSER,
	SYSZ_INS_MSGC,
	SYSZ_INS_MSGRKC,
	SYSZ_INS_MSRKC,
	SYSZ_INS_MSTA,
	SYSZ_INS_MVCDK,
	SYSZ_INS_MVCIN,
	SYSZ_INS_MVCK,
	SYSZ_INS_MVCL,
	SYSZ_INS_MVCLE,
	SYSZ_INS_MVCLU,
	SYSZ_INS_MVCOS,
	SYSZ_INS_MVCP,
	SYSZ_INS_MVCS,
	SYSZ_INS_MVCSK,
	SYSZ_INS_MVN,
	SYSZ_INS_MVO,
	SYSZ_INS_MVPG,
	SYSZ_INS_MVZ,
	SYSZ_INS_MXD,
	SYSZ_INS_MXDR,
	SYSZ_INS_MXR,
	SYSZ_INS_MXTR,
	SYSZ_INS_MXTRA,
	SYSZ_INS_MY,
	SYSZ_INS_MYH,
	SYSZ_INS_MYHR,
	SYSZ_INS_MYL,
	SYSZ_INS_MYLR,
	SYSZ_INS_MYR,
	SYSZ_INS_NIAI,
	SYSZ_INS_NTSTG,
	SYSZ_INS_PACK,
	SYSZ_INS_PALB,
	SYSZ_INS_PC,
	SYSZ_INS_PCC,
	SYSZ_INS_PCKMO,
	SYSZ_INS_PFMF,
	SYSZ_INS_PFPO,
	SYSZ_INS_PGIN,
	SYSZ_INS_PGOUT,
	SYSZ_INS_PKA,
	SYSZ_INS_PKU,
	SYSZ_INS_PLO,
	SYSZ_INS_POPCNT,
	SYSZ_INS_PPA,
	SYSZ_INS_PPNO,
	SYSZ_INS_PR,
	SYSZ_INS_PRNO,
	SYSZ_INS_PT,
	SYSZ_INS_PTF,
	SYSZ_INS_PTFF,
	SYSZ_INS_PTI,
	SYSZ_INS_PTLB,
	SYSZ_INS_QADTR,
	SYSZ_INS_QAXTR,
	SYSZ_INS_QCTRI,
	SYSZ_INS_QSI,
	SYSZ_INS_RCHP,
	SYSZ_INS_RISBGN,
	SYSZ_INS_RP,
	SYSZ_INS_RRBE,
	SYSZ_INS_RRBM,
	SYSZ_INS_RRDTR,
	SYSZ_INS_RRXTR,
	SYSZ_INS_RSCH,
	SYSZ_INS_SAC,
	SYSZ_INS_SACF,
	SYSZ_INS_SAL,
	SYSZ_INS_SAM24,
	SYSZ_INS_SAM31,
	SYSZ_INS_SAM64,
	SYSZ_INS_SAR,
	SYSZ_INS_SCCTR,
	SYSZ_INS_SCHM,
	SYSZ_INS_SCK,
	SYSZ_INS_SCKC,
	SYSZ_INS_SCKPF,
	SYSZ_INS_SD,
	SYSZ_INS_SDR,
	SYSZ_INS_SDTR,
	SYSZ_INS_SDTRA,
	SYSZ_INS_SE,
	SYSZ_INS_SER,
	SYSZ_INS_SFASR,
	SYSZ_INS_SFPC,
	SYSZ_INS_SGH,
	SYSZ_INS_SHHHR,
	SYSZ_INS_SHHLR,
	SYSZ_INS_SIE,
	SYSZ_INS_SIGA,
	SYSZ_INS_SIGP,
	SYSZ_INS_SLA,
	SYSZ_INS_SLAG,
	SYSZ_INS_SLAK,
	SYSZ_INS_SLDA,
	SYSZ_INS_SLDL,
	SYSZ_INS_SLDT,
	SYSZ_INS_SLHHHR,
	SYSZ_INS_SLHHLR,
	SYSZ_INS_SLXT,
	SYSZ_INS_SP,
	SYSZ_INS_SPCTR,
	SYSZ_INS_SPKA,
	SYSZ_INS_SPM,
	SYSZ_INS_SPT,
	SYSZ_INS_SPX,
	SYSZ_INS_SQD,
	SYSZ_INS_SQDR,
	SYSZ_INS_SQE,
	SYSZ_INS_SQER,
	SYSZ_INS_SQXR,
	SYSZ_INS_SRDA,
	SYSZ_INS_SRDL,
	SYSZ_INS_SRDT,
	SYSZ_INS_SRNM,
	SYSZ_INS_SRNMB,
	SYSZ_INS_SRNMT,
	SYSZ_INS_SRP,
	SYSZ_INS_SRSTU,
	SYSZ_INS_SRXT,
	SYSZ_INS_SSAIR,
	SYSZ_INS_SSAR,
	SYSZ_INS_SSCH,
	SYSZ_INS_SSKE,
	SYSZ_INS_SSM,
	SYSZ_INS_STAM,
	SYSZ_INS_STAMY,
	SYSZ_INS_STAP,
	SYSZ_INS_STCK,
	SYSZ_INS_STCKC,
	SYSZ_INS_STCKE,
	SYSZ_INS_STCKF,
	SYSZ_INS_STCM,
	SYSZ_INS_STCMH,
	SYSZ_INS_STCMY,
	SYSZ_INS_STCPS,
	SYSZ_INS_STCRW,
	SYSZ_INS_STCTG,
	SYSZ_INS_STCTL,
	SYSZ_INS_STFL,
	SYSZ_INS_STFLE,
	SYSZ_INS_STFPC,
	SYSZ_INS_STGSC,
	SYSZ_INS_STIDP,
	SYSZ_INS_STM,
	SYSZ_INS_STMH,
	SYSZ_INS_STMY,
	SYSZ_INS_STNSM,
	SYSZ_INS_STOCFH,
	SYSZ_INS_STOCFHE,
	SYSZ_INS_STOCFHH,
	SYSZ_INS_STOCFHHE,
	SYSZ_INS_STOCFHL,
	SYSZ_INS_STOCFHLE,
	SYSZ_INS_STOCFHLH,
	SYSZ_INS_STOCFHM,
	SYSZ_INS_STOCFHNE,
	SYSZ_INS_STOCFHNH,
	SYSZ_INS_STOCFHNHE,
	SYSZ_INS_STOCFHNL,
	SYSZ_INS_STOCFHNLE,
	SYSZ_INS_STOCFHNLH,
	SYSZ_INS_STOCFHNM,
	SYSZ_INS_STOCFHNO,
	SYSZ_INS_STOCFHNP,
	SYSZ_INS_STOCFHNZ,
	SYSZ_INS_STOCFHO,
	SYSZ_INS_STOCFHP,
	SYSZ_INS_STOCFHZ,
	SYSZ_INS_STOCGM,
	SYSZ_INS_STOCGNM,
	SYSZ_INS_STOCGNP,
	SYSZ_INS_STOCGNZ,
	SYSZ_INS_STOCGP,
	SYSZ_INS_STOCGZ,
	SYSZ_INS_STOCM,
	SYSZ_INS_STOCNM,
	SYSZ_INS_STOCNP,
	SYSZ_INS_STOCNZ,
	SYSZ_INS_STOCP,
	SYSZ_INS_STOCZ,
	SYSZ_INS_STOSM,
	SYSZ_INS_STPQ,
	SYSZ_INS_STPT,
	SYSZ_INS_STPX,
	SYSZ_INS_STRAG,
	SYSZ_INS_STRVH,
	SYSZ_INS_STSCH,
	SYSZ_INS_STSI,
	SYSZ_INS_STURA,
	SYSZ_INS_STURG,
	SYSZ_INS_SU,
	SYSZ_INS_SUR,
	SYSZ_INS_SVC,
	SYSZ_INS_SW,
	SYSZ_INS_SWR,
	SYSZ_INS_SXR,
	SYSZ_INS_SXTR,
	SYSZ_INS_SXTRA,
	SYSZ_INS_TABORT,
	SYSZ_INS_TAM,
	SYSZ_INS_TAR,
	SYSZ_INS_TB,
	SYSZ_INS_TBDR,
	SYSZ_INS_TBEDR,
	SYSZ_INS_TBEGIN,
	SYSZ_INS_TBEGINC,
	SYSZ_INS_TCDB,
	SYSZ_INS_TCEB,
	SYSZ_INS_TCXB,
	SYSZ_INS_TDCDT,
	SYSZ_INS_TDCET,
	SYSZ_INS_TDCXT,
	SYSZ_INS_TDGDT,
	SYSZ_INS_TDGET,
	SYSZ_INS_TDGXT,
	SYSZ_INS_TEND,
	SYSZ_INS_THDER,
	SYSZ_INS_THDR,
	SYSZ_INS_TP,
	SYSZ_INS_TPI,
	SYSZ_INS_TPROT,
	SYSZ_INS_TR,
	SYSZ_INS_TRACE,
	SYSZ_INS_TRACG,
	SYSZ_INS_TRAP2,
	SYSZ_INS_TRAP4,
	SYSZ_INS_TRE,
	SYSZ_INS_TROO,
	SYSZ_INS_TROT,
	SYSZ_INS_TRT,
	SYSZ_INS_TRTE,
	SYSZ_INS_TRTO,
	SYSZ_INS_TRTR,
	SYSZ_INS_TRTRE,
	SYSZ_INS_TRTT,
	SYSZ_INS_TS,
	SYSZ_INS_TSCH,
	SYSZ_INS_UNPK,
	SYSZ_INS_UNPKA,
	SYSZ_INS_UNPKU,
	SYSZ_INS_UPT,
	SYSZ_INS_VA,
	SYSZ_INS_VAB,
	SYSZ_INS_VAC,
	SYSZ_INS_VACC,
	SYSZ_INS_VACCB,
	SYSZ_INS_VACCC,
	SYSZ_INS_VACCCQ,
	SYSZ_INS_VACCF,
	SYSZ_INS_VACCG,
	SYSZ_INS_VACCH,
	SYSZ_INS_VACCQ,
	SYSZ_INS_VACQ,
	SYSZ_INS_VAF,
	SYSZ_INS_VAG,
	SYSZ_INS_VAH,
	SYSZ_INS_VAP,
	SYSZ_INS_VAQ,
	SYSZ_INS_VAVG,
	SYSZ_INS_VAVGB,
	SYSZ_INS_VAVGF,
	SYSZ_INS_VAVGG,
	SYSZ_INS_VAVGH,
	SYSZ_INS_VAVGL,
	SYSZ_INS_VAVGLB,
	SYSZ_INS_VAVGLF,
	SYSZ_INS_VAVGLG,
	SYSZ_INS_VAVGLH,
	SYSZ_INS_VBPERM,
	SYSZ_INS_VCDG,
	SYSZ_INS_VCDGB,
	SYSZ_INS_VCDLG,
	SYSZ_INS_VCDLGB,
	SYSZ_INS_VCEQ,
	SYSZ_INS_VCEQB,
	SYSZ_INS_VCEQBS,
	SYSZ_INS_VCEQF,
	SYSZ_INS_VCEQFS,
	SYSZ_INS_VCEQG,
	SYSZ_INS_VCEQGS,
	SYSZ_INS_VCEQH,
	SYSZ_INS_VCEQHS,
	SYSZ_INS_VCGD,
	SYSZ_INS_VCGDB,
	SYSZ_INS_VCH,
	SYSZ_INS_VCHB,
	SYSZ_INS_VCHBS,
	SYSZ_INS_VCHF,
	SYSZ_INS_VCHFS,
	SYSZ_INS_VCHG,
	SYSZ_INS_VCHGS,
	SYSZ_INS_VCHH,
	SYSZ_INS_VCHHS,
	SYSZ_INS_VCHL,
	SYSZ_INS_VCHLB,
	SYSZ_INS_VCHLBS,
	SYSZ_INS_VCHLF,
	SYSZ_INS_VCHLFS,
	SYSZ_INS_VCHLG,
	SYSZ_INS_VCHLGS,
	SYSZ_INS_VCHLH,
	SYSZ_INS_VCHLHS,
	SYSZ_INS_VCKSM,
	SYSZ_INS_VCLGD,
	SYSZ_INS_VCLGDB,
	SYSZ_INS_VCLZ,
	SYSZ_INS_VCLZB,
	SYSZ_INS_VCLZF,
	SYSZ_INS_VCLZG,
	SYSZ_INS_VCLZH,
	SYSZ_INS_VCP,
	SYSZ_INS_VCTZ,
	SYSZ_INS_VCTZB,
	SYSZ_INS_VCTZF,
	SYSZ_INS_VCTZG,
	SYSZ_INS_VCTZH,
	SYSZ_INS_VCVB,
	SYSZ_INS_VCVBG,
	SYSZ_INS_VCVD,
	SYSZ_INS_VCVDG,
	SYSZ_INS_VDP,
	SYSZ_INS_VEC,
	SYSZ_INS_VECB,
	SYSZ_INS_VECF,
	SYSZ_INS_VECG,
	SYSZ_INS_VECH,
	SYSZ_INS_VECL,
	SYSZ_INS_VECLB,
	SYSZ_INS_VECLF,
	SYSZ_INS_VECLG,
	SYSZ_INS_VECLH,
	SYSZ_INS_VERIM,
	SYSZ_INS_VERIMB,
	SYSZ_INS_VERIMF,
	SYSZ_INS_VERIMG,
	SYSZ_INS_VERIMH,
	SYSZ_INS_VERLL,
	SYSZ_INS_VERLLB,
	SYSZ_INS_VERLLF,
	SYSZ_INS_VERLLG,
	SYSZ_INS_VERLLH,
	SYSZ_INS_VERLLV,
	SYSZ_INS_VERLLVB,
	SYSZ_INS_VERLLVF,
	SYSZ_INS_VERLLVG,
	SYSZ_INS_VERLLVH,
	SYSZ_INS_VESL,
	SYSZ_INS_VESLB,
	SYSZ_INS_VESLF,
	SYSZ_INS_VESLG,
	SYSZ_INS_VESLH,
	SYSZ_INS_VESLV,
	SYSZ_INS_VESLVB,
	SYSZ_INS_VESLVF,
	SYSZ_INS_VESLVG,
	SYSZ_INS_VESLVH,
	SYSZ_INS_VESRA,
	SYSZ_INS_VESRAB,
	SYSZ_INS_VESRAF,
	SYSZ_INS_VESRAG,
	SYSZ_INS_VESRAH,
	SYSZ_INS_VESRAV,
	SYSZ_INS_VESRAVB,
	SYSZ_INS_VESRAVF,
	SYSZ_INS_VESRAVG,
	SYSZ_INS_VESRAVH,
	SYSZ_INS_VESRL,
	SYSZ_INS_VESRLB,
	SYSZ_INS_VESRLF,
	SYSZ_INS_VESRLG,
	SYSZ_INS_VESRLH,
	SYSZ_INS_VESRLV,
	SYSZ_INS_VESRLVB,
	SYSZ_INS_VESRLVF,
	SYSZ_INS_VESRLVG,
	SYSZ_INS_VESRLVH,
	SYSZ_INS_VFA,
	SYSZ_INS_VFADB,
	SYSZ_INS_VFAE,
	SYSZ_INS_VFAEB,
	SYSZ_INS_VFAEBS,
	SYSZ_INS_VFAEF,
	SYSZ_INS_VFAEFS,
	SYSZ_INS_VFAEH,
	SYSZ_INS_VFAEHS,
	SYSZ_INS_VFAEZB,
	SYSZ_INS_VFAEZBS,
	SYSZ_INS_VFAEZF,
	SYSZ_INS_VFAEZFS,
	SYSZ_INS_VFAEZH,
	SYSZ_INS_VFAEZHS,
	SYSZ_INS_VFASB,
	SYSZ_INS_VFCE,
	SYSZ_INS_VFCEDB,
	SYSZ_INS_VFCEDBS,
	SYSZ_INS_VFCESB,
	SYSZ_INS_VFCESBS,
	SYSZ_INS_VFCH,
	SYSZ_INS_VFCHDB,
	SYSZ_INS_VFCHDBS,
	SYSZ_INS_VFCHE,
	SYSZ_INS_VFCHEDB,
	SYSZ_INS_VFCHEDBS,
	SYSZ_INS_VFCHESB,
	SYSZ_INS_VFCHESBS,
	SYSZ_INS_VFCHSB,
	SYSZ_INS_VFCHSBS,
	SYSZ_INS_VFD,
	SYSZ_INS_VFDDB,
	SYSZ_INS_VFDSB,
	SYSZ_INS_VFEE,
	SYSZ_INS_VFEEB,
	SYSZ_INS_VFEEBS,
	SYSZ_INS_VFEEF,
	SYSZ_INS_VFEEFS,
	SYSZ_INS_VFEEH,
	SYSZ_INS_VFEEHS,
	SYSZ_INS_VFEEZB,
	SYSZ_INS_VFEEZBS,
	SYSZ_INS_VFEEZF,
	SYSZ_INS_VFEEZFS,
	SYSZ_INS_VFEEZH,
	SYSZ_INS_VFEEZHS,
	SYSZ_INS_VFENE,
	SYSZ_INS_VFENEB,
	SYSZ_INS_VFENEBS,
	SYSZ_INS_VFENEF,
	SYSZ_INS_VFENEFS,
	SYSZ_INS_VFENEH,
	SYSZ_INS_VFENEHS,
	SYSZ_INS_VFENEZB,
	SYSZ_INS_VFENEZBS,
	SYSZ_INS_VFENEZF,
	SYSZ_INS_VFENEZFS,
	SYSZ_INS_VFENEZH,
	SYSZ_INS_VFENEZHS,
	SYSZ_INS_VFI,
	SYSZ_INS_VFIDB,
	SYSZ_INS_VFISB,
	SYSZ_INS_VFKEDB,
	SYSZ_INS_VFKEDBS,
	SYSZ_INS_VFKESB,
	SYSZ_INS_VFKESBS,
	SYSZ_INS_VFKHDB,
	SYSZ_INS_VFKHDBS,
	SYSZ_INS_VFKHEDB,
	SYSZ_INS_VFKHEDBS,
	SYSZ_INS_VFKHESB,
	SYSZ_INS_VFKHESBS,
	SYSZ_INS_VFKHSB,
	SYSZ_INS_VFKHSBS,
	SYSZ_INS_VFLCDB,
	SYSZ_INS_VFLCSB,
	SYSZ_INS_VFLL,
	SYSZ_INS_VFLLS,
	SYSZ_INS_VFLNDB,
	SYSZ_INS_VFLNSB,
	SYSZ_INS_VFLPDB,
	SYSZ_INS_VFLPSB,
	SYSZ_INS_VFLR,
	SYSZ_INS_VFLRD,
	SYSZ_INS_VFM,
	SYSZ_INS_VFMA,
	SYSZ_INS_VFMADB,
	SYSZ_INS_VFMASB,
	SYSZ_INS_VFMAX,
	SYSZ_INS_VFMAXDB,
	SYSZ_INS_VFMAXSB,
	SYSZ_INS_VFMDB,
	SYSZ_INS_VFMIN,
	SYSZ_INS_VFMINDB,
	SYSZ_INS_VFMINSB,
	SYSZ_INS_VFMS,
	SYSZ_INS_VFMSB,
	SYSZ_INS_VFMSDB,
	SYSZ_INS_VFMSSB,
	SYSZ_INS_VFNMA,
	SYSZ_INS_VFNMADB,
	SYSZ_INS_VFNMASB,
	SYSZ_INS_VFNMS,
	SYSZ_INS_VFNMSDB,
	SYSZ_INS_VFNMSSB,
	SYSZ_INS_VFPSO,
	SYSZ_INS_VFPSODB,
	SYSZ_INS_VFPSOSB,
	SYSZ_INS_VFS,
	SYSZ_INS_VFSDB,
	SYSZ_INS_VFSQ,
	SYSZ_INS_VFSQDB,
	SYSZ_INS_VFSQSB,
	SYSZ_INS_VFSSB,
	SYSZ_INS_VFTCI,
	SYSZ_INS_VFTCIDB,
	SYSZ_INS_VFTCISB,
	SYSZ_INS_VGBM,
	SYSZ_INS_VGEF,
	SYSZ_INS_VGEG,
	SYSZ_INS_VGFM,
	SYSZ_INS_VGFMA,
	SYSZ_INS_VGFMAB,
	SYSZ_INS_VGFMAF,
	SYSZ_INS_VGFMAG,
	SYSZ_INS_VGFMAH,
	SYSZ_INS_VGFMB,
	SYSZ_INS_VGFMF,
	SYSZ_INS_VGFMG,
	SYSZ_INS_VGFMH,
	SYSZ_INS_VGM,
	SYSZ_INS_VGMB,
	SYSZ_INS_VGMF,
	SYSZ_INS_VGMG,
	SYSZ_INS_VGMH,
	SYSZ_INS_VISTR,
	SYSZ_INS_VISTRB,
	SYSZ_INS_VISTRBS,
	SYSZ_INS_VISTRF,
	SYSZ_INS_VISTRFS,
	SYSZ_INS_VISTRH,
	SYSZ_INS_VISTRHS,
	SYSZ_INS_VL,
	SYSZ_INS_VLBB,
	SYSZ_INS_VLC,
	SYSZ_INS_VLCB,
	SYSZ_INS_VLCF,
	SYSZ_INS_VLCG,
	SYSZ_INS_VLCH,
	SYSZ_INS_VLDE,
	SYSZ_INS_VLDEB,
	SYSZ_INS_VLEB,
	SYSZ_INS_VLED,
	SYSZ_INS_VLEDB,
	SYSZ_INS_VLEF,
	SYSZ_INS_VLEG,
	SYSZ_INS_VLEH,
	SYSZ_INS_VLEIB,
	SYSZ_INS_VLEIF,
	SYSZ_INS_VLEIG,
	SYSZ_INS_VLEIH,
	SYSZ_INS_VLGV,
	SYSZ_INS_VLGVB,
	SYSZ_INS_VLGVF,
	SYSZ_INS_VLGVG,
	SYSZ_INS_VLGVH,
	SYSZ_INS_VLIP,
	SYSZ_INS_VLL,
	SYSZ_INS_VLLEZ,
	SYSZ_INS_VLLEZB,
	SYSZ_INS_VLLEZF,
	SYSZ_INS_VLLEZG,
	SYSZ_INS_VLLEZH,
	SYSZ_INS_VLLEZLF,
	SYSZ_INS_VLM,
	SYSZ_INS_VLP,
	SYSZ_INS_VLPB,
	SYSZ_INS_VLPF,
	SYSZ_INS_VLPG,
	SYSZ_INS_VLPH,
	SYSZ_INS_VLR,
	SYSZ_INS_VLREP,
	SYSZ_INS_VLREPB,
	SYSZ_INS_VLREPF,
	SYSZ_INS_VLREPG,
	SYSZ_INS_VLREPH,
	SYSZ_INS_VLRL,
	SYSZ_INS_VLRLR,
	SYSZ_INS_VLVG,
	SYSZ_INS_VLVGB,
	SYSZ_INS_VLVGF,
	SYSZ_INS_VLVGG,
	SYSZ_INS_VLVGH,
	SYSZ_INS_VLVGP,
	SYSZ_INS_VMAE,
	SYSZ_INS_VMAEB,
	SYSZ_INS_VMAEF,
	SYSZ_INS_VMAEH,
	SYSZ_INS_VMAH,
	SYSZ_INS_VMAHB,
	SYSZ_INS_VMAHF,
	SYSZ_INS_VMAHH,
	SYSZ_INS_VMAL,
	SYSZ_INS_VMALB,
	SYSZ_INS_VMALE,
	SYSZ_INS_VMALEB,
	SYSZ_INS_VMALEF,
	SYSZ_INS_VMALEH,
	SYSZ_INS_VMALF,
	SYSZ_INS_VMALH,
	SYSZ_INS_VMALHB,
	SYSZ_INS_VMALHF,
	SYSZ_INS_VMALHH,
	SYSZ_INS_VMALHW,
	SYSZ_INS_VMALO,
	SYSZ_INS_VMALOB,
	SYSZ_INS_VMALOF,
	SYSZ_INS_VMALOH,
	SYSZ_INS_VMAO,
	SYSZ_INS_VMAOB,
	SYSZ_INS_VMAOF,
	SYSZ_INS_VMAOH,
	SYSZ_INS_VME,
	SYSZ_INS_VMEB,
	SYSZ_INS_VMEF,
	SYSZ_INS_VMEH,
	SYSZ_INS_VMH,
	SYSZ_INS_VMHB,
	SYSZ_INS_VMHF,
	SYSZ_INS_VMHH,
	SYSZ_INS_VML,
	SYSZ_INS_VMLB,
	SYSZ_INS_VMLE,
	SYSZ_INS_VMLEB,
	SYSZ_INS_VMLEF,
	SYSZ_INS_VMLEH,
	SYSZ_INS_VMLF,
	SYSZ_INS_VMLH,
	SYSZ_INS_VMLHB,
	SYSZ_INS_VMLHF,
	SYSZ_INS_VMLHH,
	SYSZ_INS_VMLHW,
	SYSZ_INS_VMLO,
	SYSZ_INS_VMLOB,
	SYSZ_INS_VMLOF,
	SYSZ_INS_VMLOH,
	SYSZ_INS_VMN,
	SYSZ_INS_VMNB,
	SYSZ_INS_VMNF,
	SYSZ_INS_VMNG,
	SYSZ_INS_VMNH,
	SYSZ_INS_VMNL,
	SYSZ_INS_VMNLB,
	SYSZ_INS_VMNLF,
	SYSZ_INS_VMNLG,
	SYSZ_INS_VMNLH,
	SYSZ_INS_VMO,
	SYSZ_INS_VMOB,
	SYSZ_INS_VMOF,
	SYSZ_INS_VMOH,
	SYSZ_INS_VMP,
	SYSZ_INS_VMRH,
	SYSZ_INS_VMRHB,
	SYSZ_INS_VMRHF,
	SYSZ_INS_VMRHG,
	SYSZ_INS_VMRHH,
	SYSZ_INS_VMRL,
	SYSZ_INS_VMRLB,
	SYSZ_INS_VMRLF,
	SYSZ_INS_VMRLG,
	SYSZ_INS_VMRLH,
	SYSZ_INS_VMSL,
	SYSZ_INS_VMSLG,
	SYSZ_INS_VMSP,
	SYSZ_INS_VMX,
	SYSZ_INS_VMXB,
	SYSZ_INS_VMXF,
	SYSZ_INS_VMXG,
	SYSZ_INS_VMXH,
	SYSZ_INS_VMXL,
	SYSZ_INS_VMXLB,
	SYSZ_INS_VMXLF,
	SYSZ_INS_VMXLG,
	SYSZ_INS_VMXLH,
	SYSZ_INS_VN,
	SYSZ_INS_VNC,
	SYSZ_INS_VNN,
	SYSZ_INS_VNO,
	SYSZ_INS_VNX,
	SYSZ_INS_VO,
	SYSZ_INS_VOC,
	SYSZ_INS_VONE,
	SYSZ_INS_VPDI,
	SYSZ_INS_VPERM,
	SYSZ_INS_VPK,
	SYSZ_INS_VPKF,
	SYSZ_INS_VPKG,
	SYSZ_INS_VPKH,
	SYSZ_INS_VPKLS,
	SYSZ_INS_VPKLSF,
	SYSZ_INS_VPKLSFS,
	SYSZ_INS_VPKLSG,
	SYSZ_INS_VPKLSGS,
	SYSZ_INS_VPKLSH,
	SYSZ_INS_VPKLSHS,
	SYSZ_INS_VPKS,
	SYSZ_INS_VPKSF,
	SYSZ_INS_VPKSFS,
	SYSZ_INS_VPKSG,
	SYSZ_INS_VPKSGS,
	SYSZ_INS_VPKSH,
	SYSZ_INS_VPKSHS,
	SYSZ_INS_VPKZ,
	SYSZ_INS_VPOPCT,
	SYSZ_INS_VPOPCTB,
	SYSZ_INS_VPOPCTF,
	SYSZ_INS_VPOPCTG,
	SYSZ_INS_VPOPCTH,
	SYSZ_INS_VPSOP,
	SYSZ_INS_VREP,
	SYSZ_INS_VREPB,
	SYSZ_INS_VREPF,
	SYSZ_INS_VREPG,
	SYSZ_INS_VREPH,
	SYSZ_INS_VREPI,
	SYSZ_INS_VREPIB,
	SYSZ_INS_VREPIF,
	SYSZ_INS_VREPIG,
	SYSZ_INS_VREPIH,
	SYSZ_INS_VRP,
	SYSZ_INS_VS,
	SYSZ_INS_VSB,
	SYSZ_INS_VSBCBI,
	SYSZ_INS_VSBCBIQ,
	SYSZ_INS_VSBI,
	SYSZ_INS_VSBIQ,
	SYSZ_INS_VSCBI,
	SYSZ_INS_VSCBIB,
	SYSZ_INS_VSCBIF,
	SYSZ_INS_VSCBIG,
	SYSZ_INS_VSCBIH,
	SYSZ_INS_VSCBIQ,
	SYSZ_INS_VSCEF,
	SYSZ_INS_VSCEG,
	SYSZ_INS_VSDP,
	SYSZ_INS_VSEG,
	SYSZ_INS_VSEGB,
	SYSZ_INS_VSEGF,
	SYSZ_INS_VSEGH,
	SYSZ_INS_VSEL,
	SYSZ_INS_VSF,
	SYSZ_INS_VSG,
	SYSZ_INS_VSH,
	SYSZ_INS_VSL,
	SYSZ_INS_VSLB,
	SYSZ_INS_VSLDB,
	SYSZ_INS_VSP,
	SYSZ_INS_VSQ,
	SYSZ_INS_VSRA,
	SYSZ_INS_VSRAB,
	SYSZ_INS_VSRL,
	SYSZ_INS_VSRLB,
	SYSZ_INS_VSRP,
	SYSZ_INS_VST,
	SYSZ_INS_VSTEB,
	SYSZ_INS_VSTEF,
	SYSZ_INS_VSTEG,
	SYSZ_INS_VSTEH,
	SYSZ_INS_VSTL,
	SYSZ_INS_VSTM,
	SYSZ_INS_VSTRC,
	SYSZ_INS_VSTRCB,
	SYSZ_INS_VSTRCBS,
	SYSZ_INS_VSTRCF,
	SYSZ_INS_VSTRCFS,
	SYSZ_INS_VSTRCH,
	SYSZ_INS_VSTRCHS,
	SYSZ_INS_VSTRCZB,
	SYSZ_INS_VSTRCZBS,
	SYSZ_INS_VSTRCZF,
	SYSZ_INS_VSTRCZFS,
	SYSZ_INS_VSTRCZH,
	SYSZ_INS_VSTRCZHS,
	SYSZ_INS_VSTRL,
	SYSZ_INS_VSTRLR,
	SYSZ_INS_VSUM,
	SYSZ_INS_VSUMB,
	SYSZ_INS_VSUMG,
	SYSZ_INS_VSUMGF,
	SYSZ_INS_VSUMGH,
	SYSZ_INS_VSUMH,
	SYSZ_INS_VSUMQ,
	SYSZ_INS_VSUMQF,
	SYSZ_INS_VSUMQG,
	SYSZ_INS_VTM,
	SYSZ_INS_VTP,
	SYSZ_INS_VUPH,
	SYSZ_INS_VUPHB,
	SYSZ_INS_VUPHF,
	SYSZ_INS_VUPHH,
	SYSZ_INS_VUPKZ,
	SYSZ_INS_VUPL,
	SYSZ_INS_VUPLB,
	SYSZ_INS_VUPLF,
	SYSZ_INS_VUPLH,
	SYSZ_INS_VUPLHB,
	SYSZ_INS_VUPLHF,
	SYSZ_INS_VUPLHH,
	SYSZ_INS_VUPLHW,
	SYSZ_INS_VUPLL,
	SYSZ_INS_VUPLLB,
	SYSZ_INS_VUPLLF,
	SYSZ_INS_VUPLLH,
	SYSZ_INS_VX,
	SYSZ_INS_VZERO,
	SYSZ_INS_WCDGB,
	SYSZ_INS_WCDLGB,
	SYSZ_INS_WCGDB,
	SYSZ_INS_WCLGDB,
	SYSZ_INS_WFADB,
	SYSZ_INS_WFASB,
	SYSZ_INS_WFAXB,
	SYSZ_INS_WFC,
	SYSZ_INS_WFCDB,
	SYSZ_INS_WFCEDB,
	SYSZ_INS_WFCEDBS,
	SYSZ_INS_WFCESB,
	SYSZ_INS_WFCESBS,
	SYSZ_INS_WFCEXB,
	SYSZ_INS_WFCEXBS,
	SYSZ_INS_WFCHDB,
	SYSZ_INS_WFCHDBS,
	SYSZ_INS_WFCHEDB,
	SYSZ_INS_WFCHEDBS,
	SYSZ_INS_WFCHESB,
	SYSZ_INS_WFCHESBS,
	SYSZ_INS_WFCHEXB,
	SYSZ_INS_WFCHEXBS,
	SYSZ_INS_WFCHSB,
	SYSZ_INS_WFCHSBS,
	SYSZ_INS_WFCHXB,
	SYSZ_INS_WFCHXBS,
	SYSZ_INS_WFCSB,
	SYSZ_INS_WFCXB,
	SYSZ_INS_WFDDB,
	SYSZ_INS_WFDSB,
	SYSZ_INS_WFDXB,
	SYSZ_INS_WFIDB,
	SYSZ_INS_WFISB,
	SYSZ_INS_WFIXB,
	SYSZ_INS_WFK,
	SYSZ_INS_WFKDB,
	SYSZ_INS_WFKEDB,
	SYSZ_INS_WFKEDBS,
	SYSZ_INS_WFKESB,
	SYSZ_INS_WFKESBS,
	SYSZ_INS_WFKEXB,
	SYSZ_INS_WFKEXBS,
	SYSZ_INS_WFKHDB,
	SYSZ_INS_WFKHDBS,
	SYSZ_INS_WFKHEDB,
	SYSZ_INS_WFKHEDBS,
	SYSZ_INS_WFKHESB,
	SYSZ_INS_WFKHESBS,
	SYSZ_INS_WFKHEXB,
	SYSZ_INS_WFKHEXBS,
	SYSZ_INS_WFKHSB,
	SYSZ_INS_WFKHSBS,
	SYSZ_INS_WFKHXB,
	SYSZ_INS_WFKHXBS,
	SYSZ_INS_WFKSB,
	SYSZ_INS_WFKXB,
	SYSZ_INS_WFLCDB,
	SYSZ_INS_WFLCSB,
	SYSZ_INS_WFLCXB,
	SYSZ_INS_WFLLD,
	SYSZ_INS_WFLLS,
	SYSZ_INS_WFLNDB,
	SYSZ_INS_WFLNSB,
	SYSZ_INS_WFLNXB,
	SYSZ_INS_WFLPDB,
	SYSZ_INS_WFLPSB,
	SYSZ_INS_WFLPXB,
	SYSZ_INS_WFLRD,
	SYSZ_INS_WFLRX,
	SYSZ_INS_WFMADB,
	SYSZ_INS_WFMASB,
	SYSZ_INS_WFMAXB,
	SYSZ_INS_WFMAXDB,
	SYSZ_INS_WFMAXSB,
	SYSZ_INS_WFMAXXB,
	SYSZ_INS_WFMDB,
	SYSZ_INS_WFMINDB,
	SYSZ_INS_WFMINSB,
	SYSZ_INS_WFMINXB,
	SYSZ_INS_WFMSB,
	SYSZ_INS_WFMSDB,
	SYSZ_INS_WFMSSB,
	SYSZ_INS_WFMSXB,
	SYSZ_INS_WFMXB,
	SYSZ_INS_WFNMADB,
	SYSZ_INS_WFNMASB,
	SYSZ_INS_WFNMAXB,
	SYSZ_INS_WFNMSDB,
	SYSZ_INS_WFNMSSB,
	SYSZ_INS_WFNMSXB,
	SYSZ_INS_WFPSODB,
	SYSZ_INS_WFPSOSB,
	SYSZ_INS_WFPSOXB,
	SYSZ_INS_WFSDB,
	SYSZ_INS_WFSQDB,
	SYSZ_INS_WFSQSB,
	SYSZ_INS_WFSQXB,
	SYSZ_INS_WFSSB,
	SYSZ_INS_WFSXB,
	SYSZ_INS_WFTCIDB,
	SYSZ_INS_WFTCISB,
	SYSZ_INS_WFTCIXB,
	SYSZ_INS_WLDEB,
	SYSZ_INS_WLEDB,
	SYSZ_INS_XSCH,
	SYSZ_INS_ZAP,

	SYSZ_INS_ENDING,   // <-- mark the end of the list of instructions
} sysz_insn;

/// Group of SystemZ instructions
typedef enum sysz_insn_group {
	SYSZ_GRP_INVALID = 0, ///< = CS_GRP_INVALID

	// Generic groups
	// all jump instructions (conditional+direct+indirect jumps)
	SYSZ_GRP_JUMP,	///< = CS_GRP_JUMP

	// Architecture-specific groups
	SYSZ_GRP_DISTINCTOPS = 128,
	SYSZ_GRP_FPEXTENSION,
	SYSZ_GRP_HIGHWORD,
	SYSZ_GRP_INTERLOCKEDACCESS1,
	SYSZ_GRP_LOADSTOREONCOND,
	SYSZ_GRP_DFPPACKEDCONVERSION,
	SYSZ_GRP_DFPZONEDCONVERSION,
	SYSZ_GRP_ENHANCEDDAT2,
	SYSZ_GRP_EXECUTIONHINT,
	SYSZ_GRP_GUARDEDSTORAGE,
	SYSZ_GRP_INSERTREFERENCEBITSMULTIPLE,
	SYSZ_GRP_LOADANDTRAP,
	SYSZ_GRP_LOADANDZERORIGHTMOSTBYTE,
	SYSZ_GRP_LOADSTOREONCOND2,
	SYSZ_GRP_MESSAGESECURITYASSIST3,
	SYSZ_GRP_MESSAGESECURITYASSIST4,
	SYSZ_GRP_MESSAGESECURITYASSIST5,
	SYSZ_GRP_MESSAGESECURITYASSIST7,
	SYSZ_GRP_MESSAGESECURITYASSIST8,
	SYSZ_GRP_MISCELLANEOUSEXTENSIONS,
	SYSZ_GRP_MISCELLANEOUSEXTENSIONS2,
	SYSZ_GRP_NOVECTOR,
	SYSZ_GRP_POPULATIONCOUNT,
	SYSZ_GRP_PROCESSORASSIST,
	SYSZ_GRP_RESETREFERENCEBITSMULTIPLE,
	SYSZ_GRP_TRANSACTIONALEXECUTION,
	SYSZ_GRP_VECTOR,
	SYSZ_GRP_VECTORENHANCEMENTS1,
	SYSZ_GRP_VECTORPACKEDDECIMAL,

	SYSZ_GRP_ENDING,   // <-- mark the end of the list of groups
} sysz_insn_group;

#ifdef __cplusplus
}
#endif

#endif
