///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#undef _RHEOLEF_PARANO
#include "rheolef/geo_tree.h"
#include "rheolef/geo-visu-aux.h"
using namespace rheolef;
using namespace std;

// initialization of static data member
const geo_tree::big_integer_type geo_tree::coord_max
  = (geo_tree::big_integer_type(1) << geo_tree::deep_max);

// les algorithmes sont inspires de bamg-0.68/QuadTree.cpp

// =========================================================================
// visualisation
// =========================================================================
geo_tree::tree::tree()
 : _n(0), _u(), _l()
{
}
geo_tree::tree::~tree()
{
}
void
geo_tree::tree::_init(bool is_node, size_type two_pow_dim)
{
    _u._node = (tree**)(this + 1);
    for (size_type i = 0; i < two_pow_dim; i++)
        _u._node [i] = 0;
    if (is_node) {
	_n = - two_pow_dim;
    } else {
	_n = 0;
    }
}
geo_tree::tree* 
geo_tree::new_node(bool is_node)
{
    check_macro (sizeof(tree*) == sizeof(size_type), "incompat.");
    tree* pt = _heap.new_bucket();
    pt -> _init (is_node, _two_pow_dim);
    return pt;
}
// x in [0,h[ -> 0, x in [h,2h[ -> 1
geo_tree::size_type
geo_tree::sector_i (const ipoint& x, const big_integer_type& h) const
{
    switch (_dim) {
    case 1:
	return    ((x[0] & h) != 0) ? 1 : 0;
    case 2:
	return ((x[1] & h) != 0) ? 
	         (((x[0] & h) != 0) ? 3 : 2) :
	         (((x[0] & h) != 0) ? 1 : 0) ;
    case 3:
    default:
	return 
	     (((x[2] & h) != 0) ? 
	      (((x[1] & h) != 0) ? 
	         (((x[0] & h) != 0) ? 7 : 6) :
	         (((x[0] & h) != 0) ? 5 : 4)):
	      (((x[1] & h) != 0) ? 
	         (((x[0] & h) != 0) ? 3 : 2) :
	         (((x[0] & h) != 0) ? 1 : 0)));
    }
}
// if sect and 2^i then h else 0, i=1,2,3
geo_tree::ipoint
geo_tree::bbox_component_i (const ipoint& x, size_type sect, const big_integer_type& h) const
{
    ipoint y;
    for (size_type i = 0; i < _dim; i++)
      y[i] = x[i] + ((sect & (1 << i)) ? h : 0);
    return y;
}
void
geo_tree::insert (size_type x_idx)
{
    const ipoint& ix = _ivertex [x_idx];
    size_type level = 0;
    big_integer_type h = coord_max;
    tree* curr = _root;
    tree* prev = 0;
    size_type sect = numeric_limits<size_type>::max();
    while (curr != 0 && curr->is_node()) {
	curr->incr_node_level();
        h /= 2;
	level++;
	sect = sector_i (ix, h);
	prev = curr;
	curr = curr->node(sect);
    }
    // here: curr == 0 || curr->is_leaf()
    if (curr != 0) {
	// curr is a leaf
	for (size_type i = 0; i < curr->n_leaf(); i++) {
	    if (curr->leaf(i) == x_idx) {
	        // x has been aleardy inserted
	        return;
            }
        }
    }
    while (curr != 0 && curr->is_leaf() && curr->n_leaf() == n_leaf_max()) {
	// leaf is full : reorganize leafes in sub-levels
	vector<size_type> tmp (n_leaf_max());
	for (size_type i = 0; i < n_leaf_max(); i++) {
	    tmp[i] = curr->leaf(i);
        }
	set_node(curr);
	h /= 2;
	level++;
	for (size_type i = 0; i < n_leaf_max(); i++) {
	    size_type sect_i = sector_i (_ivertex[tmp[i]], h);
	    if (curr->node(sect_i) == 0) {
	        curr->node(sect_i) = new_leaf();
            }
	    tree* curr_i = curr->node(sect_i);
	    curr_i->add_leaf(tmp[i]);
        }
	sect = sector_i (ix, h);
	prev = curr;
	curr = curr->node(sect);
    }
    // here: curr == 0 || curr->is_leaf() && curr->n_leaf() < n_leaf_max()
    if (curr == 0) {
	if (_root == 0) {
	    _root = new_leaf();
	    curr = _root;
	} else if (prev == 0) {
	    // curr == _root
	    check_macro (curr == _root, "unexpected non-root");
	} else {
	    prev->node(sect) = new_leaf();
	    curr = prev->node(sect);
	}
    } else {
	check_macro (curr->is_leaf(), "unexpected non-leaf");
    }
    curr->add_leaf(x_idx);
}
geo_tree::geo_tree ()
 : _dim(0), 
   _two_pow_dim(1),
   _ivertex(0),
   _xmin(),
   _xmax(),
   _xmin_exact(),
   _xmax_exact(),
   _scale(),
   _root(0), 
   _heap(),
   _element(0),
   _fvertex(0),
   _initialized(false)
{
}
geo_tree::geo_tree (const geo_tree& x)
 : _dim(0), 
   _two_pow_dim(1),
   _ivertex(0),
   _xmin(),
   _xmax(),
   _xmin_exact(),
   _xmax_exact(),
   _scale(),
   _root(0), 
   _heap(),
   _element(0),
   _fvertex(0),
   _initialized(false)
{
  fatal_macro ("copy of geo_tree should not happend !");
}
void
geo_tree::initialize (
    size_type dim,
    const point& xmin,
    const point& xmax,
    size_type n_vertex,
    vector<point>::const_iterator vertex,
    size_type n_elt,
    vector<geo_element>::const_iterator element)
{
    if (_initialized) return;
    trace_macro ("initialize...");
    _dim = dim;
    _two_pow_dim = (1 << dim);
    _ivertex.resize (n_vertex);
    _xmin = xmin;
    _xmax = xmax;
    _xmin_exact = xmin;
    _xmax_exact = xmax;
    _root = 0;
    _element.resize (n_elt);
    _fvertex.resize (n_vertex);

    _heap.reinitialize(sizeof(tree) + (1 << dim)*sizeof(tree::node_or_leaf));

    // add a delta to normalize
    point delta = (_xmax - _xmin)/20.;
    _xmin = _xmin - delta;
    _xmax = _xmax + delta;

    for (size_type i = 0; i < _dim; i++) {
        _scale[i] = Float(coord_max)/(_xmax[i] - _xmin[i]);
    }
    // insert vertices
    for (size_type idx = 0; idx < n_vertex; idx++) {
        _fvertex[idx] = vertex[idx]; // for DEBUG: copy float vertices
        _ivertex[idx] = to_icoord (vertex[idx]);
        insert(idx);
    }
    // insert elements
    for (size_type idx = 0; idx < n_elt; idx++) {
	_element [idx] = element[idx]; // // for DEBUG: copy elts
        insert(element[idx]);
    }
    _initialized = true;
    trace_macro ("initialize done");
}
geo_tree::geo_tree (
    size_type dim,
    const point& xmin,
    const point& xmax,
    size_type n_vertex,
    vector<point>::const_iterator vertex,
    size_type n_elt,
    vector<geo_element>::const_iterator element)
 : _dim(0), 
   _two_pow_dim(1),
   _ivertex(0),
   _xmin(),
   _xmax(),
   _xmin_exact(),
   _xmax_exact(),
   _scale(),
   _root(0), 
   _heap(),
   _element(0),
   _fvertex(0),
   _initialized(false)
{
    trace_macro ("non-empty cstor...");
    initialize (dim, xmin, xmax, n_vertex, vertex, n_elt, element);
    trace_macro ("non-empty cstor done");
}
geo_tree::ipoint
geo_tree::to_icoord (const point& x) const
{
    ipoint y;
    for (size_type i = 0; i < _dim; i++) {
        y[i] = big_integer_type(_scale[i]*(x[i] - _xmin[i]) + 0.5);
	if (y[i] < 0 || y[i] > coord_max) {
	   warning_macro ("coord " << y[i] << " out of bound 0:" << coord_max);
	}
    }
    return y;
}
point
geo_tree::to_fcoord (const ipoint& x) const
{
    point y;
    for (size_type i = 0; i < _dim; i++) {
	if (x[i] < 0 || x[i] > coord_max) {
	   warning_macro ("coord " << x[i] << " out of bound 0:" << coord_max);
	}
	y[i] = bigfloat_to_float(_xmin[i] + x[i]/_scale[i]);
    }
    return y;
}
// =========================================================================
// visualisation
// =========================================================================
void
geo_tree::put_gnuplot (string basename) const
{
    // -----------------------------------------------
    // gnuplot files
    // -----------------------------------------------
    bool verbose = true;
    bool all_boxes = false;
    string plot_name = basename+".plot";
    string box_name = basename+"-box.gdat";
    string cross_name = basename+"-cross.gdat";
    string mark_name = basename+"-mark.gdat";
    ofstream plot  (plot_name.c_str());
    ofstream box   (box_name.c_str());
    ofstream cross (cross_name.c_str());
    ofstream mark  (mark_name.c_str());
    if (verbose) {
        cerr << "! file \"" << plot_name << "\" created." << endl; 
        cerr << "! file \"" << box_name << "\" created." << endl; 
        cerr << "! file \"" << cross_name << "\" created." << endl; 
        cerr << "! file \"" << mark_name << "\" created." << endl; 
    }
    point delta = 0.1*(_xmax - _xmin);
    if (_dim != 3) {
        plot << "set size square" << endl;
	if (_dim == 1) {
            plot << "plot [" << _xmin[0]-delta[0] << ":" << _xmax[0]+delta[0] 
		     << "][" << _xmin[0]-delta[0] << ":" << _xmax[0]+delta[0]
		     << "] \\"
		 << endl;
	} else {
            plot << "plot [" << _xmin[0]-delta[0] << ":" << _xmax[0]+delta[0] 
		     << "][" << _xmin[1]-delta[1] << ":" << _xmax[1]+delta[1]
		     << "] \\" << endl;
	}
	plot << "   \"" << cross_name << "\" notitle with lines 2, \\" << endl
             << "   \"" << box_name << "\" notitle with lines lt 1 linewidth 2, \\" << endl
             << "   \"" << mark_name << "\" notitle with points pointsize 1" << endl
             << "pause -1 \"<retour>\"" << endl;
    } else {
        string rot_name = basename+".rot";
        plot << "set parametric" << endl
	     << "set nokey" << endl
	     << "xrot=50" << endl
	     << "yrot=10" << endl
	     << "zrot=10" << endl
             << "splot \\" << endl
             << "   \"" << cross_name << "\" notitle with lines 2, \\" << endl
             << "   \"" << box_name << "\" notitle with lines lt 1 linewidth 2, \\" << endl
             << "   \"" << mark_name << "\" notitle with points pointsize 1" << endl
             << "load \"" << rot_name << "\"" << endl
             << "pause -1 \"<retour>\"" << endl;
        ofstream rot  (rot_name.c_str());
        if (verbose) {
            cerr << "! file \"" << rot_name << "\" created." << endl; 
	}
	rot << "yrot=(yrot+1)%360" << endl
            << "set view xrot,yrot" << endl
            << "replot" << endl
            << "reread" << endl;
	rot.close();
    }
    plot.close();
    // -----------------------------------------------
    // reccursive function, flattened by using stacks
    // -----------------------------------------------
    tree*     stack [deep_max];
    size_type sect  [deep_max];
    ipoint    ix    [deep_max]; // left corner of the bbox
    size_type level = 0;
    stack [0] = _root;
    sect  [0] = _two_pow_dim;
    ix [0] = ipoint(0,0,0);
    big_integer_type h = coord_max;
    if (all_boxes) put_box_i (box, ix[0], h);
    do {
        tree* curr = stack [level];
	while (sect [level] != 0) {
	    sect [level]--;
	    if (curr->is_leaf()) {
		if (!all_boxes) put_box_i (box, ix[level], h);
	        for (size_type i = 0; i < curr->n_leaf(); i++)
	          put_marker(mark, curr->leaf(i));
		break;
	    } else {
		size_type sect1 = sect [level];
		tree* prev = stack [level];
		curr = curr->node(sect1);
		if (curr != 0) {
		    h /= 2;
		    level++;
		    ix[level] = bbox_component_i (ix[level-1], sect1, h);
		    stack [level] = curr;
		    sect [level] = _two_pow_dim;
		    if (all_boxes) put_box_i (box, ix[level], h);
	        } else {
		    ipoint ix1 = bbox_component_i (ix[level], sect1, h/2);
		    put_cross_i (cross, ix1, h/2);
		    curr = prev;
		}
	    }
	}
	if (level == 0) break;
	level--;
	h *= 2;
    } while (true);
    box.close();
    mark.close();
    cross.close();
}
void 
geo_tree::put_marker (ostream& out, size_type x_idx) const
{
  point x = to_fcoord (_ivertex[x_idx]);
  switch (_dim) {
    case 1:
    case 2:
        x.put (out, 2);
	break;
    case 3:
    default: 
        out << x;
  }
  out << endl;
}
void
geo_tree::put_box_i (ostream& out, const ipoint& x, const big_integer_type& h) const
{
  switch (_dim) {
    case 1: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h));
	double hf = x1[0]-x0[0];
	out << x0[0] << " 0" << endl
            << x1[0] << " 0" << endl
            << x1[0] << " " << hf << endl
	    << x0[0] << " " << hf << endl
	    << x0[0] << " 0" << endl
	    << endl;
	break;
    }
    case 2: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h,x[1]));
        point x2 = to_fcoord (ipoint(x[0]+h,x[1]+h));
        point x3 = to_fcoord (ipoint(x[0],  x[1]+h));
        x0.put (out, _dim); out << endl;
        x1.put (out, _dim); out << endl;
        x2.put (out, _dim); out << endl;
        x3.put (out, _dim); out << endl;
        x0.put (out, _dim); out << endl;
        out << endl;
	break;
    }
    case 3:
    default: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h,x[1],   x[2]));
        point x2 = to_fcoord (ipoint(x[0]+h,x[1]+h, x[2]));
        point x3 = to_fcoord (ipoint(x[0],  x[1]+h, x[2]));
        point x4 = to_fcoord (ipoint(x[0],  x[1],   x[2]+h));
        point x5 = to_fcoord (ipoint(x[0]+h,x[1],   x[2]+h));
        point x6 = to_fcoord (ipoint(x[0]+h,x[1]+h, x[2]+h));
        point x7 = to_fcoord (ipoint(x[0],  x[1]+h, x[2]+h));
	out << x0 << endl << x1 << endl << x2 << endl << x3 << endl << x0 << endl
	    << x4 << endl << x5 << endl << x6 << endl << x7 << endl << x4 << endl
	    << endl
	    << x1 << endl << x5 << endl << endl
	    << x2 << endl << x6 << endl << endl
	    << x3 << endl << x7 << endl << endl;
    }
  }
}
void 
geo_tree::put_cross_i (ostream& out, const ipoint& x, const big_integer_type& h) const
{
  switch (_dim) {
    case 1: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h));
	out << x0[0] << " 0" << endl
            << x1[0] << " 1" << endl
	    << endl
            << x1[0] << " 0" << endl
	    << x0[0] << " 1" << endl
	    << endl;
	break;
    }
    case 2: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h,x[1]));
        point x2 = to_fcoord (ipoint(x[0]+h,x[1]+h));
        point x3 = to_fcoord (ipoint(x[0],  x[1]+h));
        x0.put (out, _dim); out << endl;
        x2.put (out, _dim); out << endl;
        out << endl;
        x1.put (out, _dim); out << endl;
        x3.put (out, _dim); out << endl;
        out << endl;
	break;
    }
    default: {
        point x0 = to_fcoord (x);
        point x1 = to_fcoord (ipoint(x[0]+h,x[1],   x[2]));
        point x2 = to_fcoord (ipoint(x[0]+h,x[1]+h, x[2]));
        point x3 = to_fcoord (ipoint(x[0],  x[1]+h, x[2]));
        point x4 = to_fcoord (ipoint(x[0],  x[1],   x[2]+h));
        point x5 = to_fcoord (ipoint(x[0]+h,x[1],   x[2]+h));
        point x6 = to_fcoord (ipoint(x[0]+h,x[1]+h, x[2]+h));
        point x7 = to_fcoord (ipoint(x[0],  x[1]+h, x[2]+h));
	out << x0 << endl << x6 << endl << endl
	    << x1 << endl << x7 << endl << endl
	    << x2 << endl << x4 << endl << endl
	    << x3 << endl << x5 << endl << endl;
    }
  }
}
void
geo_tree::dump_reccursive (ostream& out, geo_tree::tree* p, size_type level) const
{
    if (p == 0) return;
    if (p->is_leaf()) {
        for (size_type i = 0; i < p->n_leaf(); i++) {
	    out << level << ": leaf[" << i << "] = " << p->leaf(i) << endl;
        }
    } else {
        for (size_type i = 0; i < _two_pow_dim; i++) {
	  if (p->node(i) == 0) {
	      out << level << ": node[" << i << "] = empty" << endl;
	  } else {
	      out << level << ": node[" << i << "] = no-empty" << endl;
              dump_reccursive (out, p->node(i), level+1);
          }
        }
    }
}
void
geo_tree::dump (ostream& out) const
{
    out << "begin dump" << endl;
    dump_reccursive (out, _root, 0);
    out << "end dump" << endl;
}
void
geo_tree::put_element_list (
    vector<geo_element>::const_iterator elements,
    const set<size_type>& l) const
{
    ofstream lst ("output-elt-list.gdat");
    cerr << "! file \"output-elt-list.gdat\" created." << endl;
    lst << "# list size = " << l.size() << endl;
    lst << "#";
    for (set<size_type>::const_iterator i = l.begin(); i != l.end(); i++) {
        lst << " " << *i;
    }
    lst << endl;
    for (set<size_type>::const_iterator i = l.begin(); i != l.end(); i++) {
        gnuplot_element (lst, elements[*i], _fvertex.begin(), _dim);
    }
}
// =========================================================================
// localization
// =========================================================================
geo_tree::ipoint
geo_tree::round_point_i(const ipoint& ix) const
{
    ipoint iy;
    for (size_type i = 0; i < _dim; i++) {
        iy[i] = (ix[i] < coord_max ?
		 (ix[i] < 0 ? 0 : ix[i]) :
		 coord_max);
    }
    return iy;
}
geo_tree::size_type
geo_tree::find_close_vertex_i (const ipoint& ix) const
{
    if (_root == 0) return numeric_limits<size_type>::max();
    // -------------------------------------------------------------
    // try to find it first by a deepest descent
    // -------------------------------------------------------------
    ipoint ixr = round_point_i (ix);
    big_integer_type h0 = coord_max;
    ipoint ix0(0,0,0);
    tree* curr0 = _root;
    while (curr0->is_node()) {
	big_integer_type h1 = h0/2;
	// sect0 is the box of size h1 containing x
	size_type sect1 = sector_i (ixr,h1);
        tree* curr1 = curr0->node(sect1);
	if (curr1 == 0 || (curr1->is_leaf() && curr1->n_leaf() == 0)) {
	   // empty box
	   break;
	}
	curr0 = curr1;
	ix0 = bbox_component_i (ix0, sect1, h1);
	h0 = h1;
    }
    // -------------------------------------------------------------
    // the deepest decent gives a non-empty leaf : yeah !
    // -------------------------------------------------------------
    if (curr0 != 0 && curr0->is_leaf() && curr0->n_leaf() != 0) {
	big_integer_type d = h0;
        size_type x_idx = numeric_limits<size_type>::max();
	for (size_type i = 0; i < curr0->n_leaf(); i++) {
	    size_type xi_idx = curr0->leaf(i);
	    big_integer_type di = dist_infty(ix,_ivertex[xi_idx]);
	    if (di < d) {
		d = di;
		x_idx = xi_idx;
	    }
	}
	return x_idx;
    }
    // -------------------------------------------------------------
    // we are not lucky : we get a node => general case
    // -------------------------------------------------------------
    tree*     stack [deep_max];
    size_type sect  [deep_max];
    ipoint    ix_stack [deep_max]; // left corner of the bbox
    size_type level = 0;
    stack [0] = curr0;
    sect  [0] = _two_pow_dim;
    ix_stack [0] = ix0;
    big_integer_type d = h0;
    size_type x_idx = numeric_limits<size_type>::max();
    do {
        tree* curr = stack [level];
	while (sect [level] != 0) {
	    sect [level]--;
	    size_type sect1 = sect [level];
	    if (curr->is_leaf()) {
	        size_type x1_idx = curr->leaf(sect1);
		big_integer_type d1 = dist_infty(ix,_ivertex[x1_idx]);
		if (d1 < d) {
		    d = d1;
		    x_idx = x1_idx;
		}
	    } else {
		tree* prev = curr;
		curr = curr->node(sect1);
		if (curr != 0) {
		    h0 /= 2;
		    ipoint ix1 = bbox_component_i (ix_stack[level], sect1, h0);
		    if (box_intersect_box_i (ix1,h0,ix,d)) {
			level++;
			stack [level] = curr;
			ix_stack [level] = ix1;
			if (curr->is_node()) {
			  sect  [level] = _two_pow_dim;
		        } else {
			  sect  [level] = curr->n_leaf();
	    		}
		    } else {
			curr = prev;
			h0 *= 2;
		    }
		} else {
		    curr = prev;
		}
	    }
	}
	h0 *= 2;
	if (level == 0) break;
	level--;
    } while (true);
    return x_idx;
}
bool
geo_tree::belongs_to_bounding_box (const point& x) const
{
    for (size_type i = 0; i < _dim; i++) {
        if (x[i] < _xmin_exact[i] || x[i] > _xmax_exact[i])
	    return false;
    }
    return true;
}
point
geo_tree::project_into_bounding_box (const point& x0) const
{
    point x;
    for (size_type i = 0; i < _dim; i++) {
        x[i] = min(max(x0[i], _xmin_exact[i]), _xmax_exact[i]);
    }
#ifdef TO_CLEAN
    if (dist(x,x0) != 0) {
        warning_macro ("project " << x0 << " into bbox as " << x);
    }
#endif // TO_CLEAN
    return x;
}
geo_tree::size_type
geo_tree::find_close_vertex (const point& x0) const
{
    point x = project_into_bounding_box(x0);
    return find_close_vertex_i (to_icoord(x));
}
// =========================================================================
// element insert
// =========================================================================
void
geo_tree::insert (const geo_element& K)
{
    trace_macro ("insert " << K.index() << "...");
    if (_root == 0) {
        error_macro ("empty tree");
    }
    // -------------------------------------------------------------
    // try to find it first by a deepest descent
    // -------------------------------------------------------------
    big_integer_type h0 = coord_max;
    ipoint ix0(0,0,0);
    tree* prev0 = 0;
    tree* curr0 = _root;
    size_type sect0 = numeric_limits<size_type>::max();
    if (!element_subset_box_i (K, ix0, h0)) {
        error_macro ("root tree does not contains the element");
    }
    trace_macro ("insert " << K.index() << ": deepest descent...");
    while (curr0 && curr0->is_node()) {
	big_integer_type h1 = h0/2;
	ipoint ix1;
	size_type sect1 = numeric_limits<size_type>::max();
	for (size_type i = 0; i < _two_pow_dim; i++) {
	    ix1 = bbox_component_i (ix0, i, h1);
	    if (element_subset_box_i (K, ix1, h1)) {
               sect1 = i;
	       break; 
	    }
	}
	if (sect1 != numeric_limits<size_type>::max()) {
	    // a subbox contains [a,b]
	    ix0 = ix1;
	    h0 = h1;
	    prev0 = curr0;
	    sect0 = sect1;
 	    curr0 = curr0->node(sect1);
        } else {
	    break;
        }
    }
    // -------------------------------------------------------------
    // the deepest decent gives 
    // - either a leaf that contains [a,b] (could be empty)
    // - or     a node that contains [a,b] and subnodes doesn not
    // -------------------------------------------------------------
    if (curr0 == 0 || curr0->is_leaf()) {
        trace_macro ("insert " << K.index() << ": deepest descent successfull...");
	if (curr0 == 0) {
	    if (prev0 == 0) {
	        _root = new_leaf();
		curr0 = _root;
	    } else {
	        prev0->node(sect0) = new_leaf();
		curr0 = prev0;
	    }
	}
	// curr0 is leaf
	curr0->insert_data (K.index());
        trace_macro ("insert " << K.index() << ": done by deepest descent.");
	return;
    }
    // -------------------------------------------------------------
    // we are not lucky : the segment is long and splitted between 
    // several subboxes 
    // => general case
    // -------------------------------------------------------------
    trace_macro ("insert " << K.index() << ": general case...");
    tree*     stack [deep_max];
    size_type sect  [deep_max];
    ipoint    ix_stack [deep_max]; // left corner of the bbox
    size_type level = 0;
    stack [0] = curr0;
    sect  [0] = _two_pow_dim;
    ix_stack [0] = ix0;
    big_integer_type h = h0;
    do {
        tree* curr = stack [level];
	while (sect [level] != 0) {
	    sect [level]--;
	    big_integer_type h1 = h/2;
	    size_type sect1 = sect [level];
	    ipoint ix1 = bbox_component_i (ix_stack[level], sect1, h1);
            trace_macro ("insert " << K.index() << ": call element_intersect_box...");
	    if (!bbox_element_intersect_box_i (K, ix1, h1)) {
                trace_macro ("insert " << K.index() << ": bbox_element_intersect_box -> false");
		continue;
	    }
            trace_macro ("insert " << K.index() << ": bbox_element_intersect_box -> true");
	    tree* curr1 = curr->node(sect1);
	    if (curr1 == 0 || curr1->is_leaf()) {
	        if (curr1 == 0) {
		    curr->node(sect1) = new_leaf();
		    curr1 = curr->node(sect1);
		}
		curr1->insert_data (K.index());
	    } else {
		level++;
		h = h1;
		stack [level] = curr1;
		ix_stack [level] = ix1;
		sect  [level] = _two_pow_dim;
		curr = curr1;
	    }
	}
	h *= 2;
	if (level == 0) break;
	level--;
    } while (true);
    trace_macro ("insert " << K.index() << ": done by general case.");
}
// =========================================================================
// search element containing a point
// =========================================================================
geo_tree::size_type
geo_tree::search_element_containing  (
    vector<geo_element>::const_iterator elements,
    const set<size_type>& l, 
    const point& x) const
{
#ifdef TO_CLEAN
    put_element_list (elements, l);
    warning_macro ("element list size = " << l.size());
#endif // TO_CLEAN
    for (set<size_type>::const_iterator i = l.begin(); i != l.end(); i++) {
#ifdef TO_CLEAN
        warning_macro ("element " << *i << " contains x ?");
#endif // TO_CLEAN
        if (belongs_to_element (x, elements[*i]))
	    return *i;
    }
#ifdef TO_CLEAN
    warning_macro ("no element contains x in the leaf: outside of the mesh");
#endif // TO_CLEAN
    return numeric_limits<size_type>::max();
}
geo_tree::size_type
geo_tree::search_element_containing (
    vector<geo_element>::const_iterator elements,
    const point& x) const
{
#ifdef TO_CLEAN
    warning_macro ("search: " << x);
#endif // TO_CLEAN
    if (_root == 0) {
	error_macro ("empty tree");
	return numeric_limits<size_type>::max();
    }
    point ixr = round_point_i(to_icoord(x));
    // -------------------------------------------------------------
    // try to find it first by a deepest descent
    // -------------------------------------------------------------
    big_integer_type h0 = coord_max;
    ipoint ix0(0,0,0);
    tree* curr0 = _root;
    while (curr0 && curr0->is_node()) {
	big_integer_type h1 = h0/2;
	// sect0 is the box of size h1 containing x
	size_type sect1 = sector_i (ixr,h1);
        tree* curr1 = curr0->node(sect1);
	curr0 = curr1;
	ix0 = bbox_component_i (ix0, sect1, h1);
	h0 = h1;
    }
    // -------------------------------------------------------------
    // the deepest decent gives a non-empty leaf : yeah !
    // -------------------------------------------------------------
    if (curr0 != 0 && curr0->is_leaf()) {
#ifdef TO_CLEAN
	ofstream thebox ("output-thebox.gdat");
	cerr << "! file \"output-thebox.gdat\" created." << endl;
	put_box_i (thebox, ix0, h0);
	thebox.close();
#endif // TO_CLEAN
	return search_element_containing (elements, curr0->get_element_list(), x);
    } else {
	// no box contains x : outside of the mesh
#ifdef TO_CLEAN
	if (curr0 == 0) {
	    warning_macro ("null box: no box contains x : outside of the mesh");
	} else if (curr0->is_node()) {
	    warning_macro ("get a node: no box contains x : outside of the mesh");
	} else {
	    warning_macro ("get a ???");
	}
#endif // TO_CLEAN
	return numeric_limits<size_type>::max();
    }
}
#ifdef TO_CLEAN
geo_tree::size_type
geo_tree::search_element_containing (
    vector<geo_element>::const_iterator elements,
    const point& x) const
{
    if (!belongs_to_bounding_box(x)) {
	return numeric_limits<size_type>::max();
    }
    return search_element_containing_i (elements, to_icoord(x));
}
#endif // TO_CLEAN
// =========================================================================
// find an element close to a point
// =========================================================================
geo_tree::size_type
geo_tree::find_close_element (
    vector<geo_element>::const_iterator elements,
    const point& x, point& y) const
{
    trace_macro ("find_close_element: vertex " << x);
    size_type K_idx = search_element_containing (elements, x);
    if (K_idx != numeric_limits<size_type>::max()) {
      y = x;
      trace_macro ("vertex " << x << " in element " << K_idx);
      return K_idx;
    }
    error_macro ("vertex " << x << " not in mesh; find_close_element: not yet");
    return numeric_limits<size_type>::max();
}
#ifdef TO_CLEAN
geo_tree::size_type
geo_tree::find_close_element (
    vector<geo_element>::const_iterator elements,
    const point& x, point& y) const
{
    ipoint iy;
    size_type K_idx = find_close_element_i (elements, to_icoord(x), iy);
    y = to_fcoord(iy);
    return K_idx;
}
#endif // TO_CLEAN
