# ifndef _RHEO_URM_ABTBC_H
# define _RHEO_URM_ABTBC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:urm_abtbc
NAME: @code{urm_abtbc} -- generalized Stokes solver
@findex urm\_abtbc
@cindex Stokes problem
@cindex Uzawa algorithm
@cindex stabilized finite element method
DESCRIPTION:
  @noindent
  Uzawa Minimal Residual algorithm applied to the stabilized Stokes system
  with the bock diagonal preconditionner : (inv(A),I).
  References :
  @example
        El Montasser Abdalass,
        Resolution performante du probleme de Stokes
        par mini-elements, maillages auto-adaptatifs
        et methodes multigrilles - applications.
        These se l'ecole centrale de Lyon, 1987.
        page 33.

	Michel Fortin and Roland Glowinski,
	Augmented Lagrangian methods - applications to the
	numerical solution of boundary-value problems,
	Elsevier, 1983,
	page 4.

  @end example
AUTHOR:
  | Pierre.Saramito@imag.fr
  LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:   11 april 2001
METHODS: @urm_abtbc
End:
*/

#include "rheolef/matrix_solver.h"
namespace rheolef { 

template<
    class MatrixSolver, 
    class MatrixPreconditionner, 
    class MatrixMetric, 
    class Matrix, 
    class Vector, 
    class Real>
int
urm_abtbc(
    const MatrixSolver&          m_solver,
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const Matrix&                c,
    const MatrixMetric&		 m,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    int&                         max_iter,
    Real&                        tol,
    ostream*                     p_cres = 0)
{
    m_solver(a, ap, u, f - b.trans_mult(p));
    Real   residu = 1;
    for (int k=1; k <= max_iter; k++) {
        Vector r = g - b*u - c*p;
        Real   residu = norm(r);
        if (p_cres) *p_cres << k << " " << residu << "\n" ;
        if (residu <= tol) {
	    tol = residu;
	    max_iter = k;
	    return 0;
        }
        Vector z (u.size());
	m_solver(a, ap, z, b.trans_mult(r));
        Real rho = dot(m*r, r) / dot(m*r, b*z - c*r);
	p -= rho*r;
	u += rho*z;
    }
    tol = residu;
    return 1;
}
template<
    class MatrixPreconditionner, 
    class Matrix, 
    class Vector, 
    class Real>
int
urm_abtbc(
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const Matrix&                c,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    int&                         max_iter,
    Real&                        tol,
    ostream*                     p_cres = &std::cerr)
//>uzawa_abtb:
{
    return urm_abtbc (ldlt_solver<MatrixPreconditionner, Matrix, Vector, Vector>(),
        a, ap, b, c, EYE, u, p, f, g, max_iter, tol, p_cres);
}
}// namespace rheolef
# endif // _RHEO_URM_ABTBC_H
