///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// compute:
//  \int_\Omega weight(x) dx = mes(\Omega) with weight
//
// check value on the unit cube [0,1]^d
//
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

typedef Float (*function_type)(const point&);

struct list_type {
	string name;
	function_type function;
	Float  expect;
};
Float weight_1    (const point& x) { return 1; }		//  0: 1
Float weight_x1   (const point& x) { return x[0]; }		//  1: x
Float weight_x2   (const point& x) { return x[1]; }		//  2: y
Float weight_x3   (const point& x) { return x[2]; }		//  3: z
Float weight_x1p2 (const point& x) { return x[0]*x[0]; }	//  4: x^2
Float weight_x1x2 (const point& x) { return x[0]*x[1]; }	//  5: xy
Float weight_x2p2 (const point& x) { return x[1]*x[1]; }	//  6: y^2
Float weight_x1x3 (const point& x) { return x[0]*x[2]; }	//  7: xz
Float weight_x2x3 (const point& x) { return x[1]*x[2]; }	//  8: yz
Float weight_x3p2 (const point& x) { return x[0]*x[0]; }	//  9: z^2

// cartesian

list_type weight_list[] = {
	{"1",	weight_1, 	1.},
	{"x",	weight_x1,	0.5},
	{"x2",	weight_x1p2,	Float(1)/3},
	{"y",	weight_x2,	0.5},
	{"xy",	weight_x1x2,	0.25},
	{"y2",	weight_x2p2,	Float(1)/3},
	{"z",	weight_x3,	0.5},
	{"xz",	weight_x1x3,	0.25},
	{"yz",	weight_x2x3,	0.25},
	{"z2",	weight_x3p2,	Float(1)/3}
};

// axisymmetric rz

list_type rz_weight_list[] = {
	{"1",	weight_1, 	0.5},
	{"x",	weight_x1,	Float(1)/3},
	{"x2",	weight_x1p2,	0.25},
	{"y",	weight_x2,	0.25},
	{"xy",	weight_x1x2,	Float(1)/6},
	{"y2",	weight_x2p2,	Float(1)/6},
	{"z",	weight_x3,	0.25},
	{"xz",	weight_x1x3,	Float(1)/6},
	{"yz",	weight_x2x3,	0.125},
	{"z2",	weight_x3p2,	Float(1)/6}
};
const unsigned int weight_max = sizeof(weight_list)/sizeof(list_type);

void usage()
{
      cerr << "form_mass_tst: usage: form_mass_tst"
	   << " -app approx"
	   << " [-cartesian|-rz]"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " [-weight string]"
	   << endl;
      cerr << "example:\n";
      cerr << "  form_mass_tst -app P1 -I../data carre\n";
      exit (1);
}
int main(int argc, char**argv)
{
    //
    // load geometry and options
    //
    geo omega;  
    string approx1 = "P1";
    string approx2 = "";
    string weight_id = "";
    bool mesh_done = false;
    function_type weight_function = 0;
    unsigned int weight_idx = 0;
    string coord_sys = "cartesian";

    if (argc <= 1) usage() ;

    for (int i = 1; i < argc; i++ ) {

      if (argv [i][0] == '-' && argv [i][1] == 'I')
	  append_dir_to_rheo_path (argv[i]+2) ;
      else if (strcmp(argv[i], "-app") == 0)
	  approx1 = argv[++i];
      else if (strcmp(argv[i], "-proj") == 0)
	  approx2 = argv[++i];
      else if (strcmp(argv[i], "-cartesian") == 0)
	  coord_sys = "cartesian";
      else if (strcmp(argv[i], "-rz") == 0)
	  coord_sys = "rz";
      else if (strcmp(argv[i], "-weight") == 0) {
	  weight_id = argv[++i];
          for (unsigned int i = 0; i < weight_max; i++) {
	      if (weight_id == weight_list [i].name) {
		weight_function = weight_list [i].function;
		weight_idx = i;
	      }
          }
      } else if (strcmp(argv[i], "-") == 0) {
	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! load geo on stdin" << endl ;
	  cin >> omega ;
	  omega.set_coordinate_system (coord_sys);
	  mesh_done = true ;
      } else {
	  // input geo on file
	  if (mesh_done) usage() ;
	  cerr << " ! load " << argv[i] << endl ;
	  omega = geo(argv[i], coord_sys);
	  mesh_done = true ;
      }
    }
    if (!mesh_done) usage() ;
    if (!weight_function) {
	error_macro("invalid weight identifier: " << weight_id);
    }
    if (approx2 == "") {
	approx2 = approx1;
    }
    cerr << "weight = " << weight_id << endl;
    space V1(omega, approx1);
    space V2(omega, approx2);
    field weight = interpolate (V1, weight_function);
    field one(V2,1);
    form  m(V1,V2,"mass");
    Float mes_omega = m(weight, one);
    cerr << setprecision(numeric_limits<Float>::digits10)
         << "mes(omega," << approx1 << "," << approx2 << ") = " << double(mes_omega) << endl;
    Float expect;
    if (omega.coordinate_system() == "cartesian") {
	 expect = weight_list[weight_idx].expect;
    } else {
	 expect = rz_weight_list[weight_idx].expect;
    }
    Float tol = 1e-10;

    if (fabs(mes_omega - expect) <= tol) {
        cerr << "ok" << endl;
        return 0;
    } else {
        cerr << "but it was expected " << expect << endl;
	cerr << "and error = " << fabs(mes_omega - expect) << endl;
	cerr << "and tol = " << tol << endl;
        return 1;
    }
}
