% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval-tidy.R
\name{eval_tidy}
\alias{eval_tidy}
\title{Evaluate an expression with quosures and pronoun support}
\usage{
eval_tidy(expr, data = NULL, env = caller_env())
}
\arguments{
\item{expr}{An \link[=topic-defuse]{expression} or
\link[=topic-quosure]{quosure} to evaluate.}

\item{data}{A data frame, or named list or vector. Alternatively, a
data mask created with \code{\link[=as_data_mask]{as_data_mask()}} or
\code{\link[=new_data_mask]{new_data_mask()}}. Objects in \code{data} have priority over those in
\code{env}. See the section about data masking.}

\item{env}{The environment in which to evaluate \code{expr}. This
environment is not applicable for quosures because they have
their own environments.}
}
\description{
\code{eval_tidy()} is a variant of \code{\link[base:eval]{base::eval()}} that powers the tidy
evaluation framework. Like \code{eval()} it accepts user data as
argument. Whereas \code{eval()} simply transforms the data to an
environment, \code{eval_tidy()} transforms it to a \link[=topic-data-mask]{data mask} with \code{\link[=as_data_mask]{as_data_mask()}}. Evaluating in a data
mask enables the following features:
\itemize{
\item \link[=topic-quosure]{Quosures}. Quosures are expressions bundled with
an environment. If \code{data} is supplied, objects in the data mask
always have precedence over the quosure environment, i.e. the
data masks the environment.
\item \link[=.data]{Pronouns}. If \code{data} is supplied, the \code{.env} and \code{.data}
pronouns are installed in the data mask. \code{.env} is a reference to
the calling environment and \code{.data} refers to the \code{data}
argument. These pronouns are an escape hatch for the \link[=topic-data-mask-ambiguity]{data mask ambiguity} problem.
}
}
\section{When should eval_tidy() be used instead of eval()?}{


\code{base::eval()} is sufficient for simple evaluation. Use
\code{eval_tidy()} when you'd like to support expressions referring to
the \code{.data} pronoun, or when you need to support quosures.

If you're evaluating an expression captured with
\link[=topic-inject]{injection} support, it is recommended to use
\code{eval_tidy()} because users may inject quosures.

Note that unwrapping a quosure with \code{\link[=quo_get_expr]{quo_get_expr()}} does not
guarantee that there is no quosures inside the expression. Quosures
might be unquoted anywhere in the expression tree. For instance,
the following does not work reliably in the presence of nested
quosures:

\if{html}{\out{<div class="sourceCode">}}\preformatted{my_quoting_fn <- function(x) \{
  x <- enquo(x)
  expr <- quo_get_expr(x)
  env <- quo_get_env(x)
  eval(expr, env)
\}

# Works:
my_quoting_fn(toupper(letters))

# Fails because of a nested quosure:
my_quoting_fn(toupper(!!quo(letters)))
}\if{html}{\out{</div>}}
}

\section{Stack semantics of \code{eval_tidy()}}{


\code{eval_tidy()} always evaluates in a data mask, even when \code{data} is
\code{NULL}. Because of this, it has different stack semantics than
\code{\link[base:eval]{base::eval()}}:
\itemize{
\item Lexical side effects, such as assignment with \verb{<-}, occur in the
mask rather than \code{env}.
\item Functions that require the evaluation environment to correspond
to a frame on the call stack do not work. This is why \code{return()}
called from a quosure does not work.
\item The mask environment creates a new branch in the tree
representation of backtraces (which you can visualise in a
\code{\link[=browser]{browser()}} session with \code{lobstr::cst()}).
}

See also \code{\link[=eval_bare]{eval_bare()}} for more information about these differences.
}

\examples{

# With simple defused expressions eval_tidy() works the same way as
# eval():
fruit <- "apple"
vegetable <- "potato"
expr <- quote(paste(fruit, vegetable, sep = " or "))
expr

eval(expr)
eval_tidy(expr)

# Both accept a data mask as argument:
data <- list(fruit = "banana", vegetable = "carrot")
eval(expr, data)
eval_tidy(expr, data)

# The main difference is that eval_tidy() supports quosures:
with_data <- function(data, expr) {
  quo <- enquo(expr)
  eval_tidy(quo, data)
}
with_data(NULL, fruit)
with_data(data, fruit)

# eval_tidy() installs the `.data` and `.env` pronouns to allow
# users to be explicit about variable references:
with_data(data, .data$fruit)
with_data(data, .env$fruit)
}
\seealso{
\itemize{
\item \ifelse{html}{\link[=topic-data-mask]{What is data-masking and why do I need \{\{?}}{\link[=topic-data-mask]{What is data-masking and why do I need curly-curly?}}.
\item \ifelse{html}{\link[=topic-quosure]{What are quosures and when are they needed?}}{\link[=topic-quosure]{What are quosures and when are they needed?}}.
\item \ifelse{html}{\link[=topic-defuse]{Defusing R expressions}}{\link[=topic-defuse]{Defusing R expressions}}.
\item \code{\link[=new_data_mask]{new_data_mask()}} and \code{\link[=as_data_mask]{as_data_mask()}} for manually creating data masks.
}
}
