% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_predictions.R
\name{get_predictions}
\alias{get_predictions}
\alias{get_predictions.default}
\title{S3-class definition for the ggeffects package}
\usage{
get_predictions(model, ...)

\method{get_predictions}{default}(
  model,
  data_grid = NULL,
  terms = NULL,
  ci_level = 0.95,
  type = NULL,
  typical = NULL,
  vcov = NULL,
  vcov_args = NULL,
  condition = NULL,
  interval = "confidence",
  bias_correction = FALSE,
  link_inverse = insight::link_inverse(model),
  model_info = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model, terms, ci_level, type, typical, vcov, vcov_args, condition, interval, bias_correction, verbose}{Arguments
from the call to \code{predict_response()} that are passed down to \code{get_predictions()}.
Note that \code{bias_correction} is usally already processed in \code{predict_response()}
and thus doesn't need further handling in \code{get_predictions()}, unless you
need to re-calculate the link-inverse-function (argument \code{link_inverse})
inside the \code{get_predictions()} method.}

\item{...}{Further arguments, passed to \code{predict()} or other methods used
in \code{get_predictions()}.}

\item{data_grid}{A data frame containing the data grid (or reference grid)
with all relevant values of predictors for which the adjusted predictions
should be made. Typically the data frame that is passed to the \code{newdata}
argument in \code{predict()}. A data grid can be created with functions like
\code{\link[=data_grid]{data_grid()}} or \code{\link[insight:get_datagrid]{insight::get_datagrid()}}.}

\item{link_inverse}{The model's family link-inverse function. Can be retrieved
using \code{insight::link_inverse()}.}

\item{model_info}{An object returned by \code{\link[insight:model_info]{insight::model_info()}}.}
}
\value{
A data frame that contains
\itemize{
\item the data grid (from the argument \code{data_grid})
\item the columns \code{predicted}, \code{conf.low}, and \code{conf.high}
\item optionally, the attribute \code{"std.error"} with the standard errors.
}

Note that predictions and confidence intervals should already be transformed
to the \emph{response} scale (e.g., by using \code{insight::link_inverse()}). The
\emph{standard errors} are always on the link scale (not transformed).

If values are not available (for example, confidence intervals), set their
value to \code{NA}.
}
\description{
\code{get_predictions()} is the core function to return adjusted
predictions for a model, when calling \code{ggpredict()} or \code{predict_response()}
with \code{margin = "mean_reference"} (the default option for \code{margin}).
Basically, the input contains the model object and a data grid that is
typically used for the \code{newdata} argument of the \code{predict()} method.
\code{get_predictions()} can be used as S3-method for own classes, to add support
for new models in \strong{ggeffects} and is only relevant for package developers.

There are no S3-class definitions for \code{ggemmeans()} or \code{ggaverage()}, because
these functions simply call methods from the \strong{emmeans} or \strong{marginaleffects}
packages. Hence, methods should be written for those packages, too, if a
model-object should work with \code{ggemmeans()} or \code{ggaverage()}.
}
\details{
Adding support for \strong{ggeffects} is quite easy. The user-level function is
\code{predict_response()}, which either calls \code{ggpredict()}, \code{ggemmeans()} or
\code{ggaverage()}. These function, in turn, call \code{predict()}, \code{emmeans::emmeans()}
or \code{marginaleffects::avg_predictions()}. Following needs to be done to add
support for new model classes:
\itemize{
\item \strong{emmeans}: if your model is supported by emmeans, it is automatically
supported by \code{ggemmeans()}. Thus, you need to add the corresponding methods
to your package so that your model class is supported by **emmeans.
\item \strong{marginaleffects}: similar to \strong{emmeans}, if your package is supported
by the \strong{marginaleffects} package, it works with \code{ggaverage()}.
\item \strong{predict}: in order to make your model class work with \code{ggpredict()},
you need to add a \code{get_predictions()} method. The here documented arguments
are \emph{all} passed from \code{predict_response()} to \code{get_predictions()}, no
matter if they are required to calculate predictions or not. Thus, it is
not necessary to process all of those arguments, but they can be used to
modulate certain settings when calculating predictions. Note that if your
method does not define all mentioned arguments, these are still passed via
\code{...} - make sure that further methods in your \code{get_predictions()} method
still work when they process the \code{...}. It is important that the function
returns a data frame with a specific structure, namely the data grid and
the columns \code{predicted}, \code{conf.low}, and \code{conf.high}. Predictions and
intervals should be on the response scale.
}

A simple example for an own class-implementation for Gaussian-alike models
could look like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{get_predictions.own_class <- function(model, data_grid, ci_level = 0.95, ...) \{
  predictions <- predict(
    model,
    newdata = data_grid,
    type = "response",
    se.fit = !is.na(ci_level),
    ...
  )

  # do we have standard errors?
  if (is.na(ci_level)) \{
    # copy predictions
    data_grid$predicted <- as.vector(predictions)
  \} else \{
    # copy predictions
    data_grid$predicted <- predictions$fit

    # calculate CI
    data_grid$conf.low <- predictions$fit - qnorm(0.975) * predictions$se.fit
    data_grid$conf.high <- predictions$fit + qnorm(0.975) * predictions$se.fit

    # optional: copy standard errors
    attr(data_grid, "std.error") <- predictions$se.fit
  \}

  data_grid
\}
}\if{html}{\out{</div>}}

A simple example for an own class-implementation for non-Gaussian-alike models
could look like this (note the use of the link-inverse function \code{link_inverse()},
which is passed to the \code{link_inverse} argument):

\if{html}{\out{<div class="sourceCode">}}\preformatted{get_predictions.own_class <- function(model,
                                      data_grid,
                                      ci_level = 0.95,
                                      link_inverse = insight::link_inverse(model),
                                      ...) \{
  predictions <- predict(
    model,
    newdata = data_grid,
    type = "link", # for non-Gaussian, return on link-scale
    se.fit = !is.na(ci_level),
    ...
  )

  # do we have standard errors?
  if (is.na(ci_level)) \{
    # copy predictions
    data_grid$predicted <- link_inverse(as.vector(predictions))
  \} else \{
    # copy predictions, use link-inverse to back-transform
    data_grid$predicted <- link_inverse(predictions$fit)

    # calculate CI
    data_grid$conf.low <- link_inverse(
      predictions$fit - qnorm(0.975) * predictions$se.fit
    )
    data_grid$conf.high <- link_inverse(
      predictions$fit + qnorm(0.975) * predictions$se.fit
    )

    # optional: copy standard errors
    attr(data_grid, "std.error") <- predictions$se.fit
  \}

  data_grid
\}
}\if{html}{\out{</div>}}
}
