/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "statuseditwgt.h"
#include "globals.h"

#include <QAction>
#include "qchaticon.h"

StatusEditWgt::StatusEditWgt(QWidget *parent)
 : QWidget(parent)
{
  m_grid             = new QGridLayout(this);
  m_editStatusBtn    = new QToolButton(this);
  m_statusEdit       = new QLineEdit  (this);
  m_statusCmbx       = new QComboBox  (this);
  m_descriptionsMenu = new QMenu(this);

  m_grid->addWidget(m_editStatusBtn   , 0, 0);
  m_grid->addWidget(m_statusCmbx      , 0, 1);
  m_grid->addWidget(m_statusEdit      , 1, 0, 1, 2);

  m_editStatusBtn->setMenu(m_descriptionsMenu);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-ready-for-chat"),
  Globals::StatusStr[Globals::READY4CHAT], Globals::READY4CHAT);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-online"),
  Globals::StatusStr[Globals::FREE]      , Globals::FREE);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-busy"),
  Globals::StatusStr[Globals::BUSY]      , Globals::BUSY);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-dnd"),
  Globals::StatusStr[Globals::DND]       , Globals::DND);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-away"),
  Globals::StatusStr[Globals::INACTIVE]  , Globals::INACTIVE);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-away-extended"),
  Globals::StatusStr[Globals::AWAY]      , Globals::AWAY);

  m_statusCmbx->addItem(QChatIcon::icon("status/user-invisible"),
  Globals::StatusStr[Globals::INVISIBLE] , Globals::INVISIBLE);

  m_grid->setMargin(0);

  m_statusCmbx->setMaximumHeight(25);
  m_statusCmbx->setCurrentIndex(1);

  m_statusEdit->setMaximumHeight(25);
  m_statusEdit->hide();

  m_editStatusBtn->setIcon(QChatIcon::icon("im-status-message-edit"));

  connect(m_statusEdit   , SIGNAL(textChanged(QString))    , this, SLOT(slot_setStatusDescription(QString)));

  connect(m_editStatusBtn, SIGNAL(pressed())               , this, SLOT(slot_editStatusClicked()));
  connect(m_statusCmbx   , SIGNAL(currentIndexChanged(int)), this, SLOT(slot_statusChanged ()));

  connect(m_descriptionsMenu, SIGNAL(triggered(QAction*)), this, SLOT(setDescription(QAction*)));
  connect(m_descriptionsMenu, SIGNAL(aboutToHide())      , this, SLOT(changeStatus()));

  connect(m_statusEdit   , SIGNAL(returnPressed()), this, SLOT(hideEditor()));

  setMaximumHeight(25);

  retranslate();
}

void StatusEditWgt::retranslate()
{
  m_editStatusBtn->setToolTip(tr("Edit status description"));
}

void StatusEditWgt::slot_statusChanged()
{
  if(status() != Globals::INVISIBLE)
    m_statusEdit->setText(QChatSettings::settings()->statusDescription(status()));

  m_statusEdit->setReadOnly(status() == Globals::INVISIBLE);

  if(!m_statusEdit->isHidden())
    m_statusEdit->setText(QChatSettings::settings()->statusDescription(status()));

  showDescriptionsMenu();
}

void StatusEditWgt::showDescriptionsMenu()
{
  m_descriptionsMenu->clear();

  m_descriptionsMenu->addAction(tr("New Message..."))->setData(0);

  m_descriptionsMenu->addAction(tr("Empty Message")) ->setData(1);

  if(!QChatSettings::settings()->nowListening().isEmpty())
    m_descriptionsMenu->addAction(tr("Now Listening")) ->setData(2);

  QString s = QChatSettings::settings()->statusDescription(status());
  if(qobject_cast<QToolButton*>(sender()) && !s.isEmpty())
    m_descriptionsMenu->addAction(tr("Remove Description (%1)").arg(s))->setData(4);

  m_descriptionsMenu->addSeparator();

  foreach(QString s, QChatSettings::settings()->statusDescriptions())
    m_descriptionsMenu->addAction(s)->setData(3);

  emit editing();

  if(qobject_cast<QComboBox*>(sender()))
    m_descriptionsMenu->popup(QCursor::pos() + QPoint(1, 1));
  else
    m_editStatusBtn->showMenu();
}

void StatusEditWgt::setDescription(QAction* act)
{
  switch(act->data().toInt())
  {
    case 0 :
      m_lastText = QChatSettings::settings()->statusDescription(status());

      m_statusEdit->setText(m_lastText);
      m_statusEdit->show();
      m_statusEdit->setFocus();
      m_statusEdit->selectAll();

      setMaximumHeight(55);
      setMinimumHeight(55);

      emit editing();
      break;

    case 1 :
      QChatSettings::settings()->setStatusDescription("", status());
      m_description = "";

      emit statusChanged();
      break;

    case 2 :
      m_lastText = QChatSettings::settings()->nowListening();
      m_description = m_lastText;

      emit statusChanged();
      break;

    case 3 :
      m_lastText = act->text();

      QChatSettings::settings()->setStatusDescription(m_lastText, status());
      QChatSettings::settings()->removeStatusDescription(m_lastText);
      QChatSettings::settings()->appendStatusDescription(m_lastText);

      m_description = m_lastText;

      emit statusChanged();
      break;

    case 4 :
      QChatSettings::settings()->removeStatusDescription(QChatSettings::settings()->statusDescription(status()));

      if(QChatSettings::settings()->statusDescriptions().size())
        m_description = QChatSettings::settings()->statusDescriptions()[0];
      else
        m_description = "";

      QChatSettings::settings()->setStatusDescription(m_description, status());

      break;
  }
}

void StatusEditWgt::hideEditor()
{
  m_statusEdit->hide();

  setMaximumHeight(25);
  setMinimumHeight(25);

  changeStatus();
}

void StatusEditWgt::slot_editStatusClicked()
{
  if(m_statusEdit->isHidden())
    showDescriptionsMenu();
  else
    hideEditor();
}

void StatusEditWgt::changeStatus()
{
  if(m_lastText != m_statusEdit->text() && !qobject_cast<QMenu*>(sender()))
    QChatSettings::settings()->appendStatusDescription(m_statusEdit->text());

  if(m_statusEdit->isHidden())
    emit statusChanged();
}
