#!/usr/bin/python

"""Read power meters available via sysfs."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import discovery
import pwrkap_data
import datetime
import os
import dircache
import util

class sysfs_energy_meter(pwrkap_data.energy_meter):
	"""Driver for energy meters available via sysfs."""

	def __init__(self, sensor_filename):
		self.sensor_filename = sensor_filename
		self.latency = None
		self.initial_value = self.__read()

	def __getstate__(self):
		state = {}
		for key in self.__dict__.keys():
			value = self.__dict__[key]
			state[key] = value
		del state["initial_value"]
		return state

	def __setstate__(self, state):
		self.__dict__ = state
		self.initial_value = self.__read()

	def get_chip_name(self):
		"""Determine the name of the chip that controls this sensor."""
		last_slash = self.sensor_filename.rfind("/")
		if last_slash == -1:
			return None
		base_dir = self.sensor_filename[:last_slash + 1]
		base_dir = base_dir + "/name"
		x = util.read_line_as_array(base_dir)
		if x == None:
			return None
		return x[0]

	def __read(self):
		"""Read the energy meter."""
		x = util.read_line_as_array(self.sensor_filename)
		if x == None:
			return None
		return float(x[0]) / 1000000

	def read(self):
		try:
			before = datetime.datetime.utcnow()
			return self.__read() - self.initial_value
		finally:
			after = datetime.datetime.utcnow()
			if self.latency == None:
				self.latency = (after - before)
			else:
				self.latency = (8 * self.latency + 2 * (after - before)) / 10

	def get_latency(self):
		return self.latency

	def inventory(self):
		return ("sysfsmeter", {
			"name": self.sensor_filename, \
			"chip": self.get_chip_name(), \
			})

class sysfs_power_meter(pwrkap_data.power_meter):
	"""Driver for power meters available via sysfs."""

	def __init__(self, sensor_filename):
		self.sensor_filename = sensor_filename
		self.latency = None

	def get_chip_name(self):
		"""Determine the name of the chip that controls this sensor."""
		last_slash = self.sensor_filename.rfind("/")
		if last_slash == -1:
			return None
		base_dir = self.sensor_filename[:last_slash + 1]
		base_dir = base_dir + "/name"
		x = util.read_line_as_array(base_dir)
		if x == None:
			return None
		return x[0]

	def read(self):
		try:
			before = datetime.datetime.utcnow()
			x = util.read_line_as_array(self.sensor_filename)
			if x == None:
				return None
			return float(x[0]) / 1000000
		finally:
			after = datetime.datetime.utcnow()
			if self.latency == None:
				self.latency = (after - before)
			else:
				self.latency = (8 * self.latency + 2 * (after - before)) / 10

	def get_latency(self):
		return self.latency

	def inventory(self):
		return ("sysfsmeter", {
			"name": self.sensor_filename, \
			"chip": self.get_chip_name(), \
			})

SYSFS_HWMON_DIR = "/sys/class/hwmon/"
def sysfs_meter_discover():
	"""Discover sysfs meters."""
	global SYSFS_HWMON_DIR

	# This will load ibmpex/ibmaem drivers
	os.system("modprobe -q hwmon")
	os.system("modprobe -q ipmi-si")
	os.system("modprobe -q ibmpex")
	os.system("modprobe -q ibmaem")

	# Now look for /sys/class/hwmon/hwmon*/device/power*_input
	for hwmon_dev in dircache.listdir(SYSFS_HWMON_DIR):
		if not hwmon_dev.startswith("hwmon"):
			continue
		hwmon_dev_dir = SYSFS_HWMON_DIR + hwmon_dev + "/device/"
		for sensor_dev in dircache.listdir(hwmon_dev_dir):
			if sensor_dev.startswith("power") \
			and (sensor_dev.endswith("_input") or sensor_dev.endswith("_average")):
				meter = sysfs_power_meter(hwmon_dev_dir + sensor_dev)
				discovery.PWRKAP_POWER_METERS.append(meter)
			if sensor_dev.startswith("energy") \
			and sensor_dev.endswith("_input"):
				meter = sysfs_energy_meter(hwmon_dev_dir + sensor_dev)
				discovery.PWRKAP_ENERGY_METERS.append(meter)

def sysfs_meter_init():
	"""Set up sysfs meter discovery functions."""
	discovery.PWRKAP_METER_DISCOVERY.append(sysfs_meter_discover)

	return True

sysfs_meter_init()
