/*
 * Copyright (C) 2000-2004 by Oswald Buddenhagen <puf@ossi.cjb.net>
 * based on puf 0.1.x (C) 1999,2000 by Anders Gavare <gavare@hotmail.com>
 *
 * You may modify and distribute this code under the terms of the GPL.
 * There is NO WARRANTY of any kind. See COPYING for details.
 *
 * getopts.c - command line parsing
 *
 */

#include "puf.h"

enum { O_ST_I, O_ST_O, O_ST_CI, O_RF_S, 
       O_LST_I, O_LST_O, O_LST_CI, O_LRF_S, 
       O_ACC, O_REJ, O_DOM,
       O_HELP, O_DISP, O_DPATH, O_URLF, O_ASTR, O_THROT, O_STAMP,
       O_PRX, O_PRXF, O_PPRX, O_SPPRX, O_BIND, O_BINDF, O_AGENT, O_AGENTF };

static const char *onams[] = { 
    "NR", "NR", "", "STR", 
    "NR", "NR", "", "STR", 
    "STR", "STR", "STR",
    "", "FILE", "DIR", "FILE", "STR", "NR", "DATE",
    "PRX", "FILE", "PRX", "PRX", "IP", "FILE", "STR", "FILE" };

/* Some compilers, like SunOS4 cc, don't have offsetof in <stddef.h>.  */
#ifndef offsetof
# define offsetof(type,ident) ((size_t)&(((type*)0)->ident))
#endif
#define uo(o) ((void*)offsetof(options_t, o))

#define stringify(s)    tostring(s)
#define tostring(s)     #s

/* *INDENT-OFF* */
static struct {                      
    const char *opt;
    int todo;
    void *argptr;
    int argval;
    int defargval;
    const char *desc;
} options[] = {
    {0, -1, 0, 0, 0, 
"\nURL format: [http://][user:pass@]host[.domain][:port][/path]\n"
"\nAll options except those marked as global have effect only on the following\n"
"URLs. Their effect can be cancelled by specifying <original option>- without\n"
"any parameters possibly required by the original option, or by overriding them\n"
"with another option with an opposite effect. All URL-local options can be\n"
"reverted to their default state by specifying a single comma as an argument.\n"
"The scope of URL-local options can be limited by enclosing portions of the\n"
"command line in brackets.\n"
	"\nWhat to download:"},
    {"p", O_LST_CI, uo(follow_src), SAMEDIR_RECURSIVE, NOT_RECURSIVE, "Download page requisites from same directory"},
    {"pr", O_LST_CI, uo(follow_src), SUBDIR_RECURSIVE, NOT_RECURSIVE, "Download page requisites also from subdirectories"},
    {"pr+", O_LST_CI, uo(follow_src), HOST_RECURSIVE, NOT_RECURSIVE, "Download page requisites from anywhere one the server"},
    {"pr++", O_LST_CI, uo(follow_src), GLOBAL_RECURSIVE, NOT_RECURSIVE, "Download page requisites from anywhere on the internet"},
    {"r", O_LST_CI, uo(follow_href), SUBDIR_RECURSIVE, NOT_RECURSIVE, "Recurse download into subdirectories (ups -p to -pr)"},
    {"r+", O_LST_CI, uo(follow_href), HOST_RECURSIVE, NOT_RECURSIVE, "Recurse download across whole server (ups -p to -pr+)"},
    {"r++", O_LST_CI, uo(follow_href), GLOBAL_RECURSIVE, NOT_RECURSIVE, "Recurse download across whole internet (ups -p to -pr++; caution!)"},
    {"A", O_ACC, uo(filter_list), 0, 0, "Accept file extension/pattern/mimetype STR (default: all)"},
    {"R", O_REJ, uo(filter_list), 0, 0, "Reject file extension/pattern/mimetype STR (default: none)"},
    {"D", O_DOM, 0, 0, 0, "Accept additional domain STR (with -r+)"},
    {"Dl", O_ASTR, uo(ldom_list), 0, 0, "Accept domain STR when following links (with -r+)"},
    {"Dr", O_ASTR, uo(rdom_list), 0, 0, "Accept domain STR when getting requisites (with -pr+)"},
    {"ld", O_LST_I, uo(max_depth), 0, -1, "Limit directory nesting level to NR (with -r)"},
    {"l", O_LST_I, uo(max_recurse), 0, 0, "Limit recursion depth to NR (with -r, -r+ & -r++)"},
    {"lb", O_LST_O, uo(max_bytes), 0, 0, "Download only first NR bytes of every file"},
    {"xs", O_LST_O, uo(buff_size), 0, 0, "Set writeout buffer size to NR bytes (default is " stringify(DEFAULT_MAX_BUFFER) "MB for multi-src)"},
    {"xg", O_LST_CI, uo(inhibit_cgiget), -1, 0, "Allow recursion into URLs with ? signs (i.e., CGIs)"},
    {"ng", O_LST_CI, uo(inhibit_cgiget), 1, 0, "Disallow ?-URLs, even if given on the command line"},
    {"F", O_LST_CI, uo(force_html), 1, 0, "Treat all files as HTML (scan for links)"},
    {"B", O_LRF_S, uo(url_prefix), 0, 0, "Prefix to add to every URL on the command line"},
    {"i", O_URLF, 0, 0, 0, "Read switches and URLs from FILE"},
    {0, -1, 0, 0, 0, "\nWhat to to with existing files:"},
    {"u", O_LST_CI, uo(update_mode), EX_UPDATE, EX_CLOBBER, "Update existing "/* "and delete obsolete "*/"files, continue partial"},
    {"c", O_LST_CI, uo(update_mode), EX_CONTINUE, EX_CLOBBER, "Continue download of partial files"},
    {"nc", O_LST_CI, uo(update_mode), EX_NO_CLOBBER, EX_CLOBBER, "Don't clobber existing files"},
    {0, -1, 0, 0, 0, "\nStorage of files:"},
    {"na", O_ST_CI, &always_primary_name, 1, 0, "Don't use hostname aliases for directory names (global)"},
    {"nd", O_LST_CI, uo(dir_mode), DIRS_NONE, DIRS_NORMAL, "Don't create subdirectories"},
    {"xd", O_LST_CI, uo(dir_mode), DIRS_ALWAYS, DIRS_NORMAL, "Create all subdirectories (default for -r+ & -r++)"},
    {"O", O_DISP, 0, 0, 0, "Dump files to FILE; \"-\" means stdout"},
    {"xO", O_LST_CI, uo(ext_dump), 2, 0, "Dump in a rfc822-style format (depends on -xE/-O)"},
    {"xo", O_LST_CI, uo(ext_dump), 1, 0, "Like -xO, but don't report errors from recursed URLs"},
    {"P", O_DPATH, 0, 0, 0, "Save files in directory DIR/"},
    {"xi", O_LRF_S, uo(index_filename), 0, 0, "Set the name for anonymous index files (default is " DEFAULT_INDEX_FILE_NAME ")"},
    {"xE", O_LST_CI, uo(enumerate_urls), 1, 0, "Enumerate files in command line order. Implies -nd"},
#ifndef HAVE_CYGWIN
    {"xq", O_LST_CI, uo(fat_quotes), 1, 0, "Quote file names suitably for storage on FAT file systems"},
#endif
    {"nt", O_LST_CI, uo(no_touch), 1, 0, "Don't timestamp files according to server response"},
    {"nb", O_LST_CI, uo(delete_broken), 1, 0, "Delete partial files from broken downloads"},
    {"xh", O_ASTR, uo(save_headers), 1, 0, "Save HTTP headers starting with STR (use \"\" or \"*\" for all)"},
    {"xr", O_LST_CI, uo(dump_refs), 1, 0, "Dump HTML references (as special headers)"},
    {0, -1, 0, 0, 0, "\nNetwork options:"},
    {"ni", O_LST_CI, uo(send_if_range), 0, 1, "Don't send \"If-Range:\" (assume up-to-date partial files)"},
    {"xu", O_STAMP, 0, 0, 0, "Send If-Modified-Since:/If-Range: DATE"},
    {"nR", O_LST_CI, uo(send_referer), 0, 1, "Don't send \"Referer:\""},
    {"U", O_AGENT, uo(user_agents), 0, 0, "Send \"User-Agent: STR\" (use \"\" for none)"},
    {"iU", O_AGENTF, uo(user_agents), 0, 0, "Send random User-Agent:s. FILE format: ratio name"},
    {"xH", O_ASTR, uo(aux_headers), 0, 0, "Add arbitrary header STR to HTTP requests"},
    {"Tl", O_ST_I, &timeout_dns, 0, 0, "Set DNS lookup timeout to NR seconds (global; default is " stringify(DEFAULT_TIMEOUT_DNS) ")"},
    {"Tc", O_LST_I, uo(timeout_connect), 0, DEFAULT_TIMEOUT_CONNECT, "Set connect timeout to NR seconds (default is " stringify(DEFAULT_TIMEOUT_CONNECT) ")"},
    {"Td", O_LST_I, uo(timeout_data), 0, DEFAULT_TIMEOUT_DATA, "Set data timeout to NR seconds (default is " stringify(DEFAULT_TIMEOUT_DATA) ")"},
    {"t", O_LST_I, uo(max_attempts), 0, DEFAULT_MAX_ATTEMPTS, "Set maximum number of download attempts per URL (default is " stringify(DEFAULT_MAX_ATTEMPTS) ")"},
    {"nw", O_LST_CI, uo(fail_no_wait), 1, 0, "Don't wait before reconnecting a busy/dead host"},
    {"xT", O_LST_CI, uo(http_err_trans), 1, 0, "Treat HTTP errors 403 and 404 as transient"},
    {"xb", O_BIND, uo(bind_addrs), 0, 0, "Bind outgoing connections to IP"},
    {"ib", O_BINDF, uo(bind_addrs), 0, 0, "Bind outgoing connections to random IPs from FILE"},
    {"y", O_PRX, uo(proxies), 0, 0, "Use proxy PRX. Multiple -y's are allowed"},
    {"yi", O_PRXF, uo(proxies), 0, 0, "Read proxies from FILE. PRX format: URL[*ratio]"},
    {"xy", O_PPRX, 0, 0, 0, "Prefer proxy PRX from -y list"},
    {"xyy", O_SPPRX, 0, 0, 0, "Use only proxy PRX from -y list"},
    {0, -1, 0, 0, 0, "\nResource usage quotas (global):"},
    {"Q", O_ST_O, &max_bytes, 0, 0, "Abort puf after NR bytes (unlimited by default)"},
    {"Qu", O_ST_I, &max_urls, 0, 0, "Abort puf after NR URLs (unlimited by default)"},
    {"Qt", O_ST_I, &max_time, 0, 0, "Abort puf after NR seconds (unlimited by default)"},
    {"lc", O_ST_I, &max_urls_active, 0, 0, "Max NR simultaneous connections (default is " stringify(DEFAULT_MAX_ACTIVE) ")"},
    {"ll", O_ST_I, &max_dnss_active, 0, 0, "Max NR simultaneous DNS lookups (default is " stringify(DEFAULT_MAX_DNS_FORKS) ")"},
    {"nf", O_ST_CI, &economize_files, 1, 0, "Use fewer file descriptors. Slightly slower"},
    {"nh", O_ST_CI, &economize_dns, 1, 0, "Do fewer DNS lookups. May miss some references"},
    {"dc", O_THROT, 0, 0, 0, "Delay consecutive connects by NR milliseconds"},
    {0, -1, 0, 0, 0, "\nLogging (global):"},
    {"ns", O_ST_CI, &show_stat, 0, 0, "Disable download progress statistics"},
    {"v", O_ST_CI, &verbose, ERR, 0, "Be verbose (show errors). Implies -ns"},
    {"vv", O_ST_CI, &verbose, WRN, 0, "Be very verbose (show warnings). Implies -v"},
    {"vvv", O_ST_CI, &verbose, NFO, 0, "Be extremely verbose (show infos). Implies -vv"},
    {"d", O_ST_I, &debug, 0, 0, "Debug: URL=1 DNS=2 QUE=4 CON=8 HDR=16 REF=32 MEM=64"},
    {"h", O_HELP, 0, 0, 0, "This help screen"},
    {0, -1, 0, 0, 0, "\nExamples:\n"
	"puf -P stuff -r+ www.foo.com -r www.bar.com -r- www.some.org , www.blub.de\n"
	"puf [ -r+ -xg www.foo.com ] -P /var/tmp/dl -r -u www.bar.com"},
};
/* *INDENT-ON* */

/*  Did user specify URL?  */
static int tried_url;

static void 
cdie(int num, const char *argfn, const char *msg, ...)
{
    va_list va;
    unsigned i, j;
    char fmt[SHORTSTR], qargfn[SHORTSTR];

    if (argfn) {
	for (i = j = 0; argfn[i] && j < sizeof(qargfn) - 1; i++) {
	    if (argfn[i] == '%')
		qargfn[j++] = '%';
	    qargfn[j++] = argfn[i];
	}
	qargfn[j] = 0;
	snprintf(fmt, SHORTSTR, "\n%s: %s (%s:%d)\n", progname, msg, qargfn, num);
    } else
	snprintf(fmt, SHORTSTR, "\n%s: %s (arg %d)\n", progname, msg, num);
    va_start(va, msg);
    vfprintf(stderr, fmt, va);
    va_end(va);
    exit(2);
}

static void *
irealloc (void *ptr, size_t size)
{
    void *ret;
    if (!(ret = realloc(ptr, size)))
	die(2, "out of memory.");
    return ret;
}

static void *
imalloc (size_t size)
{
    return irealloc(0, size);
}

static char *
istrdup(const char *str)
{
    char *ret;
    if (!(ret = strdup(str)))
	die(2, "out of memory.");
    return ret;
}

static void
init_ptrarr(ptrarr_t **arr)
{
    *arr = imalloc(sizeof(**arr));
    memset(*arr, 0, sizeof(**arr));
}

static void 
clr_ptrarr(ptrarr_t **arr)
{
    if ((*arr)->cow)
	init_ptrarr(arr);
    else {
	(*arr)->nents = 0;
	(*arr)->spare = 0;
    }
}

static void 
dext_arr(ptrarr_t **arr, size_t sz)
{
    ptrarr_t *narr;
    int nrents;

    if ((*arr)->cow) {
	nrents = (*arr)->nents * 2 + 1;
	narr = imalloc(sizeof(*narr) + nrents * sz);
	narr->rents = nrents;
	narr->spare = (*arr)->spare;
	narr->cow = 0;
	narr->nents = (*arr)->nents;
	memcpy(narr->ents, (*arr)->ents, narr->nents * sz);
	*arr = narr;
    } else {
	if ((*arr)->nents == (*arr)->rents) {
	    (*arr)->rents = (*arr)->rents * 2 + 1;
	    *arr = irealloc(*arr, sizeof(**arr) + (*arr)->rents * sz);
	}
    }
}

static void 
dext_proxyarr(proxyarr_t **arr, proxy_t *prox, int ratio)
{
    proxyent_t *pe;

    dext_arr((ptrarr_t **)arr, sizeof(*pe));
    pe = (*arr)->ents + (*arr)->nents++;
    pe->proxy = prox;
    pe->ratio = ratio;
    pe->score = 0;
/*    pe->cur_conn = 0;*/
}

static void 
dext_ptrarr(ptrarr_t **arr, void *data)
{
    dext_arr(arr, sizeof(void *));
    (*arr)->ents[(*arr)->nents++] = data;
}

static void *
ext_ptrarr(ptrarr_t **arr, int size)
{
    void *data = imalloc(size);
    dext_ptrarr(arr, data);
    return data;
}

static options_t *lopt;
static url_parm_t *lparm;

static void 
initlopt(void)
{
    u_int i;

    lopt = imalloc(sizeof(*lopt));
    memset (lopt, 0, sizeof(*lopt));
    for (i = 0; i < sizeof(options) / sizeof(options[0]); i++)
	switch (options[i].todo) {
	    case O_LST_CI:
	    case O_LST_I:
		*(int *)((char *)lopt + (size_t)options[i].argptr) = 
		    options[i].defargval;
		break;
	}
    init_ptrarr(&lopt->user_agents);
    init_ptrarr(&lopt->aux_headers);
    init_ptrarr(&lopt->save_headers);
    init_ptrarr(&lopt->bind_addrs);
    init_ptrarr((ptrarr_t **)&lopt->proxies);
    init_ptrarr(&lopt->filter_list);
    init_ptrarr(&lopt->ldom_list);
    init_ptrarr(&lopt->rdom_list);

    lparm = imalloc(sizeof(*lparm));
    memset(lparm, 0, sizeof(*lparm));
    lparm->opt = lopt;
}

static void
do_idetach_parm(void)
{
    url_parm_t *parm;

    parm = imalloc(sizeof(*parm));
    memcpy(parm, lparm, sizeof(*parm));
    lparm = parm;
    lparm->ref_count = 0;
    lparm->opt = lopt;
    if (lparm->disposition)
	lparm->disposition->multi = 1;
}


static void 
do_idetach_opt(void)
{
    options_t *opt;

    opt = imalloc(sizeof(*opt));
    memcpy(opt, lopt, sizeof(*opt));
    lopt = opt;
    lopt->user_agents->cow = 1;
    lopt->aux_headers->cow = 1;
    lopt->save_headers->cow = 1;
    lopt->bind_addrs->cow = 1;
    lopt->proxies->cow = 1;
    lopt->filter_list->cow = 1;
    lopt->ldom_list->cow = 1;
    lopt->rdom_list->cow = 1;
    lopt->cow = 0;
    do_idetach_parm();
}

static void 
idetach_opt(void)
{
    if (lopt->cow)
	do_idetach_opt();
}

static void
idetach_parm(void)
{
    if (lparm->ref_count)
	do_idetach_parm();
}

static void 
set_dpath(const char *path)
{
    int len;

    len = strlen(path) + 1;
    lopt->disp_path = imalloc(sizeof(*lopt->disp_path) + len);
    lopt->disp_path->file_num = 0;
    memcpy (&lopt->disp_path->path, path, len);
}

static void
set_disp(url_parm_t *parm, const char *dbuf, int len)
{
  if (len == 9 && !memcmp(dbuf, "/dev/null", 9)) {
    parm->disposition = imalloc(sizeof(disp_t) + 1);
    parm->disposition->devnull = 1;
    parm->disposition->multi = parm->disposition->created = 0;
    parm->disposition->disp[0] = 0;
  } else {
    parm->disposition = imalloc(sizeof(disp_t) + len + 1);
    parm->disposition->devnull =
    parm->disposition->multi = parm->disposition->created = 0;
    memcpy(parm->disposition->disp, dbuf, len + 1);
  }
}

static void 
adden(const char *srct, const char *url)
{
    proxy_t *prox;
    char *ptr;
    url_parm_t *parm;
    unsigned dplen;
    char buf[SHORTSTR], dbuf[20];

    if (!lopt->disp_path)
	set_dpath("");

    if (!lopt->proxies->nents && (ptr = getenv("http_proxy")) &&
	(prox = parse_add_proxy("$http_proxy", ptr)))
	dext_proxyarr(&lopt->proxies, prox, 100);

    if (lopt->follow_src != NOT_RECURSIVE &&
	lopt->follow_src < lopt->follow_href)
	lopt->follow_src = lopt->follow_href;
    lopt->follows_max = lopt->follow_src > lopt->follow_href ?
			lopt->follow_src : lopt->follow_href;

    if (lopt->follow_href != SUBDIR_RECURSIVE && lopt->max_depth >= 0)
	die(2, "-ld works only with -r.");

    if (lopt->ldom_list->nents) {
	if (lopt->ldom_list->nents == lopt->rdom_list->nents &&
	    !memcmp(lopt->ldom_list->ents, lopt->rdom_list->ents,
		    sizeof(char *) * lopt->rdom_list->nents))
	{
	    if (lopt->follow_href != HOST_RECURSIVE &&
		lopt->follow_src != HOST_RECURSIVE)
		die(2, "-D works only with -r+/-pr+.");	
	    goto donedom;
	}
	if (lopt->follow_href != HOST_RECURSIVE)
	    die(2, "-Dl works only with -r+.");	
    }
    if (lopt->rdom_list->nents && lopt->follow_src != HOST_RECURSIVE)
	die(2, "-Dr works only with -pr+.");	
  donedom:

    parm = lparm;

    if (lopt->enumerate_urls) {
	if (lparm->disposition)
	    die(2, "-xE and -O are mutually exclusive.");
	/* note that this will put a whole recursive download in one file */
	do_idetach_parm();
	set_disp(parm, dbuf, sprintf(dbuf, "%d.puf",
					   ++lopt->disp_path->file_num));
    }

    dplen = lopt->disp_path->path[0] ? strlen(lopt->disp_path->path) + 1 : 0;
    if (parm->disposition) {
	    if (dplen + strlen(parm->disposition->disp) >
		SHORTSTR - sizeof(PART_EXT))
		die(2, "user-supplied disposition (-P + -O/-xE) too long.");
    } else {
	if (dplen > SHORTSTR - sizeof(PART_EXT))
	    die(2, "disposition path (-P) too long.");
    }

    if (parm->disposition) {
	if (lopt->follows_max != NOT_RECURSIVE || parm->ref_count)
	    parm->disposition->multi = 1;
	if (parm->disposition->multi) {
	    if (lopt->update_mode == EX_CONTINUE ||
		lopt->update_mode == EX_UPDATE)
		die(2, "-c/-u and multi-source -O/-xE are mutually exclusive.");
	    if ((lopt->save_headers->nents || lopt->dump_refs) &&
		!lopt->ext_dump)
		die(2, "-xh/-xr and multi-source -O/-xE without -xo/-xO "
		       "are mutually exclusive.");
	}
	if (!parm->disposition->disp[0]) {
	    /*  Could warn about -c/-u/-nc here ...  */
	    if ((lopt->save_headers->nents || lopt->dump_refs) &&
		!lopt->ext_dump)
		die(2, "-xh/-xr and '-O -' without -xo/-xO "
		       "are mutually exclusive.");
	    show_stat = 0;
	}
    } else {
	if (lopt->ext_dump)
	    die(2, "-xo/-xO require -O/-xE.");
    }

    snprintf(buf, SHORTSTR, "%s%s",
	     lopt->url_prefix ? lopt->url_prefix : "", url);
    dbg(URL, ("Trying URL '%s' from %s\n", buf, srct));
    if (!parse_add_url(srct, buf, strlen(buf), 0, 0, parm, 0, 0, 0, 0)) {
	prx(ERR, "Invalid URL '%s'.\n", buf);
	write_psts(parm, buf, INT_MAX, 1, 450);
	/* possibly leak the parm */
    } else
	lopt->cow = 1;
    tried_url = 1;
}

static void 
prx_adden(char *proxy, int num, const char *argfn)
{
    proxy_t *prox;
    char *ptr;
    int ratio;

    if ((ptr = strchr(proxy, '*'))) {
	*ptr = '\0';
	ratio = atoi(ptr + 1);
	if (ratio <= 0)
	    cdie(num, argfn, "invalid load ratio '%s'", ptr + 1);
    } else
	ratio = 100;

    if (!(prox = parse_add_proxy(argfn ? argfn : "command line", proxy)))
	cdie(num, argfn, "invalid proxy specification '%s'", proxy);
    else
	dext_proxyarr(&lopt->proxies, prox, ratio);
}

static void 
add_bind_ip(const char *ip, int num, const char *argfn)
{
    int s;

    if ((bind_addr.sin_addr.s_addr = inet_addr(ip)) == (unsigned)-1)
	cdie(num, argfn, "'%s' is not a valid IP address", ip);
    if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0 )
	die(2, "cannot open test socket.");
    if (bind(s, (struct sockaddr *)&bind_addr, sizeof(struct sockaddr)))
	cdie(num, argfn, "cannot bind to %s", ip);
    close(s);
    *(struct in_addr *)ext_ptrarr(&lopt->bind_addrs, sizeof(struct in_addr)) = 
	bind_addr.sin_addr;
}

static void 
adden_agent(const char *agent, int perc)
{
    agent_t *ag;
    int len = strlen(agent) + 1;

    ag = (agent_t *)ext_ptrarr(&lopt->user_agents, sizeof(*ag) + len);
    ag->ratio = perc;
    lopt->uar_total += perc;
    memcpy(ag->agent, agent, len);
}

static void 
add_agent(const char *agent, int num, const char *argfn)
{
    int perc = -1, nch;

    sscanf(agent, "%i %n", &perc, &nch);
    if (perc < 0)
	cdie(num, argfn, "invalid percentage/agent spec '%s'", agent);
    adden_agent(agent + nch, perc);
}

static void 
showhelp_advanced(void)
{
    char ona[SHORTSTR];
    unsigned i;

    for (i = 0; i < sizeof(options) / sizeof(options[0]); i++) {
	if (options[i].opt) {
	    sprintf(ona, "%s %s", options[i].opt, onams[options[i].todo]);
	    printf("  -%-9s", ona);
	}
	puts(options[i].desc);
    }
}

static void 
showhelp_basic(void)
{
    printf("Usage: %s [options] [URL...]\n", progname);
}

static void 
showhelp(void)
{
    showhelp_basic();
    printf("\nTry '%s -h' for more information.\n", progname);
}

static char *
mfgets(char *buf, int len, FILE *f)
{
    int ba, be, bp;

    for (;;) {
      nxtl:
	if (fgets(buf, len, f)) {
	    for (ba = 0; buf[ba] <= ' '; ba++)
		if (!buf[ba])
		    goto nxtl;
	    if (buf[ba] == '#')
		continue;
	    for (be = (bp = ba) - 1; buf[bp]; bp++)
		if (buf[bp] > ' ')
		    be = bp;
	    buf[be + 1] = 0;
	    return buf + ba;
	}
	return 0;
    }
}

static off_t 
matoll(const char *val, const char *opt, int num, const char *argfn)
{
    const char *fbad;
    off_t oll;

    fbad = val;
    oll = 0;
    if (*fbad == '0' && fbad[1] == 'x') {
	fbad += 2;
	while (isxdigit((int)*fbad)) {
	    oll = oll * 16 + (isdigit((int)*fbad) ? 
			      *fbad - '0' : 
			      tolower((int)*fbad) - 'a' + 10);
	    fbad++;
	}
    } else
	while (isdigit((int)*fbad)) {
	    oll = oll * 10 + (*fbad - '0');
	    fbad++;
	}
    if (*fbad == 'k') {
	oll *= 1024; fbad++;
    } else if (*fbad == 'm') {
	oll *= 1024 * 1024; fbad++;
    } else if (*fbad == 'g') {
	oll *= 1024 * 1024 * 1024; fbad++;
    }
    if (*fbad)
	cdie(num, argfn, "invalid numeric argument '%s' to option '%s'", val, opt);
    return oll;
}

static int
matoi(const char *val, const char *opt, int num, const char *argfn)
{
    char *fbad;
    int oint;

    oint = strtol(val, &fbad, 0);
    if (*fbad || oint < 0)
	cdie(num, argfn, "invalid numeric argument '%s' to option '%s'", val, opt);
    return oint;
}

struct ostack {
    struct ostack *next;
    options_t *opt;
    url_parm_t *parm;
};

static void
procopts(int argc, char *argv[], FILE *argf, const char *argfn)
{
    const char *srct;
    FILE *f;
    char *fbad, *arg1, *arg2;
    filter_t *filter;
    struct ostack *ostack = 0, *nostack;
    int oind, dopts, t, acc, inv;
    unsigned i, olen;
    char buf[SHORTSTR], cbuf[SHORTSTR];

    srct = argfn ? argfn : "command line";
    for (oind = 1, dopts = 0;; oind++) {
	if (argfn) {
	    if (!(arg1 = mfgets(cbuf, sizeof(cbuf), argf)))
		break;
	} else {
	    if (oind >= argc)
		break;
	    arg1 = argv[oind];
	}
	if (!dopts && arg1[0] == '-') {
	    if (argfn) {
		if ((arg2 = strchr(arg1, ' ')))
		    *arg2++ = 0;
	    } else
		arg2 = argv[oind + 1];
	    olen = strlen(arg1 + 1);
	    inv = 0;
	    if (arg1[olen] == '-') {
		olen--;
		inv = 1;
	    }
	    for (i = 0; i < sizeof(options) / sizeof(options[0]); i++)
		if (options[i].opt && strlen(options[i].opt) == olen &&
		    !memcmp(arg1 + 1, options[i].opt, olen))
		    goto fopt;
	    cdie(oind, argfn, "unrecognized option '%s', try \"%s -h\"", arg1, 
		 progname);
	  fopt:
	    if (inv) {
		if (options[i].todo == O_DISP) {
		    idetach_parm();
		    lparm->disposition = 0;
		    goto next;
		} else if (options[i].todo == O_PPRX || options[i].todo == O_SPPRX) {
		    idetach_parm();
		    lparm->proxy = 0;
		    goto next;
		} else if (options[i].todo == O_STAMP) {
		    idetach_parm();
		    lparm->time_stamp = 0;
		    goto next;
		}
		idetach_opt();
		switch (options[i].todo) {
		    case O_LST_CI:
		    case O_LST_I:
			*(int *)((char *)lopt + (size_t)options[i].argptr) = 
			    options[i].defargval;
			break;
		    case O_LST_O:
			*(off_t *)((char *)lopt + (size_t)options[i].argptr) = 
			    0;
			break;
		    case O_LRF_S:
			*(char **)((char *)lopt + (size_t)options[i].argptr) = 
			    0;
			break;
		    case O_DPATH:
			lopt->disp_path = 0;
			break;
		    case O_BIND:
		    case O_BINDF:
		    case O_AGENT:
		    case O_AGENTF:
		    case O_PRX:
		    case O_PRXF:
		    case O_ACC:
		    case O_REJ:
			/*  clever, huh? :)  */
		    case O_ASTR:
			clr_ptrarr((ptrarr_t **)
			    ((char *)lopt + (size_t)options[i].argptr));
			break;
		    case O_DOM:
			clr_ptrarr(&lopt->ldom_list);
			clr_ptrarr(&lopt->rdom_list);
			break;
		    default:
			cdie(oind, argfn, "'%.*s' has no inverse option", olen + 1, arg1);
			break;
		}
		goto next;
	    }
	    if (options[i].todo == O_ST_CI)
		*(int *)options[i].argptr = options[i].argval;
	    else if (options[i].todo == O_LST_CI) {
		idetach_opt();
		*(int *)((char *)lopt + (size_t)options[i].argptr) = 
		    options[i].argval;
	    } else if (options[i].todo == O_HELP) {
		showhelp_basic();
		showhelp_advanced();
		exit(0);
	    } else {
		if (!arg2)
		    cdie(oind, argfn, "missing argument to option '%s', try \"%s -h\"",
			 arg1, progname);
		if (!argfn)
		    oind++;
		switch (options[i].todo) {
		    case O_ST_I:
			*(int *)options[i].argptr = matoi(arg2, arg1, oind, argfn);
			break;
		    case O_ST_O:
			*(off_t *)options[i].argptr = matoll(arg2, arg1, oind, argfn);
			break;
		    case O_RF_S:
			*(char **)options[i].argptr = argfn ? istrdup(arg2) : arg2;
			break;
		    case O_LST_I:
			idetach_opt();
			*(int *)((char *)lopt + (size_t)options[i].argptr) = 
			    matoi(arg2, arg1, oind, argfn);
			break;
		    case O_LST_O:
			idetach_opt();
			*(off_t *)((char *)lopt + (size_t)options[i].argptr) = 
			    matoll(arg2, arg1, oind, argfn);
			break;
		    case O_LRF_S:
			idetach_opt();
			*(char **)((char *)lopt + (size_t)options[i].argptr) = 
			    argfn ? istrdup(arg2) : arg2;
			break;
		    case O_ASTR:
			idetach_opt();
			if (!strcmp(arg2, "*") || !arg2[0]) {
			    if (!options[i].argval)
				cdie(oind, argfn, "empty argument to '%s' not allowed", arg1);
			    fbad = (char *)"";
			} else
			    fbad = argfn ? istrdup(arg2) : arg2;
			dext_ptrarr((ptrarr_t **)
			    ((char *)lopt + (size_t)options[i].argptr), fbad);
			break;
		    case O_DPATH:
			idetach_opt();
			set_dpath(arg2);
			break;
		    case O_BIND:
			idetach_opt();
			add_bind_ip(arg2, oind, argfn);
			break;
		    case O_BINDF:
			idetach_opt();
			if (!(f = fopen(arg2, "r")))
			    cdie(oind, argfn, "cannot open IP list '%s'", arg2);
			i = 0;
		    	while ((fbad = mfgets(buf, sizeof(buf), f)) != 0)
			    add_bind_ip(fbad, ++i, arg2);
			fclose(f);
			break;
		    case O_AGENT:
			idetach_opt();
			adden_agent(arg2, 1);
			break;
		    case O_AGENTF:
			idetach_opt();
			if (!(f = fopen(arg2, "r")))
			    cdie(oind, argfn, "cannot open User-Agent list '%s'", arg2);
			i = 0;
		    	while ((fbad = mfgets(buf, sizeof(buf), f)) != 0)
			    add_agent(fbad, ++i, arg2);
			fclose(f);
			break;
		    case O_PRX:
			idetach_opt();
			prx_adden(arg2, oind, argfn);
			break;
		    case O_PRXF:
			idetach_opt();
			if (!(f = fopen(arg2, "r")))
			    cdie(oind, argfn, "cannot open proxy list '%s'", arg2);
			i = 0;
		    	while ((fbad = mfgets(buf, sizeof(buf), f)) != 0)
			    prx_adden(fbad, ++i, arg2);
			fclose(f);
			break;
		    case O_URLF:
			if (!strcmp(arg2, "-"))
			    procopts(0, 0, stdin, "stdin");
			else {
			    if (!(f = fopen(arg2, "r")))
				cdie(oind, argfn, "cannot open command file '%s'", arg2);
			    procopts(0, 0, f, arg2);
			    fclose(f);
			}
			break;
		    case O_THROT:
			t = matoi(arg2, arg1, oind, argfn);
			throttle.tv_sec = t / 1000;
			throttle.tv_usec = t % 1000 * 1000;
			break;
		    case O_ACC:
			acc = 1;
			goto acre;
		    case O_REJ:
			acc = 0;
		      acre:
			idetach_opt();
			i = strlen(arg2);
			filter = imalloc(sizeof(*filter) + i + 1);
			filter->acc = acc;
			if (!(filter->type = memchr(arg2, '/', i) != 0))
			    filter->pat = strpbrk(arg2, "*?\\") != 0;
			else if (!lopt->filter_list->spare)
			    lopt->filter_list->spare = lopt->filter_list->nents + 1;
			memcpy(filter->data, arg2, i + 1);
			dext_ptrarr(&lopt->filter_list, filter);
			break;
		    case O_DOM:
			fbad = argfn ? istrdup(arg2) : arg2;
			dext_ptrarr(&lopt->ldom_list, fbad);
			dext_ptrarr(&lopt->rdom_list, fbad);
			break;
		    case O_DISP:
			idetach_parm();
			if (!strcmp(arg2, "-"))
			    set_disp(lparm, "", 0);
			else
			    set_disp(lparm, arg2, strlen(arg2));
			break;
		    case O_PPRX:
			idetach_parm();
			lparm->strictproxy = 0;
			goto pprx;
		    case O_SPPRX:
			idetach_parm();
			lparm->strictproxy = 1;
		      pprx:
			if (!(lparm->proxy = parse_add_proxy(srct, arg2)))
			    cdie(oind, argfn, "invalid proxy '%s' supplied to %s", arg2, arg1);
			break;
		    case O_STAMP:
			idetach_parm();
			lparm->time_stamp = strtol(arg2, &fbad, 0);
			if (*fbad) {
			    lparm->time_stamp = parseHTTPdate(arg2);
			    if (lparm->time_stamp == BAD_DATE)
				cdie(oind, argfn, "invalid date '%s' supplied to %s", arg2, arg1);
			}
			break;
		}
		continue;
	    }
	  next:
	    if (argfn && arg2)
		cdie(oind, argfn, "unexpected argument to command '%s'", arg1);
	} else if (!strcmp(arg1, ",")) {
	    initlopt();
	    dopts = 0;
	} else if (!strcmp(arg1, "]")) {
	    if (!ostack)
		cdie(oind, argfn, "unexpected closing bracket");
	    lopt = ostack->opt;
	    lparm = ostack->parm;
	    nostack = ostack->next;
	    free(ostack);
	    ostack = nostack;
	} else if (!dopts && !strcmp(arg1, "[")) {
	    nostack = imalloc(sizeof(*nostack));
	    nostack->opt = lopt;
	    nostack->parm = lparm;
	    nostack->next = ostack;
	    ostack = nostack;
	    do_idetach_opt();
	} else if (!dopts && !strcmp(arg1, "--"))
	    dopts++;
	else
	    adden(srct, arg1);
    }

    if (ostack)
	cdie(oind, argfn, "expecting closing bracket(s)");
}

void 
getopts(int argc, char *argv[])
{
    initlopt();
    procopts(argc, argv, 0, 0);
    /*  If no url was given, show help message:  */
    if (!tried_url) {
	showhelp();
	exit(2);
    }

    if ((verbose
#ifdef DEBUG
	 || debug
#endif
	) && isatty(2))
	show_stat = 0;

}
