/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include "pgmglesxbackend.h"
#include "pgmglesviewport.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gles_xbackend_debug);
#define GST_CAT_DEFAULT pgm_gles_xbackend_debug

/* Prototypes */
static gboolean event_prepare  (GSource *source,
                                gint *timeout);
static gboolean event_check    (GSource *source);
static gboolean event_dispatch (GSource *source,
                                GSourceFunc callback,
                                gpointer data);

/* Event handling functions */
static GSourceFuncs event_funcs = {
  event_prepare, event_check, event_dispatch, NULL
};

/*  */
typedef struct {
  EGLint       error;
  const gchar *string;
} EglError;

/*  */
static EglError egl_error_map[] = {
  { EGL_NOT_INITIALIZED,     "not initialized"     },
  { EGL_BAD_ACCESS,          "bad access"          },
  { EGL_BAD_ALLOC,           "bad allocation"      },
  { EGL_BAD_ATTRIBUTE,       "bad attribute"       },
  { EGL_BAD_CONFIG,          "bad configuration"   },
  { EGL_BAD_CONTEXT,         "bad context"         },
  { EGL_BAD_CURRENT_SURFACE, "bad current surface" },
  { EGL_BAD_DISPLAY,         "bad display"         },
  { EGL_BAD_MATCH,           "bad match"           },
  { EGL_BAD_NATIVE_PIXMAP,   "bad native pixmap"   },
  { EGL_BAD_NATIVE_WINDOW,   "bad native window"   },
  { EGL_BAD_PARAMETER,       "bad parameter"       },
  { EGL_BAD_SURFACE,         "bad surface"         },
  { 0,                       "unknown error"       }
};

/* X atom index IDs */
typedef enum {
  ATOM_NET_WM_STATE_FULLSCREEN = 0,
  ATOM_NET_WM_STATE            = 1,
  ATOM_NET_WM_PING             = 2,
  ATOM_NET_WM_USER_TIME        = 3,
  ATOM_NET_WM_NAME             = 4,
  ATOM_NET_ACTIVE_WINDOW       = 5,
  ATOM_MOTIF_WM_HINTS          = 6,
  ATOM_UTF8_STRING             = 7,
  ATOM_WM_PROTOCOLS            = 8,
  ATOM_WM_DELETE_WINDOW        = 9,
} XAtomIndexId;

/* X atom names */
static const gchar *atom_name[] = {
  "_NET_WM_STATE_FULLSCREEN", /* 0 */
  "_NET_WM_STATE",            /* 1 */
  "_NET_WM_PING",             /* 2 */
  "_NET_WM_USER_TIME",        /* 3 */
  "_NET_WM_NAME",             /* 4 */
  "_NET_ACTIVE_WINDOW",       /* 5 */
  "_MOTIF_WM_HINTS",          /* 6 */
  "UTF8_STRING",              /* 7 */
  "WM_PROTOCOLS",             /* 8 */
  "WM_DELETE_WINDOW",         /* 9 */
};

/* Number of atoms, atom strings and index IDs must be kept synchronised! */
static const guint nb_atoms = G_N_ELEMENTS (atom_name);

static PgmGlesBackendClass *parent_class = NULL;

/* Private functions */

/* */
static gboolean
get_egl_error (const gchar *function)
{
  guint i = 0;
  EGLint error;

  error = eglGetError ();

  if (error == EGL_SUCCESS)
    return TRUE;

  /* Search for the error */
  while (egl_error_map[i].error && error != egl_error_map[i].error)
    i++;

  /* Then log it */
  GST_ERROR ("%s failed: %s\n", function, egl_error_map[i].string);

  return FALSE;
}

/* Specifies the WM_PROTOCOLS property of the window */
static void
set_wm_protocols (PgmGlesXBackend *glesxbackend)
{
  Atom protocols[2];

  protocols[0] = glesxbackend->atom[ATOM_WM_DELETE_WINDOW];
  protocols[1] = glesxbackend->atom[ATOM_NET_WM_PING];

  XSetWMProtocols (glesxbackend->xdisplay, glesxbackend->win, protocols, 2);
}

/* Updates the _NET_WM_USER_TIME property of the window */
static void
set_wm_user_time (PgmGlesXBackend *glesxbackend,
                  glong timestamp)
{
  if (timestamp)
    XChangeProperty (glesxbackend->xdisplay, glesxbackend->win,
                     glesxbackend->atom[ATOM_NET_WM_USER_TIME], XA_CARDINAL, 32,
                     PropModeReplace, (guchar *) &timestamp, 1);
}

/* Handles the protocols event and returns an event to push if any */
static PgmEvent *
handle_wm_protocols_event (PgmGlesXBackend *glesxbackend,
                           XEvent *xevent)
{
  PgmEvent *pgmevent = NULL;

  /* Delete request */
  if (xevent->xclient.data.l[0] == glesxbackend->atom[ATOM_WM_DELETE_WINDOW]
      && xevent->xany.window == glesxbackend->win)
    {
      pgmevent = pgm_event_new (PGM_DELETE);
      set_wm_user_time (glesxbackend, xevent->xclient.data.l[1]);
    }

  /* Ping request */
  else if (xevent->xclient.data.l[0] == glesxbackend->atom[ATOM_NET_WM_PING]
           && xevent->xany.window == glesxbackend->win)
    {
      xevent->xclient.window = glesxbackend->root;

      XSendEvent (glesxbackend->xdisplay, glesxbackend->root, False,
                  SubstructureRedirectMask | SubstructureNotifyMask,
                  (XEvent *) &xevent->xclient);
    }

  return pgmevent;
}

/* Compresses events of the same type */
static void
compress_events (PgmGlesXBackend *glesxbackend,
                 XEvent *xevent)
{
  while (XCheckTypedWindowEvent (glesxbackend->xdisplay, glesxbackend->win,
                                 xevent->type, xevent));
}

/* Fill a PgmEventKey giving the corresponding keyboard xevent */
static void
translate_key_event (PgmGlesXBackend *glesxbackend,
                     PgmEventKey *event,
                     XEvent *xevent)
{
  gint index;

  event->time = xevent->xkey.time;
  event->modifier = (PgmModifierType) xevent->xkey.state;

  /* FIXME: This is a really basic support, a lot of keys are not handled */
  index = (event->modifier & (PGM_SHIFT_MASK | PGM_CAPSLOCK_MASK)) ? 1 : 0;
  event->keyval = XKeycodeToKeysym (glesxbackend->xdisplay, xevent->xkey.keycode,
                                    index);

  event->hardware_keycode = xevent->xkey.keycode;
}

/* Fill a PgmEventMotion giving the corresponding motion xevent */
static void
translate_motion_event (PgmGlesXBackend *glesxbackend,
                        PgmEventMotion *event,
                        XEvent *xevent)
{
  event->time = xevent->xmotion.time;
  event->x = xevent->xmotion.x;
  event->y = xevent->xmotion.y;
}

/* Fill a PgmEventButton giving the corresponding button xevent */
static void
translate_button_press_event (PgmGlesXBackend *glesxbackend,
                              PgmEventButton *event,
                              XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Fill a PgmEventScroll giving the corresponding button xevent */
static void
translate_scroll_event (PgmGlesXBackend *glesxbackend,
                        PgmEventScroll *event,
                        XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Create a button event (Release) giving the corresponding button xevent */
static PgmEvent *
create_button_release_event (PgmGlesXBackend *glesxbackend,
                             XEvent *xevent)
{
  PgmEventButton *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_LEFT;
      break;

    case 2:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_MIDDLE;
      break;

    case 3:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_RIGHT;
      break;

    /* No release */
    default:
      return NULL;
    }

  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;

  return (PgmEvent *) event;
}

/* Create a button event (Scroll, Press) given an XEvent */
static PgmEvent *
create_button_event (PgmGlesXBackend *glesxbackend,
                     XEvent *xevent)
{
  PgmEvent *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_LEFT;
      translate_button_press_event (glesxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 2:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_MIDDLE;
      translate_button_press_event (glesxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 3:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_RIGHT;
      translate_button_press_event (glesxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 4:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_UP;
      translate_scroll_event (glesxbackend, (PgmEventScroll *) event, xevent);
      break;

    case 5:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_DOWN;
      translate_scroll_event (glesxbackend, (PgmEventScroll *) event, xevent);
      break;

    default:
      event = NULL;
      break;
    }

  return event;
}

/* Dispatches events pushing them on the viewport queue */
static gboolean
dispatch_x_events (PgmGlesBackend *glesbackend)
{
  static guint8 button_mask = 0;
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);
  PgmGlesViewport *glesviewport = glesbackend->context->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);

  if (!XPending (glesxbackend->xdisplay))
    return TRUE;

  while (XEventsQueued (glesxbackend->xdisplay, QueuedAlready))
    {
      PgmEvent *pgmevent = NULL;
      XEvent xevent;

      XNextEvent (glesxbackend->xdisplay, &xevent);

      switch (xevent.type)
        {
          /* --- Expose event --- */
        case Expose:
          GST_DEBUG_OBJECT (glesxbackend, "Expose event");

          compress_events (glesxbackend, &xevent);
          pgmevent = pgm_event_new (PGM_EXPOSE);
          pgm_gles_context_update (glesbackend->context);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Key press event --- */
        case KeyPress:
          GST_DEBUG_OBJECT (glesxbackend, "KeyPress event: (time=%u, state=0x%x, "
                            "keycode=0x%x)", xevent.xkey.time, xevent.xkey.state,
                            xevent.xkey.keycode);

          pgmevent = pgm_event_new (PGM_KEY_PRESS);
          translate_key_event (glesxbackend, (PgmEventKey *) pgmevent, &xevent);
          set_wm_user_time (glesxbackend, xevent.xkey.time);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Key release event --- */
        case KeyRelease:
          GST_DEBUG_OBJECT (glesxbackend, "KeyRelease event: (time=%u, "
                            "state=0x%x, keycode=0x%x)", xevent.xkey.time,
                            xevent.xkey.state, xevent.xkey.keycode);

          pgmevent = pgm_event_new (PGM_KEY_RELEASE);
          translate_key_event (glesxbackend, (PgmEventKey *) pgmevent, &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Button press event --- */
        case ButtonPress:
          GST_DEBUG_OBJECT (glesxbackend, "ButtonPress event: (time=%u, "
                            "button=%d x=%d y=%d)", xevent.xbutton.time,
                            xevent.xbutton.button, xevent.xbutton.x,
                            xevent.xbutton.y);

          pgmevent = create_button_event (glesxbackend, &xevent);
          if (pgmevent)
            {
              set_wm_user_time (glesxbackend, xevent.xbutton.time);
              pgm_viewport_push_event (viewport, pgmevent);

              /* Events here could also be scroll events, so we check it's not
               * and then grab the pointer */
              if (pgmevent->type == PGM_BUTTON_PRESS)
                {
                  button_mask |= ((PgmEventButton *) pgmevent)->button;
                  XGrabPointer (glesxbackend->xdisplay, glesxbackend->win, False,
                                ButtonPressMask | ButtonReleaseMask
                                | PointerMotionMask,
                                GrabModeAsync, GrabModeAsync,
                                None, None, CurrentTime);
                }
            }
          break;

          /* --- Button release event --- */
        case ButtonRelease:
          GST_DEBUG_OBJECT (glesxbackend, "ButtonRelease event: (time=%u, "
                            "button=%d x=%d y=%d)", xevent.xbutton.time,
                            xevent.xbutton.button, xevent.xbutton.x,
                            xevent.xbutton.y);

          pgmevent = create_button_release_event (glesxbackend, &xevent);
          if (pgmevent)
            {
              pgm_viewport_push_event (viewport, pgmevent);
              button_mask &= ~((PgmEventButton *) pgmevent)->button;
              if (button_mask == 0)
                XUngrabPointer(glesxbackend->xdisplay, CurrentTime);
            }
          break;

          /* --- Motion notify event --- */
        case MotionNotify:
          GST_DEBUG_OBJECT (glesxbackend, "MotionNotify event: (time=%u, x=%d, "
                            "y=%d)", xevent.xmotion.time, xevent.xmotion.x,
                            xevent.xmotion.y);

          compress_events (glesxbackend, &xevent);
          pgmevent = pgm_event_new (PGM_MOTION_NOTIFY);
          translate_motion_event (glesxbackend, (PgmEventMotion *) pgmevent,
                                  &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Client message event --- */
        case ClientMessage:
          GST_DEBUG_OBJECT (glesxbackend, "ClientMessage event: "
                            "message_type='%s')", XGetAtomName
                            (glesxbackend->xdisplay, xevent.xclient.message_type));
          {
            Atom atom = xevent.xclient.message_type;

            if (atom == glesxbackend->atom[ATOM_WM_PROTOCOLS])
              pgmevent = handle_wm_protocols_event (glesxbackend, &xevent);
            if (pgmevent)
              pgm_viewport_push_event (viewport, pgmevent);
          }

        default:
          break;
        }
    }

  return TRUE;
}

/* Events preparing */
static gboolean
event_prepare (GSource *source,
               gint *timeout)
{
  PgmGlesXBackendSource *glesxbackend_source = (PgmGlesXBackendSource *) source;
  PgmGlesXBackend *glesxbackend = glesxbackend_source->glesxbackend;
  PgmGlesBackend *backend = PGM_GLES_BACKEND (glesxbackend);

  if (XEventsQueued (glesxbackend->xdisplay, QueuedAlready))
    return TRUE;

  *timeout = backend->context->update_tag ? 0 : -1;

  return FALSE;
}

/* Events checking */
static gboolean
event_check (GSource *source)
{
  PgmGlesXBackendSource *glesxbackend_source = (PgmGlesXBackendSource *) source;

  return (glesxbackend_source->poll_fd.revents & G_IO_IN) != 0;
}

/* Events dispatching */
static gboolean
event_dispatch (GSource *source,
                GSourceFunc callback,
                gpointer data)
{
  if (callback)
    return callback (data);

  return FALSE;
}

/* Add event handling source to the rendering main context */
static gboolean
add_event_source (PgmGlesXBackend *glesxbackend)
{
  PgmGlesBackend *glesbackend = PGM_GLES_BACKEND (glesxbackend);
  GMainContext *render_context = glesbackend->context->render_context;
  PgmGlesXBackendSource *source;

  /* Filter handled events */
  XSelectInput (glesxbackend->xdisplay, glesxbackend->win,
                KeyPressMask | KeyReleaseMask | ButtonPressMask
                | ButtonReleaseMask | ExposureMask | PointerMotionMask
                | PropertyChangeMask);

  /* Create and initialize the event handling dedicated source */
  source = (PgmGlesXBackendSource *)
    g_source_new (&event_funcs, sizeof (PgmGlesXBackendSource));
  source->poll_fd.fd = ConnectionNumber (glesxbackend->xdisplay);
  source->poll_fd.events = G_IO_IN;
  source->glesxbackend = glesxbackend;
  g_source_add_poll ((GSource *) source, &source->poll_fd);

  /* Attach it */
  g_source_set_callback ((GSource *) source, (GSourceFunc) dispatch_x_events,
                         glesxbackend, NULL);
  g_source_set_priority ((GSource *) source, G_PRIORITY_DEFAULT-20);
  glesxbackend->event_id = g_source_attach ((GSource *) source, render_context);
  g_source_unref ((GSource *) source);

  return TRUE;
}

/* PgmGlesBackend methods */

static gboolean
pgm_gles_x_backend_create_window (PgmGlesBackend *glesbackend)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);
  const gchar *title = "Pigment OpenGL ES-CM 1.X plugin (PC emulation)";
  gint swap_interval = 1;
  const gchar *env_var;
  EGLConfig config;
  gint num_configs;
  PgmViewport *viewport;
  gint width, height;
  XSizeHints *size_hints;

  gint attrib[] = {
    EGL_RED_SIZE, 5,
    EGL_GREEN_SIZE, 6,
    EGL_BLUE_SIZE, 5,
    EGL_DEPTH_SIZE, 0,
    EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
    EGL_NONE };

  GST_DEBUG_OBJECT (glesxbackend, "create_window");

  viewport = PGM_VIEWPORT (glesbackend->context->glesviewport);

  /* Initialize display */
  glesxbackend->xdisplay = XOpenDisplay (NULL);
  if (!glesxbackend->xdisplay)
    {
      GST_ERROR_OBJECT (glesxbackend, "couldn't open default display");
      return FALSE;
    }
  glesxbackend->egldisplay =
    eglGetDisplay ((NativeDisplayType) glesxbackend->xdisplay);
  if (glesxbackend->egldisplay == EGL_NO_DISPLAY)
    {
      GST_ERROR_OBJECT (glesxbackend, "cannot get a display");
      return FALSE;
    }
  if (!eglInitialize (glesxbackend->egldisplay, NULL, NULL))
    {
      get_egl_error ("eglInitialize");
      return FALSE;
    }

  glesxbackend->screen = DefaultScreen (glesxbackend->xdisplay);
  glesxbackend->root = DefaultRootWindow (glesxbackend->xdisplay);

  /* Get and log EGL informations */
  glesxbackend->vendor = eglQueryString (glesxbackend->egldisplay, EGL_VENDOR);
  glesxbackend->version = eglQueryString (glesxbackend->egldisplay, EGL_VERSION);
  glesxbackend->extensions =
    eglQueryString (glesxbackend->egldisplay, EGL_EXTENSIONS);

  GST_DEBUG_OBJECT (glesxbackend, "EGL vendor: %s", glesxbackend->vendor);
  GST_DEBUG_OBJECT (glesxbackend, "EGL version: %s", glesxbackend->version);
  GST_DEBUG_OBJECT (glesxbackend, "EGL extensions: %s",
                    glesxbackend->extensions);

  /* Create the window */
  pgm_viewport_get_size (viewport, &width, &height);
  glesxbackend->win = XCreateSimpleWindow (glesxbackend->xdisplay,
                                           glesxbackend->root,
                                           0, 0, width, height, 0, 0,
                                           BlackPixel (glesxbackend->xdisplay,
                                                       glesxbackend->screen));
  if (!glesxbackend->win)
    {
      GST_ERROR_OBJECT (glesxbackend, "couldn't create window");
      return FALSE;
    }

  /* Create the context and make it current */
  if (!eglChooseConfig (glesxbackend->egldisplay, attrib, &config, 1,
                        &num_configs))
    {
      get_egl_error ("eglChooseConfig");
      return FALSE;
    }

  glesxbackend->surface =
    eglCreateWindowSurface (glesxbackend->egldisplay, config,
                            (NativeWindowType) glesxbackend->win, NULL);
  if (!get_egl_error ("eglCreateWindowSurface"))
    return FALSE;
  glesxbackend->context = eglCreateContext (glesxbackend->egldisplay,
                                            config, EGL_NO_CONTEXT, NULL);
  if (!get_egl_error ("eglCreateContext"))
    return FALSE;
  eglMakeCurrent (glesxbackend->egldisplay, glesxbackend->surface,
                  glesxbackend->surface, glesxbackend->context);
  if (!get_egl_error ("eglMakeCurrent"))
    return FALSE;

  /* Event source */
  if (!add_event_source (glesxbackend))
    {
      GST_ERROR_OBJECT (glesxbackend, "couldn't add event handling source\n");
      return FALSE;
    }

  /* Fill X atoms arrays */
  glesxbackend->atom = g_malloc (nb_atoms * sizeof (Atom));
  XInternAtoms (glesxbackend->xdisplay, (gchar**) atom_name, nb_atoms, False,
                glesxbackend->atom);

  /* Don't allow resizing */
  size_hints = XAllocSizeHints ();
  size_hints->max_width = size_hints->min_width = width;
  size_hints->max_height = size_hints->min_height = height;
  size_hints->flags = PMinSize | PMaxSize;
  XSetWMNormalHints (glesxbackend->xdisplay, glesxbackend->win, size_hints);
  XFree (size_hints);

  /* Set the title */
  XChangeProperty (glesxbackend->xdisplay, glesxbackend->win,
                   glesxbackend->atom[ATOM_NET_WM_NAME],
                   glesxbackend->atom[ATOM_UTF8_STRING], 8, PropModeReplace,
                   (guchar*) title, strlen (title));

  /* Set a hand icon */
/*   XDefineCursor (glesxbackend->xdisplay, glesxbackend->win, */
/*                  XCreateFontCursor (glesxbackend->xdisplay, XC_hand2)); */

  /* X protocols initialization */
  set_wm_protocols (glesxbackend);

  glesxbackend->resolution_width = width;
  glesxbackend->resolution_height = height;
  glesxbackend->size_mm_width = width;
  glesxbackend->size_mm_height = height;

  /* Specify the minimum number of video frame periods per buffer swap, this
   * is actually how vertical synchronization is handled */
  env_var = g_getenv ("PGM_GLES_SWAP_INTERVAL");
  if (env_var)
    swap_interval = atoi (env_var);
  eglSwapInterval (glesxbackend->egldisplay, swap_interval);
  GST_DEBUG_OBJECT (glesxbackend, "using a swap interval of %d", swap_interval);

  glesxbackend->created = TRUE;

  return TRUE;
}

static gboolean
pgm_gles_x_backend_destroy_window (PgmGlesBackend *glesbackend)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);

  GST_DEBUG_OBJECT (glesxbackend, "destroy_window");

  if (glesxbackend->created)
    {
      /* Remove event handling source */
      g_source_remove (glesxbackend->event_id);

      /* Clean EGL related data */
      eglMakeCurrent (glesxbackend->egldisplay, EGL_NO_SURFACE,
                      EGL_NO_SURFACE, EGL_NO_CONTEXT);
      eglTerminate (glesxbackend->egldisplay);

      /* Clean up X related data */
      XDestroyWindow (glesxbackend->xdisplay, glesxbackend->win);
      XCloseDisplay (glesxbackend->xdisplay);
      glesxbackend->created = FALSE;

      g_free (glesxbackend->atom);
      glesxbackend->atom = NULL;
    }

  return TRUE;
}

static gboolean
pgm_gles_x_backend_set_visibility (PgmGlesBackend *glesbackend,
                                   gboolean visible)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);

  if (visible)
    XMapWindow (glesxbackend->xdisplay, glesxbackend->win);
  else
    XUnmapWindow (glesxbackend->xdisplay, glesxbackend->win);

  XSync (glesxbackend->xdisplay, False);

  return TRUE;
}

static void
pgm_gles_x_backend_swap_buffers (PgmGlesBackend *glesbackend)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);

  eglSwapBuffers (glesxbackend->egldisplay, glesxbackend->surface);
}

static gpointer
pgm_gles_x_backend_get_proc_address (PgmGlesBackend *glesbackend,
                                     const gchar *proc_name)
{
  return eglGetProcAddress (proc_name);
}

static void
pgm_gles_x_backend_get_screen_size_mm (PgmGlesBackend *glesbackend,
                                       gint *width,
                                       gint *height)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);

  *width = glesxbackend->size_mm_width;
  *height = glesxbackend->size_mm_height;
}

static void
pgm_gles_x_backend_get_screen_resolution (PgmGlesBackend *glesbackend,
                                          gint *width,
                                          gint *height)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (glesbackend);

  *width = glesxbackend->resolution_width;
  *height = glesxbackend->resolution_height;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesXBackend, pgm_gles_x_backend,
                         PGM_TYPE_GLES_BACKEND)

static void
pgm_gles_x_backend_dispose (GObject *object)
{
  PgmGlesXBackend *glesxbackend = PGM_GLES_X_BACKEND (object);

  if (glesxbackend->created)
    pgm_gles_x_backend_destroy_window (PGM_GLES_BACKEND (glesxbackend));

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_x_backend_class_init (PgmGlesXBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlesBackendClass *glesbackend_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_xbackend_debug, "pgm_gles_xbackend", 0,
                           "OpenGL ES plugin: PgmGlesXBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  glesbackend_class = PGM_GLES_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_x_backend_dispose);

  /* PgmGlesBackend virtual table */
  glesbackend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_create_window);
  glesbackend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_destroy_window);
  glesbackend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_swap_buffers);
  glesbackend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_get_proc_address);
  glesbackend_class->set_visibility =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_set_visibility);
  glesbackend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_get_screen_size_mm);
  glesbackend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gles_x_backend_get_screen_resolution);
}

static void
pgm_gles_x_backend_class_finalize (PgmGlesXBackendClass *klass)
{
  return;
}

static void
pgm_gles_x_backend_init (PgmGlesXBackend *glesxbackend)
{
  glesxbackend->atom = NULL;
  glesxbackend->created = FALSE;
}

/* Public methods */

void
pgm_gles_x_backend_register (GTypeModule *module)
{
  pgm_gles_x_backend_register_type (module);
}

PgmGlesBackend *
pgm_gles_x_backend_new (PgmGlesContext *glescontext)
{
  PgmGlesBackend *glesbackend;

  glesbackend = g_object_new (PGM_TYPE_GLES_X_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_GLES_X_BACKEND (glesbackend),
                    "created new glesxbackend");

  glesbackend->context = glescontext;

  return glesbackend;
}
