///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __POVRAY_RENDERER_H
#define __POVRAY_RENDERER_H

#include <core/Core.h>
#include <core/rendering/PluginRenderer.h>
#include <core/scene/animation/controller/Controller.h>
#include "../POVRay.h"

namespace POVRay {

/******************************************************************************
* A scene renderer that integrates the external POV-Ray program.
******************************************************************************/
class POVRAY_DLLEXPORT POVRayRenderer : public PluginRenderer
{
public:
	/// Default constructor.
	POVRayRenderer(bool isLoading = false);

	///	Prepares the renderer for rendering of the given scene.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool startRender(DataSet* dataset);

	/// Renders a single animation frame into the given frame buffer.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer);

	///	Finishes the rendering pass. This is called after all animation frames have been rendered
	/// or when the rendering operation has been aborted.
	virtual void endRender();

public:		// Global configuration

	/// Returns the path to the POV-Ray executable.
	static QString renderExecutable();

	/// Sets the path to the POV-Ray executable.
	static void setRenderExecutable(const QString& path);

private:

	/// The data set to be rendered.
	/// This field is only used during a render pass.
	DataSet* dataset;

	/// The POV-Ray quality level to use for rendering (0 <= level <= 11).
	/// See POV-Ray documentation for details.
	ReferenceField<IntegerController> _qualityLevel;

	/// Turns anti-aliasing on/off
	ReferenceField<BooleanController> _enableAntialiasing;

	/// Controls the AA sampling method (only 1 or 2 are valid).
	ReferenceField<IntegerController> _samplingMethod;

	/// Controls the anti-aliasing threshold.
	ReferenceField<FloatController> _AAThreshold;

	/// Controls the number of AA samples.
	ReferenceField<IntegerController> _antialiasDepth;

	/// Turns on AA-jitter.
	ReferenceField<BooleanController> _enableJitter;

	/// Supresses the POV-Ray rendering window.
	ReferenceField<BooleanController> _hidePOVRayDisplay;

	/// Turns on radiosity.
	ReferenceField<BooleanController> _enableRadiosity;
	/// Controls the number of rays that are sent out whenever a new radiosity value has to be calculated.
	ReferenceField<IntegerController> _radiosityRayCount;
	/// Determines how many recursion levels are used to calculate the diffuse inter-reflection.
	ReferenceField<IntegerController> _radiosityRecursionLimit;
	/// Controls the fraction of error tolerated for the radiosity calculation.
	ReferenceField<FloatController> _radiosityErrorBound;

    friend class POVRayRendererEditor;
    friend class POVRayExporter;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(POVRayRenderer)
	DECLARE_REFERENCE_FIELD(_qualityLevel)
	DECLARE_REFERENCE_FIELD(_enableAntialiasing)
	DECLARE_REFERENCE_FIELD(_samplingMethod)
	DECLARE_REFERENCE_FIELD(_AAThreshold)
	DECLARE_REFERENCE_FIELD(_antialiasDepth)
	DECLARE_REFERENCE_FIELD(_enableJitter)
	DECLARE_REFERENCE_FIELD(_hidePOVRayDisplay)
	DECLARE_REFERENCE_FIELD(_enableRadiosity)
	DECLARE_REFERENCE_FIELD(_radiosityRayCount)
	DECLARE_REFERENCE_FIELD(_radiosityRecursionLimit)
	DECLARE_REFERENCE_FIELD(_radiosityErrorBound)
};

};	// End of namespace POVRay

#endif // __POVRAY_RENDERER_H
