///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file OpenGLInterface.h
 * \brief This header file contains several macro definitions to make OpenGL function calls
 *        independent of the floating-point precision used for the build. It also contains the
 *        definition of the Core::OpenGLExtensions class.
 */

#ifndef __OPENGL_INTERFACE_H
#define __OPENGL_INTERFACE_H

#if defined(Q_WS_MAC)

// OpenGL extensions are listed in the system header file.
#include <OpenGL/glext.h>

typedef void (*PFNGLLOCKARRAYSEXTPROC) (GLint first, GLsizei count);
typedef void (*PFNGLUNLOCKARRAYSEXTPROC) (void);

typedef void (*PFNGLPOINTPARAMETERFARBPROC) (GLenum pname, GLfloat param);
typedef void (*PFNGLPOINTPARAMETERFVARBPROC) (GLenum pname, const GLfloat *params);

typedef void (*PFNGLBINDBUFFERARBPROC) (GLenum target, GLuint buffer);
typedef void (*PFNGLDELETEBUFFERSARBPROC) (GLsizei n, const GLuint *buffers);
typedef void (*PFNGLGENBUFFERSARBPROC) (GLsizei n, GLuint *buffers);
typedef GLboolean (*PFNGLISBUFFERARBPROC) (GLuint buffer);
typedef void (*PFNGLBUFFERDATAARBPROC) (GLenum target, GLsizeiptrARB size, const GLvoid *data, GLenum usage);
typedef void (*PFNGLBUFFERSUBDATAARBPROC) (GLenum target, GLintptrARB offset, GLsizeiptrARB size, const GLvoid *data);
typedef void (*PFNGLGETBUFFERSUBDATAARBPROC) (GLenum target, GLintptrARB offset, GLsizeiptrARB size, GLvoid *data);
typedef GLvoid* (*PFNGLMAPBUFFERARBPROC) (GLenum target, GLenum access);
typedef GLboolean (*PFNGLUNMAPBUFFERARBPROC) (GLenum target);
typedef void (*PFNGLGETBUFFERPARAMETERIVARBPROC) (GLenum target, GLenum pname, GLint *params);
typedef void (*PFNGLGETBUFFERPOINTERVARBPROC) (GLenum target, GLenum pname, GLvoid* *params);

typedef GLboolean (*PFNGLISRENDERBUFFEREXTPROC) (GLuint renderbuffer);
typedef void (*PFNGLBINDRENDERBUFFEREXTPROC) (GLenum target, GLuint renderbuffer);
typedef void (*PFNGLDELETERENDERBUFFERSEXTPROC) (GLsizei n, const GLuint *renderbuffers);
typedef void (*PFNGLGENRENDERBUFFERSEXTPROC) (GLsizei n, GLuint *renderbuffers);
typedef void (*PFNGLRENDERBUFFERSTORAGEEXTPROC) (GLenum target, GLenum internalformat, GLsizei width, GLsizei height);
typedef void (*PFNGLGETRENDERBUFFERPARAMETERIVEXTPROC) (GLenum target, GLenum pname, GLint *params);
typedef GLboolean (*PFNGLISFRAMEBUFFEREXTPROC) (GLuint framebuffer);
typedef void (*PFNGLBINDFRAMEBUFFEREXTPROC) (GLenum target, GLuint framebuffer);
typedef void (*PFNGLDELETEFRAMEBUFFERSEXTPROC) (GLsizei n, const GLuint *framebuffers);
typedef void (*PFNGLGENFRAMEBUFFERSEXTPROC) (GLsizei n, GLuint *framebuffers);
typedef GLenum (*PFNGLCHECKFRAMEBUFFERSTATUSEXTPROC) (GLenum target);
typedef void (*PFNGLFRAMEBUFFERTEXTURE1DEXTPROC) (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level);
typedef void (*PFNGLFRAMEBUFFERTEXTURE2DEXTPROC) (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level);
typedef void (*PFNGLFRAMEBUFFERTEXTURE3DEXTPROC) (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint zoffset);
typedef void (*PFNGLFRAMEBUFFERRENDERBUFFEREXTPROC) (GLenum target, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer);
typedef void (*PFNGLGETFRAMEBUFFERATTACHMENTPARAMETERIVEXTPROC) (GLenum target, GLenum attachment, GLenum pname, GLint *params);
typedef void (*PFNGLGENERATEMIPMAPEXTPROC) (GLenum target);

typedef void (*PFNGLDELETEOBJECTARBPROC) (GLhandleARB obj);
typedef GLhandleARB (*PFNGLGETHANDLEARBPROC) (GLenum pname);
typedef void (*PFNGLDETACHOBJECTARBPROC) (GLhandleARB containerObj, GLhandleARB attachedObj);
typedef GLhandleARB (*PFNGLCREATESHADEROBJECTARBPROC) (GLenum shaderType);
typedef void (*PFNGLSHADERSOURCEARBPROC) (GLhandleARB shaderObj, GLsizei count, const GLcharARB* *string, const GLint *length);
typedef void (*PFNGLCOMPILESHADERARBPROC) (GLhandleARB shaderObj);
typedef GLhandleARB (*PFNGLCREATEPROGRAMOBJECTARBPROC) (void);
typedef void (*PFNGLATTACHOBJECTARBPROC) (GLhandleARB containerObj, GLhandleARB obj);
typedef void (*PFNGLLINKPROGRAMARBPROC) (GLhandleARB programObj);
typedef void (*PFNGLUSEPROGRAMOBJECTARBPROC) (GLhandleARB programObj);
typedef void (*PFNGLVALIDATEPROGRAMARBPROC) (GLhandleARB programObj);
typedef void (*PFNGLUNIFORM1FARBPROC) (GLint location, GLfloat v0);
typedef void (*PFNGLUNIFORM2FARBPROC) (GLint location, GLfloat v0, GLfloat v1);
typedef void (*PFNGLUNIFORM3FARBPROC) (GLint location, GLfloat v0, GLfloat v1, GLfloat v2);
typedef void (*PFNGLUNIFORM4FARBPROC) (GLint location, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3);
typedef void (*PFNGLUNIFORM1IARBPROC) (GLint location, GLint v0);
typedef void (*PFNGLUNIFORM2IARBPROC) (GLint location, GLint v0, GLint v1);
typedef void (*PFNGLUNIFORM3IARBPROC) (GLint location, GLint v0, GLint v1, GLint v2);
typedef void (*PFNGLUNIFORM4IARBPROC) (GLint location, GLint v0, GLint v1, GLint v2, GLint v3);
typedef void (*PFNGLUNIFORM1FVARBPROC) (GLint location, GLsizei count, const GLfloat *value);
typedef void (*PFNGLUNIFORM2FVARBPROC) (GLint location, GLsizei count, const GLfloat *value);
typedef void (*PFNGLUNIFORM3FVARBPROC) (GLint location, GLsizei count, const GLfloat *value);
typedef void (*PFNGLUNIFORM4FVARBPROC) (GLint location, GLsizei count, const GLfloat *value);
typedef void (*PFNGLUNIFORM1IVARBPROC) (GLint location, GLsizei count, const GLint *value);
typedef void (*PFNGLUNIFORM2IVARBPROC) (GLint location, GLsizei count, const GLint *value);
typedef void (*PFNGLUNIFORM3IVARBPROC) (GLint location, GLsizei count, const GLint *value);
typedef void (*PFNGLUNIFORM4IVARBPROC) (GLint location, GLsizei count, const GLint *value);
typedef void (*PFNGLUNIFORMMATRIX2FVARBPROC) (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value);
typedef void (*PFNGLUNIFORMMATRIX3FVARBPROC) (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value);
typedef void (*PFNGLUNIFORMMATRIX4FVARBPROC) (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value);
typedef void (*PFNGLGETOBJECTPARAMETERFVARBPROC) (GLhandleARB obj, GLenum pname, GLfloat *params);
typedef void (*PFNGLGETOBJECTPARAMETERIVARBPROC) (GLhandleARB obj, GLenum pname, GLint *params);
typedef void (*PFNGLGETINFOLOGARBPROC) (GLhandleARB obj, GLsizei maxLength, GLsizei *length, GLcharARB *infoLog);
typedef void (*PFNGLGETATTACHEDOBJECTSARBPROC) (GLhandleARB containerObj, GLsizei maxCount, GLsizei *count, GLhandleARB *obj);
typedef GLint (*PFNGLGETUNIFORMLOCATIONARBPROC) (GLhandleARB programObj, const GLcharARB *name);
typedef void (*PFNGLGETACTIVEUNIFORMARBPROC) (GLhandleARB programObj, GLuint index, GLsizei maxLength, GLsizei *length, GLint *size, GLenum *type, GLcharARB *name);
typedef void (*PFNGLGETUNIFORMFVARBPROC) (GLhandleARB programObj, GLint location, GLfloat *params);
typedef void (*PFNGLGETUNIFORMIVARBPROC) (GLhandleARB programObj, GLint location, GLint *params);
typedef void (*PFNGLGETSHADERSOURCEARBPROC) (GLhandleARB obj, GLsizei maxLength, GLsizei *length, GLcharARB *source);

typedef void (*PFNGLVERTEXATTRIB1DARBPROC) (GLuint index, GLdouble x);
typedef void (*PFNGLVERTEXATTRIB1DVARBPROC) (GLuint index, const GLdouble *v);
typedef void (*PFNGLVERTEXATTRIB1FARBPROC) (GLuint index, GLfloat x);
typedef void (*PFNGLVERTEXATTRIB1FVARBPROC) (GLuint index, const GLfloat *v);
typedef void (*PFNGLVERTEXATTRIB1SARBPROC) (GLuint index, GLshort x);
typedef void (*PFNGLVERTEXATTRIB1SVARBPROC) (GLuint index, const GLshort *v);
typedef void (*PFNGLVERTEXATTRIB2DARBPROC) (GLuint index, GLdouble x, GLdouble y);
typedef void (*PFNGLVERTEXATTRIB2DVARBPROC) (GLuint index, const GLdouble *v);
typedef void (*PFNGLVERTEXATTRIB2FARBPROC) (GLuint index, GLfloat x, GLfloat y);
typedef void (*PFNGLVERTEXATTRIB2FVARBPROC) (GLuint index, const GLfloat *v);
typedef void (*PFNGLVERTEXATTRIB2SARBPROC) (GLuint index, GLshort x, GLshort y);
typedef void (*PFNGLVERTEXATTRIB2SVARBPROC) (GLuint index, const GLshort *v);
typedef void (*PFNGLVERTEXATTRIB3DARBPROC) (GLuint index, GLdouble x, GLdouble y, GLdouble z);
typedef void (*PFNGLVERTEXATTRIB3DVARBPROC) (GLuint index, const GLdouble *v);
typedef void (*PFNGLVERTEXATTRIB3FARBPROC) (GLuint index, GLfloat x, GLfloat y, GLfloat z);
typedef void (*PFNGLVERTEXATTRIB3FVARBPROC) (GLuint index, const GLfloat *v);
typedef void (*PFNGLVERTEXATTRIB3SARBPROC) (GLuint index, GLshort x, GLshort y, GLshort z);
typedef void (*PFNGLVERTEXATTRIB3SVARBPROC) (GLuint index, const GLshort *v);
typedef void (*PFNGLVERTEXATTRIB4NBVARBPROC) (GLuint index, const GLbyte *v);
typedef void (*PFNGLVERTEXATTRIB4NIVARBPROC) (GLuint index, const GLint *v);
typedef void (*PFNGLVERTEXATTRIB4NSVARBPROC) (GLuint index, const GLshort *v);
typedef void (*PFNGLVERTEXATTRIB4NUBARBPROC) (GLuint index, GLubyte x, GLubyte y, GLubyte z, GLubyte w);
typedef void (*PFNGLVERTEXATTRIB4NUBVARBPROC) (GLuint index, const GLubyte *v);
typedef void (*PFNGLVERTEXATTRIB4NUIVARBPROC) (GLuint index, const GLuint *v);
typedef void (*PFNGLVERTEXATTRIB4NUSVARBPROC) (GLuint index, const GLushort *v);
typedef void (*PFNGLVERTEXATTRIB4BVARBPROC) (GLuint index, const GLbyte *v);
typedef void (*PFNGLVERTEXATTRIB4DARBPROC) (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w);
typedef void (*PFNGLVERTEXATTRIB4DVARBPROC) (GLuint index, const GLdouble *v);
typedef void (*PFNGLVERTEXATTRIB4FARBPROC) (GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w);
typedef void (*PFNGLVERTEXATTRIB4FVARBPROC) (GLuint index, const GLfloat *v);
typedef void (*PFNGLVERTEXATTRIB4IVARBPROC) (GLuint index, const GLint *v);
typedef void (*PFNGLVERTEXATTRIB4SARBPROC) (GLuint index, GLshort x, GLshort y, GLshort z, GLshort w);
typedef void (*PFNGLVERTEXATTRIB4SVARBPROC) (GLuint index, const GLshort *v);
typedef void (*PFNGLVERTEXATTRIB4UBVARBPROC) (GLuint index, const GLubyte *v);
typedef void (*PFNGLVERTEXATTRIB4UIVARBPROC) (GLuint index, const GLuint *v);
typedef void (*PFNGLVERTEXATTRIB4USVARBPROC) (GLuint index, const GLushort *v);
typedef void (*PFNGLVERTEXATTRIBPOINTERARBPROC) (GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, const GLvoid *pointer);
typedef void (*PFNGLENABLEVERTEXATTRIBARRAYARBPROC) (GLuint index);
typedef void (*PFNGLDISABLEVERTEXATTRIBARRAYARBPROC) (GLuint index);
typedef void (*PFNGLPROGRAMSTRINGARBPROC) (GLenum target, GLenum format, GLsizei len, const GLvoid *string);
typedef void (*PFNGLBINDPROGRAMARBPROC) (GLenum target, GLuint program);
typedef void (*PFNGLDELETEPROGRAMSARBPROC) (GLsizei n, const GLuint *programs);
typedef void (*PFNGLGENPROGRAMSARBPROC) (GLsizei n, GLuint *programs);
typedef void (*PFNGLPROGRAMENVPARAMETER4DARBPROC) (GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w);
typedef void (*PFNGLPROGRAMENVPARAMETER4DVARBPROC) (GLenum target, GLuint index, const GLdouble *params);
typedef void (*PFNGLPROGRAMENVPARAMETER4FARBPROC) (GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w);
typedef void (*PFNGLPROGRAMENVPARAMETER4FVARBPROC) (GLenum target, GLuint index, const GLfloat *params);
typedef void (*PFNGLPROGRAMLOCALPARAMETER4DARBPROC) (GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w);
typedef void (*PFNGLPROGRAMLOCALPARAMETER4DVARBPROC) (GLenum target, GLuint index, const GLdouble *params);
typedef void (*PFNGLPROGRAMLOCALPARAMETER4FARBPROC) (GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w);
typedef void (*PFNGLPROGRAMLOCALPARAMETER4FVARBPROC) (GLenum target, GLuint index, const GLfloat *params);
typedef void (*PFNGLGETPROGRAMENVPARAMETERDVARBPROC) (GLenum target, GLuint index, GLdouble *params);
typedef void (*PFNGLGETPROGRAMENVPARAMETERFVARBPROC) (GLenum target, GLuint index, GLfloat *params);
typedef void (*PFNGLGETPROGRAMLOCALPARAMETERDVARBPROC) (GLenum target, GLuint index, GLdouble *params);
typedef void (*PFNGLGETPROGRAMLOCALPARAMETERFVARBPROC) (GLenum target, GLuint index, GLfloat *params);
typedef void (*PFNGLGETPROGRAMIVARBPROC) (GLenum target, GLenum pname, GLint *params);
typedef void (*PFNGLGETPROGRAMSTRINGARBPROC) (GLenum target, GLenum pname, GLvoid *string);
typedef void (*PFNGLGETVERTEXATTRIBDVARBPROC) (GLuint index, GLenum pname, GLdouble *params);
typedef void (*PFNGLGETVERTEXATTRIBFVARBPROC) (GLuint index, GLenum pname, GLfloat *params);
typedef void (*PFNGLGETVERTEXATTRIBIVARBPROC) (GLuint index, GLenum pname, GLint *params);
typedef void (*PFNGLGETVERTEXATTRIBPOINTERVARBPROC) (GLuint index, GLenum pname, GLvoid* *pointer);
typedef GLboolean (*PFNGLISPROGRAMARBPROC) (GLuint program);

typedef void (*PFNGLBINDATTRIBLOCATIONARBPROC) (GLhandleARB programObj, GLuint index, const GLcharARB *name);
typedef void (*PFNGLGETACTIVEATTRIBARBPROC) (GLhandleARB programObj, GLuint index, GLsizei maxLength, GLsizei *length, GLint *size, GLenum *type, GLcharARB *name);
typedef GLint (*PFNGLGETATTRIBLOCATIONARBPROC) (GLhandleARB programObj, const GLcharARB *name);

typedef void (*PFNGLFOGCOORDFEXTPROC) (GLfloat coord);
typedef void (*PFNGLFOGCOORDFVEXTPROC) (const GLfloat *coord);
typedef void (*PFNGLFOGCOORDDEXTPROC) (GLdouble coord);
typedef void (*PFNGLFOGCOORDDVEXTPROC) (const GLdouble *coord);
typedef void (*PFNGLFOGCOORDPOINTEREXTPROC) (GLenum type, GLsizei stride, const GLvoid *pointer);

#elif defined(Q_WS_X11)
	// GLX
	#include <GL/glx.h>
#else
	// OpenGL extensions are listed in our own header.
	#include "glext.h"
#endif

/******************************************************************************
* This defines some macros based on the current floating point format.
******************************************************************************/

#if defined(FLOATTYPE_FLOAT)

#define glColor3v glColor3fv
#define glColor4v glColor4fv
#define glLoadMatrix glLoadMatrixf
#define glMultMatrix glMultMatrixf
#define glLoadTransposeMatrix glLoadTransposeMatrixf
#define glMultTransposeMatrix glMultTransposeMatrixf
#define glTranslate glTranslatef
#define glRasterPos3 glRasterPos3f
#define glVertex3v glVertex3fv
#define glVertex3 glVertex3f
#define glVertex4v glVertex4fv
#define glNormal3v glNormal3fv
#define glGetFloatTypev glGetFloatv
#define GL_FLOATING_POINT GL_FLOAT

#elif defined(FLOATTYPE_DOUBLE)

#define glColor3v glColor3dv
#define glColor4v glColor4dv
#define glLoadMatrix glLoadMatrixd
#define glMultMatrix glMultMatrixd
#define glLoadTransposeMatrix glLoadTransposeMatrixd
#define glMultTransposeMatrix glMultTransposeMatrixd
#define glTranslate glTranslated
#define glRasterPos3 glRasterPos3d
#define glVertex3v glVertex3dv
#define glVertex3 glVertex3d
#define glVertex4v glVertex4dv
#define glNormal3v glNormal3dv
#define glGetFloatTypev glGetDoublev
#define GL_FLOATING_POINT GL_DOUBLE
#else

#error "No floating point type defined."

#endif

#ifdef WIN32
#define GLSTDCALL __stdcall
#else
#define GLSTDCALL
#endif

// GL_KTX_buffer_region extension:
typedef GLuint (GLSTDCALL *PFNGLNEWBUFFERREGIONPROC)(GLenum type);
typedef void (GLSTDCALL *PFNGLDELETEBUFFERREGIONPROC)(GLuint region);
typedef GLuint (GLSTDCALL *PFNGLBUFFERREGIONENABLEDPROC)(void);
typedef void (GLSTDCALL *PFNGLREADBUFFERREGIONPROC)(GLuint region, GLint x, GLint y, GLsizei width, GLsizei height);
typedef void (GLSTDCALL *PFNGLDRAWBUFFERREGIONPROC)(GLuint region, GLint x, GLint y, GLsizei width, GLsizei height, GLint xDest, GLint yDest);
// Parameter values for the glNewBufferRegion() OpenGL function.
#define GL_KTX_FRONT_REGION		0
#define GL_KTX_BACK_REGION		1
#define GL_KTX_Z_REGION			2
#define GL_KTX_STENCIL_REGION	3

// GL_WIN_swap_hint extension:
typedef void (GLSTDCALL *PFNGLADDSWAPHINTRECTWINPROC)(GLint x, GLint y, GLsizei width, GLsizei height);

// WGL_ARB_pbuffer extension:
#ifdef WIN32
DECLARE_HANDLE(HPBUFFERARB);
typedef HPBUFFERARB (GLSTDCALL *PFNWGLCREATEPBUFFERARB)(HDC hDC, int iPixelFormat, int iWidth, int iHeight, const int *piAttribList);
typedef HDC (GLSTDCALL *PFNWGLGETPBUFFERDVARB)(HPBUFFERARB hPbuffer);
typedef int (GLSTDCALL *PFNWGLRELEASEPBUFFERDVARB)(HPBUFFERARB hPbuffer, HDC hDC);
typedef BOOL (GLSTDCALL *PFNWGLDESTROYPBUFFERARB)(HPBUFFERARB hPbuffer);
typedef BOOL (GLSTDCALL *PFNWGLQUERYPBUFFERARB)(HPBUFFERARB hPbuffer, int iAttribute, int *piValue);
#define WGL_DRAW_TO_PBUFFER_ARB			0x202D
#define WGL_DRAW_TO_PBUFFER_ARB			0x202D
#define WGL_MAX_PBUFFER_PIXELS_ARB		0x202E
#define WGL_MAX_PBUFFER_WIDTH_ARB		0x202F
#define WGL_MAX_PBUFFER_HEIGHT_ARB		0x2030
#define WGL_PBUFFER_LARGEST_ARB			0x2033
#define WGL_PBUFFER_WIDTH_ARB			0x2034
#define WGL_PBUFFER_HEIGHT_ARB			0x2035
#define WGL_PBUFFER_LOST_ARB			0x2036
#endif

namespace Core {

/******************************************************************************
* Error handling function.
* Returns a human-readable string that describes the OpenGL error with
* the given code.
******************************************************************************/
CORE_DLLEXPORT const char* openglErrorString(GLenum errorCode);

/******************************************************************************
* Debugging macro
******************************************************************************/
#ifdef _DEBUG
#define CHECK_OPENGL(cmd)									\
	cmd;													\
	{														\
		GLenum error;										\
		while((error = ::glGetError()) != GL_NO_ERROR) {	\
			qDebug() << "WARNING: OpenGL call" << #cmd << "failed "	\
			"in line" << __LINE__ << "of file" << __FILE__ 	\
			<< "with error" << Core::openglErrorString(error);	\
		}													\
	}
#else
#define CHECK_OPENGL(cmd)			cmd
#endif

/**
 * \brief Stores the entry points of detected OpenGL extensions.
 *
 * Calling detectOpenGLExtensions() detects all OpenGL extensions
 * available on the current computer.
 *
 * The availability of a specific extension can afterwards be checked
 * using the \c hasXXXExtension() family of methods.
 *
 * The Window3DContainer class is derived from the OpenGLExtensions class.
 * It therefore provides all methods defined here including the entry points for
 * the available OpenGL extensions.
 *
 * \author Alexander Stukowski
 * \sa Window3DContainer
 */
class OpenGLExtensions
{
public:
	/// \brief Detects the available OpenGL extensions.
	void detectOpenGLExtensions();

	// GL_EXT_compiled_vertex_array extension:
	PFNGLLOCKARRAYSEXTPROC glLockArrays;
	PFNGLUNLOCKARRAYSEXTPROC glUnlockArrays;
	/// Returns true if the GL_EXT_compiled_vertex_array OpenGL extensions is available.
	bool hasCompiledVertexArraysExtension() const { return _hasCompiledVertexArrays; }

	// GL_KTX_buffer_region extension:
	PFNGLNEWBUFFERREGIONPROC glNewBufferRegion;
	PFNGLDELETEBUFFERREGIONPROC glDeleteBufferRegion;
	PFNGLBUFFERREGIONENABLEDPROC glBufferRegionEnabled;
	PFNGLREADBUFFERREGIONPROC glReadBufferRegion;
	PFNGLDRAWBUFFERREGIONPROC glDrawBufferRegion;
	/// Returns true if the GL_KTX_buffer_region OpenGL extensions is available.
	bool hasBufferRegionsExtension() const { return _hasBufferRegions; }

	// GL_WIN_swap_hint extension:
	PFNGLADDSWAPHINTRECTWINPROC glAddSwapHintRectWIN;
	/// Returns true if the GL_WIN_swap_hint OpenGL extensions is available.
	bool hasSwapHintExtension() const { return _hasSwapHint; }

	// GL_ARB_point_parameters extension:
	PFNGLPOINTPARAMETERFARBPROC  glPointParameterfARB;
	PFNGLPOINTPARAMETERFVARBPROC glPointParameterfvARB;
	/// Returns true if the GL_ARB_point_parameters OpenGL extensions is available.
	bool hasPointParametersExtension() const { return _hasPointParameters; }

	// Vertex buffer objects extension
	PFNGLGENBUFFERSARBPROC glGenBuffersARB;					// VBO Name Generation Procedure
	PFNGLBINDBUFFERARBPROC glBindBufferARB;					// VBO Bind Procedure
	PFNGLBUFFERDATAARBPROC glBufferDataARB;					// VBO Data Loading Procedure
	PFNGLDELETEBUFFERSARBPROC glDeleteBuffersARB;			// VBO Deletion Procedure
	PFNGLMAPBUFFERARBPROC glMapBufferARB;					// VBO Memory Mapping Procedure
	PFNGLUNMAPBUFFERARBPROC glUnmapBufferARB;				// VBO Memory UnMapping Procedure
	PFNGLGETBUFFERPARAMETERIVARBPROC glGetBufferParameterivARB;		// VBO Parameter Procedure
	PFNGLGETBUFFERPOINTERVARBPROC glGetBufferPointervARB;		// VBO Pointer Procedure
	/// Returns true if the GL_ARB_vertex_buffer_object OpenGL extensions is available.
	bool hasVertexBufferObjectsExtension() const { return _hasVertexBufferObjects; }

	// GL_EXT_framebuffer_object extensions:
	PFNGLISRENDERBUFFEREXTPROC glIsRenderbufferEXT;
	PFNGLBINDRENDERBUFFEREXTPROC glBindRenderbufferEXT;
	PFNGLDELETERENDERBUFFERSEXTPROC glDeleteRenderbuffersEXT;
	PFNGLGENRENDERBUFFERSEXTPROC glGenRenderbuffersEXT;
	PFNGLRENDERBUFFERSTORAGEEXTPROC glRenderbufferStorageEXT;
	PFNGLGETRENDERBUFFERPARAMETERIVEXTPROC glGetRenderbufferParameterivEXT;
	PFNGLISFRAMEBUFFEREXTPROC glIsFramebufferEXT;
	PFNGLBINDFRAMEBUFFEREXTPROC glBindFramebufferEXT;
	PFNGLDELETEFRAMEBUFFERSEXTPROC glDeleteFramebuffersEXT;
	PFNGLGENFRAMEBUFFERSEXTPROC glGenFramebuffersEXT;
	PFNGLCHECKFRAMEBUFFERSTATUSEXTPROC glCheckFramebufferStatusEXT;
	PFNGLFRAMEBUFFERTEXTURE1DEXTPROC glFramebufferTexture1DEXT;
	PFNGLFRAMEBUFFERTEXTURE2DEXTPROC glFramebufferTexture2DEXT;
	PFNGLFRAMEBUFFERTEXTURE3DEXTPROC glFramebufferTexture3DEXT;
	PFNGLFRAMEBUFFERRENDERBUFFEREXTPROC glFramebufferRenderbufferEXT;
	PFNGLGETFRAMEBUFFERATTACHMENTPARAMETERIVEXTPROC glGetFramebufferAttachmentParameterivEXT;
	PFNGLGENERATEMIPMAPEXTPROC glGenerateMipmapEXT;
	/// Returns true if the GL_EXT_framebuffer_object OpenGL extensions is available.
	bool hasFrameBufferExtension() const { return _hasFrameBuffer; }

	// GL_ARB_shader_objects extension:
	PFNGLCREATESHADEROBJECTARBPROC glCreateShaderObjectARB;
	PFNGLSHADERSOURCEARBPROC glShaderSourceARB;
	PFNGLCOMPILESHADERARBPROC glCompileShaderARB;
	PFNGLDELETEOBJECTARBPROC glDeleteObjectARB;
	PFNGLCREATEPROGRAMOBJECTARBPROC glCreateProgramObjectARB;
	PFNGLATTACHOBJECTARBPROC glAttachObjectARB;
	PFNGLLINKPROGRAMARBPROC glLinkProgramARB;
	PFNGLUSEPROGRAMOBJECTARBPROC glUseProgramObjectARB;
	PFNGLGETINFOLOGARBPROC glGetInfoLogARB;
	PFNGLGETHANDLEARBPROC glGetHandleARB;
	PFNGLGETOBJECTPARAMETERFVARBPROC glGetObjectParameterfvARB;
	PFNGLGETOBJECTPARAMETERIVARBPROC glGetObjectParameterivARB;
	PFNGLGETUNIFORMLOCATIONARBPROC glGetUniformLocationARB;
	PFNGLUNIFORM1FARBPROC glUniform1fARB;
	PFNGLUNIFORM2FARBPROC glUniform2fARB;
	PFNGLUNIFORM3FARBPROC glUniform3fARB;
	PFNGLUNIFORM4FARBPROC glUniform4fARB;
	PFNGLUNIFORM1IARBPROC glUniform1iARB;
	PFNGLUNIFORM2IARBPROC glUniform2iARB;
	PFNGLUNIFORM3IARBPROC glUniform3iARB;
	PFNGLUNIFORM4IARBPROC glUniform4iARB;
	PFNGLUNIFORM1FVARBPROC glUniform1fvARB;
	PFNGLUNIFORM2FVARBPROC glUniform2fvARB;
	PFNGLUNIFORM3FVARBPROC glUniform3fvARB;
	PFNGLUNIFORM4FVARBPROC glUniform4fvARB;
	PFNGLUNIFORM1IVARBPROC glUniform1ivARB;
	PFNGLUNIFORM2IVARBPROC glUniform2ivARB;
	PFNGLUNIFORM3IVARBPROC glUniform3ivARB;
	PFNGLUNIFORM4IVARBPROC glUniform4ivARB;
	PFNGLUNIFORMMATRIX2FVARBPROC glUniformMatrix2fvARB;
	PFNGLUNIFORMMATRIX3FVARBPROC glUniformMatrix3fvARB;
	PFNGLUNIFORMMATRIX4FVARBPROC glUniformMatrix4fvARB;
	/// Returns true if the GL_ARB_shader_objects OpenGL extensions is available.
	bool hasShaderObjectsExtension() const { return _hasShaderObjects; }

	// GL_ARB_vertex_shader extension:
	PFNGLENABLEVERTEXATTRIBARRAYARBPROC glEnableVertexAttribArrayARB;
	PFNGLDISABLEVERTEXATTRIBARRAYARBPROC glDisableVertexAttribArrayARB;
	PFNGLBINDATTRIBLOCATIONARBPROC glBindAttribLocationARB;
	PFNGLGETATTRIBLOCATIONARBPROC glGetAttribLocationARB;
	PFNGLVERTEXATTRIBPOINTERARBPROC glVertexAttribPointerARB;
	PFNGLGETACTIVEATTRIBARBPROC glGetActiveAttribARB;
	PFNGLVERTEXATTRIB1FARBPROC glVertexAttrib1fARB;
	PFNGLVERTEXATTRIB1SARBPROC glVertexAttrib1sARB;
	PFNGLVERTEXATTRIB1DARBPROC glVertexAttrib1dARB;
	PFNGLVERTEXATTRIB2FARBPROC glVertexAttrib2fARB;
	PFNGLVERTEXATTRIB2SARBPROC glVertexAttrib2sARB;
	PFNGLVERTEXATTRIB2DARBPROC glVertexAttrib2dARB;
	PFNGLVERTEXATTRIB3FARBPROC glVertexAttrib3fARB;
	PFNGLVERTEXATTRIB3SARBPROC glVertexAttrib3sARB;
	PFNGLVERTEXATTRIB3DARBPROC glVertexAttrib3dARB;
	PFNGLVERTEXATTRIB4FARBPROC glVertexAttrib4fARB;
	PFNGLVERTEXATTRIB4SARBPROC glVertexAttrib4sARB;
	PFNGLVERTEXATTRIB4DARBPROC glVertexAttrib4dARB;
	PFNGLVERTEXATTRIB4NUBARBPROC glVertexAttrib4NubARB;
	PFNGLVERTEXATTRIB1FVARBPROC glVertexAttrib1fvARB;
	PFNGLVERTEXATTRIB2FVARBPROC glVertexAttrib2fvARB;
	PFNGLVERTEXATTRIB3FVARBPROC glVertexAttrib3fvARB;
	PFNGLVERTEXATTRIB4FVARBPROC glVertexAttrib4fvARB;
	/// Returns true if the GL_ARB_vertex_shader OpenGL extensions is available.
	bool hasVertexShaderExtension() const { return _hasVertexShader; }

	// GL_EXT_fog_coord extension:
	PFNGLFOGCOORDFEXTPROC glFogCoordfEXT;
	PFNGLFOGCOORDDEXTPROC glFogCoorddEXT;
	PFNGLFOGCOORDFVEXTPROC glFogCoordfvEXT;
	PFNGLFOGCOORDDVEXTPROC glFogCoorddvEXT;
	PFNGLFOGCOORDPOINTEREXTPROC glFogCoordPointerEXT;
	/// Returns true if the GL_EXT_fog_coord OpenGL extensions is available.
	bool hasFogCoordExtension() const { return _hasFogCoord; }

	// GL_ARB_fragment_shader extension:
	/// Returns true if the GL_ARB_fragment_shader OpenGL extensions is available.
	bool hasFragmentShaderExtension() const { return _hasFragmentShader; }

	/// Returns the driver vendor string returned by the OpenGL implemenation.
	const QByteArray& driverVendor() const { return _driverVendor; }
	
	/// Returns the renderer string returned by the OpenGL implemenation.
	const QByteArray& rendererName() const { return _rendererName; }


private:
	/// Checks if a single OpenGL extension is available.
	bool isExtensionSupported(const char* szTargetExtension);
	/// Returns the address of an OpenGL extension procedure.
	void* GetProcAddress(const char* procedureName);

	bool _hasCompiledVertexArrays;
	bool _hasFogCoord;
	bool _hasFragmentShader;
	bool _hasVertexShader;
	bool _hasShaderObjects;
	bool _hasFrameBuffer;
	bool _hasVertexBufferObjects;
	bool _hasBufferRegions;
	bool _hasPointParameters;
	bool _hasSwapHint;
	
	QByteArray _driverVendor;
	QByteArray _rendererName;
};

};

#endif // __OPENGL_INTERFACE_H
