///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/undo/UndoManager.h>
#include <core/data/DataSetManager.h>
#include <core/actions/ActionManager.h>

namespace Core {

/// The singleton instance of this class.
UndoManager* UndoManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the undo manager.
******************************************************************************/
UndoManager::UndoManager() : currentIndex(-1), suspendCount(0), maxUndoSteps(40),
	_isUndoing(false), _isRedoing(false)
{
}

/******************************************************************************
* Opens a new compound operation and assigns it the given display name.
******************************************************************************/
CompoundOperation* UndoManager::beginCompoundOperation(const QString& displayName)
{
	CompoundOperation* cop = new CompoundOperation(displayName);
	compoundStack.push_back(cop);
	return cop;
}

/******************************************************************************
* Closes the current compound operation.
******************************************************************************/
void UndoManager::endCompoundOperation()
{
	OVITO_ASSERT(!compoundStack.empty());
	if(compoundStack.empty()) 
		throw Exception("Invalid operation.");
	CompoundOperation* cop = compoundStack.back();
	CHECK_POINTER(cop);
	compoundStack.pop_back();
	if(suspendCount > 0 || !cop->isSignificant()) {
		delete cop;
		return;
	}
	if(compoundStack.empty()) {
		for(int i=currentIndex + 1; i<operations.size(); i++)
			delete operations[i];
		operations.remove(currentIndex + 1, operations.size() - (currentIndex + 1));
		operations.push_back(cop);
		currentIndex++;
		OVITO_ASSERT(currentIndex == operations.size()-1);
		limitUndoStack();
		updateUI();

		// Set dirty flag of data set.
		DATASET_MANAGER.currentSet()->setDirty();
	}
	else {
		CompoundOperation* parentOp = compoundStack.back();
		parentOp->addOperation(cop);
	}
}

/******************************************************************************
* Registers a aingle undoable operation. 
* This object will be put onto the undo stack.
******************************************************************************/
void UndoManager::addOperation(UndoableOperation* operation)
{
	CHECK_POINTER(operation);
	OVITO_ASSERT_MSG(isUndoingOrRedoing() == false, "UndoManager::addOperation()", "Cannot record an operation while undoing or redoing another operation.");
	if(suspendCount > 0 || compoundStack.empty()) {
		delete operation;
		return;
	}
	compoundStack.back()->addOperation(operation);
}

/******************************************************************************
* Shrinks the undo stack to maximum number of undo steps.
******************************************************************************/
void UndoManager::limitUndoStack()
{
	int n = operations.size() - maxUndoSteps;
	if(maxUndoSteps >= 0 && n > 0) {
		if(currentIndex >= n) {
			for(int i=0; i<n; i++)
				delete operations[i];
			operations.remove(0, n);
			currentIndex -= n;
		}
	}
}

/******************************************************************************
* Resets the undo system. The undo stack will be cleared.
******************************************************************************/
void UndoManager::reset()
{
	Q_FOREACH(UndoableOperation* op, operations)
		delete op;
	operations.clear();
	Q_FOREACH(CompoundOperation* op, compoundStack)
		delete op;
	compoundStack.clear();
	currentIndex = -1;
	updateUI();
}

/******************************************************************************
* Undoes the last operation in the undo stack.
******************************************************************************/
void UndoManager::undo()
{
	OVITO_ASSERT_MSG(compoundStack.empty(), "UndoManager::undo()", "Cannot undo last operation while a compound operation is open.");
	if(currentIndex < 0) return;
	
	UndoSuspender noUndoRecording;

	UndoableOperation* curOp = currentOperation();
	CHECK_POINTER(curOp);
	_isUndoing = true;
	try {
		curOp->undo();
	}
	catch(const Exception& ex) {
		ex.showError();
	}
	_isUndoing = false;
	currentIndex--;
	updateUI();

	// Set dirty flag of data set.
	DATASET_MANAGER.currentSet()->setDirty();
}

/******************************************************************************
* Redoes the last undone operation in the undo stack.
******************************************************************************/
void UndoManager::redo()
{
	OVITO_ASSERT_MSG(compoundStack.empty(), "UndoManager::redo()", "Cannot redo operation while a compound operation is open.");
	if(currentIndex >= (int)operations.size() - 1) return;
	
	UndoSuspender noUndoRecording;
	
	UndoableOperation* nextOp = operations[currentIndex + 1];
	CHECK_POINTER(nextOp);
	_isRedoing = true;
	try {
		nextOp->redo();
	}
	catch(const Exception& ex) {
		ex.showError();
	}
	_isRedoing = false;
	currentIndex++;
	updateUI();

	// Set dirty flag of data set.
	DATASET_MANAGER.currentSet()->setDirty();
}

/******************************************************************************
* Updates the Undo/Redo menu items.
******************************************************************************/
void UndoManager::updateUI()
{	
	statusChanged();
	
	// Update the UndoManager's menu items.
	
	ActionProxy* undoAction = ACTION_MANAGER.findActionProxy(ACTION_EDIT_UNDO);
	CHECK_POINTER(undoAction);
	undoAction->setEnabled(currentIndex >= 0);
	if(currentIndex >= 0)
		undoAction->setText(tr("Undo %1").arg(currentOperation()->displayName()));
	else
		undoAction->setText(tr("Undo"));
		
	ActionProxy* redoAction = ACTION_MANAGER.findActionProxy(ACTION_EDIT_REDO);
	CHECK_POINTER(redoAction);
	redoAction->setEnabled(currentIndex < (int)operations.size()-1);
	if(currentIndex < (int)operations.size()-1)
		redoAction->setText(tr("Redo %1").arg(operations[currentIndex+1]->displayName()));
	else
		redoAction->setText(tr("Redo"));
}

/******************************************************************************
* Undo the compound edit operation that was made.
******************************************************************************/
void CompoundOperation::undo()
{
	UndoSuspender noUndoRecording;
	for(int i=subOperations.size()-1; i>=0; --i) {
		CHECK_POINTER(subOperations[i]);
		subOperations[i]->undo();
	}
}

/******************************************************************************
* Re-apply the compound change, assuming that it has been undone. 
******************************************************************************/
void CompoundOperation::redo()
{
	UndoSuspender noUndoRecording;
	for(int i=0; i<subOperations.size(); i++) {
		CHECK_POINTER(subOperations[i]);
		subOperations[i]->redo();
	}		
}

/******************************************************************************
* Provides a localized, human readable description of this operation. 
******************************************************************************/
QString SimplePropertyChangeOperation::displayName() const 
{ 
	return UndoManager::tr("Change %1").arg(propertyName); 
}

};
