///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_ACTIONMANAGER_H
#define __OVITO_ACTIONMANAGER_H

#include <core/Core.h>
#include "ActionProxy.h"
#include "Action.h"

namespace Core {

//////////////////////// Predefined action identifier ///////////////////////////

/// This action closes the main window and exits the application.
#define ACTION_EXIT				"App.Exit"
/// This action loads an empty scene file.
#define ACTION_FILE_NEW			"App.File.New"
/// This action resets the application to its initial state.
#define ACTION_FILE_RESET		"App.File.Reset"
/// This action shows the file open dialog.
#define ACTION_FILE_OPEN		"App.File.Open"
/// This action saves the current file.
#define ACTION_FILE_SAVE		"App.File.Save"
/// This action shows the file save as dialog.
#define ACTION_FILE_SAVEAS		"App.File.SaveAs"
/// This action shows the file import dialog.
#define ACTION_FILE_IMPORT		"App.File.Import"
/// This action shows the file export dialog.
#define ACTION_FILE_EXPORT		"App.File.Export"

/// This action shows the about dialog.
#define ACTION_HELP_ABOUT				"App.Help.About"
/// This action shows the online help.
#define ACTION_HELP_SHOW_ONLINE_HELP	"App.Help.ShowOnlineHelp"

/// This action undoes the last operation.
#define ACTION_EDIT_UNDO		"App.Edit.Undo"
/// This action redoues the last undone operation.
#define ACTION_EDIT_REDO		"App.Edit.Redo"
/// This action deletes the selected objects.
#define ACTION_EDIT_DELETE		"App.Edit.Delete"

/// This action maximizes the active viewport.
#define ACTION_VIEWPORT_MAXIMIZE	"App.Viewport.Maximize"
/// This action activates the viewport zoom mode.
#define ACTION_VIEWPORT_ZOOM		"App.Viewport.Zoom"
/// This action activates the viewport pan mode.
#define ACTION_VIEWPORT_PAN			"App.Viewport.Pan"
/// This action activates the viewport orbit mode.
#define ACTION_VIEWPORT_ORBIT		"App.Viewport.Orbit"
/// This action activates the field of view viewport mode.
#define ACTION_VIEWPORT_FOV			"App.Viewport.FOV"
/// This action activates the 'pick center of rotation' input mode.
#define ACTION_VIEWPORT_PICK_ORBIT_CENTER		"App.Viewport.Orbit.PickCenter"
/// This zooms the current viewport to the scene extents
#define ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS		"App.Viewport.Zoom.SceneExtents"
/// This zooms the current viewport to the selection extents
#define ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS	"App.Viewport.Zoom.SelectionExtents"
/// This zooms all viewports to the scene extents
#define ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL		"App.Viewport.SceneExtents.All"
/// This zooms all viewports to the selection extents
#define ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL	"App.Viewport.SelectionExtents.All"

/// This action activates the object selection mode.
#define ACTION_SELECTION_MODE		"App.Mode.Selection"
/// This action activates the object move mode.
#define ACTION_MOVE_MODE			"App.Mode.Move"
/// This action activates the object rotation mode.
#define ACTION_ROTATION_MODE		"App.Mode.Rotate"
/// This action activates the object scaling mode.
#define ACTION_SCALING_MODE			"App.Mode.Scale"

/// This action activates transformation system selector.
#define ACTION_XFORM_SYSTEM			"App.XForm.System"
/// This action activates transformation center selector.
#define ACTION_XFORM_CENTER			"App.XForm.Center"

/// This action activates/deactivates the snapping on objects/grid.
#define ACTION_SNAPPING_OBJECT		"App.Snapping.Object"
/// This action activates/deactivates the snapping for angle values.
#define ACTION_SNAPPING_ANGLE		"App.Snapping.Angle"
/// This action activates/deactivates the snapping for percent values.
#define ACTION_SNAPPING_PERCENT		"App.Snapping.Percent"

/// This action deletes the currently selected modifier from the modifier stack.
#define ACTION_MODIFIER_DELETE		"App.Modifier.Delete"
/// This action moves the currently selected modifer up one entry in the modifier stack.
#define ACTION_MODIFIER_MOVE_UP		"App.Modifier.MoveUp"
/// This action moves the currently selected modifer up down entry in the modifier stack.
#define ACTION_MODIFIER_MOVE_DOWN	"App.Modifier.MoveDown"
/// This action toggles the enabled/disable state of the currently selected modifer.
#define ACTION_MODIFIER_TOGGLE_STATE	"App.Modifier.ToggleEnabledState"

/// This action jumps to the start of the animation
#define ACTION_GOTO_START_OF_ANIMATION		"App.Animation.Goto.Start"
/// This action jumps to the end of the animation
#define ACTION_GOTO_END_OF_ANIMATION		"App.Animation.Goto.End"
/// This action jumps to previous frame in the animation
#define ACTION_GOTO_PREVIOUS_FRAME			"App.Animation.Goto.PreviousFrame"
/// This action jumps to next frame in the animation
#define ACTION_GOTO_NEXT_FRAME				"App.Animation.Goto.NextFrame"
/// This action toggles animation playback
#define ACTION_TOGGLE_ANIMATION_PLAYBACK	"App.Animation.TogglePlayback"
/// This action starts the animation playback
#define ACTION_START_ANIMATION_PLAYBACK		"App.Animation.StartPlayback"
/// This action starts the animation playback
#define ACTION_STOP_ANIMATION_PLAYBACK		"App.Animation.StopPlayback"
/// This action shows the animation settings dialog
#define ACTION_ANIMATION_SETTINGS			"App.Animation.Settings"
/// This action activates/deactivates the animation mode
#define ACTION_ANIMATION_MODE_TOGGLE		"App.Animation.ToggleRecording"

/// This action starts rendering of the current view.
#define ACTION_RENDER_ACTIVE_VIEWPORT		"App.Render.ActiveViewport"
/// This action shows a dialog box that lets the user select the renderer plugin.
#define ACTION_SELECT_RENDERER_DIALOG		"App.Render.SelectRenderer"
/// This action displays the frame buffer windows showing the last rendered image.
#define ACTION_SHOW_FRAME_BUFFER			"App.Render.ShowFrameBuffer"

/// This actions open the application's "Settings" dialog.
#define ACTION_SETTINGS_DIALOG				"App.Settings"

/// \def ACTION_MANAGER
/// \brief The predefined instance of the Core::ActionManager class.
///
/// Always use this macro to access the Core::ActionManager class instance.
#define ACTION_MANAGER		(*ActionManager::getSingletonInstance())

/**
 * \brief Manages the application-wide actions.
 *
 * This is a singleton class with only one predefined instance of this class.
 * You can access the instance of this class using the ACTION_MANAGER macro.
 *
 * \author Alexander Stukowski
 * \sa Viewport
 */
class CORE_DLLEXPORT ActionManager : public QObject
{
	Q_OBJECT

public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the ActionManager singleton class.
	/// \note You should use the ViewportManager macro to access the ActionManager instance instead
	///       of this method.
	inline static ActionManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "ActionManager::getSingletonInstance", "ActionManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns the proxy for the action with the given ID.
	/// \param actionId The unique identifier string of the action to return.
	/// \return The action's proxy or \c NULL if there is no such action.
	/// \sa getActionProxies()
	ActionProxy* findActionProxy(const QString& actionId);

	/// \brief Returns the list of all registered action proxies.
	/// \return The complete list of ActionProxy objects.
	/// \sa findActionProxy()
	QList<ActionProxy*> getActionProxies();

	/// \brief Returns an action with the given ID.
	/// \param actionId The unique identifier string of the action to return.
	/// \return The action object or \c NULL if there is no such action.
	Action::SmartPtr getAction(const QString& actionId);

	/// \brief Invokes the command action with the given ID.
	/// \param actionId The unique identifier string of the action to invoke.
	/// \throw Exception if the action does not exist or if an error occurs during the action invokation.
	void invokeAction(const QString& actionId);

	/// \brief Registers a new action proxy with the ActionManager.
	/// \param actionProxy The action proxy to be registered. The ActionManager will become
	///                    the owner (parent) of the object.
	void addAction(ActionProxy* actionProxy);

	/// \brief Registers a new action with the ActionManager.
	/// \param action The action to be registered.
	/// \return The newly created ActionProxy that encapsulates the action object.
	ActionProxy* addAction(const Action::SmartPtr& action);

	/// \brief Registers a new action with the ActionManager.
	/// \param action The action to be registered.
	/// \param title The title string for the action.
	/// \param icon The icon for the action.
	/// \return The newly created ActionProxy that encapsulates the action object.
	ActionProxy* addAction(const Action::SmartPtr& action, const QString& title, const QIcon& icon) {
		ActionProxy* proxy = addAction(action);
		proxy->setText(title);
		proxy->setIcon(icon);
		return proxy;
	}

	/// \brief Registers a new action with the ActionManager.
	/// \param action The action to be registered.
	/// \param title The title string for the action.
	/// \param iconPath The file path of the icon image for the action.
	/// \return The newly created ActionProxy that encapsulates the action object.
	ActionProxy* addAction(const Action::SmartPtr& action, const QString& title, const char* iconPath = NULL) {
		ActionProxy* proxy = addAction(action);
		proxy->setText(title);
		if(iconPath != NULL && APPLICATION_MANAGER.guiMode())
			proxy->setIcon(QIcon(QString(iconPath)));
		return proxy;
	}

private:

	/// Creates and registers the standard actions with the action manager.
	void registerStandardActions();

	/// This is a singleton class. No public instances allowed.
	ActionManager() {}

	/// Initializes the ActionManager.
	static void initialize() {
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new ActionManager();
		_singletonInstance->registerStandardActions();
	}

	/// ActionManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	/// The singleton instance of this class.
	static ActionManager* _singletonInstance;

	friend class ApplicationManager;
};

};

#endif // __OVITO_ACTIONMANAGER_H
