///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file SimulationCell.h
 * \brief Contains the definition of the AtomViz::SimulationCell class.
 */

#ifndef __SIMULATION_CELL_H
#define __SIMULATION_CELL_H

#include <atomviz/AtomViz.h>

#include <core/reference/RefTarget.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <core/gui/properties/FloatPropertyUI.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/ObjectNode.h>
#include <core/viewport/Viewport.h>

namespace AtomViz {

/**
 * \brief Stores the geometry and periodic boundary conditions of the
 *        simulation box.
 *
 * Each instance of the AtomsObject class owns one SimulationCell object
 * that describes the geometry of the simulation box. It can be accessed via
 * AtomsObject::simulationCell() and changed via AtomsObject::setSimulationCell().
 *
 * The simulation box is defined by three edge vectors as a parallelepiped.
 * A fourth vector defines the origin of the simulation box in space.
 *
 * For each of the three spatial directions a boolean flags controls whether
 * perdiodic boundary conditions are active. This information is used by
 * several modifiers that work on the relative position of atoms.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT SimulationCell : public RefTarget
{
public:

	/// \brief Creates an empty simulation cell.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	SimulationCell(bool isLoading = false) : RefTarget(isLoading),
		_cellVector1(NULL_VECTOR), _cellVector2(NULL_VECTOR), _cellVector3(NULL_VECTOR),
		_cellOrigin(ORIGIN), _pbcX(false), _pbcY(false), _pbcZ(false) {
		init(isLoading);
	}

	/// \brief Constructs a cell from three vectors specifiying the cell's axes.
	/// \param a1 The first edge vector.
	/// \param a2 The second edge vector.
	/// \param a3 The third edge vector.
	///
	/// The three edge vectors must form a right-handed coordinate system, i.e.
	/// the cell volume must not be negative.
	///
	/// The origin of the simulation cell is initialized to (0,0,0).
	SimulationCell(const Vector3& a1, const Vector3& a2, const Vector3& a3) : RefTarget(false),
		_cellVector1(a1), _cellVector2(a2), _cellVector3(a3),
		_cellOrigin(ORIGIN), _pbcX(false), _pbcY(false), _pbcZ(false) {
		init();
	}

	/// \brief Constructs a cell from the origin point and three vectors specifiying the cell's axes.
	/// \param origin The origin point of the simulation cell.
	/// \param a1 The first edge vector.
	/// \param a2 The second edge vector.
	/// \param a3 The third edge vector.
	///
	/// The three edge vectors must form a right-handed coordinate system, i.e.
	/// the cell volume must not be negative.
	SimulationCell(const Point3& origin, const Vector3& a1, const Vector3& a2, const Vector3& a3) : RefTarget(false),
		_cellVector1(a1), _cellVector2(a2), _cellVector3(a3),
		_cellOrigin(origin), _pbcX(false), _pbcY(false), _pbcZ(false) {
		init();
	}

	/// \brief Constructs a cell from a matrix specifying its shape.
	/// \param cellMatrix The matrix
	SimulationCell(const AffineTransformation& cellMatrix) : RefTarget(false),
		_cellVector1(cellMatrix.column(0)), _cellVector2(cellMatrix.column(1)), _cellVector3(cellMatrix.column(2)),
		_cellOrigin(ORIGIN + cellMatrix.column(3)), _pbcX(false), _pbcY(false), _pbcZ(false) {
		init();
	}

	/// \brief Constructs a cell with an axis-aligned box shape.
	/// \param box The axis-aligned box.
	/// \param pbcX Specifies whether perdiodic boundary conditions are applied in the X direction.
	/// \param pbcY Specifies whether perdiodic boundary conditions are applied in the Y direction.
	/// \param pbcZ Specifies whether perdiodic boundary conditions are applied in the Z direction.
	SimulationCell(const Box3& box, bool pbcX = false, bool pbcY = false, bool pbcZ = false) : RefTarget(false),
		_cellVector1(box.sizeX(), 0, 0), _cellVector2(0, box.sizeY(), 0), _cellVector3(0, 0, box.sizeZ()),
		_cellOrigin(box.minc), _pbcX(pbcX), _pbcY(pbcY), _pbcZ(pbcZ) {
		init();
		OVITO_ASSERT_MSG(box.sizeX() >= 0 && box.sizeY() >= 0 && box.sizeZ() >= 0, "SimulationCell constructor", "The simulation box must have a non-negative volume.");
		OVITO_ASSERT(volume() >= 0.0);
	}

	/// \brief Changes the cell vectors to the given axis-aligned orthogonal box.
	/// \param box The axis-aligned box.
	/// \undoable
	void setBoxShape(const Box3& box) {
		OVITO_ASSERT_MSG(box.sizeX() >= 0 && box.sizeY() >= 0 && box.sizeZ() >= 0, "SimulationCell::setBoxShape", "The simulation box must have a non-negative volume.");
		_cellVector1 = Vector3(box.sizeX(), 0, 0);
		_cellVector2 = Vector3(0, box.sizeY(), 0);
		_cellVector3 = Vector3(0, 0, box.sizeZ());
		_cellOrigin = box.minc;
	}

	/// \brief Changes the cell's origin point and its three vectors specifiying the cell's axes.
	/// \param origin The origin point of the simulation cell.
	/// \param a1 The first edge vector.
	/// \param a2 The second edge vector.
	/// \param a3 The third edge vector.
	///
	/// The three edge vectors must form a right-handed coordinate system, i.e.
	/// the cell volume must not be negative.
	/// \undoable
	void setCellShape(const Point3& origin, const Vector3& a1, const Vector3& a2, const Vector3& a3) {
		_cellVector1 = a1;
		_cellVector2 = a2;
		_cellVector3 = a3;
		_cellOrigin = origin;
	}

	/// \brief Changes the cell's shape.
	/// \param shape Specifies the new shape matrix for the simulation cell.
	///              The first three matrix columns contain the three edge vectors
	///              and the fourth matrix column specifies the translation of the cell origin.
	///
	/// The three edge vectors must form a right-handed coordinate system, i.e.
	/// the cell volume must not be negative.
	/// \undoable
	/// \sa cellMatrix()
	void setCellMatrix(const AffineTransformation& shape) {
		_cellVector1 = shape.column(0);
		_cellVector2 = shape.column(1);
		_cellVector3 = shape.column(2);
		_cellOrigin = ORIGIN + shape.getTranslation();
	}

	/// \brief Returns the dimensions of the simulation cell as a 3x4 matrix.
	/// \return Contains the shape matrix of the simulation cell.
	///         The first three matrix columns contain the three edge vectors
	///         and the fourth matrix column specifies the translation of the cell origin.
	/// \sa setCellMatrix()
	AffineTransformation cellMatrix() const {
		AffineTransformation tm;
		tm.setColumn(0, _cellVector1);
		tm.setColumn(1, _cellVector2);
		tm.setColumn(2, _cellVector3);
		tm.setTranslation(_cellOrigin.value() - ORIGIN);
		return tm;
	}

	/// \brief Returns the periodicity of the simulation cell in each direction.
	/// \return An array of boolean flags that indicate whether perdiodic boundary conditions
	///         are applied in each of the three spatial directions.
	/// \sa setPeriodicity()
	array<bool,3> periodicity() const {
		array<bool, 3> pbc;
		pbc[0] = _pbcX;
		pbc[1] = _pbcY;
		pbc[2] = _pbcZ;
		return pbc;
	}

	/// \brief Sets the periodicity of the simulation cell in each spatial direction.
	/// \param pbcX Specifies whether periodic boundary conditions are applied in the direction of the first edge vector of the cell.
	/// \param pbcY Specifies whether periodic boundary conditions are applied in the direction of the second edge vector of the cell.
	/// \param pbcZ Specifies whether periodic boundary conditions are applied in the direction of the third edge vector of the cell.
	/// \undoable
	/// \sa periodicity()
	void setPeriodicity(bool pbcX, bool pbcY, bool pbcZ) {
		 _pbcX = pbcX;
		 _pbcY = pbcY;
		 _pbcZ = pbcZ;
	}

	/// \brief Computes the volume of the simulation cell.
	/// \return The volume of the simulation box. This is equal to the determinant
	///         of the cemmMatrix().
	FloatType volume() const { return cellMatrix().determinant(); }

	/// \brief Returns the axis-aligned bounding box of the simulation cell.
	/// \return An axis-aligned box that completely covers the simulation cell.
	Box3 boundingBox() const {
		return Box3(ORIGIN, Point3(1,1,1)).transformed(cellMatrix());
	}

	/// \brief Returns the line width used to render the simulation cell box.
	/// \return The line with in world units or zero if the simulation box is not rendered.
	/// \sa setSimulationCellLineWidth()
	FloatType simulationCellLineWidth() const { return _simulationCellLineWidth; }

	/// \brief Sets the line width used to render the simulation cell box.
	/// \param newWidth The new line width in world units or zero to not render the box at all.
	/// \undoable
	/// \sa simulationCellLineWidth()
	void setSimulationCellLineWidth(FloatType newWidth) { _simulationCellLineWidth = newWidth; }

	/// \brief Returns whether the simulation cell is visible in the rendered image.
	/// \return The visibility flag.
	/// \sa setRenderSimulationCell()
	bool renderSimulationCell() const { return _renderSimulationCell; }

	/// \brief Sets whether the simulation cell is visible in the rendered image.
	/// \param on The visibility flag.
	/// \undoable
	/// \sa renderSimulationCell()
	void setRenderSimulationCell(bool on) { _renderSimulationCell = on; }

	/// \brief Returns the color used for rendering the simulation cell.
	/// \return The line color
	/// \sa setSimulationCellRenderingColor()
	Color simulationCellRenderingColor() const { return _simulationCellColor; }

	/// \brief Sets the color to be used for rendering the simulation cell.
	/// \param Color The new line color.
	/// \undoable
	/// \sa simulationCellRenderingColor()
	void setSimulationCellRenderingColor(const Color& color) { _simulationCellColor = color; }

	/// \brief Renders the simulation cell in the viewports.
	/// \param time The current animation time.
	/// \param vp The viewport into which the box should be rendered.
	/// \param contextNode The scene object the AtomsObject belongs to.
	void render(TimeTicks time, Viewport* vp, ObjectNode* contextNode);

	/// \brief Renders the simulation cell in high-quality mode to an offscreen buffer.
	void renderHQ(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

public:

	Q_PROPERTY(FloatType simulationCellLineWidth READ simulationCellLineWidth WRITE setSimulationCellLineWidth)
	Q_PROPERTY(bool renderSimulationCell READ renderSimulationCell WRITE setRenderSimulationCell)

protected:

	/// Creates the storage for the internal parameters.
	void init(bool isLoading = false);

	/// Stores the first cell edge.
	PropertyField<Vector3> _cellVector1;
	/// Stores the second cell edge.
	PropertyField<Vector3> _cellVector2;
	/// Stores the third cell edge.
	PropertyField<Vector3> _cellVector3;
	/// Stores the cell origin.
	PropertyField<Point3> _cellOrigin;

	/// Specifies periodic boundary condition in the X direction.
	PropertyField<bool> _pbcX;
	/// Specifies periodic boundary condition in the Y direction.
	PropertyField<bool> _pbcY;
	/// Specifies periodic boundary condition in the Z direction.
	PropertyField<bool> _pbcZ;

	/// Controls the line width used to render the simulation cell.
	PropertyField<FloatType> _simulationCellLineWidth;

	/// Controls whether the simulation cell is visible in the rendered image.
	PropertyField<bool> _renderSimulationCell;

	/// Controls the rendering color of the simulation cell.
	PropertyField<Color> _simulationCellColor;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(SimulationCell)
	DECLARE_PROPERTY_FIELD(_cellVector1)
	DECLARE_PROPERTY_FIELD(_cellVector2)
	DECLARE_PROPERTY_FIELD(_cellVector3)
	DECLARE_PROPERTY_FIELD(_cellOrigin)
	DECLARE_PROPERTY_FIELD(_pbcX)
	DECLARE_PROPERTY_FIELD(_pbcY)
	DECLARE_PROPERTY_FIELD(_pbcZ)
	DECLARE_PROPERTY_FIELD(_renderSimulationCell)
	DECLARE_PROPERTY_FIELD(_simulationCellLineWidth)
	DECLARE_PROPERTY_FIELD(_simulationCellColor)
};

/**
 * \brief A properties editor for the SimulationCell class.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT SimulationCellEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when a spinner's value has changed.
	void onSizeSpinnerValueChanged(int dim);

	/// Is called when the user begins dragging a spinner interactively.
	void onSizeSpinnerDragStart(int dim);

	/// Is called when the user stops dragging a spinner interactively.
	void onSizeSpinnerDragStop(int dim);

	/// Is called when the user aborts dragging a spinner interactively.
	void onSizeSpinnerDragAbort(int dim);

	/// After the simulation cell size has changed, updates the UI controls.
	void updateSimulationBoxSize();

private:

	/// After the user has changed a spinner value, this method changes the
	/// simulation cell geometry.
	void changeSimulationBoxSize(int dim);

	SpinnerWidget* simCellSizeSpinners[3];

	FloatPropertyUI* lineWidthUI;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(SimulationCellEditor)
};

};	// End of namespace AtomViz

#endif // __SIMULATION_CELL_H
