# 12 "omake_ast_lex.mll"
 

include Omake_pos.Make (struct let name = "Omake_ast_lex" end)


let debug_lex =
   Lm_debug.create_debug (**)
      { debug_name = "debug-ast-lex";
        debug_description = "Print tokens as they are scanned";
        debug_value = false
      }

(*
 * Current mode:
 *    ModeNormal: normal lexing mode
 *    ModeString s: parsing a literal string, dollar sequences are still expanded,
 *       s is the quotation delimiter
 *    ModeSkipString s :parsing a literal string, dollar sequences are still expanded,
 *       s is the quotation delimiter, skip the token if it is a quote that is not s
 *    ModeQuote s: parsing a literal string, dollar sequences are still expanded,
 *       escape sequences are allowed, s is the quotation delimiter.
 *
 * GS. The main entry is lex_line (below). Depending on the current mode,
 * a different lexer function is invoked:
 *
 * ModeNormal:      calls lex_main
 * ModeString:      calls lex_string, for text in $+dquote (e.g. $"")
 * ModeSkipString:  calls lex_skip_string. This is used after newlines inside
 *                  $-dquoted-text for checking whether the matching end
 *                  quote is following. Fairly technical.
 * ModeQuote:       calls lex_quote, for text after dquote
 *)
type mode =
   ModeNormal
 | ModeSkipString of string
 | ModeString of string
 | ModeQuote of string

(*
 * The lexing mode.
 *    ModeInitial: lexbuf is ready to be used
 *    ModeIndent i: initial indentation has been scanned
 *    ModeNormal: normal processing
 *
 * GS. LexModeInitial means we are at the beginning of the line. LexModeNormal
 * means that we've just lexed the left indentation.
 *)
type lexmode =
   LexModeInitial
 | LexModeNormal of int

(*
 * Parsing results.
 *)
type parse_item =
   ParseExp of Omake_ast.exp list
 | ParseError
 | ParseEOF

(*
 * This is the info for each indentation level.
 *)
type info =
   { info_mode     : mode;
     info_indent   : int;
     info_parens   : int option
   }

(*
 * State of the lexer.
 *)
type session =
   { (* The current location *)
     current_file            : Lm_symbol.t;
     mutable current_line    : int;
     mutable current_off     : int;
     mutable current_loc     : Lm_location.t;

     (* GS TODO: line/off/loc is now tracked by lexbuf (it wasn't in ancient
        versions of OCaml). Remove this here, and rely on lexbuf only.
      *)

     (* The current input buffer *)
     mutable current_buffer  : string;
     mutable current_index   : int;
     mutable current_prompt  : string;
     mutable current_fill_ok : bool;
     mutable current_eof     : bool;
     readline                : (string -> string);
     mutable is_interactive  : bool;

     (* The current lexbuf *)
     mutable current_lexbuf  : Lexing.lexbuf;
     mutable current_lexmode : lexmode;
     mutable current_token   : Omake_ast_parse.token;

     (* The current mode *)
     mutable current_mode    : mode;
     mutable current_parens  : int option;
     mutable current_indent  : int;
     mutable current_stack   : info list
   }

(************************************************************************
 * Printing.  NOTICE: if new tokens are added, please update
 * the token list in omake_gen_parse.ml!!!
 *)
let pp_print_token buf = function
    Omake_ast_parse.TokEof _ ->
       Lm_printf.pp_print_string buf "<eof>"
  | TokEol _ ->
       Lm_printf.pp_print_string buf "<eol>"
  | TokWhite (s, _) ->
       Format.fprintf buf "whitespace: \"%s\"" s
  | TokLeftParen (s, _) ->
       Format.fprintf buf "left parenthesis: %s" s
  | TokRightParen (s, _) ->
       Format.fprintf buf "right parenthesis: %s" s
  | TokArrow (s, _) ->
       Format.fprintf buf "arrow: %s" s
  | TokComma (s, _) ->
       Format.fprintf buf "comma: %s" s
  | TokColon (s, _) ->
       Format.fprintf buf "colon: %s" s
  | TokDoubleColon (s, _) ->
       Format.fprintf buf "doublecolon: %s" s
  | TokNamedColon (s, _) ->
       Format.fprintf buf "named colon: %s" s
  | TokDollar (s, strategy, _) ->
       Format.fprintf buf "dollar: %s%a" s Omake_ast_print.pp_print_strategy strategy
  | TokEq (s, _) ->
       Format.fprintf buf "equals: %s" s
  | TokArray (s, _) ->
       Format.fprintf buf "array: %s" s
  | TokDot (s, _) ->
       Format.fprintf buf "dot: %s" s
  | TokId (s, _) ->
       Format.fprintf buf "id: %s" s
  | TokInt (s, _) ->
       Format.fprintf buf "int: %s" s
  | TokFloat (s, _) ->
       Format.fprintf buf "float: %s" s
  | TokKey (s, _) ->
       Format.fprintf buf "key: %s" s
  | TokKeyword (s, _) ->
       Format.fprintf buf "keyword: %s" s
  | TokCatch (s, _) ->
       Format.fprintf buf "catch: %s" s
  | TokClass (s, _) ->
       Format.fprintf buf "class: %s" s
  | TokVar (_, s, _) ->
       Format.fprintf buf "var: %s" s
  | TokOp (s, _) ->
       Format.fprintf buf "op: %s" s
  | TokString (s, _) ->
       Format.fprintf buf "string: \"%s\"" (String.escaped s)
  | TokBeginQuote (s, _) ->
       Format.fprintf buf "begin-quote: %s" s
  | TokEndQuote (s, _) ->
       Format.fprintf buf "end-quote: %s" s
  | TokBeginQuoteString (s, _) ->
       Format.fprintf buf "begin-quote-string: %s" s
  | TokEndQuoteString (s, _) ->
       Format.fprintf buf "end-quote-string: %s" s
  | TokStringQuote (s, _) ->
       Format.fprintf buf "quote: %s" s
  | TokVarQuote (_, s, _) ->
       Format.fprintf buf "key: %s" s

(*
 * Set state.
 *)
let create name readline =
   let loc = Lm_location.bogus_loc name in
      { current_file    = Lm_symbol.add name;
        current_line    = 1;
        current_off     = 0;
        current_loc     = loc;
        current_buffer  = "";
        current_index   = 0;
        current_prompt  = ">";
        current_fill_ok = true;
        current_eof     = true;
        readline        = readline;
        is_interactive  = false;
        current_lexbuf  = Lexing.from_string "";
        current_lexmode = LexModeInitial;
        current_token   = TokEof loc;
        current_mode    = ModeNormal;
        current_parens  = None;
        current_indent  = 0;
        current_stack   = []
      }

(* let set_current_loc state loc = *)
(*    state.current_loc <- loc *)

let current_location state =
   state.current_loc

(*
 * Advance a line.
 *)
let set_next_line state lexbuf =
   let { current_line = line;
         current_file = file;
         _
       } = state
   in
   let line = succ line in
      state.current_line  <- line;
      state.current_off   <- Lexing.lexeme_start lexbuf;
      state.current_loc   <- Lm_location.create_loc file line 0 line 0

(*
 * Save the state.
 *)
let save_mode state =
   let { current_mode   = mode';
         current_parens = parens;
         current_indent = indent;
         current_stack  = stack;
         _
       } = state
   in
   let info =
      { info_mode   = mode';
        info_parens = parens;
        info_indent = indent
      }
   in
      info :: stack

(*
 * Restore the state.
 *)
let restore_mode state stack =
   match stack with
      info :: stack ->
         state.current_mode    <- info.info_mode;
         state.current_parens  <- info.info_parens;
         state.current_indent  <- info.info_indent;
         state.current_stack   <- stack
    | [] ->
         ()

(*
 * Push the new mode.
 *)
let push_mode state mode =
   let stack = save_mode state in
      state.current_mode   <- mode;
      state.current_parens <- None;
      state.current_stack  <- stack

(*
 * Pop the mode.
 *)
let pop_mode state =
   restore_mode state state.current_stack

(*
 * We are moving from a quotation to normal mode.
 * Start collecting parentheses.
 *)
let push_dollar state mode =
   push_mode state mode;
   state.current_parens <- Some 0

(* GS. The reason for counting open parentheses (in current_parens) is that
   a line feed is interpreted differently while there is an open parenthesis.
 *)

(*
 * Push a paren.
 *)
let push_paren state =
   let { current_parens = parens ; _} = state in
      match parens with
         Some i ->
            state.current_parens <- Some (succ i)
       | None ->
            ()

(*
 * When a paren is popped, if the level becomes zero,
 * then return to the previous mode.
 *)
let pop_paren state =
   let { current_parens = parens ; _} = state in
      match parens with
         Some i ->
            let i = pred i in
               if i = 0 then
                  pop_mode state
               else
                  state.current_parens <- Some i
       | None ->
            ()

(*
 * Get the location of the current lexeme.
 * We assume it is all on one line.
 *)
let lexeme_loc state lexbuf =
   let { current_line = line;
         current_off  = off;
         current_file = file;
         _
       } = state
   in
   let schar = Lexing.lexeme_start lexbuf - off in
   let echar = Lexing.lexeme_end lexbuf - off in
   let loc = Lm_location.create_loc file line schar line echar in
      state.current_loc <- loc;
      loc
(* GS TODO: use Lexing.lexeme_start_p and Lexing.lexeme_end_p instead *)


(*
 * Raise a syntax error exception.
 *)
let parse_error state =
   let lexbuf = state.current_lexbuf in
   let loc = lexeme_loc state lexbuf in
   let print_error buf =
      Format.fprintf buf "unexpected token: %a" pp_print_token state.current_token
   in
      raise (Omake_value_type.OmakeException (loc_exp_pos loc, LazyError print_error))

let syntax_error state s lexbuf =
   let loc = lexeme_loc state lexbuf in
      raise (Omake_value_type.OmakeException (loc_exp_pos loc, SyntaxError s))

(*
 * Get the string in the lexbuf.
 *)
let lexeme_string state lexbuf =
   let loc = lexeme_loc state lexbuf in
   let s = Lexing.lexeme lexbuf in
      s, loc

(*
 * Remove any trailing dots from the string.
 *)
(* let split_nl_string s = *)
(*    let len = String.length s in *)
(*    let rec search i = *)
(*       if i = len then *)
(*          s, "" *)
(*       else *)
(*          match s.[i] with *)
(*              '\n' *)
(*            | '\r' -> *)
(*                  search (succ i) *)
(*            | _ -> *)
(*                  String.sub s 0 i, String.sub s i (len - i) *)
(*    in *)
(*       search 0 *)

(*
 * Process a name.
 *)
let lexeme_name state lexbuf =
   let id, loc = lexeme_string state lexbuf in
      match id with
         "if"
       | "elseif"
       | "else"
       | "switch"
       | "match"
       | "select"
       | "case"
       | "default"
       | "section"
       | "include"
       | "extends"
       | "import"
       | "try"
       | "when"
       | "finally"
       | "raise"
       | "return"
       | "export"
       | "open"
       | "autoload"
       | "declare"
       | "value"
       | "with"
       | "as"
       | "while"
       | "do"
       | "set"
       | "program-syntax" ->
             Omake_ast_parse.TokKeyword (id, loc)
       | "catch" ->
             TokCatch (id, loc)
       | "class" ->
             TokClass (id, loc)
       | _ ->
             TokId (id, loc)

let lexeme_key state lexbuf =
    let id, loc = lexeme_string state lexbuf in
    Omake_ast_parse.TokKey (id, loc)

(*
 * Get the escaped char.
 * GS. e.g. "\X" -> "X"
 *)
let lexeme_esc state lexbuf =
   let s, loc = lexeme_string state lexbuf in
      String.make 1 s.[1], loc

(*
 * Single character variable.
 * GS. $x (not $(...)). Also $`x and $,x.
 *)
let lexeme_var state lexbuf =
  let s, loc = lexeme_string state lexbuf in
  let strategy, s =
    match s.[1] with
    | '`' -> Omake_ast.LazyApply, String.sub s 2 1
    | ',' -> EagerApply, String.sub s 2 1
    | _ -> NormalApply, String.sub s 1 1
  in
  Omake_ast_parse.TokVar (strategy, s, loc)

(*
 * Dollar sequence.
 *)
let lexeme_dollar_pipe state lexbuf =
   let s, loc = lexeme_string state lexbuf in
   let len = String.length s in
   let strategy, off =
      if len >= 2 then
         match s.[1] with
            '`'  -> Omake_ast.LazyApply, 2
          | ','  -> EagerApply, 2
          | '|'  -> NormalApply, 1
          | _    -> syntax_error state ("illegal character: " ^ s) lexbuf
      else
         NormalApply, 1
   in
   let s = String.sub s off (String.length s - off) in
      strategy, s, loc

(* GS. Unclear why there are two versions of this function. lexeme_dollar
   seems to be the usual function, for all of $` $, $$
 *)

let lexeme_dollar state lexbuf =
   let s, loc = lexeme_string state lexbuf in
   let len = String.length s in
      if len >= 2 then
         match s.[1] with
            '`'  -> Omake_ast_parse.TokDollar (s, LazyApply, loc)
          | ','  -> TokDollar (s, EagerApply, loc)
          | '$'  -> TokString ("$", loc)
          | _    -> syntax_error state ("illegal character: " ^ s) lexbuf
      else
         TokDollar (s, NormalApply, loc)

(*
 * Special character.
 * Keep track of paren nesting.
 *)
let lexeme_char state lexbuf =
  let s, loc = lexeme_string state lexbuf in
  match s.[0] with
    '$' ->
    Omake_ast_parse.TokDollar (s, NormalApply, loc)
  | ':' ->
    TokColon (s, loc)
  | ',' ->
    TokComma (s, loc)
  | '=' ->
    TokEq (s, loc)
  | '.' ->
    TokDot (s, loc)
  | '%' ->
    TokVar (NormalApply, s, loc)
  | '(' ->
    push_paren state;
    TokLeftParen (s, loc)
  | ')' ->
    pop_paren state;
    TokRightParen (s, loc)
  | _   ->
    TokOp (s, loc)

(*
 * Special string.
 *)
let lexeme_special_string state lexbuf =
  let s, loc = lexeme_string state lexbuf in
  match s with
    "=>" ->
    Omake_ast_parse.TokArrow (s, loc)
  | "::" ->
    TokDoubleColon (s, loc)
  | "+=" ->
    TokEq (s, loc)
  | "[]" ->
    TokArray (s, loc)
  | _ ->
    TokOp (s, loc)

(*
 * Count the indentation in a string of characters.
 *)
let indent_of_string s =
   let len = String.length s in
   let rec loop col i =
      if i = len then
         col
      else
         match s.[i] with
            '\r'
          | '\n' ->
             loop 0 (succ i)
          | '\t' ->
             loop ((col + 8) land (lnot 7)) (succ i)
          | _ ->
             loop (succ col) (succ i)
   in
      loop 0 0

(*
 * Use lexer positions.
 *)
let lexeme_pos lexbuf =
   let s = Lexing.lexeme lexbuf in
   let pos1 = Lexing.lexeme_start_p lexbuf in
   let pos2 = Lexing.lexeme_end_p lexbuf in
   let { Lexing.pos_fname = file;
         Lexing.pos_lnum = line1;
         Lexing.pos_bol = bol1;
         Lexing.pos_cnum = cnum1
       } = pos1
   in
   let { Lexing.pos_lnum = line2;
         Lexing.pos_bol = bol2;
         Lexing.pos_cnum = cnum2;
         _
       } = pos2
   in
   let loc = Lm_location.create_loc (Lm_symbol.add file) line1 (cnum1 - bol1) line2 (cnum2 - bol2) in
      s, loc

# 553 "omake_ast_lex.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\236\255\118\000\160\000\243\255\001\000\058\000\061\000\
    \069\000\069\000\237\000\070\000\098\001\249\255\216\001\087\000\
    \253\255\088\000\040\002\120\002\199\002\022\003\135\000\255\255\
    \001\000\151\000\098\001\099\001\071\001\102\003\181\003\004\004\
    \083\004\182\001\162\004\241\004\077\000\000\005\010\005\020\005\
    \119\000\101\000\242\255\103\000\058\000\109\000\069\005\147\005\
    \006\006\086\006\120\000\098\000\244\255\245\255\041\000\200\006\
    \129\000\137\000\116\000\024\007\241\255\113\000\139\007\208\000\
    \237\255\004\000\238\255\252\007\247\255\047\008\253\255\000\008\
    \013\008\255\255\005\000\248\255\006\000\249\255\251\255\211\008\
    \252\255\211\008\248\255\250\255\007\000\042\009\143\000\141\000\
    \014\008\156\009\100\001\249\255\008\000\252\255\218\009\253\255\
    \174\000\058\000\150\000\146\000\183\008\252\255\063\000\154\000\
    \150\000\186\009\255\255\009\000\175\000\066\000\157\000\154\000\
    \201\006\255\255\010\000\085\004\087\004\123\010\247\255\248\255\
    \132\004\251\255\011\000\252\255\253\255\165\008\209\006\255\255\
    \201\010\027\011\067\001\210\004\243\004\249\255\012\000\250\255\
    \254\255\233\007\253\255\254\255\234\007\099\001\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\016\000\015\000\255\255\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\255\255\015\000\012\000\
    \255\255\002\000\002\000\002\000\002\000\002\000\020\000\255\255\
    \000\000\001\000\255\255\000\000\255\255\002\000\002\000\002\000\
    \002\000\255\255\002\000\004\000\004\000\255\255\004\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\015\000\005\000\
    \005\000\005\000\255\255\255\255\255\255\255\255\009\000\011\000\
    \008\000\007\000\255\255\255\255\255\255\013\000\015\000\255\255\
    \255\255\018\000\255\255\255\255\255\255\005\000\255\255\001\000\
    \001\000\255\255\000\000\255\255\007\000\255\255\255\255\005\000\
    \255\255\255\255\255\255\255\255\005\000\004\000\001\000\001\000\
    \000\000\000\000\000\000\255\255\006\000\255\255\004\000\255\255\
    \000\000\000\000\000\000\000\000\255\255\255\255\002\000\002\000\
    \002\000\001\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \001\000\255\255\000\000\255\255\000\000\255\255\255\255\255\255\
    \007\000\255\255\004\000\255\255\255\255\000\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\004\000\255\255\006\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255";
  Lexing.lex_default =
   "\003\000\000\000\255\255\003\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\046\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\026\000\000\000\
    \255\255\255\255\026\000\026\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\046\000\255\255\
    \046\000\255\255\255\255\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\003\000\255\255\
    \000\000\255\255\000\000\071\000\000\000\255\255\000\000\071\000\
    \255\255\000\000\255\255\000\000\255\255\000\000\000\000\255\255\
    \000\000\088\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \088\000\088\000\255\255\000\000\255\255\000\000\255\255\000\000\
    \255\255\255\255\255\255\255\255\105\000\000\000\255\255\255\255\
    \255\255\105\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\115\000\115\000\119\000\000\000\000\000\
    \131\000\000\000\255\255\000\000\000\000\255\255\255\255\000\000\
    \255\255\255\255\255\255\131\000\131\000\000\000\255\255\000\000\
    \000\000\140\000\000\000\000\000\140\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\025\000\023\000\023\000\025\000\024\000\064\000\073\000\
    \075\000\083\000\091\000\106\000\113\000\121\000\133\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \025\000\000\000\013\000\022\000\012\000\004\000\006\000\013\000\
    \004\000\004\000\004\000\009\000\004\000\004\000\015\000\004\000\
    \021\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\010\000\004\000\008\000\011\000\007\000\014\000\
    \018\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\017\000\002\000\016\000\004\000\018\000\
    \042\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\061\000\005\000\042\000\014\000\066\000\
    \064\000\042\000\042\000\065\000\042\000\045\000\041\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\023\000\037\000\043\000\027\000\044\000\066\000\042\000\
    \066\000\066\000\066\000\042\000\050\000\066\000\066\000\066\000\
    \025\000\023\000\066\000\025\000\024\000\054\000\003\000\046\000\
    \056\000\255\255\255\255\057\000\255\255\255\255\060\000\042\000\
    \066\000\086\000\037\000\066\000\087\000\042\000\097\000\025\000\
    \098\000\099\000\026\000\102\000\103\000\104\000\109\000\110\000\
    \255\255\111\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \098\000\110\000\066\000\003\000\046\000\099\000\111\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\255\255\255\255\255\255\255\255\003\000\
    \001\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\000\000\255\255\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\042\000\
    \000\000\000\000\097\000\109\000\003\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \058\000\000\000\058\000\000\000\059\000\000\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \000\000\000\000\000\000\000\000\023\000\023\000\091\000\027\000\
    \027\000\092\000\127\000\000\000\000\000\000\000\064\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\000\000\000\000\000\000\057\000\142\000\052\000\255\255\
    \053\000\056\000\142\000\000\000\053\000\053\000\055\000\000\000\
    \000\000\000\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\000\000\000\000\053\000\127\000\
    \255\255\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\000\000\053\000\
    \053\000\053\000\055\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\000\000\054\000\000\000\
    \053\000\255\255\255\255\000\000\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\047\000\255\255\255\255\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \047\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\255\255\255\255\255\255\255\255\047\000\
    \000\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\000\000\255\255\018\000\048\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\255\255\255\255\091\000\000\000\000\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\000\000\000\000\000\000\000\000\018\000\
    \000\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\000\000\000\000\018\000\018\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\040\000\000\000\000\000\000\000\000\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\000\000\000\000\000\000\000\000\018\000\
    \255\255\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\000\000\018\000\028\000\018\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\000\000\000\000\000\000\000\000\000\000\000\000\018\000\
    \018\000\018\000\018\000\018\000\029\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\000\000\000\000\000\000\000\000\018\000\000\000\
    \018\000\018\000\018\000\018\000\029\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\000\000\018\000\028\000\018\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\018\000\018\000\
    \018\000\018\000\018\000\029\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\000\000\000\000\000\000\000\000\018\000\000\000\018\000\
    \032\000\018\000\018\000\029\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\031\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\030\000\018\000\
    \018\000\033\000\000\000\034\000\018\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\000\000\000\000\000\000\000\000\018\000\000\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\000\000\018\000\000\000\018\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\018\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \000\000\000\000\000\000\000\000\018\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \000\000\018\000\000\000\018\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\018\000\018\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\113\000\
    \000\000\113\000\116\000\018\000\116\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\000\000\
    \018\000\000\000\018\000\032\000\032\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\000\000\121\000\000\000\
    \000\000\132\000\000\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\000\000\000\000\
    \000\000\000\000\018\000\000\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\000\000\018\000\
    \000\000\018\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\121\000\000\000\000\000\132\000\
    \000\000\000\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\121\000\000\000\000\000\
    \132\000\018\000\000\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\000\000\000\000\000\000\
    \018\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\039\000\000\000\039\000\000\000\000\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \000\000\255\255\255\255\000\000\000\000\113\000\000\000\113\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \047\000\255\255\000\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\000\000\000\000\
    \000\000\000\000\047\000\255\255\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\000\000\255\255\
    \255\255\047\000\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\047\000\255\255\255\255\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\047\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\047\000\000\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\000\000\255\255\047\000\048\000\000\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \050\000\000\000\000\000\000\000\000\000\000\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\000\000\000\000\000\000\000\000\047\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\000\000\112\000\113\000\047\000\112\000\114\000\000\000\
    \000\000\000\000\126\000\121\000\000\000\126\000\122\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\112\000\000\000\000\000\115\000\000\000\053\000\000\000\
    \000\000\126\000\053\000\053\000\131\000\000\000\000\000\000\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\000\000\000\000\053\000\000\000\255\255\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\000\000\053\000\053\000\053\000\
    \000\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\000\000\054\000\059\000\053\000\000\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\060\000\000\000\000\000\000\000\000\000\000\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\000\000\000\000\000\000\000\000\059\000\
    \000\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\000\000\255\255\255\255\059\000\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\063\000\255\255\255\255\
    \255\255\255\255\000\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\255\255\255\255\
    \255\255\255\255\000\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\073\000\255\255\
    \000\000\074\000\255\255\139\000\255\255\255\255\000\000\000\000\
    \139\000\255\255\000\000\000\000\000\000\000\000\000\000\075\000\
    \255\255\000\000\076\000\255\255\000\000\000\000\070\000\000\000\
    \069\000\000\000\255\255\070\000\255\255\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \255\255\000\000\255\255\000\000\077\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\141\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\080\000\000\000\078\000\000\000\000\000\
    \072\000\078\000\078\000\079\000\255\255\000\000\000\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\077\000\255\255\078\000\000\000\000\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\255\255\078\000\078\000\078\000\079\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\000\000\000\000\000\000\078\000\135\000\133\000\
    \000\000\000\000\134\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\106\000\000\000\000\000\107\000\135\000\000\000\135\000\
    \135\000\135\000\000\000\000\000\135\000\135\000\135\000\000\000\
    \000\000\135\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\103\000\000\000\000\000\000\000\083\000\104\000\136\000\
    \084\000\000\000\135\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\138\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\086\000\000\000\085\000\
    \000\000\078\000\087\000\000\000\068\000\078\000\078\000\000\000\
    \255\255\135\000\000\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\075\000\255\255\078\000\
    \000\000\000\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\090\000\
    \078\000\078\000\078\000\102\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\095\000\089\000\
    \093\000\078\000\000\000\000\000\093\000\093\000\094\000\000\000\
    \000\000\000\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\000\000\000\000\093\000\000\000\
    \000\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\000\000\093\000\
    \093\000\093\000\094\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\133\000\255\255\000\000\
    \093\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\101\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \255\255\000\000\000\000\255\255\255\255\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\082\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \093\000\000\000\000\000\000\000\093\000\093\000\000\000\000\000\
    \000\000\000\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\000\000\000\000\093\000\000\000\
    \089\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\255\255\093\000\
    \093\000\093\000\000\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\000\000\000\000\000\000\
    \093\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\126\000\121\000\000\000\126\000\
    \122\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\126\000\255\255\123\000\120\000\000\000\
    \000\000\000\000\123\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\124\000\000\000\000\000\
    \000\000\000\000\255\255\128\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\127\000\125\000\
    \127\000\000\000\128\000\000\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\128\000\000\000\
    \000\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\000\000\000\000\000\000\000\000\
    \128\000\000\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\000\000\000\000\000\000\128\000\
    \128\000\000\000\000\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\130\000\000\000\000\000\
    \000\000\000\000\000\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\000\000\000\000\
    \000\000\000\000\128\000\118\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\000\000\000\000\
    \000\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\024\000\000\000\000\000\065\000\074\000\
    \076\000\084\000\092\000\107\000\114\000\122\000\134\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\007\000\000\000\005\000\000\000\002\000\
    \002\000\008\000\009\000\002\000\011\000\015\000\017\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\022\000\036\000\041\000\022\000\043\000\002\000\044\000\
    \002\000\002\000\002\000\045\000\051\000\002\000\002\000\002\000\
    \025\000\025\000\002\000\025\000\025\000\054\000\040\000\050\000\
    \056\000\003\000\003\000\057\000\003\000\003\000\058\000\061\000\
    \002\000\086\000\036\000\002\000\087\000\017\000\097\000\025\000\
    \098\000\099\000\025\000\102\000\103\000\104\000\109\000\110\000\
    \003\000\111\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\255\255\003\000\003\000\
    \096\000\108\000\002\000\040\000\050\000\096\000\108\000\255\255\
    \255\255\255\255\003\000\003\000\003\000\003\000\003\000\255\255\
    \255\255\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\063\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\255\255\003\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \255\255\255\255\096\000\108\000\063\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\255\255\010\000\255\255\010\000\255\255\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \255\255\255\255\255\255\255\255\026\000\027\000\090\000\026\000\
    \027\000\090\000\130\000\255\255\255\255\255\255\002\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\255\255\255\255\255\255\012\000\141\000\012\000\022\000\
    \012\000\012\000\141\000\255\255\012\000\012\000\012\000\255\255\
    \255\255\255\255\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\255\255\255\255\012\000\130\000\
    \003\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\255\255\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\255\255\012\000\255\255\
    \012\000\014\000\014\000\255\255\014\000\014\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \014\000\255\255\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\255\255\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \255\255\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\255\255\014\000\018\000\014\000\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\026\000\027\000\090\000\255\255\255\255\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\255\255\255\255\255\255\255\255\018\000\
    \255\255\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\255\255\255\255\019\000\018\000\255\255\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\255\255\255\255\255\255\255\255\255\255\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\255\255\255\255\255\255\255\255\019\000\
    \014\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\255\255\020\000\020\000\019\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\255\255\255\255\255\255\255\255\255\255\255\255\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\255\255\255\255\255\255\255\255\020\000\255\255\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\255\255\021\000\021\000\020\000\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\255\255\255\255\255\255\255\255\021\000\255\255\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\021\000\021\000\021\000\021\000\021\000\021\000\021\000\
    \021\000\029\000\255\255\029\000\021\000\255\255\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\255\255\255\255\255\255\255\255\029\000\255\255\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\255\255\030\000\255\255\029\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \255\255\255\255\255\255\255\255\030\000\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \255\255\031\000\255\255\030\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\115\000\
    \255\255\116\000\115\000\031\000\116\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \032\000\255\255\031\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\120\000\255\255\
    \255\255\120\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\034\000\
    \255\255\032\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\131\000\255\255\255\255\131\000\
    \255\255\255\255\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\132\000\255\255\255\255\
    \132\000\034\000\255\255\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\255\255\255\255\255\255\
    \034\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\037\000\255\255\037\000\255\255\255\255\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\046\000\046\000\
    \255\255\046\000\046\000\255\255\255\255\115\000\255\255\116\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\046\000\255\255\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\120\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\255\255\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \047\000\046\000\255\255\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\255\255\255\255\255\255\
    \255\255\255\255\131\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\255\255\255\255\
    \255\255\255\255\047\000\132\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\255\255\048\000\
    \048\000\047\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\048\000\255\255\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\046\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\255\255\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\255\255\048\000\049\000\048\000\255\255\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\255\255\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\049\000\255\255\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\112\000\112\000\049\000\112\000\112\000\255\255\
    \255\255\255\255\126\000\126\000\255\255\126\000\126\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\112\000\255\255\255\255\112\000\255\255\055\000\255\255\
    \255\255\126\000\055\000\055\000\126\000\255\255\255\255\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\255\255\255\255\055\000\255\255\048\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\055\000\055\000\055\000\
    \255\255\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\255\255\055\000\059\000\055\000\255\255\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\255\255\255\255\255\255\255\255\255\255\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\255\255\255\255\255\255\255\255\059\000\
    \255\255\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\255\255\062\000\062\000\059\000\062\000\
    \062\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\062\000\255\255\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \255\255\062\000\062\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\062\000\062\000\062\000\
    \062\000\062\000\255\255\255\255\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\255\255\255\255\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\067\000\062\000\
    \255\255\067\000\071\000\137\000\140\000\071\000\255\255\255\255\
    \137\000\140\000\255\255\255\255\255\255\255\255\255\255\072\000\
    \088\000\255\255\072\000\088\000\255\255\255\255\067\000\255\255\
    \067\000\255\255\071\000\067\000\071\000\255\255\255\255\071\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \088\000\255\255\088\000\255\255\072\000\088\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\137\000\140\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\069\000\255\255\069\000\255\255\255\255\
    \067\000\069\000\069\000\069\000\071\000\255\255\255\255\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\072\000\088\000\069\000\255\255\255\255\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\062\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\255\255\255\255\255\255\069\000\125\000\125\000\
    \255\255\255\255\125\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\100\000\255\255\255\255\100\000\125\000\255\255\125\000\
    \125\000\125\000\255\255\255\255\125\000\125\000\125\000\255\255\
    \255\255\125\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\100\000\255\255\255\255\255\255\081\000\100\000\125\000\
    \081\000\255\255\125\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\137\000\140\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\081\000\255\255\081\000\
    \255\255\079\000\081\000\255\255\067\000\079\000\079\000\255\255\
    \071\000\125\000\255\255\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\072\000\088\000\079\000\
    \255\255\255\255\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\081\000\
    \079\000\079\000\079\000\100\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\085\000\081\000\
    \085\000\079\000\255\255\255\255\085\000\085\000\085\000\255\255\
    \255\255\255\255\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\255\255\255\255\085\000\255\255\
    \255\255\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\255\255\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\125\000\089\000\255\255\
    \085\000\089\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\100\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\255\255\
    \089\000\255\255\255\255\089\000\105\000\255\255\255\255\105\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\081\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\105\000\255\255\255\255\255\255\
    \255\255\105\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \089\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \094\000\255\255\255\255\255\255\094\000\094\000\255\255\255\255\
    \255\255\255\255\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\255\255\255\255\094\000\255\255\
    \089\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\105\000\094\000\
    \094\000\094\000\255\255\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\255\255\255\255\255\255\
    \094\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\117\000\117\000\255\255\117\000\
    \117\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\117\000\089\000\117\000\117\000\255\255\
    \255\255\255\255\117\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\255\255\255\255\
    \255\255\255\255\105\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\255\255\117\000\255\255\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\128\000\255\255\
    \255\255\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\255\255\255\255\255\255\255\255\
    \128\000\255\255\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \128\000\128\000\128\000\128\000\255\255\255\255\255\255\128\000\
    \129\000\255\255\255\255\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\255\255\255\255\
    \255\255\255\255\255\255\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\255\255\255\255\
    \255\255\255\255\129\000\117\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\255\255\255\255\
    \255\255\129\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec lex_main state lexbuf =
   __ocaml_lex_lex_main_rec state lexbuf 0
and __ocaml_lex_lex_main_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 669 "omake_ast_lex.mll"
   ( let loc = state.current_loc in
     let _ = lexeme_loc state lexbuf in
        set_next_line state lexbuf;
        Omake_ast_parse.TokEol loc
   )
# 1415 "omake_ast_lex.ml"

  | 1 ->
# 675 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokWhite (s, loc)
   )
# 1422 "omake_ast_lex.ml"

  | 2 ->
# 688 "omake_ast_lex.mll"
   ( lexeme_name state lexbuf )
# 1427 "omake_ast_lex.ml"

  | 3 ->
# 690 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokInt (s, loc)
   )
# 1434 "omake_ast_lex.ml"

  | 4 ->
# 694 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokFloat (s, loc)
   )
# 1441 "omake_ast_lex.ml"

  | 5 ->
# 698 "omake_ast_lex.mll"
   ( lexeme_key state lexbuf )
# 1446 "omake_ast_lex.ml"

  | 6 ->
# 700 "omake_ast_lex.mll"
   ( let id, loc = lexeme_string state lexbuf in
     let mode = ModeQuote id in
        push_mode state mode;
        TokBeginQuoteString (id, loc)
   )
# 1455 "omake_ast_lex.ml"

  | 7 ->
# 707 "omake_ast_lex.mll"
   ( let id, loc = lexeme_string state lexbuf in
     let id = String.sub id 1 (pred (String.length id)) in
              (* GS TODO: use "as" *)
     let mode = ModeString id in
        push_mode state mode;
        TokBeginQuote ("", loc)
   )
# 1466 "omake_ast_lex.ml"

  | 8 ->
# 715 "omake_ast_lex.mll"
   ( let id, _ = lexeme_string state lexbuf in
     let id = String.sub id 1 (pred (String.length id)) in
              (* GS TODO: use "as" *)
     let s, loc = lex_literal state (Buffer.create 32) id lexbuf in
     (* GS: lex_literal is a sublexer. Returns the quoted string *)
        TokStringQuote (s, loc)
   )
# 1477 "omake_ast_lex.ml"

  | 9 ->
# 724 "omake_ast_lex.mll"
   ( let strategy, id, _ = lexeme_dollar_pipe state lexbuf in
     let s, loc = lex_literal state (Buffer.create 32) id lexbuf in
        TokVarQuote (strategy, s, loc)
   )
# 1485 "omake_ast_lex.ml"

  | 10 ->
# 730 "omake_ast_lex.mll"
   ( lexeme_var state lexbuf )
# 1490 "omake_ast_lex.ml"

  | 11 ->
# 733 "omake_ast_lex.mll"
   ( lexeme_dollar state lexbuf )
# 1495 "omake_ast_lex.ml"

  | 12 ->
# 736 "omake_ast_lex.mll"
   ( lexeme_char state lexbuf )
# 1500 "omake_ast_lex.ml"

  | 13 ->
# 739 "omake_ast_lex.mll"
   ( lexeme_special_string state lexbuf )
# 1505 "omake_ast_lex.ml"

  | 14 ->
# 741 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokNamedColon (s, loc)
   )
# 1512 "omake_ast_lex.ml"

  | 15 ->
# 745 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokString (s, loc)
   )
# 1519 "omake_ast_lex.ml"

  | 16 ->
# 749 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        TokString (s, loc)
   )
# 1526 "omake_ast_lex.ml"

  | 17 ->
# 753 "omake_ast_lex.mll"
   ( let s, loc = lexeme_esc state lexbuf in
        TokStringQuote (s, loc)
   )
# 1533 "omake_ast_lex.ml"

  | 18 ->
# 757 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
        set_next_line state lexbuf;
        state.current_prompt <- "\\";
        state.current_fill_ok <- true;
        TokString (" ", loc)
   )
# 1543 "omake_ast_lex.ml"

  | 19 ->
# 764 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
        match state.current_token with
           TokEol _
         | TokEof _ ->
              TokEof loc
         | _ ->
              TokEol loc
   )
# 1555 "omake_ast_lex.ml"

  | 20 ->
# 773 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        syntax_error state ("illegal character: " ^ String.escaped s) lexbuf
   )
# 1562 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_main_rec state lexbuf __ocaml_lex_state

and lex_quote state lexbuf =
   __ocaml_lex_lex_quote_rec state lexbuf 67
and __ocaml_lex_lex_quote_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 787 "omake_ast_lex.mll"
   ( set_next_line state lexbuf;
     syntax_error state "unterminated string" lexbuf
   )
# 1576 "omake_ast_lex.ml"

  | 1 ->
# 792 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
   Omake_ast_parse.TokString (s, loc)
   )
# 1583 "omake_ast_lex.ml"

  | 2 ->
# 796 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        match state.current_mode with
           ModeQuote s' when s' = s ->
              pop_mode state;
              TokEndQuoteString (s, loc)
         | _ ->
              TokString (s, loc)
   )
# 1595 "omake_ast_lex.ml"

  | 3 ->
# 805 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
        TokString ("$", loc)
   )
# 1602 "omake_ast_lex.ml"

  | 4 ->
# 809 "omake_ast_lex.mll"
   ( lexeme_var state lexbuf )
# 1607 "omake_ast_lex.ml"

  | 5 ->
# 811 "omake_ast_lex.mll"
   ( push_dollar state ModeNormal;
     lexeme_dollar state lexbuf
   )
# 1614 "omake_ast_lex.ml"

  | 6 ->
# 815 "omake_ast_lex.mll"
   ( let s, loc = lexeme_esc state lexbuf in
        TokString (s, loc)
   )
# 1621 "omake_ast_lex.ml"

  | 7 ->
# 819 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
        set_next_line state lexbuf;
        state.current_fill_ok <- true;
        TokString ("", loc)
   )
# 1630 "omake_ast_lex.ml"

  | 8 ->
# 825 "omake_ast_lex.mll"
   ( syntax_error state "unterminated string" lexbuf )
# 1635 "omake_ast_lex.ml"

  | 9 ->
# 827 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        syntax_error state ("illegal character in string constant: " ^ String.escaped s) lexbuf
   )
# 1642 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_quote_rec state lexbuf __ocaml_lex_state

and lex_string state lexbuf =
   __ocaml_lex_lex_string_rec state lexbuf 81
and __ocaml_lex_lex_string_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 841 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        Omake_ast_parse.TokString (s, loc)
   )
# 1656 "omake_ast_lex.ml"

  | 1 ->
# 845 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        match state.current_mode with
           ModeString s' when s' = s ->
              pop_mode state;
              TokEndQuote ("", loc)
         | _ ->
              TokString (s, loc)
   )
# 1668 "omake_ast_lex.ml"

  | 2 ->
# 854 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
        TokString ("$", loc)
   )
# 1675 "omake_ast_lex.ml"

  | 3 ->
# 858 "omake_ast_lex.mll"
   ( lexeme_var state lexbuf )
# 1680 "omake_ast_lex.ml"

  | 4 ->
# 860 "omake_ast_lex.mll"
   ( push_dollar state ModeNormal;
     lexeme_dollar state lexbuf
   )
# 1687 "omake_ast_lex.ml"

  | 5 ->
# 864 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
     let () =
        match state.current_mode with
           ModeString s ->
              push_mode state (ModeSkipString s)
         | _ ->
             (* GS CHECK: When is this possible? *)
              ()
     in
        set_next_line state lexbuf;
        state.current_fill_ok <- true;
        TokString (s, loc)
   )
# 1704 "omake_ast_lex.ml"

  | 6 ->
# 878 "omake_ast_lex.mll"
   ( let loc = lexeme_loc state lexbuf in
     let () =
        match state.current_mode with
           ModeString s ->
              push_mode state (ModeSkipString s)
         | _ ->
              ()
     in
        set_next_line state lexbuf;
        state.current_fill_ok <- true;
        TokString ("", loc)
   )
# 1720 "omake_ast_lex.ml"

  | 7 ->
# 891 "omake_ast_lex.mll"
   ( syntax_error state "unterminated string" lexbuf )
# 1725 "omake_ast_lex.ml"

  | 8 ->
# 893 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        syntax_error state ("illegal character: " ^ String.escaped s) lexbuf
   )
# 1732 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_string_rec state lexbuf __ocaml_lex_state

and lex_skip_string state lexbuf =
   __ocaml_lex_lex_skip_string_rec state lexbuf 96
and __ocaml_lex_lex_skip_string_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 902 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        pop_mode state;
        match state.current_mode with
           ModeString s' when s' = s ->
              pop_mode state;
              Omake_ast_parse.TokEndQuote ("", loc)
         | _ ->
              TokString ("", loc)
   )
# 1752 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_skip_string_rec state lexbuf __ocaml_lex_state

and lex_literal state buf equote lexbuf =
   __ocaml_lex_lex_literal_rec state buf equote lexbuf 100
and __ocaml_lex_lex_literal_rec state buf equote lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 918 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        set_next_line state lexbuf;
        state.current_fill_ok <- true;
        Buffer.add_string buf s;
        lex_literal_skip state buf equote lexbuf
   )
# 1769 "omake_ast_lex.ml"

  | 1 ->
# 925 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        Buffer.add_string buf s;
        lex_literal state buf equote lexbuf
   )
# 1777 "omake_ast_lex.ml"

  | 2 ->
# 930 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        if s = equote then
           let s = Buffer.contents buf in
              s, loc
        else
           begin
              Buffer.add_string buf s;
              lex_literal state buf equote lexbuf
           end
   )
# 1791 "omake_ast_lex.ml"

  | 3 ->
# 941 "omake_ast_lex.mll"
   ( syntax_error state "unterminated string" lexbuf )
# 1796 "omake_ast_lex.ml"

  | 4 ->
# 943 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
        syntax_error state ("illegal character: " ^ String.escaped s) lexbuf
   )
# 1803 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_literal_rec state buf equote lexbuf __ocaml_lex_state

and lex_literal_skip state buf equote lexbuf =
   __ocaml_lex_lex_literal_skip_rec state buf equote lexbuf 108
and __ocaml_lex_lex_literal_skip_rec state buf equote lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 949 "omake_ast_lex.mll"
   ( let s, loc = lexeme_string state lexbuf in
        if s = equote then
           let s = Buffer.contents buf in
              s, loc
        else
           lex_literal state buf equote lexbuf
   )
# 1821 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_literal_skip_rec state buf equote lexbuf __ocaml_lex_state

and lex_indent state lexbuf =
   __ocaml_lex_lex_indent_rec state lexbuf 112
and __ocaml_lex_lex_indent_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 963 "omake_ast_lex.mll"
   ( set_next_line state lexbuf;
     state.current_fill_ok <- true;
     lex_indent state lexbuf
   )
# 1836 "omake_ast_lex.ml"

  | 1 ->
# 968 "omake_ast_lex.mll"
   ( let s, _ = lexeme_string state lexbuf in
     let indent = indent_of_string s in
        indent
   )
# 1844 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_indent_rec state lexbuf __ocaml_lex_state

and lex_deps lexbuf =
   __ocaml_lex_lex_deps_rec lexbuf 117
and __ocaml_lex_lex_deps_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 981 "omake_ast_lex.mll"
   ( let s, loc = lexeme_pos lexbuf in
   Omake_ast_parse.TokString (s, loc)
   )
# 1858 "omake_ast_lex.ml"

  | 1 ->
# 985 "omake_ast_lex.mll"
   ( let _, loc = lexeme_pos lexbuf in
        TokString (":", loc)
   )
# 1865 "omake_ast_lex.ml"

  | 2 ->
# 989 "omake_ast_lex.mll"
   ( let s, loc = lexeme_pos lexbuf in
        TokColon (s, loc)
   )
# 1872 "omake_ast_lex.ml"

  | 3 ->
# 993 "omake_ast_lex.mll"
   ( let s, loc = lexeme_pos lexbuf in
     let buf = Buffer.create 64 in
        Buffer.add_string buf s;
        lex_deps_quote s buf lexbuf;
        TokString (Buffer.contents buf, loc)
   )
# 1882 "omake_ast_lex.ml"

  | 4 ->
# 1001 "omake_ast_lex.mll"
   ( let _, loc = lexeme_pos lexbuf in
        TokEol loc
   )
# 1889 "omake_ast_lex.ml"

  | 5 ->
# 1005 "omake_ast_lex.mll"
   ( let s, loc = lexeme_pos lexbuf in
     let s = String.make 1 s.[1] in
        TokStringQuote (s, loc)
   )
# 1897 "omake_ast_lex.ml"

  | 6 ->
# 1010 "omake_ast_lex.mll"
   ( let _, loc = lexeme_pos lexbuf in
        TokWhite (" ", loc)
   )
# 1904 "omake_ast_lex.ml"

  | 7 ->
# 1014 "omake_ast_lex.mll"
   ( let s, loc = lexeme_pos lexbuf in
        TokString (s, loc)
   )
# 1911 "omake_ast_lex.ml"

  | 8 ->
# 1018 "omake_ast_lex.mll"
   ( let _, loc = lexeme_pos lexbuf in
        TokEof loc
   )
# 1918 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_deps_rec lexbuf __ocaml_lex_state

and lex_deps_quote term buf lexbuf =
   __ocaml_lex_lex_deps_quote_rec term buf lexbuf 137
and __ocaml_lex_lex_deps_quote_rec term buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1026 "omake_ast_lex.mll"
   ( let s, _ = lexeme_pos lexbuf in
        Buffer.add_string buf s;
        lex_deps_quote term buf lexbuf
   )
# 1933 "omake_ast_lex.ml"

  | 1 ->
# 1031 "omake_ast_lex.mll"
   ( let s, _ = lexeme_pos lexbuf in
        Buffer.add_string buf s;
        if s <> term then
           lex_deps_quote term buf lexbuf
   )
# 1942 "omake_ast_lex.ml"

  | 2 ->
# 1038 "omake_ast_lex.mll"
   ( raise Parsing.Parse_error )
# 1947 "omake_ast_lex.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_lex_deps_quote_rec term buf lexbuf __ocaml_lex_state

;;

# 1040 "omake_ast_lex.mll"
 
(************************************************************************
 * Prompts.
 *)

(*
 * Lex and parse a line for the shell.
 *)
let tabstop = 3

let prompt_ext s =
   s ^ "> "

(* Prune the prompt to a reasonable length *)
let prompt_prune prompt indent =
   let max_len = 8 in
   let s = Bytes.make (indent * tabstop + max_len + 2) ' ' in
   let length = String.length prompt in
      if length > max_len then
         begin
            Bytes.blit_string prompt 0 s 0 max_len;
            Bytes.set s max_len '>'
         end
      else
         Bytes.blit_string prompt 0 s 0 length;
      Bytes.to_string s

let prompt_indent prompt root indent =
   if root then
      prompt
   else
      prompt_prune prompt indent

let prompt_string state root nest e =
   let prompt = prompt_ext (Omake_ast_util.key_of_exp e) in
      if state.is_interactive && root then
         Lm_printf.printf "%s%s@?" (prompt_prune prompt nest) state.current_buffer;
      prompt

(*
 * Parser for the body of an expression.
 *)
let body_parser state body =
   match body with
      Omake_ast.NoBody ->
         None
    | OptBody ->
         if state.is_interactive then
            None
         else
            Some Omake_ast_parse.shell
    | ColonBody ->
         Some Omake_ast_parse.shell
    | ArrayBody ->
         Some Omake_ast_parse.string

(************************************************************************
 * Lexing input.
 *)

(*
 * Copy into the lexbuf.
 *)
let lex_fill state buf len =
   let { 
         current_buffer = buffer;
         current_index = index;
         _
       } = state
   in
   let length = String.length buffer in
   let amount = min (length - index) len in
      if amount = 0 then
         state.current_eof <- true
      else
         begin
            String.blit buffer index buf 0 amount;
            state.current_index <- index + amount
         end;
      amount

(*
 * Refill the buffer using the readline function.
 *)
let state_refill state =
   let { current_fill_ok = fill_ok;
         current_prompt = prompt;
         readline = readline;
         _
       } = state
   in
      if fill_ok then
         let line = readline prompt in
         let line =
            if state.is_interactive && line = ".\n" then
               ""
            else
               line
         in
            state.current_buffer <- line;
            state.current_index <- 0;
            state.current_fill_ok <- false

(*
 * Lexer function to refill the buffer.
 * GS. This is for Lexing.from_function.
 *)
let lex_refill state buf len =
   let { current_buffer = buffer;
         current_index = index;
         _
       } = state
   in
   let length = String.length buffer in
   let amount = length - index in
      if amount = 0 then
         state_refill state;
      lex_fill state buf len

(************************************************************************
 * Main lexer.
 *)

(*
 * Get the input.
 *)
let lex_line state lexbuf =
   let tok =
      match state.current_mode with
         ModeNormal ->
            lex_main state lexbuf
       | ModeString _ ->
            lex_string state lexbuf
       | ModeSkipString _ ->
            lex_skip_string state lexbuf
       | ModeQuote _ ->
            lex_quote state lexbuf
   in
      if !debug_lex then
         Lm_printf.eprintf "Token: %a@." pp_print_token tok;
      state.current_token <- tok;
      tok

(************************************************************************
 * Parse main loop.
 *)

(*
 * Make sure the lexbuf is valid.
 *)
let parse_refill state prompt root nest =
   if state.current_eof then
      begin
         let lexbuf = Lexing.from_function (lex_refill state) in
            state.current_eof     <- false;
            state.current_fill_ok <- true;
            state.current_prompt  <- prompt_indent prompt root nest;
            state.current_lexbuf  <- lexbuf;
            state.current_lexmode <- LexModeInitial;
            state.current_off     <- 0
      end

(*
 * Get the current indentation level.
 *)
let parse_indent state prompt root nest =
   parse_refill state prompt root nest;
   match state.current_lexmode with
      LexModeInitial ->
         let indent =
            (* Interactive shell ignores indentation *)
            if state.is_interactive then
               nest
            else
               lex_indent state state.current_lexbuf
         in
            if !debug_lex then
               Lm_printf.eprintf "indent: %d@." indent;
            state.current_lexmode <- LexModeNormal indent;
            indent
    | LexModeNormal indent ->
         indent

(* GS. In the following, parse = Omake_ast_parse.shell, i.e. the ocamlyacc
   generated parser
 *)

(*
 * Parse a single expression.
 * GS. an "expression" is not just a $-expression, but any code block, which
 * may span several lines.
 *)
let rec parse_exp state parse prompt root nest =
  let indent = parse_indent state prompt root nest in
  if indent > state.current_indent then
    syntax_error state "illegal indentation" state.current_lexbuf
  else if indent < state.current_indent then
    raise End_of_file
  else
    parse_exp_indent state parse prompt root nest

and parse_exp_indent state parse _ root nest =
  (* GS: after the indentation... *)
  let code, e =
    try parse (lex_line state) state.current_lexbuf with
      Parsing.Parse_error ->
      parse_error state
  in  (* GS: e is the parsed expression *)
  let code = Omake_ast_util.scan_body_flag code e in
  let parse = body_parser state code in
  (* GS. parse is now None, or Some Omake_ast_parse.shell or .string *)
  match parse with
    Some parse ->
    let prompt = prompt_string state root nest e in
    let body = parse_body state parse prompt nest in
    let e = Omake_ast_util.update_body e code body in
    (match Omake_ast_util.can_continue e with
      Some prompt ->
      (try e :: parse_exp state parse (prompt_ext prompt) false nest with
        End_of_file ->
        [e])
    | None ->
      [e])
  | None ->
    [e]

and parse_body state parse prompt nest =
  let nest = succ nest in
  let indent = parse_indent state prompt false nest in
  (* GS. The body must be further indented, otherwise it is not a body
     of the preceding expr
   *)
  if indent > state.current_indent then
    begin
      push_mode state ModeNormal;
      state.current_indent <- indent;
      parse_body_indent state parse prompt nest []
    end
  else
    []

and parse_body_indent state parse prompt nest el =
  (* GS TODO: reformulate with "match ... with exception" *)
  let e =
    try ParseExp (parse_exp state parse prompt false nest) with
      End_of_file ->
      if state.is_interactive then
        Lm_printf.printf ".@.";
      pop_mode state;
      ParseEOF
    | Omake_value_type.OmakeException _ as exn when state.is_interactive ->
      Lm_printf.eprintf "%a@." Omake_exn_print.pp_print_exn exn;
      ParseError
  in
  match e with
    ParseExp e ->
    parse_body_indent state parse prompt nest (List.rev_append e el)
  | ParseError ->
    parse_body_indent state parse prompt nest el
  | ParseEOF ->
    List.rev el

(*
 * Parse a file.
 * GS: Entry point
 *)
let parse_ast name =
   let inx = open_in name in
   let readline _ =
      try input_line inx ^ "\n" with
         End_of_file ->
            ""
   in
   let state = create name readline in
   let el = parse_body_indent state Omake_ast_parse.shell "<prompt>" 0 [] in
      close_in inx;
      el

(*
 * Parse a string.
 * GS: Entry point
 *)
let parse_string s =
   let len = String.length s in
   let index = ref 0 in
   let readline _ =
      let start = !index in
      let rec search i =
         if i = len then
            if start < i then
               begin
                  index := i;
                  String.sub s start (i - start) ^ "\n"
               end
            else
               raise End_of_file
         else if s.[i] = '\n' then
            begin
               index := i + 1;
               String.sub s start (i - start + 1)
            end
         else
            search (succ i)
      in
         search start
   in
   let state = create "-" readline in
        parse_body_indent state Omake_ast_parse.shell "<prompt>" 0 []

(*
 * Parse an expression.
 *)
let create_shell () =
   let state = create "-" Lm_readline.readline in
      state.is_interactive <- Lm_readline.is_interactive ();
      state

(*
 * Copy the state, if an exception happens, then
 * restore the initial state.
 *)
let parse_shell state prompt =
   let stack = save_mode state in
      state.current_fill_ok <- true;
      try parse_exp state Omake_ast_parse.shell prompt true 0 with
         exn ->
            Lm_readline.flush ();
            restore_mode state stack;
            state.current_buffer <- "";
            state.current_index <- 0;
            raise exn

(*
 * Just dependency analysis.
 *)
let parse_deps name =
   let inx = open_in name in
   let lexbuf = Lexing.from_channel inx in
   let deps =
      try Omake_ast_parse.deps lex_deps lexbuf with
         exn ->
            close_in inx;
            Lm_printf.eprintf "%s: char %d: scanner dependency syntax error@." name (Lexing.lexeme_end lexbuf);
            raise exn
   in
      close_in inx;
      deps

# 2303 "omake_ast_lex.ml"
