/* NVTV GUI (Brooktree part) -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: gui_bt.c,v 1.23 2003/10/10 16:16:57 dthierbach Exp $
 *
 * Contents:
 *
 * The GTK graphical user interface. Brooktree part.
 */

#include "local.h" /* before everything else */

#include <gtk/gtk.h>

#include "gui.h"
#include "gui_bt.h"
#include "backend.h"
#include "calc_bt.h"
#include "data_bt.h"

/* Automatic minimum distance between overscan percentages */

#define AUTO_DIST 10.0

GtkAdjustment *update_bt;

#ifndef DISABLE_TIMEOUT
static gint gui_timeout_bt_id = -1; /* status timeout */
#endif

static TVMode bt_calc_mode = {
  {TV_SYSTEM_NONE, 0, 0, "Custom", "", 0.0, 0.0},
  descFlags: TV_CAP_DUALVIEW | TV_DESC_DUALVIEW | TV_CAP_MACROVISION | 
             TV_CAP_NONINTERLACED | TV_CAP_MONOCHROME
};

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_BT(m) FIELD(gui_regs.enc.bt,m)

/* -------- GUI Masks -------- */

struct mask_calc {
  GtkAdjustment *hres, *vres;
  GtkAdjustment *hoc_min, *hoc_max, *voc_min, *voc_max;
  GtkLabel *hoc_sel, *voc_sel;
  GtkCList *list;
  gint column; /* last column sorted */
  GtkToggleButton *dist;
  GtkButton *calc;
};

struct mask_calc gui_mask_calc;

struct mask_bt_freq {
  GtkLabel *clk, *hsyn, *vsyn, *hover, *vover, *aspect;
};

struct mask_bt_freq gui_mask_bt_freq;

struct mask_status {
  GtkToggleButton *tvon, *pll, *over, *under, *pal, *busy;
};

struct mask_status gui_mask_bt_status;

static GuiRegMask bt_mask1_reg [] = {
  {label:"h_active:",      bits:10, tick:1, FIELD_BT(h_active)},
  {label:"h_blanki:",      bits:9,  tick:1, FIELD_BT(h_blanki)},
  {label:"h_clki:",        bits:11, tick:1, FIELD_BT(h_clki)},
  {label:"h_fract:",       bits:8,  tick:1, FIELD_BT(h_fract)},
  {label:"h_blanko:",      bits:11, tick:1, FIELD_BT(h_blanko)},
  {label:"h_clko:",        bits:12, tick:1, FIELD_BT(h_clko)},
  {label:"sync_amp:",      bits:8,  tick:1, FIELD_BT(sync_amp)},
  {label:"bst_amp:",       bits:8,  tick:1, FIELD_BT(bst_amp)},
  {label:"hsync_width:",   bits:8,  tick:1, FIELD_BT(hsync_width)},
  {label:"hburst_begin:",  bits:8,  tick:1, FIELD_BT(hburst_begin)},
  {label:"hburst_end:",    bits:8,  tick:1, FIELD_BT(hburst_end)},
  {label:"v_activei:",     bits:10, tick:1, FIELD_BT(v_activei)},
  {label:"v_blanki:",      bits:8,  tick:1, FIELD_BT(v_blanki)},
  {label:"v_linesi:",      bits:10, tick:1, FIELD_BT(v_linesi)},
  {label:"v_activeo:",     bits:9,  tick:1, FIELD_BT(v_activeo)},
  {label:"v_blanko:",      bits:10, tick:1, FIELD_BT(v_blanko)},
  {label:"v_scale:",       bits:14, tick:1, FIELD_BT(v_scale)},
  {label:"pll_fract:",     bits:16, tick:1, FIELD_BT(pll_fract)},
  {label:"pll_int:",       bits:6,  tick:1, FIELD_BT(pll_int)},
  {label:NULL}
};				    

static GuiRegMask bt_mask2_reg [] = {
  {label:"hsynoffset:",    bits:-10,tick:1, FIELD_BT(hsynoffset)},
  {label:"vsynoffset:",    bits:11, tick:1, FIELD_BT(vsynoffset)},
  {label:"hsynwidth:",     bits:6,  tick:1, FIELD_BT(hsynwidth)},
  {label:"vsynwidth:",     bits:3,  tick:1, FIELD_BT(vsynwidth)},
  {label:"phase_off:",     bits:8,  tick:1, FIELD_BT(phase_off)},
  {label:"macro:",         bits:3,  tick:1, FIELD_BT(macro)},
  {label:"mcr:",           bits:8,  tick:1, FIELD_BT(mcr)},
  {label:"mcb:",           bits:8,  tick:1, FIELD_BT(mcb)},
  {label:"my:",            bits:8,  tick:1, FIELD_BT(my)},
  {label:"out_muxa:",      bits:2,  tick:1, FIELD_BT(out_muxa)},
  {label:"out_muxb:",      bits:2,  tick:1, FIELD_BT(out_muxb)},
  {label:"out_muxc:",      bits:2,  tick:1, FIELD_BT(out_muxc)},
  {label:NULL}
};				    

static GuiRegMask bt_mask2_twin [] = {
  {label:"msc:",           bits:32, tick:1, FIELD_BT(msc)},
  {label:NULL}
};				    

static GuiFlagMask bt_mask2_flag [] = {
  {label:"dacdisa",     mask:BT_FLAG3_DACDISA,     FIELD_BT(flags3)}, 
  {label:"dacdisb",     mask:BT_FLAG3_DACDISB,     FIELD_BT(flags3)}, 
  {label:"dacdisc",     mask:BT_FLAG3_DACDISC,     FIELD_BT(flags3)}, 
  {label:NULL }
};

static GuiRegMask bt_mask3_reg [] = {
  {label:"f_sely:",        bits:3,  tick:1, FIELD_BT(f_sely)},
  {label:"f_selc:",        bits:3,  tick:1, FIELD_BT(f_selc)},
  {label:"ylpf:",          bits:2,  tick:1, FIELD_BT(ylpf)},
  {label:"clpf:",          bits:2,  tick:1, FIELD_BT(clpf)},
  {label:"ycoring:",       bits:3,  tick:1, FIELD_BT(ycoring)},
  {label:"ccoring:",       bits:3,  tick:1, FIELD_BT(ccoring)},
  {label:"yattenuate:",    bits:3,  tick:1, FIELD_BT(yattenuate)},
  {label:"cattenuate:",    bits:3,  tick:1, FIELD_BT(cattenuate)},
  {label:NULL}
};				    

static GuiFlagMask bt_mask3_flag [] = {
  {label:"ni_out",      mask:BT_FLAG1_NI_OUT,      FIELD_BT(flags1)}, 
  {label:"setup",       mask:BT_FLAG1_SETUP,       FIELD_BT(flags1)}, 
  {label:"625line",     mask:BT_FLAG1_625LINE,     FIELD_BT(flags1)}, 
  {label:"vsync_dur",   mask:BT_FLAG1_VSYNC_DUR,   FIELD_BT(flags1)}, 
  {label:"dis_screset", mask:BT_FLAG1_DIS_SCRESET, FIELD_BT(flags1)}, 
  {label:"pal_md",      mask:BT_FLAG1_PAL_MD,      FIELD_BT(flags1)}, 
  {label:"dis_ffilt",   mask:BT_FLAG2_DIS_FFILT,   FIELD_BT(flags2)}, 
  {label:"dis_yflpf",   mask:BT_FLAG2_DIS_YFLPF,   FIELD_BT(flags2)}, 
  {label:"eclip",       mask:BT_FLAG1_ECLIP,	   FIELD_BT(flags1)}, 
  {label:"en_async",    mask:BT_FLAG1_EN_ASYNC,	   FIELD_BT(flags1)}, 
  {label:"dis_gmshy",   mask:BT_FLAG2_DIS_GMSHY,   FIELD_BT(flags2)}, 
  {label:"dis_gmushy",  mask:BT_FLAG2_DIS_GMUSHY,  FIELD_BT(flags2)}, 
  {label:"dis_gmshc",   mask:BT_FLAG2_DIS_GMSHC,   FIELD_BT(flags2)}, 
  {label:"dis_gmushc",  mask:BT_FLAG2_DIS_GMUSHC,  FIELD_BT(flags2)}, 
  {label:"dis_chroma",  mask:BT_FLAG2_DIS_CHROMA,  FIELD_BT(flags2)}, 
  {label:NULL }
};

/* -------- -------- */

void bt_freq_calc_cb (GtkObject *obj, struct mask_bt_freq *m)
{
  double Fxtal = 13500000; 
  double Fclk; 
  double Fhsyn;
  double Fvsyn;
  double dALO, dTLO, dATO, dTTO, dVACTIVEO;
  double dAspect, dHOC, dVOC;
  char s [20];

  if ((gui_tv_chip & TV_ENCODER) != TV_BROOKTREE && 
      (gui_tv_chip & TV_ENCODER) != TV_CONEXANT) return;

  Fclk = Fxtal * (gui_regs.enc.bt.pll_int + 
		  gui_regs.enc.bt.pll_fract / 65536.0) / 6.0;
  sprintf (s, "%3.2f MHz", Fclk / 1e6);
  gtk_label_set_text (m->clk, s);

  Fhsyn = Fclk / gui_regs.enc.bt.h_clki;
  snprintf (s, 20, "%3.2f kHz", Fhsyn / 1e3);
  gtk_label_set_text (m->hsyn, s);

  Fvsyn = Fhsyn / gui_regs.enc.bt.v_linesi;
  snprintf (s, 20, "%3.2f  Hz", Fvsyn);
  gtk_label_set_text (m->vsyn, s);

  if (gui_regs.enc.bt.flags1 & BT_FLAG1_625LINE) {
    dALO = 288.0;
    dTLO = 312.0;
    dATO = .000052;
    dTTO = .000064;
  } else {
    dALO = 243.0;
    dTLO = 262.0;
    dATO = .00005265556;
    dTTO = .00006355556;
  }
  /* if interlace then dTLO += 0.5 */
  dVACTIVEO = (int) (((gui_regs.enc.bt.v_activei * dTLO) + 
		      (gui_regs.enc.bt.v_linesi - 1.0)) / 
		     gui_regs.enc.bt.v_linesi);
  /* normally, dVACTIVEO = v_activeo - 2 */
  /* These two euqations are also possible, but that's not what happens
   *  in recalc.c, so don't use them.
   * dVOC = 1.0 - (gui_regs.enc.bt.v_activeo - 2) / dALO;
   * dVOC = 1.0 - ((gui_regs.enc.bt.v_activei / gui_regs.enc.bt.v_linesi) / 
   *              (dALO / dTLO));
   */
  dVOC = 1.0 - dVACTIVEO / dALO;
  dHOC = 1.0 - ((2.0 * gui_regs.enc.bt.h_active / gui_regs.enc.bt.h_clko) / 
               (dATO / dTTO));
  dAspect = (1.0 - dHOC) / (1.0 - dVOC);

  snprintf (s, 20, "%06.3f %%", dHOC * 100.0);
  gtk_label_set_text (m->hover, s);
  snprintf (s, 20, "%06.3f %%", dVOC * 100.0);
  gtk_label_set_text (m->vover, s);
  snprintf (s, 20, "%07.5f", dAspect);
  gtk_label_set_text (m->aspect, s);

  /* FIXME: check above, calculate aspect */
}

gint check_bt_cb (struct mask_status *m)
{
  int status;

  if ((gui_tv_chip & TV_ENCODER) != TV_BROOKTREE) return TRUE;
  status = back_card->getStatus (2);
  if (status < 0) return TRUE;
  if (! (status & 0x100)) status = 0; 
  gtk_toggle_button_set_active (m->tvon,  (status & 0x100) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->pll,   (status & 0x010) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->over,  (status & 0x008) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->under, (status & 0x004) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->pal,   (status & 0x002) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->busy,  (status & 0x001) ? TRUE : FALSE);
  return TRUE;
}

void calc_update_cb (GtkWidget *widget, struct mask_calc *m)
{
  gtk_widget_set_sensitive (GTK_WIDGET (m->calc),
			    (m->list->selection) ? TRUE : FALSE);
}

void calc_dist_min_cb (GtkAdjustment *min, GtkAdjustment *max)
{
  if (gui_mask_calc.dist->active && max->value - min->value < AUTO_DIST)
    gtk_adjustment_set_value (max, min->value + AUTO_DIST);
  if (min->value > max->value)
    gtk_adjustment_set_value (max, min->value);
}

void calc_dist_max_cb (GtkAdjustment *max, GtkAdjustment *min)
{
  if (gui_mask_calc.dist->active && max->value - min->value < AUTO_DIST)
    gtk_adjustment_set_value (min, max->value - AUTO_DIST);
  if (min->value > max->value)
    gtk_adjustment_set_value (min, max->value);
}

void calc_result_cb (double hoc, double voc, double badness,
  double aspect, int flag)
{
  char sh[10], sv[10], sb[10], sa[10], sn[4];
  char *text [5];

  snprintf (sh, 10, "%06.3f %%", hoc * 100.0);
  snprintf (sv, 10, "%06.3f %%", voc * 100.0);
  snprintf (sb, 10, "%8.5f", badness);
  snprintf (sa, 10, "%6.4f", aspect);
  snprintf (sn, 3, "%c%s", 
	    (flag & BT_CALC_CHARCLK8) ? '8' : '1', 
	    (flag & BT_CALC_RATIO32) ? "+" : "");
  text [0] = sh;
  text [1] = sv;
  text [2] = sb;
  text [3] = sa;
  text [4] = sn;
  gtk_clist_append (gui_mask_calc.list, text);
}

void calc_list_cb (GtkWidget *widget, struct mask_calc *m)
{
  int hres, vres;
  double hoc_min, hoc_max, voc_min, voc_max;

  gtk_clist_freeze (m->list);
  gtk_clist_clear (m->list);
  hres = (int) m->hres->value;
  vres = (int) m->vres->value;
  hoc_min = m->hoc_min->value / 100.0;
  voc_min = m->voc_min->value / 100.0;
  hoc_max = m->hoc_max->value / 100.0;
  voc_max = m->voc_max->value / 100.0;
  switch (gui_tv_chip & TV_ENCODER)
  {
    case TV_BROOKTREE:
      recalc_bt_find (gui_system, hres, vres, hoc_min, hoc_max, 
		      voc_min, voc_max, &calc_result_cb);
      break;
    case TV_CONEXANT:
      recalc_cx_find (gui_system, hres, vres, hoc_min, hoc_max, 
		      voc_min, voc_max, &calc_result_cb);
      break;
    default:
      break;
  }
  m->column = -1;
  gtk_clist_sort (m->list);
  gtk_clist_select_row (m->list, 0, 0);
  gtk_clist_thaw (m->list);
  calc_update_cb (NULL, m);
}

void calc_reset_cb (GtkWidget *widget, struct mask_calc *m)
{
  if ((gui_tv_chip & TV_ENCODER) != TV_BROOKTREE && 
      (gui_tv_chip & TV_ENCODER) != TV_CONEXANT) return;
  if (gui_list_mode) {
    gtk_adjustment_set_value (m->hres, gui_list_mode->spec.res_x);
    gtk_adjustment_set_value (m->vres, gui_list_mode->spec.res_y);
    gtk_adjustment_set_value (m->hoc_min, 
      gui_list_mode->spec.hoc - AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->hoc_max, 
      gui_list_mode->spec.hoc + AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->voc_min, 
      gui_list_mode->spec.voc - AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->voc_max, 
      gui_list_mode->spec.voc + AUTO_DIST/2.0);
    gtk_clist_clear (m->list);
    calc_update_cb (NULL, m);
  }
}

void calc_column_cb (GtkCList *clist, gint column, struct mask_calc *m)
{
  if (m->column == column) {
    gtk_clist_set_sort_type (m->list, GTK_SORT_DESCENDING);
    m->column = -1;
  } else {
    gtk_clist_set_sort_type (m->list, GTK_SORT_ASCENDING);
    m->column = column;
  }
  gtk_clist_set_sort_column (m->list, column); 
  gtk_clist_sort (m->list);
  gtk_clist_set_sort_column (m->list, 2); 
  gtk_clist_set_sort_type (m->list, GTK_SORT_ASCENDING);
}

void calc_select_cb (GtkCList *clist, gint row, gint column,
  GdkEventButton *event, struct mask_calc *m)
{
  gchar *text;

  g_return_if_fail (GTK_IS_CLIST (clist));
  gtk_clist_get_text (clist, row, 0, &text);
  gtk_label_set_text (m->hoc_sel, text);
  gtk_clist_get_text (clist, row, 1, &text);
  gtk_label_set_text (m->voc_sel, text);
}

void calc_calc_cb (GtkButton *button, struct mask_calc *m)
{
  int hres, vres;
  double hoc, voc;

  bt_calc_mode.spec.res_x = hres = (int) m->hres->value;
  bt_calc_mode.spec.res_y = vres = (int) m->vres->value;
  sscanf (m->hoc_sel->label, "%lf", &hoc); 
  sscanf (m->voc_sel->label, "%lf", &voc); 
  bt_calc_mode.spec.hoc = hoc;
  bt_calc_mode.spec.voc = voc;
  hoc /= 100.0;
  voc /= 100.0;

  gui_func->calc (gui_system, hres, vres, hoc, voc, &bt_calc_mode.regs);
  gui_act_mode_set (&bt_calc_mode);
}


void reset_bt_cb (GtkObject *obj, gpointer data)
{
  if ((gui_tv_chip & TV_ENCODER) != TV_BROOKTREE) return;
  if (gui_act_mode) {
    gui_regs.enc.bt = gui_act_mode->regs.enc.bt;
    gui_tv_set ();
    gtk_signal_emit_by_name (GTK_OBJECT (update_bt), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
}

void update_bt_cb (GtkObject *obj, gpointer data)
{
  if ((gui_tv_chip & TV_ENCODER) != TV_BROOKTREE) return;
  gtk_signal_emit_by_name (GTK_OBJECT (update_bt), "changed");
}

void gui_map_bt_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_map_bt_cb");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_bt_id == -1) {
    gui_timeout_bt_id = gtk_timeout_add (500 /* ms */, 
      (GtkFunction) check_bt_cb, &gui_mask_bt_status);
  }
#endif
}

void gui_unmap_bt_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_unmap_bt_cb");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_bt_id != -1) {
    gtk_timeout_remove (gui_timeout_bt_id);
    gui_timeout_bt_id = -1;
  }
#endif
}

/* -------- GUI Pages -------- */

GtkWidget *gui_bt_status_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;

  page = gtk_vbox_new (FALSE, 5);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_bt_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_bt_cb), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (page), 5);

  /* Frequencies CRT / TV */

  frame = gtk_frame_new ("Frequencies");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (3, 4, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  gui_mask_bt_freq.clk = 
    gui_mask_label (table, "Monitor dot clock:",   "---.-- MHz", 0,1,2, 0);
  gui_mask_bt_freq.hsyn = 
    gui_mask_label (table, "Monitor horiz. sync:", "---.-- kHz", 0,1,2, 1);
  gui_mask_bt_freq.vsyn = 
    gui_mask_label (table, "Monitor vert. sync:",  "---.--  Hz", 0,1,2, 2);
  gui_mask_bt_freq.hover = 
    gui_mask_label (table, "TV horiz. overscan:",  "--.--  %",   0,1,2, 3);
  gui_mask_bt_freq.vover = 
    gui_mask_label (table, "TV vert. overscan:",   "--.--  %",   0,1,2, 4);
  gui_mask_bt_freq.aspect = 
   gui_mask_label (table, "TV aspect ratio:",      "-.---  ",    0,1,2, 5);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  /* TV Status: BT Fifo overrun/underrun */

  frame = gtk_frame_new ("Brooktree Status");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  button = gtk_check_button_new_with_label ("TV on");
  gui_mask_bt_status.tvon = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 0,1, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("PLL locked");
  gui_mask_bt_status.pll = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 1,2, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("FIFO overrun");
  gui_mask_bt_status.over = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 1,2, 0,1, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("FIFO underrun");
  gui_mask_bt_status.under = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 1,2, 1,2, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("PAL pin");
  gui_mask_bt_status.pal = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 2,3, 0,1, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("Busy");
  gui_mask_bt_status.busy = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 2,3, 1,2, GTK_FILL,GTK_FILL,0,0);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  /* return */

  return page;
}

GtkWidget *gui_bt_reg1_page (void)
{
  GtkAccelGroup *gui_bt_reg_accel_group;

  gui_bt_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("BT 868/869 Timings", 
    gui_bt_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_bt), GTK_SIGNAL_FUNC (reset_bt_cb),
    0, 0, 11, bt_mask1_reg, 0, 11, NULL, 0, 0, 0, NULL);
}

GtkWidget *gui_bt_reg2_page (void)
{
  GtkAccelGroup *gui_bt_reg_accel_group;

  gui_bt_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("BT 868/869 Parameters", 
    gui_bt_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_bt), GTK_SIGNAL_FUNC (reset_bt_cb),
    0, 0, 9, bt_mask2_reg, 0, 9, bt_mask2_twin, 6, 0, 9, bt_mask2_flag);
}

GtkWidget *gui_bt_reg3_page (void)
{
  GtkAccelGroup *gui_bt_reg_accel_group;

  gui_bt_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("BT 868/869 Flags", 
    gui_bt_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_bt),  GTK_SIGNAL_FUNC (reset_bt_cb),
    0, 0, 9, bt_mask3_reg, 0, 0, NULL, 6, 0, 8, bt_mask3_flag);
}

GtkWidget *gui_bt_calc_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *spin;   
  GtkWidget *button;
  GtkWidget *scroller;
  GtkWidget *clist;
  GtkAdjustment *adj;
  GtkAccelGroup *gui_bt_calc_accel_group;

  gui_bt_calc_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (5,2,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_bt_calc_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_bt_calc_accel_group);

  frame = gtk_frame_new ("Calculate registers");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Horizontal:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  label = gtk_label_new ("Vertical:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  label = gtk_label_new ("Resolution");
  gtk_table_attach (GTK_TABLE(table), label, 1,2, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 
          1024.0, 1.0, 10.0, 0.0);
  gui_mask_calc.hres = adj;
  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 
          1024.0, 1.0, 10.0, 0.0);
  gui_mask_calc.vres = adj;
  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 1,2, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new ("Overscan range (%)");
  gtk_table_attach (GTK_TABLE(table), label, 2,4, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.hoc_min = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.hoc_max = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.hoc_min), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_min_cb), (gpointer) gui_mask_calc.hoc_max);
  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.hoc_max), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_max_cb), (gpointer) gui_mask_calc.hoc_min);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.voc_min = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 2,3, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.voc_max = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 3,4, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.voc_min), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_min_cb), (gpointer) gui_mask_calc.voc_max);
  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.voc_max), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_max_cb), (gpointer) gui_mask_calc.voc_min);

  label = gtk_label_new ("Selected");
  gtk_table_attach (GTK_TABLE(table), label, 4,5, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--- %");
  gui_mask_calc.hoc_sel = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 4,5, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);

  label = gtk_label_new ("--.--- %");
  gui_mask_calc.voc_sel = (GtkLabel *) label;
   gtk_table_attach (GTK_TABLE(table), label, 4,5, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_table_attach (GTK_TABLE(table), scroller, 0,5, 3,4,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  clist = gtk_clist_new (5);
  gui_mask_calc.list = (GtkCList *) clist;
  gui_mask_calc.column = -1;
  gtk_clist_set_column_title (GTK_CLIST (clist), 0, "H.Over.");
  gtk_clist_set_column_title (GTK_CLIST (clist), 1, "V.Over.");
  gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Badness");
  gtk_clist_set_column_title (GTK_CLIST (clist), 3, "Aspect");
  gtk_clist_set_column_title (GTK_CLIST (clist), 4, "Note");
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  gtk_clist_column_titles_active (GTK_CLIST (clist)); /* must be after show */
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_IN); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 58); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 58); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 2, 55); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 3, 45); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 4, 25); /* some guess */
  gtk_clist_set_sort_column (GTK_CLIST (clist), 2); /* badness */
  gtk_clist_set_sort_type (GTK_CLIST (clist), GTK_SORT_ASCENDING); 
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
    GTK_SIGNAL_FUNC (calc_select_cb), &gui_mask_calc);
  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
    GTK_SIGNAL_FUNC (calc_column_cb), &gui_mask_calc);
  gtk_container_add(GTK_CONTAINER(scroller), clist);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);

  button = gtk_button_new_with_label ("List");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_list_cb), &gui_mask_calc);
  gtk_table_attach(GTK_TABLE(page), button, 0,1,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Calc");
  gui_mask_calc.calc = (GtkButton *) button;
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_calc_cb), &gui_mask_calc);
  gtk_table_attach(GTK_TABLE(page), button, 1,2,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_toggle_button_new_with_label ("AutoDist");
  gui_mask_calc.dist = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach(GTK_TABLE(page), button, 2,3,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), 
    (gpointer) (PRINT_CHIP_REGS | PRINT_CRT_REGS));
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_reset_cb), &gui_mask_calc);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_bt_calc_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

void gui_bt_init (void)
{
  update_bt = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (update_chip), "changed", 
    GTK_SIGNAL_FUNC (update_bt_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (update_mode), "changed",
    GTK_SIGNAL_FUNC (calc_reset_cb), &gui_mask_calc);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
    GTK_SIGNAL_FUNC (bt_freq_calc_cb), &gui_mask_bt_freq);
}

/*

BT Regs1: old BT Regs
BT Regs2: hsynoffset, vsynoffset, hsynwidth, vsynwidth, phase_off,
          macro, msc  
          out_mux{a,b,c}
BT Regs3: old BT Flags

*/

