/*
 *	tcpblast/tcpping - test and estimate TCP thruput
 *
 *	Daniel Karrenberg   <dfk@nic.eu.net>
 */

/*
 *      tcpblast.c 2.1 91/08/06
 */

#ifndef lint
static char SccsId[] ="@(#)tcpblast.c	2.1\t8/6/91";
#endif lint

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/time.h>

#include <netinet/in.h>

#include <netdb.h>
#include <stdio.h>

#define DEFBLKSIZ 1

/* inet stuff */
struct	sockaddr_in sin;
struct	hostent *host;

/* time stuff */
unsigned long starts, startms, stops, stopms, expms;
struct timeval ti; 
struct timezone tiz;

/* buffer pointers */
int *ibuf, *obuf;

/* getopt stuff */
extern char *optarg;
extern int optind;

/* the socket to the server */
int	f;

/* flags and switches */
int pingflag = 0;	/* we are tcpping and not tcpblast */
int parent = 0;		/* we are the parent process */
int vflag = 0;		/* verbose flag */
int cflag = 0;		/* compare flag */
int fflag = 0;		/* use file flag */
int bflag = 0;		/* use nonstd blocksize flag */

/* from todays commandline */
int nkb;		/* KBytes to send */
int blksiz = DEFBLKSIZ; /* block size */
int df;			/* data  file */

main(argc, argv)
int argc; char **argv;
{
	register int i,j, cnt;
	register signed char c;

	if (strcmp(argv[0], "tcpping") == 0)
		pingflag++;

	if (argc == 1)
		usage();

	while ((c=getopt(argc, argv, pingflag ? "vcb:f:" : "vb:f:")) != EOF) {
		switch (c) {
		case 'v':
			vflag++;
			break;

		case 'c':
			if (!pingflag)
				usage();
			cflag++;
			break;

		case 'b':
			bflag++;
			blksiz = atoi(optarg);
			if (blksiz<1 || blksiz > 64) 
				ex("%s: 1KB <= blksiz <= 64KB!\n", argv[0]);
			break;

		case 'f':
			fflag++;
			if ((df = open(optarg, 0)) <0) {
				perror(optarg);
				usage();
			}
			break;

		case '?':
			usage();
			break;
		}
	}

	if (argc-optind != 2)
		usage();

	nkb = atoi(argv[optind+1]);
	if (nkb<=1 || nkb>=10000)
		ex("%s: 1 < nKB <= 10000 \n", argv[0]);

	if (cflag && fflag)
		ex("-c flag does not work with data from file, sorry");

	if (nkb < 32) 
		fprintf(stderr, "%s: nkbytes low, loosing accuracy\n", argv[0]);

	if (nkb%blksiz) {
		nkb -= nkb%blksiz;
		fprintf(stderr, "nkbytes shortened to %dKB ", nkb);
		fprintf(stderr, "to match blocksize\n");
	}
		
	bzero((char*)&sin, sizeof (sin));
	sin.sin_family = AF_INET;
	f = socket(AF_INET, SOCK_STREAM, 0);
	if (f < 0) 
		perrex("socket");
	if (bind(f, &sin, sizeof (sin)) < 0)
		perrex("bind");

	host = gethostbyname(argv[optind]);
	if (host) {
		sin.sin_family = host->h_addrtype;
		bcopy(host->h_addr, &sin.sin_addr, host->h_length);
	} else {
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = inet_addr(argv[optind]);
		if (sin.sin_addr.s_addr == -1) 
			ex("host %s unknown", argv[optind]);

	}

	sin.sin_port = pingflag ? htons(7): htons(9);

	if (connect(f, &sin, sizeof(sin)) <0)
		perrex("connect");

	if (gettimeofday(&ti, &tiz) < 0)
		perrex("time");

	if ((obuf = (int *) malloc(blksiz*1024)) == NULL)
		perrex("output buffer");

	if (pingflag && (ibuf = (int*) malloc(blksiz*1024)) == NULL)
		perrex("input buffer");

	starts =  ti.tv_sec;
	startms += ti.tv_usec / 1000L;

	if (!fflag)
		srandom((int)startms&0xffff);

	if (vflag)
		printf("Sending %dKB with blocksize %dKB\n", nkb, blksiz);

	i = nkb;

	if (pingflag  && (parent=fork()) == -1)
		perrex("fork()");

	while (i > 0)
	{
	    if (!pingflag || (pingflag&&parent)) {
		if (fflag) {
			if (read(df, (char*) obuf, blksiz*1024) < blksiz*1024)
				lseek(df, 0, 0);
		} 
		else {
			for (j=0; j<blksiz*1024/4; j++) 
				obuf[j] = random();
		}
		if (write(f, (char*)obuf, blksiz*1024) != blksiz*1024)
			perror("send:");
		if (vflag)
			write(1, ">", 1);
	    }

	    if (pingflag && !parent) {
		char *p = (char*) ibuf;

		j=blksiz*1024;
		while (j>0)
		{
			if ((cnt=read(f, p, j)) <0)
				perror("receive:");
			p += cnt;
			j -= cnt;
		}
		if (vflag)
			write(1, "<", 1);

		if (cflag & !fflag) {
			for (j=0; j<blksiz*1024/4; j++) {
				if (ibuf[j] != random())
					fprintf(stderr, "data compare error\n");
			}
		}
	    }

	    i -= blksiz;
	}

	if (pingflag) {
		if (!parent)
			exit(0);
		else
			wait(NULL);
	}

	if (gettimeofday(&ti, &tiz) < 0)
		perrex("time");
	stops =  ti.tv_sec;
	stopms += ti.tv_usec / 1000L;


	expms = (stops-starts)*1000 + (stopms-startms);
	if (vflag) {
		printf ("\n");
		printf("%d KB in %ld msec\n", nkb, expms);
	}
	printf("%.0f bit/s\n", (double) nkb*1024. / expms*8000.0);
	exit(0);
}

usage()
{
	if (pingflag)
	   fprintf(stderr, "usage: tcpping [-v] [-c] [-b blksiz] [-f file] ");
	else
	   fprintf(stderr, "usage: tcpblast [-v] [-b blksiz] [-f file] ");
	fprintf(stderr, "destination nkbytes\n");

	fprintf(stderr, "      -v  be verbose & print dots etc.\n");
    	if (pingflag)
	  fprintf(stderr, "      -c  compare sent vs. received data\n");
	fprintf(stderr, "      -b  blksiz, specify blocksize in KB\n");
	fprintf(stderr, "      -f  file, send data from the file\n");
	exit(1);
}

perrex(msg)
char *msg;
{
	perror(msg);
	exit(1);
}

ex(msg, arg)
char *msg, *arg;
{
	fprintf(stderr, msg, arg);
	fprintf(stderr, "\n");
	exit(1);
}
