/* This file was contributed by Suzanne Skinner and is copyrighted
   under the GNU General Public License. (C) 2002 Suzanne Skinner.

   The code contained in this file is free software; you can
   redistribute it and/or modify it under the terms of the GNU General
   Public License as published by the Free Software Foundation; either
   version 2, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

/* getline is a more flexible, display-independent readline library. It
 * ties in with the dynstr library in order to provide command-line editing
 * on lines of arbitrary length. It calls no display or output functions,
 * acting solely as a back-end.
 *
 * There is only one important user-visible function: gl_handle_keystroke. Call
 * it whenever you've read an input keystroke. gl_handle_keystroke takes a
 * pointer to the current string, a prefix length (see below), an integer
 * cursor (0-based), and a chtype value representing the keystroke. It returns
 * the updated cursor, or, if the user typed a key bound to glf_done (most
 * likely ENTER), returns GL_DONE to indicate that the input is complete,
 * or GL_ABORT to indicate that the input was aborted.
 *
 * If a non-0 prefix length, L, is specified, the user will not be able to
 * edit or cursor over the first L characters of the given string.
 *
 * This library includes ncurses.h to provide chtype, but does not otherwise
 * depend on ncurses.
 *
 * Default keybindings are defined in the array gl_keybindings. They may be
 * user configurable in the future; for now, you'll have to edit getline.c
 * directly.
 *
 * getline keeps an input history, much like readline, retaining up to
 * GL_HISTORY_MAXLEN-1 past inputs. However, note that only one history
 * record is kept for all calls to gl_handle_keystroke in a program.
 */

#ifndef GETLINE_H
#define GETLINE_H

#include <ncurses.h>

#define GL_DONE            -1
#define GL_ABORT           -2
#define GL_HISTORY_MAXLEN  50

#define GL_FUNC(name) int name(dynstr *str, int prefix_len, int cursor)

typedef int (*gl_func)(dynstr *, int, int);

typedef struct gl_keybinding_struct {
    chtype   key;
    gl_func  handler;
} gl_keybinding;

typedef struct gl_history_entry_struct {
    char *line;
    struct gl_history_entry_struct *next;
    struct gl_history_entry_struct *prev;
} gl_history_entry;

int gl_handle_keystroke(dynstr *str, int prefix_len, int cursor,
                        chtype keystroke);

GL_FUNC(glf_cursor_left);
GL_FUNC(glf_cursor_right);
GL_FUNC(glf_beginning_of_line);
GL_FUNC(glf_end_of_line);
GL_FUNC(glf_delete_char_back);
GL_FUNC(glf_delete_char_forward);
GL_FUNC(glf_delete_all);
GL_FUNC(glf_delete_to_end);
GL_FUNC(glf_history_back);
GL_FUNC(glf_history_forward);
GL_FUNC(glf_abort);
GL_FUNC(glf_done);

#endif
