/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

/*!
    \file   lcas.h
    \brief  API of the LCAS library
    \author Martijn Steenbakkers for the EU DataGrid.

    This header contains the declarations of the LCAS library functions:
    -# lcas_init(): To initialize the LCAS module
    -# lcas_get_fabric_authorization(): to place an authorization request
    -# lcas_term(): To cleanly terminate the module
    \ingroup LcasInterface
*/
/*!
    \defgroup LcasInterface Interface to LCAS (library)
    \brief The API is available by including the header lcas.h
*/

#ifndef LCAS_H
#define LCAS_H

#ifndef LCAS_C
#   define LCAS_C_STORAGE_CLASS extern
#else
#   define LCAS_C_STORAGE_CLASS
#endif

/******************************************************************************
                             Include header files
******************************************************************************/
#ifndef NOGLOBUS
    #include <gssapi.h>
#endif /* NOGLOBUS */
#include "lcas_types.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/

/******************************************************************************
Function:   lcas_init
Description:
    Initialize LCAS module: 
    setup logging, error handling
    read from LCAS database the plugins to be loaded

Parameters:
    fp: file handle for logging (from gatekeeper)
Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
/*!
    \fn lcas_init(
        FILE* fp
        )
    \brief Initialize the LCAS module.

    The function does the following:
    - initialize LCAS module.
    - setup logging, error handling (not yet).
    - read from LCAS database the plugins to be loaded.
    - initialize the plugins

    \param fp file handle for logging (from gatekeeper)
    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_init(
        FILE* fp
);

/******************************************************************************
Function:   lcas_init_and_log
Description:
    Select logging type
    Setup logging, error handling (not yet).
    Read from LCAS database the plugins to be loaded.
    Initialize the plugins

Parameters:
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
/*!
    \fn lcas_init_and_log(
        FILE* fp,
        unsigned short logtype
        )
    \brief Initialize the LCAS module.

    The function does the following:
    - select logging type
    - setup logging, error handling (not yet).
    - read from LCAS database the plugins to be loaded.
    - initialize the plugins

    \param fp file handle for logging (from gatekeeper or other previously opened file handle).
              If the file handle is zero, assume that only syslogging is requested.
    \param logtype type of logging (usrlog and/or syslog)

    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_init_and_log(
        FILE* fp,
        unsigned short logtype
);

/******************************************************************************
Function:   lcas_init_and_logfile
Description:
    Select logging type
    Setup logging by providing a file handle or file name, error handling (not yet).
    Read from LCAS database the plugins to be loaded.
    Initialize the plugins

Parameters:
    logfile: name of logfile 
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
/*!
    \fn lcas_init_and_logfile(
        char * logfile,
        FILE* fp,
        unsigned short logtype
        )
    \brief Initialize the LCAS module.

    The function does the following:
    - select logging type
    - setup logging by providing a file handle or file name, error handling (not yet).
    - read from LCAS database the plugins to be loaded.
    - initialize the plugins

    \param logfile name of the logfile
    \param fp file handle for logging (from gatekeeper or other previously opened file handle).
              If the file handle is zero, assume that only syslogging is requested.
    \param logtype type of logging (usrlog and/or syslog)

    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_init_and_logfile(
        char * logfile,
        FILE* fp,
        unsigned short logtype
);

/******************************************************************************
Function:   lcas_term
Description:
    Terminate LCAS module: 

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
/*!
    \fn lcas_term()
    \brief Terminate the LCAS module.

    The function does the following:
    - terminate the LCAS module
    - terminate the plugins

    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_term(void);

/******************************************************************************
Function:   lcas_get_fabric_authorization
Description:
    Call LCAS in order to get authorization on the local fabric

Parameters:
    user_cred : user globus credential handle
    request: RSL string
Returns:
    0: authorization succeeded
    1: authorization failed
******************************************************************************/
#if ALLOW_EMPTY_CREDENTIALS
/*!
    \fn lcas_get_fabric_authorization(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcas_request_t request
    )
    \brief submit authorization request to the LCAS

    This function calls the plugins for authorization.
    As a back-up for empty credentials the user DN may be specified as well.
    For this \b ALLOW_EMPTY_CREDENTIALS should be defined.

    \param user_dn_tmp user DN
    \param user_cred   GSS/GSI user credential
    \param request     authorization request as RSL string

    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_get_fabric_authorization(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcas_request_t request
);
#else
/*!
    \fn lcas_get_fabric_authorization(
        gss_cred_id_t user_cred,
        lcas_request_t request
    )
    \brief submit authorization request to the LCAS

    This function calls the plugins for authorization.

    \param user_cred GSS/GSI user credential
    \param request   authorization request as RSL string

    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
LCAS_C_STORAGE_CLASS int lcas_get_fabric_authorization(
        gss_cred_id_t user_cred,
        lcas_request_t request
);
#endif

LCAS_C_STORAGE_CLASS int getMajorVersionNumber(void);
LCAS_C_STORAGE_CLASS int getMinorVersionNumber(void);
LCAS_C_STORAGE_CLASS int getPatchVersionNumber(void);

#endif /* LCAS_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/interface/lcas.h,v $
    $Date: 2010-05-03 10:43:18 $
    $Revision: 1.8 $
    $Author: okoeroo $
******************************************************************************/
