package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ItemEvent;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Dot;
import latexDraw.figures.Figure;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.components.ButtonIcon;
import latexDraw.ui.components.ColorButton;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines the frame which allows to change the parameters of dots.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersDotFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to change the x-coordinates of the centre of the figure */
	protected JSpinner centerXField;
	
	/** Allows to change the y-coordinates of the centre of the figure */
	protected JSpinner centerYField;
	
	/** Allows to change the y-coordinates of the width of the figure */
	protected JSpinner widthField;
	
	/** The list of the styles of dots */
	protected LaTeXDrawComboBox dotChoice;
	
	
	
	/**
	 * The constructor by default
	 */
	public ParametersDotFrame(Frame parent, DrawPanel p, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  			  		
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		JPanel pParams = new JPanel(new GridBagLayout());
	  		
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 1;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	pParams.add(new JLabel(LABEL_CENTER_X), constraint);
	  		
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerXField = new JSpinner(model);
			centerXField.setEditor(new JSpinner.NumberEditor(centerXField, "0.0"));//$NON-NLS-1$
			centerXField.addChangeListener(this);
	     	centerXField.setName(LABEL_CENTER_X);
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	pParams.add(centerXField, constraint);

	    	constraint.gridx = 0;	constraint.gridy = 1;
	     	constraint.gridwidth = 1;
	     	pParams.add(new JLabel(LABEL_CENTER_Y), constraint);
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerYField = new JSpinner(model);
	     	centerYField.setEditor(new JSpinner.NumberEditor(centerYField, "0.0"));//$NON-NLS-1$
			centerYField.addChangeListener(this);
	     	centerYField.setName(LABEL_CENTER_Y);
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	pParams.add(centerYField, constraint);
	     	
	    	constraint.gridx = 0;	constraint.gridy++;
	     	constraint.gridwidth = 1;
	     	pParams.add(new JLabel(LABEL_WIDTH), constraint);

	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	widthField = new JSpinner(model);
	     	widthField.setEditor(new JSpinner.NumberEditor(widthField, "0.0"));//$NON-NLS-1$
	     	widthField.addChangeListener(this);
	     	widthField.setName(LABEL_WIDTH);
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	pParams.add(widthField, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	constraint.gridwidth = 1;
	     	pParams.add(new JLabel(LABEL_ROTATION), constraint);
	     	
	     	model = new SpinnerNumberModel(Figure.DEFAULT_ROTATION_ANGLE, -360, 360,0.1);
	     	rotationField = new JSpinner(model);
	     	rotationField.setEditor(new JSpinner.NumberEditor(rotationField, "0.0"));//$NON-NLS-1$
	     	rotationField.addChangeListener(this);
	     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	pParams.add(rotationField, constraint);
	     	
	     	bordersButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_BUTTON, new ButtonIcon(Color.BLACK));
	     	bordersButton.setText(LaTeXDrawResources.LABEL_COLOR_BUTTON);
	     	bordersButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_BUTTON);
	     	bordersButton.addActionListener(this);
	     	constraint.gridx = 0;	constraint.gridy++;
	     	pParams.add(bordersButton, constraint);
	     		     	
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	pParams.add(new JLabel(LaTeXDrawLang.getDialogFrameString("ParametersDotFrame.0")), constraint); //$NON-NLS-1$
	     	
	     	dotChoice = Dot.createDotStyleChoice();
	     	dotChoice.setName(LaTeXDrawResources.CHOICE_DOT_NAME);
	     	dotChoice.setSelectedItem(PSTricksConstants.LINE_NONE_STYLE);
	     	dotChoice.addItemListener(this);
	     	constraint.gridx = 2;	constraint.gridwidth = 1;
	     	pParams.add(dotChoice, constraint);
	     	
	     	pParams.setBorder(new CompoundBorder(new TitledBorder(null, TITLE_GENERAL, 
					  			TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	
	     	pGeneral.add(pParams);
	     	pGeneral.add(createFillPanel());
	  		hatchChoice.setVisible(false);
	  		
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(pGeneral);
			getContentPane().add(pButton);
			filledCheckBox.setVisible(false);
			pack();
			setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		Dot d = (Dot)figure;
		
		float x = Float.valueOf(centerXField.getValue().toString()).floatValue();
		float y = Float.valueOf(centerYField.getValue().toString()).floatValue();

		JLabel label = (JLabel) dotChoice.getSelectedItem();
		d.setCurrentStyle(label.getText());
		d.setWidth(Float.valueOf(widthField.getValue().toString()).floatValue());
		d.getCenter().x = x;
		d.getCenter().y = y;
		d.updateDelimitor();
		d.updateShape();
		
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}
	
	
	
	
	@Override
	public void setFigureFrameField()
	{
		Dot d = (Dot)figure;
		LaTeXDrawPoint2D center = d.getCenter();
		dotChoice.setSelectedItem(d.getCurrentStyle());
		centerXField.setValue(center.x);
		centerYField.setValue(center.y);
		widthField.setValue((double)d.getWidth());
		interiorButton.setEnabled(d.canBeFilled());
		
		if(d.canBeFilled())
			interiorButton.setColor(d.getInteriorColor());
	}

	
	
	@Override
	public void setDefaultsValues()
	{
		try
		{
			super.setDefaultsValues();
			dotChoice.setSelectedItem(Dot.DEFAULT_DOT_STYLE);
			widthField.setValue(Dot.DEFAULT_WIDTH);
			interiorButton.setEnabled(glimpsePanel.glimpseFigure.canBeFilled());
			interiorButton.setColor(Figure.DEFAULT_INTERIOR_COL);
			
		}catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}

	
	
	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Dot))
				throw new ClassCastException();

			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
	
	
	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Dot f = (Dot) glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				
				if(name.equals(LABEL_CENTER_X))
				{
					setModified(true);
					f.getCenter().x = Float.valueOf(centerXField.getValue().toString()).floatValue();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_CENTER_Y))
				{
					setModified(true);
					f.getCenter().y = Float.valueOf(centerYField.getValue().toString()).floatValue();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_WIDTH))
				{
					setModified(true);
					f.setWidth(Float.valueOf(widthField.getValue().toString()).floatValue());
					glimpsePanel.repaint();
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.stateChanged(e);
	}
	
	
	
	
	@Override
	public void itemStateChanged(ItemEvent e) 
	{	
		try
		{
			Dot f = (Dot) glimpsePanel.getGlimpseFigure();
			Object o = e.getSource();
			
			if(o instanceof LaTeXDrawComboBox)
			{
				if(((LaTeXDrawComboBox)o).getName().equals(LaTeXDrawResources.CHOICE_DOT_NAME))
				{
					JLabel l = (JLabel) ((LaTeXDrawComboBox)o).getSelectedItem();
					f.setCurrentStyle(l.getText());
					interiorButton.setEnabled(f.canBeFilled());
					
					if(f.canBeFilled())
					{
						interiorButton.setColor(f.getInteriorColor());
						interiorButton.repaint();
					}
					
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		super.itemStateChanged(e);
	}
}
