package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ActionEvent;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Circle;
import latexDraw.figures.Figure;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.figures.Square;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawPanel;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines the frame which allows to change the parameters of circles and squares.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersCircleSquareFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to change the x-coordinates of the centre of the figure */
	protected JSpinner centerXField;
	
	/** Allows to change the y-coordinates of the centre of the figure */
	protected JSpinner centerYField;
	
	/** Allows to change the radius of the circle */
	protected JSpinner radiusField;
	
	/** Allows to set if the square will be round (only for square) */
	protected JCheckBox isRound;
	
	/** The panel containing to set the roundness of the figure. */
	protected JPanel roundPanel;
	
	/** Allows to change the angle of the round corner. */
	protected JSpinner frameArcField;
	
	
	
	/**
	 * The constructor by default
	 */
	public ParametersCircleSquareFrame(Frame parent, DrawPanel p, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel coordP = new JPanel(new GridBagLayout());
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 1;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	coordP.add(new JLabel(LABEL_CENTER_X) , constraint);
	  		
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerXField = new JSpinner(model);
			centerXField.setEditor(new JSpinner.NumberEditor(centerXField, "0.0"));//$NON-NLS-1$
			centerXField.addChangeListener(this);
	     	centerXField.setName(LABEL_CENTER_X);
	     	constraint.gridx++;	
	     	coordP.add(centerXField, constraint);

	    	constraint.gridx++;	
	     	coordP.add(new JLabel(LABEL_CENTER_Y), constraint);
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerYField = new JSpinner(model);
	     	centerYField.setEditor(new JSpinner.NumberEditor(centerYField, "0.0"));//$NON-NLS-1$
			centerYField.addChangeListener(this);
	     	centerYField.setName(LABEL_CENTER_Y);
	     	constraint.gridx++;	
	     	coordP.add(centerYField, constraint);
	     	
	    	constraint.gridx = 0;	constraint.gridy++;
	    	coordP.add(new JLabel(LABEL_RADIUS), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	radiusField = new JSpinner(model);
	     	radiusField.setEditor(new JSpinner.NumberEditor(radiusField, "0.0"));//$NON-NLS-1$
	     	radiusField.addChangeListener(this);
	     	radiusField.setName(LABEL_RADIUS);
	     	constraint.gridx++;	
	     	coordP.add(radiusField, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	coordP.add(new JLabel(LABEL_ROTATION), constraint);
	     	constraint.gridx++;
	     	coordP.add(rotationField, constraint);
	     	coordP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_POSITION_DIMENSIONS, 
							  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	
	     	pGeneral.add(coordP);
	     	pGeneral.add(createBordersPanel(true));
	     	pGeneral.add(createDoubleLinePanel());
	     	pGeneral.add(createFillPanel());
	     	
	     	JPanel pOther = new JPanel();
	     	pOther.setLayout(new BoxLayout(pOther, BoxLayout.Y_AXIS));
	     	pOther.add(createHatchingPanel());
	     	pOther.add(createShadowPanel());
	     	pOther.add(createGradientPanel());
	     	
	     	roundPanel = new JPanel();
	     	roundPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
	     	
	     	isRound = new JCheckBox(LABEL_ISROUND);
	     	isRound.addActionListener(this);
	     	isRound.setActionCommand(LABEL_ISROUND);
	     	roundPanel.add(isRound);
	     	
	    	roundPanel.add(new JLabel(LABEL_ROUND_CORNER));
	    	
	     	model = new SpinnerNumberModel(LaTeXDrawRectangle.DEFAULT_FRAME_ARC, 0.01, 1,0.05);
			frameArcField = new JSpinner(model);
			frameArcField.setName(LABEL_ROUND_CORNER);
			frameArcField.setEditor(new JSpinner.NumberEditor(frameArcField, "0.00"));//$NON-NLS-1$
			frameArcField.addChangeListener(this);
			frameArcField.setEnabled(false);
	     	roundPanel.add(frameArcField);

	     	roundPanel.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.6"),  //$NON-NLS-1$
					  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	pOther.add(roundPanel);
	     	
	     	tabbedPane.addTab(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	tabbedPane.addTab(TITLE_TABBED_PANE_OTHERS, pOther);
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}
		catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		try
		{
			Object o = e.getSource();
			Figure f = glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JCheckBox)
			{
				String label = ((JCheckBox)o).getActionCommand();

				if(label.equals(LABEL_ISROUND))
				{
					((Square)f).setIsRound(isRound.isSelected());
					frameArcField.setEnabled(isRound.isSelected());
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
			}			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.actionPerformed(e);
	}
	
	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Figure f = glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				
				if(name.equals(LABEL_ROUND_CORNER))
				{
					if(f instanceof Square)
					{
						((Square)f).setFrameArc(Double.valueOf(frameArcField.getValue().toString()).doubleValue());
						setModified(true);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				LaTeXDrawPoint2D p1, p2, p3, p4;
				
				if(figure instanceof Circle)
				{
					Circle c = (Circle)f;
					p1 = c.getBordersPoint(0); 	p2 = c.getBordersPoint(1);
					p3 = c.getBordersPoint(2); 	p4 = c.getBordersPoint(3);
				}
				else
				{
					Square s = (Square)f;
					p1 = s.getPoint(0); 	p2 = s.getPoint(1);
					p3 = s.getPoint(2); 	p4 = s.getPoint(3);
				}
				
				if(name.equals(LABEL_CENTER_X))
				{
					double radius = Double.valueOf(radiusField.getValue().toString());
					double x = Double.valueOf(centerXField.getValue().toString());
					
					p1.x = x - radius;
					p2.x = x + radius;
					p3.x = x - radius;
					p4.x = x + radius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_CENTER_Y))
				{
					double radius = Double.valueOf(radiusField.getValue().toString());
					double y = Double.valueOf(centerYField.getValue().toString());
					
					p1.y = y - radius;
					p2.y = y - radius;
					p3.y = y + radius;
					p4.y = y + radius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_RADIUS))
				{
					double radius = Double.valueOf(radiusField.getValue().toString());
					double y = Double.valueOf(centerYField.getValue().toString());
					double x = Double.valueOf(centerXField.getValue().toString());
					
					p1.x = x - radius;
					p2.x = x + radius;
					p3.x = x - radius;
					p4.x = x + radius;
					p1.y = y - radius;
					p2.y = y - radius;
					p3.y = y + radius;
					p4.y = y + radius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();					
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.stateChanged(e);
	}
	
	
	
	
	
	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		LaTeXDrawPoint2D p1, p2, p3, p4;
		
		float radius = Float.valueOf(radiusField.getValue().toString()).floatValue();
		float x = Float.valueOf(centerXField.getValue().toString()).floatValue();
		float y = Float.valueOf(centerYField.getValue().toString()).floatValue();
		
		if(figure instanceof Circle)
		{
			Circle c = (Circle)figure;
			p1 = c.getBordersPoint(0); 	p2 = c.getBordersPoint(1);
			p3 = c.getBordersPoint(2); 	p4 = c.getBordersPoint(3);
		}
		else
		{
			Square s = (Square)figure;
			p1 = s.getPoint(0); 	p2 = s.getPoint(1);
			p3 = s.getPoint(2); 	p4 = s.getPoint(3);
			((Square)figure).setIsRound(isRound.isSelected());
			((Square)figure).setWidth(radius*2);
			((Square)figure).setFrameArc(
					Double.valueOf(frameArcField.getValue().toString()).doubleValue());
		}
		
		p1.x = x - radius;
		p2.x = x + radius;
		p3.x = x - radius;
		p4.x = x + radius;

		p1.y = y - radius;
		p2.y = y - radius;
		p3.y = y + radius;
		p4.y = y + radius;
		
		figure.updateShape();
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}
	
	
	

	@Override
	public void setFigureFrameField()
	{
		LaTeXDrawPoint2D NW, SE;
		NW = figure.getTheNWPoint();
		SE = figure.getTheSEPoint();
		
		centerXField.removeChangeListener(this);
		centerXField.setValue((NW.x+SE.x)/2.);
		centerXField.addChangeListener(this);
		
		centerYField.removeChangeListener(this);
		centerYField.setValue((NW.y+SE.y)/2.);
		centerYField.addChangeListener(this);
		
		radiusField.removeChangeListener(this);
		radiusField.setValue((SE.x-NW.x)/2.);
		radiusField.addChangeListener(this);
		
		if(figure instanceof Square)
		{
			 isRound.setSelected(((Square)figure).isRound());
			 frameArcField.setEnabled(isRound.isSelected());
			 frameArcField.setValue(((Square)figure).getFrameArc());
			 roundPanel.setVisible(true);
		}
		else roundPanel.setVisible(false);
	}

	
	
	
	@Override
	public void setDefaultsValues()
	{
		super.setDefaultsValues();
		
		if(figure instanceof Square)
		{
			 isRound.setSelected(LaTeXDrawRectangle.DEFAULT_ISROUND);
		 ((Square)glimpsePanel.getGlimpseFigure()).setIsRound(LaTeXDrawRectangle.DEFAULT_ISROUND);
		}
		else 
			isRound.setVisible(false);
	}
	

	
	
	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Circle || selected instanceof Square))
				throw new IllegalArgumentException();

			if(isFramedBox)
			{
				centerXField.setEnabled(false);
				centerYField.setEnabled(false);
				radiusField.setEnabled(false);
			}
			
			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);

		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
}
