package latexDraw.parsers;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Vector;

import javax.swing.*;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import latexDraw.lang.LaTeXDrawLang;


/** 
 * This class define a frame which can contains the messages that appeared during 
 * the parsing.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 * <br>
 * 
 * 02/02/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParserMessagesFrame extends JDialog implements ActionListener
{
	private static final long serialVersionUID = 1L;

	/** The label of the button "ok" */
	public static final String LABEL_OK  = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.4"); //$NON-NLS-1$

	/** If true the messages must by displayed in a frame, else in
	 * the console. */
	private boolean displayInFrame;
	
	/** The table to display */
	private JTable table;

	/** The messages of the parser */
	private Vector<ParserCmdMessage> messages;
	
	
	
	
	
	/**
	 * The constructor by default.
	 * @param displayInF True : the messages will be displayed in a frame, else in the console.
	 * @param msg The messages.
	 */
	public ParserMessagesFrame(JFrame parent, boolean displayInF, Vector<ParserCmdMessage> msg)
	{
		super(parent, true);
		
		try
		{
			displayInFrame 	= displayInF ;
			messages 		= msg;

	 		int i, size = msg.size();
	 		final String[] titles = {LaTeXDrawLang.getString1_6("ParserMessagesFrame.0"), LaTeXDrawLang.getString1_6("ParserMessagesFrame.1"), LaTeXDrawLang.getString1_6("ParserMessagesFrame.2")}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	 		final Object[][] data = new String[size][ParserCmdMessage.NB_COL];
	 		
	 		for(i=0; i<size; i++)
	 		{
	 			ParserCmdMessage m = messages.elementAt(i);
	 			data[i][0] = m.getCommand();
	 			data[i][1] = String.valueOf(m.getNLine());
	 			data[i][2] = m.getException().toString();
	 		}
	 		
	        TableModel dataModel = new AbstractTableModel() 
	        { 
				private static final long serialVersionUID = 1L;
				public int getColumnCount() { return titles.length; } 
	            public int getRowCount() { return data.length;} 
	            public Object getValueAt(int row, int col) {return data[row][col];} 
	            
	            @Override
				public String getColumnName(int column) {return titles[column];} 
	            
	            @Override
				public Class<? extends Object> getColumnClass(int c) {return getValueAt(0, c).getClass();} 
	            
			    @Override
				public boolean isCellEditable(int row, int col) {return false;} 
			    
	            @Override
				public void setValueAt(Object aValue, int row, int column) { data[row][column] = aValue; } 
	         }; 

			table = new JTable(dataModel);
	 		table.getTableHeader().setReorderingAllowed(false);
	 		TableColumnModel colmodel = table.getColumnModel();

	 		colmodel.getColumn(0).setPreferredWidth(120);
	 		colmodel.getColumn(1).setPreferredWidth(15);
	 		colmodel.getColumn(2).setPreferredWidth(205);
	 		JScrollPane scrollPane = new JScrollPane();
	 		JViewport vp = scrollPane.getViewport(); 
		 	vp.add(table); 

		 	JPanel topPanel = new JPanel();
		 	topPanel.add(new JLabel(LaTeXDrawLang.getString1_6("ParserMessagesFrame.3"))); //$NON-NLS-1$
		 	JPanel bPanel = new JPanel(new GridLayout(1, 1));
			JButton okB = new JButton(LABEL_OK);
			okB.setActionCommand(LABEL_OK);
			okB.addActionListener(this);
			bPanel.add(okB);
			
			setTitle(LaTeXDrawLang.getString1_6("ParserMessagesFrame.4")); //$NON-NLS-1$
			getContentPane().setLayout(new BorderLayout());
			getContentPane().add(topPanel,BorderLayout.NORTH);
			getContentPane().add(scrollPane, BorderLayout.CENTER);
			getContentPane().add(bPanel, BorderLayout.SOUTH);
			
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			
			if(displayInFrame)
				JOptionPane.showMessageDialog( 
			 		    this, 
			 		    LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.4") + //$NON-NLS-1$
			 		    ex.toString(), 
			 		   LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.5"),  //$NON-NLS-1$
			 		    JOptionPane.WARNING_MESSAGE 
			 		);
		}
	}
	
	
	
	
	
	@Override
	public void setVisible(boolean visible)
	{		
		if(visible)
		{
			Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();		
			setLocation(dim.width/2-300, dim.height/2-200);
			setSize(600, 400);
		}
		super.setVisible(visible);
	}
	
	
	
	
	/**
	 * Allows to display a frame which contains the messages which appeared during to 
	 * parsing.
	 * @param msg All the messages
	 * @param displayInFrame If true, the messages are printed in a frame ; else they are
	 * printed in the console.
	 */
	public static void showParserMessagesFrame(JFrame parent, boolean displayInFrame, Vector<ParserCmdMessage> msg)
	{
		ParserMessagesFrame f = new ParserMessagesFrame(parent, displayInFrame, msg);
		
		if(displayInFrame)
			f.setVisible(true);
		else
			if(msg!=null)
				for(ParserCmdMessage pcm : msg)
					System.out.println(pcm);
	}
		
	
	
	
	/**
	 * The actions to do bound to the buttons.
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof JButton)
		{
			String msg = ((JButton)o).getActionCommand();
			
			if(msg.equals(LABEL_OK))
			{
				setVisible(false);
				return ;
			}
		}
	}
}
