/************************************************************

  This example shows how to read and write integer datatypes
  to an attribute.  The program first writes integers to an
  attribute with a dataspace of DIM0xDIM1, then closes the
  file.  Next, it reopens the file, reads back the data, and
  outputs it to the screen.

  This file is intended for use with HDF5 Library verion 1.6

 ************************************************************/

package examples.datatypes;

import java.text.DecimalFormat;

import ncsa.hdf.hdf5lib.H5;
import ncsa.hdf.hdf5lib.HDF5Constants;
import ncsa.hdf.object.Attribute;
import ncsa.hdf.object.Datatype;
import ncsa.hdf.object.FileFormat;
import ncsa.hdf.object.h5.H5Datatype;
import ncsa.hdf.object.h5.H5File;
import ncsa.hdf.object.h5.H5ScalarDS;

public class H5ObjectEx_T_IntegerAttribute {
	private static String FILENAME = "H5ObjectEx_T_IntegerAttribute.h5";
	private static String DATASETNAME = "DS1";
	private static String ATTRIBUTENAME = "A1";
	private static final int DIM0 = 4;
	private static final int DIM1 = 7;

	private static void CreateDataset() {
        H5File file = null;
        H5ScalarDS dset = null;
		int dataset_id = -1;
		int attribute_id = -1;
		long[] dims = { DIM0, DIM1 };
		int[][] dset_data = new int[DIM0][DIM1];
        final H5Datatype typeInt = new H5Datatype(Datatype.CLASS_INTEGER, 4, Datatype.ORDER_LE, -1);
        final H5Datatype typeBigInt = new H5Datatype(Datatype.CLASS_INTEGER, 8, Datatype.ORDER_BE, -1);

		// Initialize data.
		for (int indx = 0; indx < DIM0; indx++)
			for (int jndx = 0; jndx < DIM1; jndx++) {
				dset_data[indx][jndx] = indx * jndx - jndx;
			}

		// Create a new file using default properties.
		try {
            file = new H5File(FILENAME, FileFormat.CREATE);
            file.open();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Create dataset with a scalar dataspace.
		try {
            dset = (H5ScalarDS) file.createScalarDS(DATASETNAME, null, typeInt, dims, null, null, 0, null);
            dataset_id = dset.open();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Create the attribute and write the array data to it.
		try {
            Attribute dataArray = new Attribute(ATTRIBUTENAME, typeBigInt, dims);
            dset.writeMetadata(dataArray);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Write the dataset.
		try {
            if (dataset_id >= 0)
                attribute_id = H5.H5Aopen_by_name(dataset_id, ".", ATTRIBUTENAME, 
                        HDF5Constants.H5P_DEFAULT, HDF5Constants.H5P_DEFAULT);
			if (attribute_id >= 0)
				H5.H5Awrite(attribute_id, HDF5Constants.H5T_NATIVE_INT, dset_data);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// End access to the dataset and release resources used by it.
		try {
			if (attribute_id >= 0)
				H5.H5Aclose(attribute_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		try {
			if (dataset_id >= 0)
                dset.close(dataset_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Close the file.
		try {
            file.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

	}

	private static void ReadDataset() {
        H5File file = null;
        H5ScalarDS dset = null;
		int dataspace_id = -1;
		int dataset_id = -1;
		int attribute_id = -1;
		long[] dims = { DIM0, DIM1 };
		int[][] dset_data;

		// Open an existing file.
		try {
            file = new H5File(FILENAME, FileFormat.READ);
            file.open();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Open an existing dataset.
		try {
            dset = (H5ScalarDS) file.get(DATASETNAME);
            dataset_id = dset.open();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		try {
			if (dataset_id >= 0)
				attribute_id = H5.H5Aopen_by_name(dataset_id, ".", ATTRIBUTENAME, 
				        HDF5Constants.H5P_DEFAULT, HDF5Constants.H5P_DEFAULT);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Get dataspace and allocate memory for read buffer.
		try {
			if (attribute_id >= 0)
				dataspace_id = H5.H5Aget_space(attribute_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		try {
			if (dataspace_id >= 0)
				H5.H5Sget_simple_extent_dims(dataspace_id, dims, null);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Allocate array of pointers to two-dimensional arrays (the
		// elements of the dataset.
		dset_data = new int[(int) dims[0]][(int) (dims[1])];

		// Read data.
		try {
			if (attribute_id >= 0)
				H5.H5Aread(attribute_id, HDF5Constants.H5T_NATIVE_INT, dset_data);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Output the data to the screen.
		DecimalFormat df = new DecimalFormat("#,##0");
		System.out.println(ATTRIBUTENAME + ":");
		for (int indx = 0; indx < dims[0]; indx++) {
			System.out.print(" [");
			for (int jndx = 0; jndx < dims[1]; jndx++) {
				System.out.print(" " + df.format(dset_data[indx][jndx]));
			}
			System.out.println("]");
		}
		System.out.println();

		// End access to the dataset and release resources used by it.
		try {
			if (attribute_id >= 0)
				H5.H5Aclose(attribute_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		try {
			if (dataset_id >= 0)
                dset.close(dataset_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Terminate access to the data space.
		try {
			if (dataspace_id >= 0)
				H5.H5Sclose(dataspace_id);
		}
		catch (Exception e) {
			e.printStackTrace();
		}

		// Close the file.
		try {
            file.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}

	}

	public static void main(String[] args) {
		H5ObjectEx_T_IntegerAttribute.CreateDataset();
		// Now we begin the read section of this example. Here we assume
		// the dataset and array have the same name and rank, but can have
		// any size. Therefore we must allocate a new array to read in
		// data using malloc().
		H5ObjectEx_T_IntegerAttribute.ReadDataset();
	}

}
