/*  HomeBank -- Free, easy, personal accounting for everyone.
 *  Copyright (C) 1995-2020 Maxime DOYEN
 *
 *  This file is part of HomeBank.
 *
 *  HomeBank is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  HomeBank is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "homebank.h"

#include "list-account.h"
#include "hub-account.h"

/****************************************************************************/
/* Debug macros																*/
/****************************************************************************/
#define MYDEBUG 0

#if MYDEBUG
#define DB(x) (x);
#else
#define DB(x);
#endif

/* our global datas */
extern struct HomeBank *GLOBALS;
extern struct Preferences *PREFS;


/*
** draw some icons according to the stored data structure
*/
static void
status_cell_data_function (GtkTreeViewColumn *col,
                           GtkCellRenderer   *renderer,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           user_data)
{
Account *acc;
gchar *iconname = NULL;
gint dt;

	gtk_tree_model_get(model, iter, 
		LST_DSPACC_DATATYPE, &dt,
		LST_DSPACC_DATAS, &acc,
		-1);

	if( dt == DSPACC_TYPE_NORMAL )
	{
	
		switch(GPOINTER_TO_INT(user_data))
		{
			case 1:
				iconname = (acc->flags & AF_ADDED) ? ICONNAME_HB_OPE_NEW : NULL;
				// override if closed account
				if( acc->flags & AF_CLOSED )
					iconname = ICONNAME_CHANGES_PREVENT;
				break;
			case 2:
				iconname = (acc->flags & AF_CHANGED) ? ICONNAME_HB_OPE_EDIT : NULL;
				break;
		}
	}

	g_object_set(renderer, "icon-name", iconname, NULL);
}


/*
** draw some text from the stored data structure
*/
static void
text_cell_data_function (GtkTreeViewColumn *col,
				GtkCellRenderer *renderer,
				GtkTreeModel *model,
				GtkTreeIter *iter,
				gpointer user_data)
{
gpointer p;
gint dt;
gint weight = PANGO_WEIGHT_BOLD;
gchar *text = NULL;
//gchar *groupname;

	gtk_tree_model_get(model, iter,
		LST_DSPACC_DATATYPE, &dt,
		LST_DSPACC_DATAS, &p,
		//LST_DSPACC_NAME, &groupname,
		-1);

	if( p == NULL )
		goto end;

	switch( dt )
	{
		case DSPACC_TYPE_NORMAL:
		{
		Account *acc = p;
			weight = PANGO_WEIGHT_NORMAL;
			text = acc->name;
		}
		break;

		case DSPACC_TYPE_HEADER:
		{
		PnlAccGrp *g = p;
			text = g->name;
		}
		break;

		case DSPACC_TYPE_SUBTOTAL:
			text = _("Total");
		break;

		case DSPACC_TYPE_TOTAL:
			text = _("Grand total");
		break;
	}

end:
	g_object_set(renderer, "weight", weight, "text", text, NULL);

}


static void
float_cell_data_function (GtkTreeViewColumn *col, GtkCellRenderer *renderer, GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data)
{
gchar buf[G_ASCII_DTOSTR_BUF_SIZE];
gpointer p;
gint dt;
gint colid = GPOINTER_TO_INT(user_data); //LST_DSPACC_(BANK/TODAY/FUTURE)
gdouble amount = -1;
gint weight = PANGO_WEIGHT_NORMAL;
gchar *color = NULL;
gchar *text = NULL;
	
	gtk_tree_model_get(model, iter,
		LST_DSPACC_DATATYPE, &dt,
		LST_DSPACC_DATAS, &p,
		-1);

	if( p == NULL )
		goto end;		

	if( dt == DSPACC_TYPE_NORMAL )
	{
	Account *acc = p;

		switch(colid)
		{
			case COL_DSPACC_BANK: amount = acc->bal_bank; break;
			case COL_DSPACC_TODAY: amount = acc->bal_today; break;
			case COL_DSPACC_FUTURE: amount = acc->bal_future; break;
		}

		hb_strfmon(buf, G_ASCII_DTOSTR_BUF_SIZE-1, amount, acc->kcur, GLOBALS->minor);
		color = get_minimum_color_amount(amount, acc->minimum);
		text = buf;
	}
	else
	{
	gboolean expanded = TRUE;

		if( dt == DSPACC_TYPE_HEADER )
		{
		GtkTreePath* tp = gtk_tree_model_get_path(model, iter);
			expanded = gtk_tree_view_row_expanded(GTK_TREE_VIEW(gtk_tree_view_column_get_tree_view(col)), tp);
			gtk_tree_path_free(tp);
		}
			
		if( dt != DSPACC_TYPE_HEADER || (expanded == FALSE) )
		{
		PnlAccGrp *g = p;

			switch(colid)
			{
				case COL_DSPACC_BANK: amount = g->bal_bank; break;
				case COL_DSPACC_TODAY: amount = g->bal_today; break;
				case COL_DSPACC_FUTURE: amount = g->bal_future; break;
			}

			//nota: header is always in base currency
			hb_strfmon(buf, G_ASCII_DTOSTR_BUF_SIZE-1, amount, GLOBALS->kcur, GLOBALS->minor);
			if(expanded)
				weight = PANGO_WEIGHT_BOLD;
			color = get_normal_color_amount(amount);
			text = buf;
		}
	}

end:
	g_object_set(renderer, 
		"weight", weight,
		"foreground",  color,
		"text", text,
		NULL);	
}


static GtkTreeViewColumn *amount_list_account_column(gchar *name, gint id)
{
GtkTreeViewColumn  *column;
GtkCellRenderer    *renderer;

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, name);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set(renderer, "xalign", 1.0, NULL);
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func(column, renderer, float_cell_data_function, GINT_TO_POINTER(id), NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_alignment (column, 0.5);
	gtk_tree_view_column_set_spacing( column, 16 );
	//gtk_tree_view_column_set_sort_column_id (column, LST_DSPACC_BANK);

	return column;
}

static gint
list_account_compare_func (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer userdata)
{
gint retval = 0;
gint dt1, dt2;
gpointer p1, p2;
gint pos1, pos2;

    gtk_tree_model_get(model, a, 
		LST_DSPACC_POS, &pos1,
    	LST_DSPACC_DATATYPE, &dt1, 
    	LST_DSPACC_DATAS, &p1,
    	-1);
    gtk_tree_model_get(model, b, 
		LST_DSPACC_POS, &pos2,
    	LST_DSPACC_DATATYPE, &dt2, 
    	LST_DSPACC_DATAS, &p2,
    	-1);

	if( p1 == NULL || p2 == NULL )
		return 0;
	
	if( dt1 == DSPACC_TYPE_NORMAL && dt2 == DSPACC_TYPE_NORMAL )
	{
	Account *entry1 = p1;
	Account *entry2 = p2;

		retval = entry1->pos - entry2->pos;
	}
	else
	if( dt1 == DSPACC_TYPE_HEADER && dt2 == DSPACC_TYPE_HEADER )
	{
	PnlAccGrp *elt1 = p1;
	PnlAccGrp *elt2 = p2;

		retval = pos1 - pos2;
		if( !retval )
			retval = hb_string_utf8_compare(elt1->name, elt2->name);
	}

    return retval;
}


/*
 *
 */ 
static
gboolean list_account_selectionfunc(
GtkTreeSelection *selection, GtkTreeModel *model, GtkTreePath *path, gboolean path_currently_selected, gpointer data)
{
GtkTreeIter iter;

	if( gtk_tree_path_get_depth( path ) < 2 )
		return FALSE;

	if(gtk_tree_model_get_iter(model, &iter, path))
	{
	gint dt;

		gtk_tree_model_get(model, &iter,
			LST_DSPACC_DATATYPE, &dt,
			-1);	

		if( dt != DSPACC_TYPE_NORMAL )
			return FALSE;
	}

	return TRUE;
}


static GtkTreeViewColumn *list_account_get_column(GtkTreeView *treeview, gint uid)
{
GtkTreeViewColumn *retval = NULL;
guint i;

	for(i=0;i<gtk_tree_view_get_n_columns(treeview);i++)
	{
	GtkTreeViewColumn *column = gtk_tree_view_get_column(treeview, i);

		if(column)
		{
			gint coluid = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(column), "uid"));
			if( coluid == uid )
				retval = column;
		}
	}
	return retval;	
}


static void list_account_destroy(GtkTreeView *treeview, gpointer user_data)
{
GtkTreeViewColumn  *column;

	DB( g_print ("\n[list_account] destroy\n") );

	column = list_account_get_column(treeview, COL_DSPACC_ACCOUNTS);
	if( column )
	{
		PREFS->pnl_acc_col_acc_width = gtk_tree_view_column_get_width(column);
	}
}


GtkWidget *create_list_account(void)
{
GtkTreeStore *store;
GtkWidget *view;
GtkCellRenderer    *renderer;
GtkTreeViewColumn  *column;

	DB( g_print ("\n[list_account] create\n") );

	/* create list store */
	store = gtk_tree_store_new(
	  	NUM_LST_DSPACC,
		G_TYPE_INT,		/* datatype */
		G_TYPE_INT,		/* pos */
		G_TYPE_POINTER  /* datas*/
		);

	//treeview
	view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	g_object_unref(store);

	gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (view), PREFS->grid_lines);
	//gtk_tree_view_set_search_column (GTK_TREE_VIEW (treeview),
	//			       COLUMN_DESCRIPTION);

	//gtk_tree_view_set_enable_tree_lines(GTK_TREE_VIEW (view), TRUE);

	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(view)), GTK_SELECTION_SINGLE);

    /* Status */
	column = gtk_tree_view_column_new();
    //gtk_tree_view_column_set_title(column, _("Status"));

    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func(column, renderer, status_cell_data_function, GINT_TO_POINTER(1), NULL);

    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func(column, renderer, status_cell_data_function, GINT_TO_POINTER(2), NULL);

	gtk_tree_view_column_set_alignment (column, 0.5);
    gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);

	/* Account */
	renderer = gtk_cell_renderer_text_new ();
	g_object_set(renderer, 
		"ellipsize", PANGO_ELLIPSIZE_END,
		"ellipsize-set", TRUE,
		//taken from nemo, not exactly a resize to content, but good compromise
	    "width-chars", 40,
		NULL);

	column = gtk_tree_view_column_new();
	g_object_set_data(G_OBJECT(column), "uid", GUINT_TO_POINTER(COL_DSPACC_ACCOUNTS));
	gtk_tree_view_column_set_title(column, _("Accounts"));
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func(column, renderer, text_cell_data_function, NULL, NULL);
	gtk_tree_view_column_set_alignment (column, 0.5);
	gtk_tree_view_column_set_min_width(column, HB_MINWIDTH_LIST/2);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);

	gtk_tree_view_column_set_fixed_width(column, PREFS->pnl_acc_col_acc_width);
	gtk_tree_view_set_expander_column(GTK_TREE_VIEW (view), column);

    /* Bank */
	column = amount_list_account_column(_("Bank"), COL_DSPACC_BANK);
	gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);

    /* Today */
	column = amount_list_account_column(_("Today"), COL_DSPACC_TODAY);
	gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);

    /* Future */
	column = amount_list_account_column(_("Future"), COL_DSPACC_FUTURE);
	gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);

    /* column 7: empty */
	column = gtk_tree_view_column_new();
	gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);


	/* disable selection for level 1 of the tree */
	
	gtk_tree_selection_set_select_function(gtk_tree_view_get_selection(GTK_TREE_VIEW(view)), list_account_selectionfunc, NULL, NULL);
	
	//sort etc
	gtk_tree_sortable_set_default_sort_func(GTK_TREE_SORTABLE(store), list_account_compare_func, NULL, NULL);
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store), GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID, GTK_SORT_ASCENDING);

	g_signal_connect (view, "destroy", G_CALLBACK (list_account_destroy), NULL);

	return(view);
}

