/* Declarations of functions and data types used for SM3 sum library
   function.
   Copyright (C) 2017-2020 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <https://www.gnu.org/licenses/>.  */

/* This module can be used to compute SM3 message digest of files or
   memory blocks according to the specification GM/T 004-2012
   Cryptographic Hash Algorithm SM3, published by State Cryptography
   Administration, China.

   The official SM3 cryptographic hash algorithm specification is
   available at
   <http://www.sca.gov.cn/sca/xwdt/2010-12/17/content_1002389.shtml>. */

#ifndef SM3_H
# define SM3_H 1

# include <stdio.h>
# include <stdint.h>

# if HAVE_OPENSSL_SM3
#  include <openssl/sm3.h>
# endif

# ifdef __cplusplus
extern "C" {
# endif

enum { SM3_DIGEST_SIZE = 256 / 8 };

# if HAVE_OPENSSL_SM3
#  define GL_OPENSSL_NAME 3
#  include "gl_openssl.h"
# else
/* Structure to save state of computation between the single steps.  */
struct sm3_ctx
{
  uint32_t state[8];

  uint32_t total[2];
  size_t buflen;       /* ≥ 0, ≤ 128 */
  uint32_t buffer[32]; /* 128 bytes; the first buflen bytes are in use */
};

/* Initialize structure containing state of computation. */
extern void sm3_init_ctx (struct sm3_ctx *ctx);

/* Starting with the result of former calls of this function (or the
   initialization function update the context for the next LEN bytes
   starting at BUFFER.
   It is necessary that LEN is a multiple of 64!!! */
extern void sm3_process_block (const void *buffer, size_t len,
                               struct sm3_ctx *ctx);

/* Starting with the result of former calls of this function (or the
   initialization function update the context for the next LEN bytes
   starting at BUFFER.
   It is NOT required that LEN is a multiple of 64.  */
extern void sm3_process_bytes (const void *buffer, size_t len,
                               struct sm3_ctx *ctx);

/* Process the remaining bytes in the buffer and put result from CTX
   in first 32 bytes following RESBUF.  The result is always in little
   endian byte order, so that a byte-wise output yields to the wanted
   ASCII representation of the message digest.  */
extern void *sm3_finish_ctx (struct sm3_ctx *ctx, void *resbuf);

/* Put result from CTX in first 32 bytes following RESBUF.  The result is
   always in little endian byte order, so that a byte-wise output yields
   to the wanted ASCII representation of the message digest.  */
extern void *sm3_read_ctx (const struct sm3_ctx *ctx, void *resbuf);

/* Compute SM3 message digest for LEN bytes beginning at BUFFER.  The
   result is always in little endian byte order, so that a byte-wise
   output yields to the wanted ASCII representation of the message
   digest.  */
extern void *sm3_buffer (const char *buffer, size_t len, void *resblock);

# endif
/* Compute SM3 message digest for bytes read from STREAM.  The
   resulting message digest number will be written into the 32 bytes
   beginning at RESBLOCK.  */
extern int sm3_stream (FILE *stream, void *resblock);


# ifdef __cplusplus
}
# endif

#endif

/*
 * Hey Emacs!
 * Local Variables:
 * coding: utf-8
 * End:
 */
