/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

/*
 * TracingSystemHandlerFactory.java
 *
 * Created on August 12, 2004, 4:04 PM
 */

package com.sun.enterprise.webservice.monitoring;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Logger;
import java.util.logging.Level;

import java.io.File;
import java.io.BufferedInputStream;
import java.io.FileInputStream;

import java.net.URL;
import java.net.URLDecoder;
import java.net.MalformedURLException;

import org.w3c.dom.Document;
import javax.xml.parsers.DocumentBuilderFactory;
import org.xml.sax.InputSource;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPath;
import javax.xml.namespace.NamespaceContext;

import javax.enterprise.deploy.shared.ModuleType;

import com.sun.logging.LogDomains;
import com.sun.enterprise.deployment.WebServiceEndpoint;

import com.sun.enterprise.webservice.monitoring.LogAuthenticationListener;

/**
 * This class acts as a factory to create TracingSystemHandler 
 * instances. It also provides an API to register listeners 
 * of SOAP messages.
 * <p><b>NOT THREAD SAFE: mutable instance variable: globalMessageListener</b>
 *
 * @author Jerome Dochez
 */
public final class WebServiceEngineImpl implements WebServiceEngine {
    
    protected final Map<String, Endpoint> endpoints = new HashMap<String, Endpoint>();
    protected final List<EndpointLifecycleListener> lifecycleListeners = 
            new ArrayList<EndpointLifecycleListener>();
    protected final List<AuthenticationListener> authListeners = 
            new ArrayList<AuthenticationListener>();
    protected volatile GlobalMessageListener globalMessageListener = null; 
           
    static final ThreadLocal servletThreadLocal = new ThreadLocal();
    public static final Logger sLogger = LogDomains.getLogger(LogDomains.CORE_LOGGER);
    
    
    /** Creates a new instance of TracingSystemHandlerFactory */
    private WebServiceEngineImpl() {
        // this is normally a bad idiom (see Java Concurrency 3.2), but 
        // this method is private, and can only be constructed in the static
        // constructor below *and* the listern maintains no reference to 'this.
        addAuthListener( new LogAuthenticationListener() );
    }
    
    private static final WebServiceEngineImpl INSTANCE    = new WebServiceEngineImpl();
    
    public static  WebServiceEngineImpl getInstance() {
        return INSTANCE;
    }
    
    public EndpointImpl createHandler(WebServiceEndpoint endpointDesc)  {
        
        EndpointImpl newEndpoint = createEndpointInfo(endpointDesc);
        if (newEndpoint==null) {
            return null;
        }
        String key = newEndpoint.getEndpointSelector();
        endpoints.put(key, newEndpoint); 
        
        // notify listeners
        for (EndpointLifecycleListener listener : lifecycleListeners) {
            listener.endpointAdded(newEndpoint);
        }
        
        return newEndpoint;
    }
    
    public EndpointImpl createHandler(com.sun.xml.rpc.spi.runtime.SystemHandlerDelegate parent, 
        WebServiceEndpoint endpointDesc)  {
        
        EndpointImpl newEndpoint = createHandler(endpointDesc);
        ((JAXRPCEndpointImpl)newEndpoint).setParent(parent);
        return newEndpoint;
    }

    public Endpoint getEndpoint(String uri) {    
        return endpoints.get(uri);
    }
    
    public Iterator<Endpoint> getEndpoints() {
        return endpoints.values().iterator();
    }
    
    public void removeHandler(WebServiceEndpoint endpointDesc) {

        EndpointImpl endpoint = (EndpointImpl) endpointDesc.getExtraAttribute(EndpointImpl.NAME);
        if (endpoint==null) 
            return;
                
        // remove this endpoint from our list of endpoints
        endpoints.remove(endpoint.getEndpointSelector());

        // notify listeners
        for (EndpointLifecycleListener listener : lifecycleListeners) {
            listener.endpointRemoved(endpoint);
        }
        
        // forcing the cleaning so we don't have DOL objects staying alive because
        // some of our clients have not released the endpoint instance.
        endpoint.setDescriptor(null);
    }
            
    public void addLifecycleListener(EndpointLifecycleListener listener) {
        lifecycleListeners.add(listener);
    }
    
    public void removeLifecycleListener(EndpointLifecycleListener listener) {
        lifecycleListeners.remove(listener);
    }
    
    public void addAuthListener(AuthenticationListener listener) {
        authListeners.add(listener);
    }
    
    public void removeAuthListener(AuthenticationListener listener) {
        authListeners.remove(listener);
    }
    
    public Collection<AuthenticationListener> getAuthListeners() {
        return authListeners;
    }
    
    public GlobalMessageListener getGlobalMessageListener() {
        return globalMessageListener;
    }
    
    public void setGlobalMessageListener(GlobalMessageListener listener) {
        globalMessageListener = listener;
    }
    
    
    public boolean hasGlobalMessageListener() {
        return globalMessageListener!=null;
    }
                
    private EndpointImpl createEndpointInfo(WebServiceEndpoint endpoint) {
        
        try { 
            String endpointURL = endpoint.getEndpointAddressUri();
            EndpointType endpointType;            
            ModuleType moduleType = endpoint.getWebService().getWebServicesDescriptor().getModuleType();
            if (moduleType.equals(ModuleType.EJB)) {
                endpointType = EndpointType.EJB_ENDPOINT;
            } else {
                endpointType = EndpointType.SERVLET_ENDPOINT;
            }

            EndpointImpl newEndpoint;
            // At this point, we can depend on presence of mapping file to distinguish between JAXRPC and JAXWS
            // service
            if(endpoint.getWebService().getMappingFileUri()==null) {
                newEndpoint = new JAXWSEndpointImpl(endpointURL, endpointType); 
            } else {
                newEndpoint = new JAXRPCEndpointImpl(endpointURL, endpointType); 
            }
            newEndpoint.setDescriptor(endpoint);
            return newEndpoint;
        
        } catch(Exception e) {
            e.printStackTrace();
        }                         
        return null;
    }    
        
    /** 
     * Callback when a web service request entered the web service container
     * before any processing is done. 
     * @return a message ID to trace the request in the subsequent callbacks
     */
    public String preProcessRequest(Endpoint endpoint) {
        
        if (globalMessageListener==null)
            return null;
        
        return globalMessageListener.preProcessRequest(endpoint);
    }
    
    /** 
     * Callback when a web service request is received on 
     * the endpoint.
     * @param messageID returned by preProcessRequest call 
     * @param msgContext the jaxrpc message trace, transport dependent.
     */
    public void processRequest(String messageID, com.sun.xml.rpc.spi.runtime.SOAPMessageContext context, 
            TransportInfo info) {
        
        if (globalMessageListener==null) 
            return;
        
        globalMessageListener.processRequest(messageID, context, info);
    }
    
    /**
     * Callback when a web service response is received on the 
     * endpoint. 
     * @param messageID returned by the preProcessRequest call
     * @param trace jaxrpc message context
     */
    public void processResponse(String messageID, com.sun.xml.rpc.spi.runtime.SOAPMessageContext context) {
        
        if (globalMessageListener==null)
            return;
        
        globalMessageListener.processResponse(messageID, context);
    }
    
    /** 
     * Callback when a 2.0 web service request is received on 
     * the endpoint.
     * @param messageID returned by preProcessRequest call 
     * @param msgContext the jaxrpc message trace, transport dependent.
     */
    public void processRequest(String messageID, com.sun.enterprise.webservice.SOAPMessageContext context, 
            TransportInfo info) {
        
        if (globalMessageListener==null) 
            return;

        globalMessageListener.processRequest(messageID, context, info);
    }
    
    /**
     * Callback when a 2.0 web service response is received on the 
     * endpoint. 
     * @param messageID returned by the preProcessRequest call
     * @param trace jaxrpc message context
     */
    public void processResponse(String messageID, com.sun.enterprise.webservice.SOAPMessageContext context) {
        
        if (globalMessageListener==null)
            return;
        globalMessageListener.processResponse(messageID, context);
    }
    
    /**
     * Callback when a web service response has finished being processed
     * by the container and was sent back to the client
     * @param messageID returned by the preProcessRequest call
     */
    public void postProcessResponse(String messageID, TransportInfo info) {
        
        if (globalMessageListener==null)
            return;
        
        globalMessageListener.postProcessResponse(messageID, info);        
    }
    
    public ThreadLocal getThreadLocal() {
        return servletThreadLocal;
    }    
}
