/* $Id: model.h,v 1.80 2003/06/17 14:45:56 bsenders Exp $ */

#ifndef FFRENZY_MODEL_H
#define FFRENZY_MODEL_H

#include <SDL_thread.h>

#include "global.h"
#include "list.h"
#include "vector.h"

/** \file
 * Functions and data structures of the game model
 */

/** \defgroup Model Model abstraction layer
 *  The game model represents the world of the game. It has it's own rules
 *  of acceleration, speed and position.
 *  @{
 */

/** Model version number.
 *  A game can be played if the models of every participant have the same
 *  model version number.
 */
#define MODEL_VERSION "1.0.2"

/* Scores */
#define SC_TICK            1    /**< Score per tick alive. */
#define SC_PICKUP          10   /**< Score for picking up food. */
#define SC_HIT             200  /**< Score for hitting other player. */
#define SC_DELIV           50   /**< Score for delivery of food. */
#define SC_WIN             1000 /**< Score for player when survived another. */

/* Global model scale */
#define MODEL_X_SCALE      10   /**< X scale factor of model. */
#define MODEL_Y_SCALE      10   /**< Y scale factor of model. */

/* Global model settings */
#define MODEL_UPD_LIFE     25   /**< Tick interval for updating life of players. */
#define MODEL_SPAWN_DELAY  25   /**< Ticks between spawning requested food. */
#define MODEL_CSPAWN_DELAY 100  /**< Ticks between constant spawning food. */

#define MODEL_MIN_FOOD     1    /**< Lower boundary for max food decrementing. */
#define MODEL_MAX_FOOD     9    /**< Max food present in model per player. */
#define MODEL_STARV_DELAY  500  /**< Number of ticks between decrementing max food of model. */
#define MODEL_STARV_AMNT   1    /**< Number of food objects to decrement the maximum with per player. */

/* Event flags */
#define EVF_THROW     (1 << EV_THROW)   /**< Event flag food is thrown. */
#define EVF_JUMP      (1 << EV_JUMP)    /**< Event flag player jumps. */
#define EVF_DELIV     (1 << EV_DELIVER) /**< Event flag food is deliverd. */
#define EVF_PICKUP    (1 << EV_PICKUP)  /**< Event flag player picks up food. */
#define EVF_HIT       (1 << EV_HIT)     /**< Event flag player got hit by
                                             food. */
#define EVF_POWERUP   (1 << EV_POWERUP) /**< Event flag player picks up 
                                             power-up. */
#define EVF_DEAD      (1 << EV_DEAD)    /**< Event flag player died. */

/* Object macros */
/** Check if object is food. */
#define IS_FOOD(o)    (((o)->type >= OT_BANANA) && ((o)->type <= OT_POWERUP))
/** Check if object is a player. */
#define IS_PLAYER(o)  ((o)->type == OT_PLAYER)
/** Check if object is a homebase. */
#define IS_HOME(o)    ((o)->type == OT_HOME)
/** Check if object is a box. */
#define IS_BOX(o)     ((o)->type == OT_BOX)
/** Check if object is part of sceney (boxes, homes). */
#define IS_SCENERY(o) (IS_HOME(o) || IS_BOX(o))
/** Check whether the object is marked for deletion and waits for graphics
  * data cleanup. */
#define IS_DELETED(o) ((o)->gfx_state == GFX_DELETED)

/* Non-scenery macros */
/** Check if food object or player is powered up! */
#define IS_POWEREDUP(o)   (IS_PLAYER(o) ? (o)->state.ps == PS_POWERUP : \
                                          (o)->state.fs == FS_POWERUP)
/** Determines whether object is standing on the floor. */
#define IS_STANDING(o)    ((o)->is_standing)
/** Check if non-scenery object is active (pos != des_pos). */
#define IS_ACTIVE(o)      ((o)->active)

/* Food macros */
/** Check whether the food object belongs to the fruitsalad special. */
#define IS_FRUITSALAD(f)  ((f)->state.fs == FS_FRUITSALAD)
 
/* Player macros */
/** Check if player is dead. */
#define IS_DEAD(p)        ((p)->state.ps == PS_DEAD)
/** Check if player is unconscious. */
#define IS_UNCONSCIOUS(p) ((p)->state.ps == PS_UNCONSCIOUS)
/** Check whether the player looks to the left or right. */
#define LOOKS_LEFT(p)     ((p)->lookdir == LD_LEFT)

/** Event types.
 *  Flags are set per tick per object. Signifies what happened with non-scenery
 *  object during tick. */
typedef enum {
  EV_FIRST = 0,
  EV_THROW = 0,
  EV_JUMP,
  EV_DELIVER,
  EV_PICKUP,
  EV_HIT,
  EV_POWERUP,
  EV_DEAD,
  EV_NREFFECTS
} Event_type;

/* String mapping for event type enum. */
extern char *event_type_string[];

/** Model object type. 
 *  OT_BANANA .. OT_CHERRIES are the six types of food; OT_PLAYER is a
 *  player; OT_HOME is a player's home (where the player starts and the
 *  collected food can be brought to); OT_BOX is a level element (either
 *  platform or wall) and OT_POWERUP is a powerup.
 */
typedef enum {
  OT_FIRST = 0,   /**< First type (bogus) */  
  OT_BANANA = 0,  /**< Banana */
  OT_PEPPER,      /**< Pepper */
  OT_PUMPKIN,     /**< Pumpkin */
  OT_CUCUMBER,    /**< Cucumber */
  OT_ORANGE,      /**< Orange */
  OT_CHERRIES,    /**< Cherries */
  OT_POWERUP,     /**< Power-up */
#define OT_NRFOOD OT_POWERUP

  OT_PLAYER,      /**< Player */
  
  OT_HOME,        /**< Home/base of a player (scenery) */
  OT_BOX,         /**< (Bounding) box, floor or platform (scenery) */
  OT_NRTYPES      /**< Number of types. */
} Object_type;

/** String mapping for food type enum. */
extern char *object_type_string[];

/** String mapping for food type enum (short (char) version). */
extern char food_type_short_string[];

/** Model player state */
typedef enum {
  PS_NORMAL = 0,  /**< Player is alive and kicking. */
  PS_UNCONSCIOUS, /**< Player is unconscious and can't move. */
  PS_DEAD,        /**< Player is dead (infinite unconsciousness). */
  PS_POWERUP      /**< Player is powered up and will throw power-upped food. */
} Player_state;

/** String mapping for player state enum. */
extern char *player_state_string[];

/** * Model food state */
typedef enum {
  FS_NORMAL = 0,  /**< Food item is in a normal state. */
  FS_POWERUP,     /**< Food is power-upped and hence has special powers. */
  FS_FRUITSALAD   /**< Food is in fruitsalad mode. */
} Food_state;

/** String mapping for food state enum. */
extern char *food_state_string[];

/** Model graphics state */
typedef enum {
  GFX_NORMAL = 0, /**< Object is visible. */
  GFX_DELETED     /**< Object is invisible and deleted, tick should clean it up 
                       after graphics data is cleared. */
} Graphics_state;

/** String mapping for graphics state enum. */
extern char *graphics_state_string[];

/** Model object state.
 *  If the object is neither a type of food nor a player, this state is
 *  undefined and shouldn't be used.
 */
typedef union {
  Food_state fs;    /**< State of food. */
  Player_state ps;  /**< State of player. */
} Object_state;

/** String mapping for looking direction enum. */
extern char *look_dir_string[];

/** Model object look direction
  * Looking direction for the player objects, either left or right.
  */
typedef enum {
  LD_LEFT = 0,  /**< Player looks to the left. */
  LD_RIGHT      /**< Player looks to the right. */
} Look_dir;

/** Struct containing information for configuring the model on creation. */
typedef struct {
  /* Game settings. */
  int spawn_powerups;             /**< Enable spawning of powerups. */
  int spawn_constant;             /**< Spawn food every \c spawn_constant_delay ticks. */
  int spawn_deliver;              /**< Spawn food object when a food object is delivered. */
  int spawn_hit;                  /**< Spawn food object when player got hit by food. */
  int spawn_pickup_powerup;       /**< Spawn food when powerup is picked up. */
  int32_t min_food;               /**< Lower boundary of food to exist in model per player. */
  int32_t max_food;               /**< Max food per player to exist in model. */
  int32_t init_food_spawn;        /**< Number of food objects to spawn per player at start. */
  int32_t spawn_delay;            /**< Number of ticks delay between food spawning (retries). */
  int32_t spawn_constant_delay;   /**< Number of ticks between constant food spawning (if enabled). */
  int32_t starvation_delay;       /**< Number of ticks between decrementing the max food. */
  int32_t starvation_amount;      /**< Number of food objects to decrement max food with. */
  
  /* Global model settings. */
  int32_t random_seed;            /**< Initial random seed. */
  Vector *gravity;                /**< Gravity acceleration vector for all objects (FIXME). */
  Vector *bounciness;             /**< Bounce factor vector. */

  /* Player settings. */
  int32_t player_max_life;        /**< Maximum life of player (ticks). */
  int32_t player_ground_friction; /**< Ground friction for players. */
  int32_t player_air_friction;    /**< Air friction for players. */
  int32_t player_ground_accel;    /**< Acceleration for walking. */
  int32_t player_air_accel;       /**< Acceleration for flying. */
  Vector *player_jump_speed;      /**< Initial jump speed vector. */
  Vector *player_max_speed;       /**< Maximum speed vector for players. */
  int32_t foodstack_size;         /**< Size of food stack of players. */
  int32_t powerup_length;         /**< Number of ticks player is powered up. */
  int32_t unconc_length;          /**< Number of ticks player is unconscious. */

  /* Food settings. */
  int32_t food_ground_friction;   /**< Ground friction for food. */
  int32_t food_air_friction;      /**< Air friction for food. */
  Vector *food_max_speed;         /**< Maximum speed vector for food. */
  Vector *food_throw_speed;       /**< Initial throw speed vector. */
  int32_t food_deliver_life;      /**< Life increase of food when delivered. */
} Model_settings;

/** Game model.
 *  There should be one of these for each player in a particular game; the
 *  models will be kept in sync via the model_tick function and by sending
 *  events to each other.
 */
typedef struct {
  /* public */
  List *objects;        /**< All the objects currently in the model. */
  int32_t width;        /**< Width of the model (i.e., the level) in world 
                             units. */
  int32_t height;       /**< Height of the model (in world units). */
  int32_t framecounter; /**< Number of frames (game time units) this model has
			      been in action. */
  /* private */
  int alive;            /**< (priv) The number of players alive. */
  Model_settings *settings;   /**< (priv) (Default) model and object settings. */
  SDL_mutex *lock;      /**< (priv) Lock for the model object. */
  SDL_mutex *changed;   /**< (priv) Signify whether model has changed 
                                    (e.g. tick is done). */	
  List *homes;          /**< (priv) Subset of home objects. */
  List *act_uplayers;   /**< (priv) Subset of active unconscious players. */
  List *pas_uplayers;   /**< (priv) Subset of passive unconscious players. */
  List *act_food;       /**< (priv) Subset of active food objects. */
  List *pas_food;       /**< (priv) Subset of passive food objects. */
  List *boxes;          /**< (priv) Subset of boxes. */
  List *act_players;    /**< (priv) Subset of active players. */
  List *pas_players;    /**< (priv) Subset of passive players. */
  List *foodplaces;     /**< (priv) Locations to spawn food. */
  int32_t min_food;     /**< (priv) Lower boundary for decrementing max food. */
  int32_t max_food;     /**< (priv) Maximum food objects present in model. */
  int32_t starvation;   /**< (priv) Amount to decrement max food with every 
                                    starvation delay. */
  int32_t spawn_food;   /**< (priv) Number of food objects to spawn at end of 
                                    tick. */
  int32_t random_seed;  /**< (priv) Random seed for food creation. */
  Vector *reg_types[OT_NRTYPES]; /**< (priv) Registered types size vectors. */
} Model;

typedef struct _Model_object Model_object;
/** Object in game model.
 *  An object consists of two parts:
 *    - Information about location, speed and state.
 *    - Information for the graphics.
 */
struct _Model_object {
  /* public */
  Object_type type;          /**< Object type. */
  Vector *pos;               /**< Position of object. */
  Vector *size;              /**< Size of object. */
  Vector *speed;             /**< Speed of object. */
  int32_t score;             /**< Score of the player object. */
  int32_t life;              /**< Life of the player object. */
  List *foodstack;           /**< Food the player has collected (stack). */
  Look_dir lookdir;          /**< Direction player objects looks to. */
  int events;                /**< Event bitflags. */
  Object_state state;        /**< Object state. */
  Graphics_state gfx_state;  /**< Object state for graphics. */
  /* private */
  int active;                /**< (priv) Flag signyfing if object is active. */
  int disown;                /**< (priv) Object is barely moving and will be
                                         disowned soon. */
  void *gfx_data;            /**< (priv) Graphics data. */
  Model_object *owner;       /**< (priv) Pointer to owner of object. */
  Vector *accel;             /**< (priv) Acceleration of object. */
  Vector *des_pos;           /**< (priv) Desired new position of object. */
  Vector *max_speed;         /**< (priv) Maximum speed of object. */
  Vector *gravity;           /**< (priv) Gravity acceleration of object. */
  Vector *bounciness;        /**< (priv) Bounciness factor of object. */
  Vector *throw_speed;       /**< (priv) Initial throw speed vector. */
  Vector *jump_speed;        /**< (priv) Initial jump speed vector. */
  int32_t friction_air;      /**< (priv) Air friction for object. */
  int32_t friction_ground;   /**< (priv) Ground friction for object. */
  int32_t accel_air;         /**< (priv) Air acceleration for object. */
  int32_t accel_ground;      /**< (priv) Ground acceleration for object. */
  int32_t frame_revive;      /**< (priv) Frame number when player should
                                         regain consciousness. */
  int32_t frame_powerup_exp; /**< (priv) Frame number when players powerup 
                                         ability expires. */
  int left;                  /**< (priv) Flag signifying object wants to 
                                         move left. */
  int right;                 /**< (priv) Flag signifying object wants to 
                                         move right. */
  int jump;                  /**< (priv) Flag signifying object wants to 
                                         jump. */
  int action;                /**< (priv) Flag signifying object wants 
                                         to perform action. */
  int hit;                   /**< (priv) Flag signifying player got hit. */
  int is_standing;           /**< (priv) Flag signifying whether object is 
                                         standing. */
};

/* Model settings ADT. */
/** Create model settings struct.
 * \return Ponter to created model settings struct.
 */
Model_settings* new_model_settings(void);

/** Copy the model settings structure.
 * \return Pointer to new created model settings struct.
 */
Model_settings* copy_model_settings(Model_settings *ms);

/** Deletes the model settings structure. */
void del_model_settings(Model_settings *ms);


/* Model API for main */

/** Create a new model.
 *  This model will model a game world of size \a width * \a height, with
 *  random events initialized with \a random_seed. This seed should be the
 *  same at every computer modelling the same game, but different each time
 *  the same sequence of levels is played, so it should be generated by the
 *  server that initiates the game.
 *  \param width       Width of world/game model.
 *  \param height      Height of world/game model.
 *  \param ms          Settings object to initialize model and all new objects
 *                     with.
 *  \return Pointer to created model.
 */
Model *new_model(int32_t width, int32_t height, Model_settings *ms);

/** Delete the model.
 *  This should be done when the model will not be used anymore (i.e., when the
 *  game will be quit). Function assumes gfx_data field has been freed
 *  by del_graphics before calling del_model!
 *  \param m Model to delete.
 */
void del_model(Model *m);

/** Reset the model to newly-initialized state.
 *  Marks all objects as deleted so graphics can clear out the data
 *  and next tick after that in model will clean everything up.
 *  Also resets framecounter.
 *  The model should be reset before a new level will be played.
 *  \param m Model to reset.
 */
void reset_model(Model *m);

/** Registers size of object type.
 *  Model assumes that at the first tick all object types that will
 *  be needed for object creation are registered correctly,
 *  otherwise default size will be used.
 *  \param m      Model the register with.
 *  \param type   Object type to register.
 *  \param width  (Default) width of an object of object type \a type.
 *  \param height (Default) height of an object of object type \a type.
 */
int model_register_type(Model *m, Object_type type, 
                        int32_t width, int32_t height);

/** Add object with specified characteristics to model.
 *  This will only be called for objects that are not created internally by
 *  the model itself.
 *  \param m      Model to add object to.
 *  \param type   Type of the object to add.
 *  \param owner  Owner of the object, \c NULL if unknown or nobody.
 *  \param x      Initial X position.
 *  \param y      Initial Y position.
 *  \param width  Width of object.
 *  \param height Height of object.
 *  \return Pointer to created object in model (\c NULL if unable to create).
 */
Model_object *model_add_object(Model *m, Object_type type, Model_object *owner,
                               int32_t x, int32_t y, 
			       int32_t width, int32_t height);


/* Model API for IO */

/** Update the model.
 *  This processes all events that happened after last tick, and updates the
 *  location and state of all moving objects.  Since all the models across the
 *  network receive the same events, all models are in sync after each of them
 *  has had a model_tick.
 *  \param m Model to update.
 *  \return \a TRUE if model is live or FALSE if model is dead, a player has won.
 */
int model_tick(Model * m);

/** Move object in model horizontally.
 *  Once an object is moving in a particular (horizontal) direction, it will
 *  remain moving in that direction until model_object_move_x is called once
 *  more. Sets flags \a left and \a right on object.
 *  \invariant \f$o.left \lor o.right \lor (!o.left \land !o.right)\f$
 *  \param o      Object to move.
 *  \param x_move X Direction to move in (postive -> left, 
 *                negative -> right, zero -> stop).
 */
void model_object_move_x(Model_object *o, int x_move);

/** Move object in model vertically.
 *  Once an object is moving in a particular (vertical) direction, it will
 *  remain moving in that direction until model_object_move_x is called once
 *  more. Moving down does nothing, moving up sets \a jump flag on object.
 *  \param o      Object that wants to move in the X direction.
 *  \param y_move Y direction to move in (positive -> up, 
 *                negative -> down, zero -> halt).
 */
void model_object_move_y(Model_object *o, int y_move);

/** Throw food from object (object wants to throw food).
  * Sets \a action flag on object.
  * \param o Object that wants to throw food.
  */
void model_object_throw(Model_object *o);


/* Model API for GFX */

/** Attach graphics data to object.
  * \param o    Object from which to retrieve the graphics data from.
  * \param data Graphics data to set on the object in question.
  */
void model_object_set_gfx_data(Model_object *o, void *data);

/** Retrieve graphics data from object.
  * \param o Object from which to retrieve the graphics data from.
  */
void *model_object_get_gfx_data(Model_object *o);

/** Iterate over objects in model, applying function to object.
  * \param m         Model with objects to iterate over.
  * \param func      Function to call on each object.
  * \param user_data Misc data for use in \a ListFunc.
  */
void model_foreach_object(Model *m, ListFunc func, void *user_data);

/** *} end of group */

#endif /* FFRENZY_MODEL_H */
