/*
 * $Id: sig_isa_bus.c,v 1.33 2012-07-05 21:09:31 potyra Exp $
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_isa_bus.h"

int
sig_isa_bus_inb(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned char *valp,
	unsigned short port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xff;
			return -1;
		}
		if (! b->member[nr].f->inb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->inb(b->member[nr].s, valp, port) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_inw(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned short *valp,
	unsigned short port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xffff;
			return -1;
		}
		if (! b->member[nr].f->inw
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->inw(b->member[nr].s, valp, port) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_outb(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned char val,
	unsigned short port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->outb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->outb(b->member[nr].s, val, port) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_outw(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned short val,
	unsigned short port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->outw
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->outw(b->member[nr].s, val, port) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_readb(
	struct sig_isa_bus_main *b,
	void *s,
	uint32_t addr,
	uint8_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xff;
			return -1;
		}
		if (! b->member[nr].f->readb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->readb(b->member[nr].s, addr, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_readw(
	struct sig_isa_bus_main *b,
	void *s,
	uint32_t addr,
	uint16_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xffff;
			return -1;
		}
		if (! b->member[nr].f->readw
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->readw(b->member[nr].s, addr, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_writeb(
	struct sig_isa_bus_main *b,
	void *s,
	uint32_t addr,
	uint8_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->writeb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->writeb(b->member[nr].s, addr, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_writew(
	struct sig_isa_bus_main *b,
	void *s,
	uint32_t addr,
	uint16_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->writew
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->writew(b->member[nr].s, addr, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_map(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned long pa,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->map
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->map(b->member[nr].s, pa,
				haddr_mr_p, haddr_mw_p) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_ack(
	struct sig_isa_bus_main *b,
	void *s,
	unsigned char *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xff;
			return -1;
		}
		if (! b->member[nr].f->ack
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->ack(b->member[nr].s, valp) == 0) {
			return 0;
		}
	}
}

void
sig_isa_bus_main_connect(
	struct sig_isa_bus_main *b,
	void *s,
	const struct sig_isa_bus_main_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_ISA_BUS_MAIN);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static int
sig_isa_bus_main_s0_inb(void *_f, unsigned char *valp, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_inb(f->s1, f, valp, port);
}

static int
sig_isa_bus_main_s0_inw(void *_f, unsigned short *valp, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_inw(f->s1, f, valp, port);
}

static int
sig_isa_bus_main_s0_outb(void *_f, unsigned char val, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_outb(f->s1, f, val, port);
}

static int
sig_isa_bus_main_s0_outw(void *_f, unsigned short val, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_outw(f->s1, f, val, port);
}

static int
sig_isa_bus_main_s0_readb(void *_f, uint32_t addr, uint8_t *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_readb(f->s1, f, addr, valp);
}

static int
sig_isa_bus_main_s0_readw(void *_f, uint32_t addr, uint16_t *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_readw(f->s1, f, addr, valp);
}

static int
sig_isa_bus_main_s0_writeb(void *_f, uint32_t addr, uint8_t val)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_writeb(f->s1, f, addr, val);
}

static int
sig_isa_bus_main_s0_writew(void *_f, uint32_t addr, uint16_t val)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_writew(f->s1, f, addr, val);
}

static int
sig_isa_bus_main_s0_map(
	void *_f,
	unsigned long pa,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_map(f->s1, f, pa, haddr_mr_p, haddr_mw_p);
}

static int
sig_isa_bus_main_s0_ack(void *_f, unsigned char *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_ack(f->s1, f, valp);
}

static int
sig_isa_bus_main_s1_inb(void *_f, unsigned char *valp, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_inb(f->s0, f, valp, port);
}

static int
sig_isa_bus_main_s1_inw(void *_f, unsigned short *valp, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_inw(f->s0, f, valp, port);
}

static int
sig_isa_bus_main_s1_outb(void *_f, unsigned char val, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_outb(f->s0, f, val, port);
}

static int
sig_isa_bus_main_s1_outw(void *_f, unsigned short val, unsigned short port)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_outw(f->s0, f, val, port);
}

static int
sig_isa_bus_main_s1_readb(void *_f, uint32_t addr, uint8_t *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_readb(f->s0, f, addr, valp);
}

static int
sig_isa_bus_main_s1_readw(void *_f, uint32_t addr, uint16_t *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_readw(f->s0, f, addr, valp);
}

static int
sig_isa_bus_main_s1_writeb(void *_f, uint32_t addr, uint8_t val)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_writeb(f->s0, f, addr, val);
}

static int
sig_isa_bus_main_s1_writew(void *_f, uint32_t addr, uint16_t val)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_writew(f->s0, f, addr, val);
}

static int
sig_isa_bus_main_s1_map(
	void *_f,
	unsigned long pa,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_map(f->s0, f, pa, haddr_mr_p, haddr_mw_p);
}

static int
sig_isa_bus_main_s1_ack(void *_f, unsigned char *valp)
{
	struct sig_isa_bus_main_merge *f
			= (struct sig_isa_bus_main_merge *) _f;

	return sig_isa_bus_ack(f->s0, f, valp);
}

struct sig_isa_bus_main_merge *
sig_isa_bus_main_merge(
	struct sig_isa_bus_main *s0,
	struct sig_isa_bus_main *s1
)
{
	static const struct sig_isa_bus_main_funcs s0_funcs = {
		.inb = sig_isa_bus_main_s0_inb,
		.inw = sig_isa_bus_main_s0_inw,
		.outb = sig_isa_bus_main_s0_outb,
		.outw = sig_isa_bus_main_s0_outw,
		.readb = sig_isa_bus_main_s0_readb,
		.readw = sig_isa_bus_main_s0_readw,
		.writeb = sig_isa_bus_main_s0_writeb,
		.writew = sig_isa_bus_main_s0_writew,
		.map = sig_isa_bus_main_s0_map,
		.ack = sig_isa_bus_main_s0_ack,
	};
	static const struct sig_isa_bus_main_funcs s1_funcs = {
		.inb = sig_isa_bus_main_s1_inb,
		.inw = sig_isa_bus_main_s1_inw,
		.outb = sig_isa_bus_main_s1_outb,
		.outw = sig_isa_bus_main_s1_outw,
		.readb = sig_isa_bus_main_s1_readb,
		.readw = sig_isa_bus_main_s1_readw,
		.writeb = sig_isa_bus_main_s1_writeb,
		.writew = sig_isa_bus_main_s1_writew,
		.map = sig_isa_bus_main_s1_map,
		.ack = sig_isa_bus_main_s1_ack,
	};
	struct sig_isa_bus_main_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_isa_bus_main_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_isa_bus_main_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_isa_bus_main_split(struct sig_isa_bus_main_merge *m)
{
	fixme();
}

struct sig_isa_bus_main *
sig_isa_bus_main_create(const char *name)
{
	struct sig_isa_bus_main *b;

	b = shm_alloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_ISA_BUS_MAIN;
	b->member_count = 0;

	return b;
}

void
sig_isa_bus_main_destroy(struct sig_isa_bus_main *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_ISA_BUS_MAIN);

	shm_free(sig);
}

void
sig_isa_bus_main_suspend(struct sig_isa_bus_main *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_isa_bus_main_resume(struct sig_isa_bus_main *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}

int
sig_isa_bus_dma_req(
	struct sig_isa_bus_dma *b,
	void *s
)
{
	unsigned int nr;
	int (*func)(void *);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->req;
		if (func
		 && func(b->member[nr].s) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_in(
	struct sig_isa_bus_dma *b,
	void *s,
	uint32_t addr,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint32_t, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_in;
		if (func
		 && func(b->member[nr].s, addr, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_out(
	struct sig_isa_bus_dma *b,
	void *s,
	uint32_t addr,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint32_t, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_out;
		if (func
		 && func(b->member[nr].s, addr, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_verify(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verify;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_inb(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc,
	uint8_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint8_t *);

		if (nr == b->member_count) {
			*valp = -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_inb;
		if (func && func(b->member[nr].s, tc, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_inw(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc,
	uint16_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint16_t *);

		if (nr == b->member_count) {
			*valp = -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_inw;
		if (func && func(b->member[nr].s, tc, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_outb(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc,
	uint8_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint8_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_outb;
		if (func && func(b->member[nr].s, tc, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_outw(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc,
	uint16_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint16_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_outw;
		if (func && func(b->member[nr].s, tc, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_verifyb(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verifyb;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_bus_dma_ack_verifyw(
	struct sig_isa_bus_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verifyw;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

void
sig_isa_bus_dma_connect(
	struct sig_isa_bus_dma *b,
	void *s,
	const struct sig_isa_bus_dma_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_ISA_BUS_DMA);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static int
sig_isa_bus_dma_s0_req(void *_f)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_req(f->s1, f);
}

static int
sig_isa_bus_dma_s0_ack_inb(void *_f, unsigned int tc, uint8_t *valp)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_inb(f->s1, f, tc, valp);
}

static int
sig_isa_bus_dma_s0_ack_outb(void *_f, unsigned int tc, uint8_t val)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_outb(f->s1, f, tc, val);
}

static int
sig_isa_bus_dma_s0_ack_inw(void *_f, unsigned int tc, uint16_t *valp)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_inw(f->s1, f, tc, valp);
}

static int
sig_isa_bus_dma_s0_ack_outw(void *_f, unsigned int tc, uint16_t val)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_outw(f->s1, f, tc, val);
}

static int
sig_isa_bus_dma_s1_req(void *_f)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_req(f->s0, f);
}

static int
sig_isa_bus_dma_s1_ack_inb(void *_f, unsigned int tc, uint8_t *valp)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_inb(f->s0, f, tc, valp);
}

static int
sig_isa_bus_dma_s1_ack_outb(void *_f, unsigned int tc, uint8_t val)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_outb(f->s0, f, tc, val);
}

static int
sig_isa_bus_dma_s1_ack_inw(void *_f, unsigned int tc, uint16_t *valp)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_inw(f->s0, f, tc, valp);
}

static int
sig_isa_bus_dma_s1_ack_outw(void *_f, unsigned int tc, uint16_t val)
{
	struct sig_isa_bus_dma_merge *f
			= (struct sig_isa_bus_dma_merge *) _f;

	return sig_isa_bus_dma_ack_outw(f->s0, f, tc, val);
}

struct sig_isa_bus_dma_merge *
sig_isa_bus_dma_merge(
	struct sig_isa_bus_dma *s0,
	struct sig_isa_bus_dma *s1
)
{
	static const struct sig_isa_bus_dma_funcs s0_funcs = {
		.req = sig_isa_bus_dma_s0_req,
		.ack_inb = sig_isa_bus_dma_s0_ack_inb,
		.ack_outb = sig_isa_bus_dma_s0_ack_outb,
		.ack_inw = sig_isa_bus_dma_s0_ack_inw,
		.ack_outw = sig_isa_bus_dma_s0_ack_outw,
	};
	static const struct sig_isa_bus_dma_funcs s1_funcs = {
		.req = sig_isa_bus_dma_s1_req,
		.ack_inb = sig_isa_bus_dma_s1_ack_inb,
		.ack_outb = sig_isa_bus_dma_s1_ack_outb,
		.ack_inw = sig_isa_bus_dma_s1_ack_inw,
		.ack_outw = sig_isa_bus_dma_s1_ack_outw,
	};
	struct sig_isa_bus_dma_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_isa_bus_dma_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_isa_bus_dma_connect(s1, m, &s1_funcs);

	return m;
}

struct sig_isa_bus_dma *
sig_isa_bus_dma_create(const char *name)
{
	struct sig_isa_bus_dma *sig;

	sig = shm_alloc(sizeof(*sig));
	assert(sig);

	sig->type = SIG_GEN_ISA_BUS_DMA;
	sig->member_count = 0;

	return sig;
}

void
sig_isa_bus_dma_destroy(struct sig_isa_bus_dma *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_ISA_BUS_DMA);

	shm_free(sig);
}

void
sig_isa_bus_dma_suspend(struct sig_isa_bus_dma *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_isa_bus_dma_resume(struct sig_isa_bus_dma *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}

struct sig_isa_bus *
sig_isa_bus_create(const char *name)
{
	struct sig_isa_bus *c;
	char n[1000];

	c = shm_alloc(sizeof(*c));
	assert(c);

	c->type = SIG_GEN_ISA_BUS;

	sprintf(n, "%s-+5V", name);
	c->p5V = sig_boolean_create(n);
	sprintf(n, "%s--5V", name);
	c->m5V = sig_boolean_create(n);
	sprintf(n, "%s-+12V", name);
	c->p12V = sig_boolean_create(n);
	sprintf(n, "%s--12V", name);
	c->m12V = sig_boolean_create(n);

	sprintf(n, "%s-n_reset", name);
	c->n_reset = sig_boolean_create(n);

	sprintf(n, "%s-main", name);
	c->main = sig_isa_bus_main_create(n);

	sprintf(n, "%s-dma0", name);
	c->dma0 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma1", name);
	c->dma1 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma2", name);
	c->dma2 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma3", name);
	c->dma3 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma5", name);
	c->dma5 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma6", name);
	c->dma6 = sig_isa_bus_dma_create(n);
	sprintf(n, "%s-dma7", name);
	c->dma7 = sig_isa_bus_dma_create(n);

	sprintf(n, "%s-int3", name);
	c->int3 = sig_boolean_or_create(n);
	sprintf(n, "%s-int4", name);
	c->int4 = sig_boolean_or_create(n);
	sprintf(n, "%s-int5", name);
	c->int5 = sig_boolean_or_create(n);
	sprintf(n, "%s-int6", name);
	c->int6 = sig_boolean_or_create(n);
	sprintf(n, "%s-int7", name);
	c->int7 = sig_boolean_or_create(n);
	sprintf(n, "%s-int9", name);
	c->int9 = sig_boolean_or_create(n);
	sprintf(n, "%s-int10", name);
	c->int10 = sig_boolean_or_create(n);
	sprintf(n, "%s-int11", name);
	c->int11 = sig_boolean_or_create(n);
	sprintf(n, "%s-int12", name);
	c->int12 = sig_boolean_or_create(n);
	sprintf(n, "%s-int14", name);
	c->int14 = sig_boolean_or_create(n);
	sprintf(n, "%s-int15", name);
	c->int15 = sig_boolean_or_create(n);

	return c;
}

void
sig_isa_bus_destroy(struct sig_isa_bus *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_ISA_BUS);

	sig_boolean_destroy(sig->p5V);
	sig_boolean_destroy(sig->m5V);
	sig_boolean_destroy(sig->p12V);
	sig_boolean_destroy(sig->m12V);

	sig_boolean_destroy(sig->n_reset);

	sig_isa_bus_main_destroy(sig->main);

	sig_isa_bus_dma_destroy(sig->dma0);
	sig_isa_bus_dma_destroy(sig->dma1);
	sig_isa_bus_dma_destroy(sig->dma2);
	sig_isa_bus_dma_destroy(sig->dma3);
	sig_isa_bus_dma_destroy(sig->dma5);
	sig_isa_bus_dma_destroy(sig->dma6);
	sig_isa_bus_dma_destroy(sig->dma7);

	sig_boolean_or_destroy(sig->int3);
	sig_boolean_or_destroy(sig->int4);
	sig_boolean_or_destroy(sig->int5);
	sig_boolean_or_destroy(sig->int6);
	sig_boolean_or_destroy(sig->int7);
	sig_boolean_or_destroy(sig->int9);
	sig_boolean_or_destroy(sig->int10);
	sig_boolean_or_destroy(sig->int11);
	sig_boolean_or_destroy(sig->int12);
	sig_boolean_or_destroy(sig->int14);
	sig_boolean_or_destroy(sig->int15);

	shm_free(sig);
}

void
sig_isa_bus_suspend(struct sig_isa_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
	
	sig_boolean_suspend(b->p5V, fSig);
	sig_boolean_suspend(b->m5V, fSig);
	sig_boolean_suspend(b->p12V, fSig);
	sig_boolean_suspend(b->m12V, fSig);	
	sig_boolean_suspend(b->n_reset, fSig);
	
	sig_isa_bus_main_suspend(b->main, fSig);
	
	sig_isa_bus_dma_suspend(b->dma0, fSig);
	sig_isa_bus_dma_suspend(b->dma1, fSig);
	sig_isa_bus_dma_suspend(b->dma2, fSig);
	sig_isa_bus_dma_suspend(b->dma3, fSig);
	sig_isa_bus_dma_suspend(b->dma5, fSig);
	sig_isa_bus_dma_suspend(b->dma6, fSig);
	sig_isa_bus_dma_suspend(b->dma7, fSig);
	
	sig_boolean_or_suspend(b->int3, fSig);
	sig_boolean_or_suspend(b->int4, fSig);
	sig_boolean_or_suspend(b->int5, fSig);
	sig_boolean_or_suspend(b->int6, fSig);
	sig_boolean_or_suspend(b->int7, fSig);
	sig_boolean_or_suspend(b->int9, fSig);
	sig_boolean_or_suspend(b->int10, fSig);
	sig_boolean_or_suspend(b->int11, fSig);
	sig_boolean_or_suspend(b->int12, fSig);
	sig_boolean_or_suspend(b->int14, fSig);
	sig_boolean_or_suspend(b->int15, fSig);
}

void
sig_isa_bus_resume(struct sig_isa_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
	
	sig_boolean_resume(b->p5V, fSig);
	sig_boolean_resume(b->m5V, fSig);
	sig_boolean_resume(b->p12V, fSig);
	sig_boolean_resume(b->m12V, fSig);	
	sig_boolean_resume(b->n_reset, fSig);
	
	sig_isa_bus_main_resume(b->main, fSig);
	
	sig_isa_bus_dma_resume(b->dma0, fSig);
	sig_isa_bus_dma_resume(b->dma1, fSig);
	sig_isa_bus_dma_resume(b->dma2, fSig);
	sig_isa_bus_dma_resume(b->dma3, fSig);
	sig_isa_bus_dma_resume(b->dma5, fSig);
	sig_isa_bus_dma_resume(b->dma6, fSig);
	sig_isa_bus_dma_resume(b->dma7, fSig);
	
	sig_boolean_or_resume(b->int3, fSig);
	sig_boolean_or_resume(b->int4, fSig);
	sig_boolean_or_resume(b->int5, fSig);
	sig_boolean_or_resume(b->int6, fSig);
	sig_boolean_or_resume(b->int7, fSig);
	sig_boolean_or_resume(b->int9, fSig);
	sig_boolean_or_resume(b->int10, fSig);
	sig_boolean_or_resume(b->int11, fSig);
	sig_boolean_or_resume(b->int12, fSig);
	sig_boolean_or_resume(b->int14, fSig);
	sig_boolean_or_resume(b->int15, fSig);
}
