/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <math.h>
#include <netinet/in.h>
#include "exroutput.h"

double voltage_to_intensity_rec709( double val )
{
    /* Handle invalid values before doing a gamma transform. */
    if( val < 0.0 ) return 0.0;
    if( val > 1.0 ) return 1.0;

    if( val <= 0.081 ) {
        return val / 4.5;
    }
    return pow( ( val + 0.099 ) / 1.099, 2.2 );
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    double *scanline;
    int width, height;
    char ppmline[ 2048 ];
    uint8_t *inscanline;
    uint16_t maxval;
    double maxvald;
    FILE *ppm;
    uint16_t min, max;
    int y;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: ppmtoexr input.ppm output.exr\n" );
        return 1;
    }

    ppm = fopen( argv[ 1 ], "r" );
    if( !ppm ) {
        fprintf( stderr, "ppmtoexr: Cannot open %s\n", argv[ 1 ] );
        return 1;
    }
    fgets( ppmline, sizeof( ppmline ), ppm );
    if( ppmline[ 0 ] != 'P' || ppmline[ 1 ] != '6' ) {
        fprintf( stderr, "ppmtoexr: %s is not a valid ppm file.\n", argv[ 1 ] );
        fclose( ppm );
        return 1;
    }

    fgets( ppmline, sizeof( ppmline ), ppm );
    while( ppmline[ 0 ] == '#' ) {
        fgets( ppmline, sizeof( ppmline ), ppm );
    }
    sscanf( ppmline, "%d %d\n", &width, &height );
    fprintf( stdout, "ppmtoexr: %s [%dx%d]\n", argv[ 1 ], width, height );

    fgets( ppmline, sizeof( ppmline ), ppm );
    while( ppmline[ 0 ] == '#' ) {
        fgets( ppmline, sizeof( ppmline ), ppm );
    }
    sscanf( ppmline, "%d\n", &maxval );
    maxvald = ((double) maxval);
    if( !maxval ) {
        fprintf( stderr, "ppmtoexr: %s is not a valid ppm file.\n", argv[ 1 ] );
        fclose( ppm );
        return 1;
    }

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "ppmtoexr: Cannot allocate memory.\n" );
        fclose( ppm );
        return 1;
    }

    if( maxval < 256 ) {
        inscanline = malloc( width * 3 );
        if( !inscanline ) {
            fprintf( stderr, "ppmtoexr: Cannot allocate memory.\n" );
            fclose( ppm );
            free( scanline );
            return 1;
        }
    } else {
        inscanline = malloc( width * 3 * sizeof( uint16_t ) );
        if( !inscanline ) {
            fprintf( stderr, "ppmtoexr: Cannot allocate memory.\n" );
            fclose( ppm );
            free( scanline );
            return 1;
        }
    }

    exrout = exroutput_new( argv[ 2 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "ppmtoexr: Cannot open %s\n", argv[ 2 ] );
        free( inscanline );
        free( scanline );
        return 1;
    }

    min = 65535;
    max = 0;

    for( y = 0; y < height; y++ ) {
        int x;

        fprintf( stderr, "ppmtoexr: Writing scanline %d\r", y );

        if( maxval < 256 ) {
            fread( inscanline, width * 3, 1, ppm );
        } else {
            fread( inscanline, width * 3, sizeof( uint16_t ), ppm );
        }

        for( x = 0; x < width; x++ ) {
            uint16_t ri, gi, bi;
            double r, g, b;

            if( maxval < 256 ) {
                ri = inscanline[ (x * 3) + 0 ];
                gi = inscanline[ (x * 3) + 1 ];
                bi = inscanline[ (x * 3) + 2 ];
                r = voltage_to_intensity_rec709( ((double) ri) / maxvald );
                g = voltage_to_intensity_rec709( ((double) gi) / maxvald );
                b = voltage_to_intensity_rec709( ((double) bi) / maxvald );
            } else {
                ri = ntohs( ((uint16_t *) inscanline)[ (x * 3) + 0 ] );
                gi = ntohs( ((uint16_t *) inscanline)[ (x * 3) + 1 ] );
                bi = ntohs( ((uint16_t *) inscanline)[ (x * 3) + 2 ] );
                r = ((double) ri) / maxvald;
                g = ((double) gi) / maxvald;
                b = ((double) bi) / maxvald;
            }

            if( ri < min ) min = ri; if( ri > max ) max = ri;
            if( gi < min ) min = gi; if( gi > max ) max = gi;
            if( bi < min ) min = bi; if( bi > max ) max = bi;

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = 1.0;
        }

        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nppmtoexr: Done (min %d, max %d).\n", min, max );

    exroutput_delete( exrout );
    fclose( ppm );
    free( inscanline );
    free( scanline );
    return 0;
}

