/*
 * Copyright (C) 2007-2009 KenD00
 * 
 * This file is part of DumpHD.
 * 
 * DumpHD is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package dumphd.core;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.TreeMap;

import dumphd.bdplus.ConversionTable;

/**
 * This class holds all information about a specific disc set. A disc set contains all files from a specific content type on that disc (e.g. HD-DVD Advanced Content Video).
 * Additionally it is used to store different information used to process the disc set, e.g. source- and destination-directory, KeyData.
 * 
 * Objects of this class are created by the DumpHD class and are used as information exchange objects between the core and the GUI.
 * 
 * @author KenD00
 */
public class DiscSet {

   public final static int HD_STANDARD_V = 1;
   public final static int HD_STANDARD_A = 2;
   public final static int HD_ADVANCED_V = 3;
   public final static int HD_ADVANCED_A = 4;
   public final static int BD_MV = 5;
   public final static int BDR_MV = 6;
   public final static int BDR_AV = 7;

   public final static String[] DISCTYPE_NAMES = { "None", "HD-DVD Standard Content Video", "HD-DVD Standard Content Audio", "HD-DVD Advanced Content Video",
      "HD-DVD Advanced Content Audio", "Blu-Ray BDMV", "Blu-Ray Recordable BDMV", "Blu-Ray Recordable BDAV" };



   /**
    * Type of this DiscSet
    */
   public final int contentType;
   /**
    * The directory that contains the files / directories to process. This is an absolute path.
    */
   public final File srcDir;
   /**
    * The destination directory, all files will be put there. This is an absolute path
    */
   public File dstDir = null;
   /**
    * The directory with the AACS files. This points to the directory with the Title / Unit Key file(s),
    * in case of BDAV the MKB is in the parent directory! This is an absolute path.
    */
   public File aacsDir = null;
   /**
    * This holds all key information about this DiscSet 
    */
   public KeyData keyData = null;
   /**
    * If true, this DiscSet is additionally BD+ protected. Only valid for Blu-Ray types.
    */
   public boolean bdplus = false;
   /**
    * The BD+ ConversionTable. Only valid for Blu-Ray types.
    */
   public ConversionTable convTable = null;
   /**
    * This TreeMap contains all "stream" source files, with a relative path starting from srcDir.
    * Currently for all media except BDAV this TreeMap contains only one ArrayList with the key 0 with the
    * files from the "base" content directory, HVDVD_TS for HD-DVD and BDMV for Blu-Ray. For BDAV there
    * can be multiple entries, the keys are corresponding to the BDAV directory number.
    */
   public final TreeMap<Integer, ArrayList<String>> streamSets = new TreeMap<Integer, ArrayList<String>>();
   /**
    * This ArrayList contains additional files, for Blu-Ray this is the Certificate folder, for HD-DVD Advanced Content
    * the ADV_OBJ directory.
    */
   public final ArrayList<String> fileSet = new ArrayList<String>(4096);
   /**
    * If true, this DiscSet will be processed, otherwise not.
    */
   public boolean selected = false;


   /**
    * Creates a new DiscSet of the given type with the given source directory.
    * 
    * @param contentType The type of the DiscSet
    * @param srcDir The source directory of the DiscSet. This will also be set as the root directory.
    */
   public DiscSet(int contentType, File srcDir) {
      this.contentType = contentType;
      this.srcDir = srcDir;
   }

   public boolean isBluRay() {
      return (contentType >= 5);
   }

   public boolean isRecordable() {
      //TODO: This won't work for HD-DVD recordables (which should reside below the BluRay-ID's)
      return (contentType >= 6);
   }

   public String toString() {
      StringBuffer sb = new StringBuffer(2048);
      sb.append("DiscSet:\n");
      sb.append("Type                 : ");
      sb.append(DiscSet.DISCTYPE_NAMES[contentType]);
      sb.append("\nSource directory     : ");
      sb.append(srcDir.toString());
      sb.append("\nDestination directory: ");
      if (dstDir != null) {
         sb.append(dstDir.toString());
      } else {
         sb.append("NONE");
      }
      sb.append("\nAACS directory       : ");
      if (aacsDir != null) {
         sb.append(aacsDir.toString());
      } else {
         sb.append("NONE");
      }
      sb.append("\nKey data             : ");
      if (keyData != null) {
         sb.append(keyData.toString());
      } else {
         sb.append("NONE");
      }
      if (isBluRay()) {
         sb.append("\nBD+ protection       : ");
         if (bdplus) {
            sb.append("YES");
            sb.append("\nConversion Table     : ");
            if (convTable != null) {
               sb.append("PRESENT");
            } else {
               sb.append("MISSING");
            }
         } else {
            sb.append("NO");
         }
      }
      sb.append("\nStreamsets           :");
      Iterator<ArrayList<String>> setIt = streamSets.values().iterator();
      while (setIt.hasNext()) {
         Iterator<String> it = setIt.next().iterator();
         while (it.hasNext()) {
            sb.append("\n");
            sb.append(it.next());
         }
      }
      sb.append("\nOther files          :");
      Iterator<String> it = fileSet.iterator();
      while (it.hasNext()) {
         sb.append("\n");
         sb.append(it.next());
      }
      return sb.toString();
   }

}
