/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2008, 2009 Alexis Megas, 
** Gunther van Dooble, and the Dooble Team. 
** All rights reserved.
**
**
** License: GPL2 only: 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://dooble.net/projects/dooble
**   http://dooble.sf.net
**   http://www.getadooble.com
****************************************************************************/


#include <QUrl>
#include <QFile>
#include <QPrinter>
#include <QSettings>
#include <QCloseEvent>
#include <QFileDialog>
#include <QTextStream>
#include <QPrintDialog>
#include <QDesktopServices>
#include <QPrintPreviewDialog>

#include "pagesourcewindow.h"

pagesourcewindow::pagesourcewindow(const QUrl &url,
				   const QString &html):QMainWindow()
{
  ui.setupUi(this);
  ui.textEdit->setPlainText(html);
  connect(ui.actionPrint, SIGNAL(triggered(void)), this,
	  SLOT(slotPrint(void)));
  connect(ui.actionPrint_Preview, SIGNAL(triggered(void)), this,
	  SLOT(slotPrintPreview(void)));
  connect(ui.actionClose, SIGNAL(triggered(void)), this,
	  SLOT(slotClose(void)));
  connect(ui.actionFind, SIGNAL(triggered(void)), this,
	  SLOT(slotShowFind(void)));
  connect(ui.actionSave_As, SIGNAL(triggered(void)), this,
	  SLOT(slotSavePageAs(void)));
  connect(ui.hideFindToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotHideFind(void)));
  connect(ui.findLineEdit, SIGNAL(returnPressed(void)), this,
	  SLOT(slotFind(void)));
  connect(ui.findLineEdit, SIGNAL(textChanged(const QString &)), this,
	  SLOT(slotFind(const QString &)));
  connect(ui.actionWrap_Lines, SIGNAL(toggled(bool)),
	  this, SLOT(slotWrapLines(bool)));
  setWindowTitle(tr("Dooble Web Browser - Page Source (") +
		 url.toString(QUrl::StripTrailingSlash) + tr(")"));

  QSettings settings;

  if(url.path().isEmpty() || url.path() == "/")
    fileName = "source";
  else if(url.path().contains("/"))
    {
      fileName = url.path();
      fileName = fileName.mid(fileName.lastIndexOf("/") + 1);
    }
  else
    fileName = url.path();

  ui.actionWrap_Lines->setChecked
    (settings.value("pageSource/wrapLines", false).toBool());
  slotWrapLines(ui.actionWrap_Lines->isChecked());
  slotSetIcons();
  show();
}

pagesourcewindow::~pagesourcewindow()
{
}

void pagesourcewindow::slotClose(void)
{
  close();
}

void pagesourcewindow::closeEvent(QCloseEvent *event)
{
  event->accept();
  deleteLater();
}

void pagesourcewindow::slotPrint(void)
{
  QPrinter printer;
  QPrintDialog printDialog(&printer, this);

  if(printDialog.exec() == QDialog::Accepted)
    ui.textEdit->print(&printer);
}

void pagesourcewindow::slotPrintPreview(void)
{
  QPrinter printer;
  QPrintPreviewDialog printDialog(&printer, this);

  connect(&printDialog,
	  SIGNAL(paintRequested(QPrinter *)),
	  this,
	  SLOT(slotTextEditPrintPreview(QPrinter *)));

  if(printDialog.exec() == QDialog::Accepted)
    ui.textEdit->print(&printer);
}

void pagesourcewindow::slotTextEditPrintPreview(QPrinter *printer)
{
  ui.textEdit->print(printer);
}

void pagesourcewindow::slotHideFind(void)
{
  ui.findFrame->setVisible(false);
}

void pagesourcewindow::slotShowFind(void)
{
  ui.findFrame->setVisible(true);
  ui.findLineEdit->setFocus();
  ui.findLineEdit->selectAll();
}

void pagesourcewindow::keyPressEvent(QKeyEvent *event)
{
  if(event->key() == Qt::Key_Escape)
    ui.findFrame->setVisible(false);
}

void pagesourcewindow::slotFind(void)
{
  slotFind(ui.findLineEdit->text());
}

void pagesourcewindow::slotFind(const QString &text)
{
  static QPalette lineEditPalette(ui.findLineEdit->palette());
  QTextDocument::FindFlags findFlags = 0;

  if(ui.matchCaseCheckBox->isChecked())
    findFlags |= QTextDocument::FindCaseSensitively;

  if(ui.textEdit->find(text, findFlags) || text.isEmpty())
    ui.findLineEdit->setPalette(lineEditPalette);
  else
    {
      if(ui.textEdit->textCursor().selectedText().isEmpty())
	{
	  QColor color(240, 128, 128); // Light Coral
	  QPalette palette(ui.findLineEdit->palette());

	  palette.setColor(ui.findLineEdit->backgroundRole(), color);
	  ui.findLineEdit->setPalette(palette);
	}
      else
	{
	  ui.textEdit->moveCursor(QTextCursor::Start);
	  slotFind(text);
	}
    }
}

void pagesourcewindow::slotSavePageAs(void)
{
  QString currentPath = QDesktopServices::storageLocation
    (QDesktopServices::DesktopLocation);

  if(currentPath.isEmpty())
    currentPath = QDir::homePath();

  QFileDialog fileDialog(this);

  fileDialog.setFileMode(QFileDialog::AnyFile);
  fileDialog.setDirectory(currentPath);
  fileDialog.setLabelText(QFileDialog::Accept, tr("&Save"));
  fileDialog.setAcceptMode(QFileDialog::AcceptSave);
  fileDialog.selectFile(fileName);

  if(fileDialog.exec() == QDialog::Accepted)
    {
      QFile file(fileDialog.selectedFiles()[0]);

      if(file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
	  QTextStream out(&file);

	  out << ui.textEdit->toPlainText();
	  file.close();
	}
    }
}

void pagesourcewindow::slotWrapLines(bool checked)
{
  QSettings settings;

  settings.setValue("pageSource/wrapLines", checked);

  if(checked)
    ui.textEdit->setLineWrapMode(QTextEdit::WidgetWidth);
  else
    ui.textEdit->setLineWrapMode(QTextEdit::NoWrap);
}

void pagesourcewindow::slotSetIcons(void)
{
  QSettings settings;
  QSettings cfgSettings
    (settings.value("iconSet").toString(), QSettings::IniFormat);

  cfgSettings.beginGroup("pageSourceWindow");
  ui.actionPrint->setIcon(QIcon(cfgSettings.value("actionPrint").toString()));
  ui.actionPrint_Preview->setIcon
    (QIcon(cfgSettings.value("actionPrint_Preview").toString()));
  ui.actionClose->setIcon(QIcon(cfgSettings.value("actionClose").toString()));
  ui.actionFind->setIcon(QIcon(cfgSettings.value("actionFind").toString()));
  ui.actionSave_As->setIcon
    (QIcon(cfgSettings.value("actionSave_As").toString()));
  ui.hideFindToolButton->setIcon
    (QIcon(cfgSettings.value("hideFindToolButton").toString()));
  ui.findFrame->setVisible(false);
  setWindowIcon(QIcon(cfgSettings.value("windowIcon").toString()));
}
