// ************************************************************************** //
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Wrap/Swig/libBornAgainSim.i
//! @brief     SWIG interface file for libBornAgainSim
//!
//!            Configuration is done in Sim/CMakeLists.txt
//!
//! @homepage  http://apps.jcns.fz-juelich.de/BornAgain
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2013
//! @authors   Scientific Computing Group at MLZ Garching
//
// ************************************************************************** //

%module(directors="1", moduleimport="import $module") "libBornAgainSim"

%include "commons.i"

%include "ignoreSample.i"

%import(module="libBornAgainFit") ""

%rename(addSimulationAndData_cpp) FitObjective::addSimulationAndData;
%rename(evaluate_residuals_cpp) FitObjective::evaluate_residuals;
%rename(evaluate_cpp) FitObjective::evaluate;
%rename(finalize_cpp) FitObjective::finalize;
%rename(initPlot_cpp) FitObjective::initPlot;
%rename(uncertainties_cpp) FitObjective::uncertainties;
%rename(uncertaintyData_cpp) FitObjective::uncertaintyData;
%rename(containsUncertainties_cpp) FitObjective::containsUncertainties;
%rename(allPairsHaveUncertainties_cpp) FitObjective::allPairsHaveUncertainties;

%feature("director") PyBuilderCallback; //TODO: used?
%feature("director") PyObserverCallback; // TODO: used?

%feature("director") FitObjective;       // used in custom_objective_function.py

// Propagate python exceptions (from https://stackoverflow.com/questions/4811492)
%feature("director:except") {
    if( $error != NULL ) {
        PyObject *ptype, *pvalue, *ptraceback;
        PyErr_Fetch( &ptype, &pvalue, &ptraceback );
        PyErr_Restore( ptype, pvalue, ptraceback );
        PyErr_Print();
        Py_Exit(1);
    }
}

%{
#include "BAVersion.h"

#include "Base/Axis/Scale.h"
#include "Base/Axis/Frame.h"

#include "Device/Histo/SimulationResult.h"

#include "Fit/Minimizer/MinimizerResult.h"

#include "Sample/Scattering/ISampleNode.h"

#include "Sim/Background/ConstantBackground.h"
#include "Sim/Background/PoissonBackground.h"
#include "Sim/Export/ExportToPython.h"
#include "Sim/Fitting/FitObjective.h"
#include "Sim/Fitting/IterationInfo.h"
#include "Sim/Fitting/PyFittingCallbacks.h"
#include "Sim/Fitting/SimDataPair.h"
#include "Sim/Residual/ChiSquaredModule.h"
#include "Sim/Residual/IIntensityFunction.h"
#include "Sim/Residual/VarianceFunctions.h"
#include "Sim/Scan/AlphaScan.h"
#include "Sim/Scan/QzScan.h"
#include "Sim/Simulation/DepthprobeSimulation.h"
#include "Sim/Simulation/ScatteringSimulation.h"
#include "Sim/Simulation/OffspecSimulation.h"
#include "Sim/Simulation/SpecularSimulation.h"
%}

// The following goes verbatim from libBornAgainSim.i to libBornAgainSim_wrap.cxx.
// Note that the order matters, as base classes must be included before derived classes.

%include "fromBase.i"
%include "fromParam.i"

%import(module="libBornAgainSample") "Sample/Scattering/ISampleNode.h"

%template(swig_dummy_type_const_inode_vector) std::vector<const INode*>;

%include "BAVersion.h"

%include "Sim/Fitting/IterationInfo.h"
%include "Sim/Fitting/PyFittingCallbacks.h"
%include "Sim/Fitting/FitObjective.h"

%include "Sim/Scan/IBeamScan.h"
%include "Sim/Scan/AlphaScan.h"
%include "Sim/Scan/QzScan.h"

%include "Sim/Simulation/ISimulation.h"
%include "Sim/Simulation/ScatteringSimulation.h"
%include "Sim/Simulation/DepthprobeSimulation.h"
%include "Sim/Simulation/SpecularSimulation.h"
%include "Sim/Simulation/OffspecSimulation.h"

%include "Sim/Background/IBackground.h"
%include "Sim/Background/ConstantBackground.h"
%include "Sim/Background/PoissonBackground.h"

%include "Sim/Export/ExportToPython.h"

%include "Sim/Residual/IIntensityFunction.h"
%include "Sim/Residual/IChiSquaredModule.h"
%include "Sim/Residual/ChiSquaredModule.h"
%include "Sim/Residual/VarianceFunctions.h"

%extend Vec3<double> {
    Vec3<double> __add__(const Vec3<double>& rhs) const {
        return *($self) + rhs; }
    Vec3<double> __mul__(double c) const {
        return c * *($self); }
    Vec3<double> __rmul__(double c) const {
        return *($self) * c; }
    Vec3<double> __neg__() const {
        return - *($self); }
};

%pythoncode %{
class SimulationBuilderWrapper(PyBuilderCallback):
    def __init__(self, f):
        super(SimulationBuilderWrapper, self).__init__()
        self.f_ = f

    def create_par_dict(self, pars):
        """
        Conversion of ba.Parameters to Python dictionary
        """
        pars_dict = dict()
        for index, p in enumerate(pars):
            pars_dict[p.name()] = p.value
        return pars_dict

    def build_simulation(self, obj):
        simulation = self.f_(self.create_par_dict(obj))
        simulation.__disown__()
        return simulation
%}

%pythoncode %{
class ObserverCallbackWrapper(PyObserverCallback):
    def __init__(self, callback):
        super(ObserverCallbackWrapper, self).__init__()
        self.callback_ = callback

    def update(self, fit_objective):
        return self.callback_(fit_objective)

%}

%extend FitObjective {
%pythoncode %{
    def addSimulationAndData(self, callback, data, *args, **kwargs):
        """
        Sets simulation and experimental data to the fit objective.
        Optionally accepts experimental data uncertainties and
        user-defined dataset weight.

        Arguments:

        callback -- user-defined function returning fully-defined bornagain.ISimulation object.
        The function must use fit parameter dictionary as its input.

        data -- numpy array with experimental data.

        uncertainties -- numpy array with experimental data uncertainties.
        Array shape must correspond to the shape of data. Optional argument.

        weight -- user-defined weight of the dataset. If not specified, defaults to 1.0.
        """
        if not hasattr(self, 'callback_container'):
            self.callback_container = []
        wrp = SimulationBuilderWrapper(callback)
        self.callback_container.append(wrp)
        return self.addSimulationAndData_cpp(wrp, data, *args, **kwargs)

    def convert_params(self, params):
        """
        Converts parameters to what FitObjective::evaluate expects
        """

        if str(params.__module__) == "lmfit.parameter":
            bapars = libBornAgainFit.Parameters()
            for p in params:
                bapars.add(p, params[p].value)
            return bapars
        else:
            return params

    def evaluate_residuals(self, params):
        return self.evaluate_residuals_cpp(self.convert_params(params))

    def evaluate(self, params):
        return self.evaluate_cpp(self.convert_params(params))

    def convert_result(self, minim_result):
        """
        Converts result reported by arbitrary minimizer to ba.MinimizerResult
        """

        if str(minim_result.__module__) == "lmfit.minimizer":
            return libBornAgainFit.MinimizerResult()
        else:
            return minim_result

    def finalize(self, minimizer_result):
        return self.finalize_cpp(self.convert_result(minimizer_result))

    def initPlot(self, every_nth, callback):
        self.wrp_plot_observer = ObserverCallbackWrapper(callback)
        return self.initPlot_cpp(every_nth, self.wrp_plot_observer)

    def uncertainties(self):
        """
        Returns one-dimensional array representing merged data uncertainties.
        If any of the associated data pairs lack uncertainties, returns None.
        """
        if self.allPairsHaveUncertainties_cpp():
            return self.uncertainties_cpp()
        return None

    def uncertaintyData(self, i=0):
        """
        Returns uncertainties for i-th simulation-data pair. If
        no uncertainties are assigned to the data pair, returns
        None.
        """
        if self.containsUncertainties_cpp(i):
            return self.uncertaintyData_cpp(i)
        return None
%}
};
