/*
 * Copyright (c) 2019-2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>

namespace arm_gemm {

void a64_hybrid_u8u32_dot_6x16_a55 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const uint8_t *B_ptr, IndirectOutputArg<uint32_t> output_arg,
    const uint32_t *, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 171f\n"
      "cmp %x[M], #0x4\n"
      "bgt 137f\n"
      "beq 103f\n"
      "cmp %x[M], #0x2\n"
      "bgt 69f\n"
      "beq 35f\n"
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "2:"  // Height 1: Column loop
      "tbz %x[flags], #0, 12f\n"
      "cmp x8, #0x10\n"
      "bge 11f\n"
      "tbz x8, #3, 6f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "tbz x8, #2, 4f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "tbz x8, #1, 3f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x8, #0, 10f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "b 10f\n"
      "3:"  // Height 1: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 10f\n"
      "ldr s11, [x17, #0x0]\n"
      "b 10f\n"
      "4:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x8, #1, 5f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x8, #0, 10f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "b 10f\n"
      "5:"  // Height 1: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 10f\n"
      "ldr s10, [x17, #0x0]\n"
      "b 10f\n"
      "6:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x8, #2, 8f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "tbz x8, #1, 7f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x8, #0, 10f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "b 10f\n"
      "7:"  // Height 1: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 10f\n"
      "ldr s9, [x17, #0x0]\n"
      "b 10f\n"
      "8:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x8, #1, 9f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x8, #0, 10f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "b 10f\n"
      "9:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "10:"  // Height 1: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 13f\n"
      "11:"  // Height 1: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "b 13f\n"
      "12:"  // Height 1: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "13:"  // Height 1: setup done
      "mov x15, #0x0\n"
      "14:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 15f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "cbnz x15, 16f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "b 16f\n"
      "15:"  // Height 1: setup direct input
      "mov x13, %x[input_ptr]\n"
      "16:"  // Height 1: input setup done
      "cmp x14, #0x10\n"
      "blt 19f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 18f\n"
      "17:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr d17, [x16, #0x20]\n"
      "ldr x20, [x16, #0x28]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr d16, [x16, #0x30]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x38]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f80e22a  // udot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr d17, [x16, #0x40]\n"
      "ldr x20, [x16, #0x48]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr d16, [x16, #0x50]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x58]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa0e228  // udot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr d17, [x16, #0x60]\n"
      "ldr x20, [x16, #0x68]\n"
      ".inst 0x6fa0e209  // udot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr d16, [x16, #0x70]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x78]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa0e22a  // udot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr d17, [x16, #0x80]\n"
      "ldr x20, [x16, #0x88]\n"
      ".inst 0x6fa0e20b  // udot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr d16, [x16, #0x90]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x98]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f80ea28  // udot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr d17, [x16, #0xa0]\n"
      "ldr x20, [x16, #0xa8]\n"
      ".inst 0x6f80ea09  // udot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr d16, [x16, #0xb0]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0xb8]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f80ea2a  // udot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr d17, [x16, #0xc0]\n"
      "ldr x20, [x16, #0xc8]\n"
      ".inst 0x6f80ea0b  // udot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr d16, [x16, #0xd0]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0xd8]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa0ea28  // udot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr d17, [x16, #0xe0]\n"
      "ldr x20, [x16, #0xe8]\n"
      ".inst 0x6fa0ea09  // udot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr d16, [x16, #0xf0]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0xf8]\n"
      "mov v16.d[1], x20\n"
      "add x13, x13, #0x10\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0ea2a  // udot v10.4s, v17.16b, v0.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      "ldr x20, [x16, #0x8]\n"
      ".inst 0x6fa0ea0b  // udot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      "sub x14, x14, #0x10\n"
      "ldr d7, [x16, #0x10]\n"
      "cmp x14, #0x20\n"
      "ldr x21, [x13, #0x8]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x16, #0x18]\n"
      "mov v0.d[1], x21\n"
      "mov v7.d[1], x20\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "bge 17b\n"
      "18:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q17, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f80e22a  // udot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x16, #0x40]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x50]\n"
      ".inst 0x6fa0e228  // udot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x16, #0x60]\n"
      ".inst 0x6fa0e209  // udot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x16, #0x70]\n"
      ".inst 0x6fa0e22a  // udot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x16, #0x80]\n"
      ".inst 0x6fa0e20b  // udot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x16, #0x90]\n"
      ".inst 0x6f80ea28  // udot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x16, #0xa0]\n"
      ".inst 0x6f80ea09  // udot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x16, #0xb0]\n"
      ".inst 0x6f80ea2a  // udot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x16, #0xc0]\n"
      ".inst 0x6f80ea0b  // udot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x16, #0xd0]\n"
      ".inst 0x6fa0ea28  // udot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr q17, [x16, #0xe0]\n"
      ".inst 0x6fa0ea09  // udot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr q16, [x16, #0xf0]\n"
      "add x13, x13, #0x10\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6fa0ea2a  // udot v10.4s, v17.16b, v0.4b[3]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6fa0ea0b  // udot v11.4s, v16.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      "19:"  // Height 1: Multiply loop: Main loop skip
      "cbz x14, 24f\n"
      "cmp x14, #0x4\n"
      "blt 21f\n"
      "20:"  // Height 1: Multiply loop: Odd block loop
      "ldr s18, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr q16, [x16, #0x0]\n"
      ".inst 0x6f92e208  // udot v8.4s, v16.16b, v18.4b[0]\n"
      "ldr q16, [x16, #0x10]\n"
      ".inst 0x6f92e209  // udot v9.4s, v16.16b, v18.4b[0]\n"
      "ldr q17, [x16, #0x20]\n"
      "cmp x14, #0x4\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f92e22a  // udot v10.4s, v17.16b, v18.4b[0]\n"
      ".inst 0x6f92e20b  // udot v11.4s, v16.16b, v18.4b[0]\n"
      "add x16, x16, #0x40\n"
      "bge 20b\n"
      "21:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x14, 24f\n"
      "tbz x14, #1, 22f\n"
      "ldr h0, [x13], #0x2\n"
      "tbz x14, #0, 23f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "b 23f\n"
      "22:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "23:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q16, [x16, #0x0]\n"
      ".inst 0x6f80e208  // udot v8.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x10]\n"
      ".inst 0x6f80e209  // udot v9.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x20]\n"
      ".inst 0x6f80e20a  // udot v10.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      "24:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 14b\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "bge 33f\n"
      "tbz x8, #3, 28f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "tbz x8, #2, 26f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "tbz x8, #1, 25f\n"
      "str d11, [x17], #0x8\n"
      "tbz x8, #0, 32f\n"
      "st1 { v11.s }[2], [x17]\n"
      "b 32f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x8, #0, 32f\n"
      "str s11, [x17, #0x0]\n"
      "b 32f\n"
      "26:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x8, #1, 27f\n"
      "str d10, [x17], #0x8\n"
      "tbz x8, #0, 32f\n"
      "st1 { v10.s }[2], [x17]\n"
      "b 32f\n"
      "27:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x8, #0, 32f\n"
      "str s10, [x17, #0x0]\n"
      "b 32f\n"
      "28:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x8, #2, 30f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "tbz x8, #1, 29f\n"
      "str d9, [x17], #0x8\n"
      "tbz x8, #0, 32f\n"
      "st1 { v9.s }[2], [x17]\n"
      "b 32f\n"
      "29:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x8, #0, 32f\n"
      "str s9, [x17, #0x0]\n"
      "b 32f\n"
      "30:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x8, #1, 31f\n"
      "str d8, [x17], #0x8\n"
      "tbz x8, #0, 32f\n"
      "st1 { v8.s }[2], [x17]\n"
      "b 32f\n"
      "31:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "32:"  // Height 1: Partial direct writeback: Done
      "b 34f\n"
      "33:"  // Height 1: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "34:"  // Height 1: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 2b\n"
      "b 206f\n"
      "35:"  // Height 2
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "36:"  // Height 2: Column loop
      "tbz %x[flags], #0, 46f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x8, #0x10\n"
      "add x24, x17, x20, LSL #2\n"
      "bge 45f\n"
      "tbz x8, #3, 40f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "tbz x8, #2, 38f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "tbz x8, #1, 37f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "tbz x8, #0, 44f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "b 44f\n"
      "37:"  // Height 2: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 44f\n"
      "ldr s11, [x17, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "b 44f\n"
      "38:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x8, #1, 39f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "tbz x8, #0, 44f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "b 44f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 44f\n"
      "ldr s10, [x17, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "b 44f\n"
      "40:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x8, #2, 42f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "tbz x8, #1, 41f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "tbz x8, #0, 44f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "b 44f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 44f\n"
      "ldr s9, [x17, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "b 44f\n"
      "42:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x8, #1, 43f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "tbz x8, #0, 44f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "b 44f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s12, [x24, #0x0]\n"
      "44:"  // Height 2: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 47f\n"
      "45:"  // Height 2: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "b 47f\n"
      "46:"  // Height 2: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "47:"  // Height 2: setup done
      "mov x15, #0x0\n"
      "48:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 49f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "ldr x12, [x20, #0x8]\n"
      "cbnz x15, 50f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "b 50f\n"
      "49:"  // Height 2: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "50:"  // Height 2: input setup done
      "cmp x14, #0x10\n"
      "blt 53f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 52f\n"
      "51:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x16, #0x28]\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr d17, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr x20, [x16, #0x38]\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr d16, [x16, #0x30]\n"
      "mov v17.d[1], x21\n"
      ".inst 0x6f80e22a  // udot v10.4s, v17.16b, v0.4b[0]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f81e22e  // udot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr d17, [x16, #0x40]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr x20, [x16, #0x48]\n"
      ".inst 0x6f81e20f  // udot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr d16, [x16, #0x50]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x58]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa0e228  // udot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr x21, [x16, #0x68]\n"
      ".inst 0x6fa1e22c  // udot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr d17, [x16, #0x60]\n"
      ".inst 0x6fa0e209  // udot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr x20, [x16, #0x78]\n"
      ".inst 0x6fa1e20d  // udot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr d16, [x16, #0x70]\n"
      "mov v17.d[1], x21\n"
      ".inst 0x6fa0e22a  // udot v10.4s, v17.16b, v0.4b[1]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa1e22e  // udot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr d17, [x16, #0x80]\n"
      ".inst 0x6fa0e20b  // udot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr x20, [x16, #0x88]\n"
      ".inst 0x6fa1e20f  // udot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr d16, [x16, #0x90]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0x98]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f80ea28  // udot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr x21, [x16, #0xa8]\n"
      ".inst 0x6f81ea2c  // udot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr d17, [x16, #0xa0]\n"
      ".inst 0x6f80ea09  // udot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr x20, [x16, #0xb8]\n"
      ".inst 0x6f81ea0d  // udot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr d16, [x16, #0xb0]\n"
      "mov v17.d[1], x21\n"
      ".inst 0x6f80ea2a  // udot v10.4s, v17.16b, v0.4b[2]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6f81ea2e  // udot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr d17, [x16, #0xc0]\n"
      ".inst 0x6f80ea0b  // udot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr x20, [x16, #0xc8]\n"
      ".inst 0x6f81ea0f  // udot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr d16, [x16, #0xd0]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x16, #0xd8]\n"
      "mov v16.d[1], x20\n"
      ".inst 0x6fa0ea28  // udot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr x21, [x16, #0xe8]\n"
      ".inst 0x6fa1ea2c  // udot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr d17, [x16, #0xe0]\n"
      ".inst 0x6fa0ea09  // udot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr x20, [x16, #0xf8]\n"
      ".inst 0x6fa1ea0d  // udot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr d16, [x16, #0xf0]\n"
      "mov v17.d[1], x21\n"
      "add x13, x13, #0x10\n"
      "mov v16.d[1], x20\n"
      "add x12, x12, #0x10\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0ea2a  // udot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea2e  // udot v14.4s, v17.16b, v1.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      "ldr x21, [x16, #0x8]\n"
      ".inst 0x6fa0ea0b  // udot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      ".inst 0x6fa1ea0f  // udot v15.4s, v16.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0x0]\n"
      "sub x14, x14, #0x10\n"
      "ldr d7, [x16, #0x10]\n"
      "cmp x14, #0x20\n"
      "ldr x20, [x13, #0x8]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x12, #0x8]\n"
      "mov v0.d[1], x20\n"
      "ldr x20, [x16, #0x18]\n"
      "mov v1.d[1], x21\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "mov v7.d[1], x20\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      "bge 51b\n"
      "52:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q17, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f80e22a  // udot v10.4s, v17.16b, v0.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f81e22e  // udot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x16, #0x40]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f81e20f  // udot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x16, #0x50]\n"
      ".inst 0x6fa0e228  // udot v8.4s, v17.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6fa1e22c  // udot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x16, #0x60]\n"
      ".inst 0x6fa0e209  // udot v9.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x6fa1e20d  // udot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x16, #0x70]\n"
      ".inst 0x6fa0e22a  // udot v10.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x6fa1e22e  // udot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x16, #0x80]\n"
      ".inst 0x6fa0e20b  // udot v11.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x6fa1e20f  // udot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x16, #0x90]\n"
      ".inst 0x6f80ea28  // udot v8.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x6f81ea2c  // udot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x16, #0xa0]\n"
      ".inst 0x6f80ea09  // udot v9.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x6f81ea0d  // udot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x16, #0xb0]\n"
      ".inst 0x6f80ea2a  // udot v10.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x6f81ea2e  // udot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x16, #0xc0]\n"
      ".inst 0x6f80ea0b  // udot v11.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x6f81ea0f  // udot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x16, #0xd0]\n"
      ".inst 0x6fa0ea28  // udot v8.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea2c  // udot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr q17, [x16, #0xe0]\n"
      ".inst 0x6fa0ea09  // udot v9.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea0d  // udot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr q16, [x16, #0xf0]\n"
      ".inst 0x6fa0ea2a  // udot v10.4s, v17.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa1ea2e  // udot v14.4s, v17.16b, v1.4b[3]\n"
      ".inst 0x6fa0ea0b  // udot v11.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea0f  // udot v15.4s, v16.16b, v1.4b[3]\n"
      "53:"  // Height 2: Multiply loop: Main loop skip
      "cbz x14, 58f\n"
      "cmp x14, #0x4\n"
      "blt 55f\n"
      "54:"  // Height 2: Multiply loop: Odd block loop
      "ldr s19, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s18, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr q17, [x16, #0x0]\n"
      ".inst 0x6f93e228  // udot v8.4s, v17.16b, v19.4b[0]\n"
      "ldr q16, [x16, #0x10]\n"
      ".inst 0x6f92e22c  // udot v12.4s, v17.16b, v18.4b[0]\n"
      "ldr q17, [x16, #0x20]\n"
      ".inst 0x6f93e209  // udot v9.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x6f92e20d  // udot v13.4s, v16.16b, v18.4b[0]\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f93e22a  // udot v10.4s, v17.16b, v19.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f92e22e  // udot v14.4s, v17.16b, v18.4b[0]\n"
      ".inst 0x6f93e20b  // udot v11.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x6f92e20f  // udot v15.4s, v16.16b, v18.4b[0]\n"
      "bge 54b\n"
      "55:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x14, 58f\n"
      "tbz x14, #1, 56f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "tbz x14, #0, 57f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "b 57f\n"
      "56:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "ldr b1, [x12, #0x0]\n"
      "57:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q17, [x16, #0x0]\n"
      ".inst 0x6f80e228  // udot v8.4s, v17.16b, v0.4b[0]\n"
      "ldr q16, [x16, #0x10]\n"
      ".inst 0x6f81e22c  // udot v12.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x16, #0x20]\n"
      ".inst 0x6f80e209  // udot v9.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x6f81e20d  // udot v13.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x16, #0x30]\n"
      ".inst 0x6f80e22a  // udot v10.4s, v17.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e22e  // udot v14.4s, v17.16b, v1.4b[0]\n"
      ".inst 0x6f80e20b  // udot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x6f81e20f  // udot v15.4s, v16.16b, v1.4b[0]\n"
      "58:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 48b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "bge 67f\n"
      "tbz x8, #3, 62f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "tbz x8, #2, 60f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "tbz x8, #1, 59f\n"
      "str d11, [x17], #0x8\n"
      "str d15, [x24], #0x8\n"
      "tbz x8, #0, 66f\n"
      "st1 { v11.s }[2], [x17]\n"
      "st1 { v15.s }[2], [x24]\n"
      "b 66f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x8, #0, 66f\n"
      "str s11, [x17, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "b 66f\n"
      "60:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x8, #1, 61f\n"
      "str d10, [x17], #0x8\n"
      "str d14, [x24], #0x8\n"
      "tbz x8, #0, 66f\n"
      "st1 { v10.s }[2], [x17]\n"
      "st1 { v14.s }[2], [x24]\n"
      "b 66f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x8, #0, 66f\n"
      "str s10, [x17, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "b 66f\n"
      "62:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x8, #2, 64f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "tbz x8, #1, 63f\n"
      "str d9, [x17], #0x8\n"
      "str d13, [x24], #0x8\n"
      "tbz x8, #0, 66f\n"
      "st1 { v9.s }[2], [x17]\n"
      "st1 { v13.s }[2], [x24]\n"
      "b 66f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x8, #0, 66f\n"
      "str s9, [x17, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "b 66f\n"
      "64:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x8, #1, 65f\n"
      "str d8, [x17], #0x8\n"
      "str d12, [x24], #0x8\n"
      "tbz x8, #0, 66f\n"
      "st1 { v8.s }[2], [x17]\n"
      "st1 { v12.s }[2], [x24]\n"
      "b 66f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "66:"  // Height 2: Partial direct writeback: Done
      "b 68f\n"
      "67:"  // Height 2: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "68:"  // Height 2: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 36b\n"
      "b 206f\n"
      "69:"  // Height 3
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "70:"  // Height 3: Column loop
      "tbz %x[flags], #0, 80f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 79f\n"
      "tbz x8, #3, 74f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x8, #2, 72f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x8, #1, 71f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x8, #0, 78f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 78f\n"
      "71:"  // Height 3: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 78f\n"
      "ldr s11, [x17, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "b 78f\n"
      "72:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x8, #1, 73f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x8, #0, 78f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 78f\n"
      "73:"  // Height 3: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 78f\n"
      "ldr s10, [x17, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "b 78f\n"
      "74:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x8, #2, 76f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "tbz x8, #1, 75f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x8, #0, 78f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 78f\n"
      "75:"  // Height 3: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 78f\n"
      "ldr s9, [x17, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "b 78f\n"
      "76:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x8, #1, 77f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "tbz x8, #0, 78f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "b 78f\n"
      "77:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s12, [x24, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "78:"  // Height 3: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 81f\n"
      "79:"  // Height 3: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "b 81f\n"
      "80:"  // Height 3: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "81:"  // Height 3: setup done
      "mov x15, #0x0\n"
      "82:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 83f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "cbnz x15, 84f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "b 84f\n"
      "83:"  // Height 3: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "84:"  // Height 3: input setup done
      "cmp x14, #0x10\n"
      "blt 87f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q2, [x11, #0x0]\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 86f\n"
      "85:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x16, #0x28]\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x38]\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr d21, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x48]\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr d20, [x16, #0x30]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6f80e2aa  // udot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x6f81e2ae  // udot v14.4s, v21.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x58]\n"
      ".inst 0x6f82e2b2  // udot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr d21, [x16, #0x40]\n"
      ".inst 0x6f80e28b  // udot v11.4s, v20.16b, v0.4b[0]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6f81e28f  // udot v15.4s, v20.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x68]\n"
      ".inst 0x6f82e293  // udot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr d20, [x16, #0x50]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6fa0e2a8  // udot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x6fa1e2ac  // udot v12.4s, v21.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x78]\n"
      ".inst 0x6fa2e2b0  // udot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr d21, [x16, #0x60]\n"
      ".inst 0x6fa0e289  // udot v9.4s, v20.16b, v0.4b[1]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6fa1e28d  // udot v13.4s, v20.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0x88]\n"
      ".inst 0x6fa2e291  // udot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr d20, [x16, #0x70]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6fa0e2aa  // udot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x6fa1e2ae  // udot v14.4s, v21.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x98]\n"
      ".inst 0x6fa2e2b2  // udot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr d21, [x16, #0x80]\n"
      ".inst 0x6fa0e28b  // udot v11.4s, v20.16b, v0.4b[1]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6fa1e28f  // udot v15.4s, v20.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0xa8]\n"
      ".inst 0x6fa2e293  // udot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr d20, [x16, #0x90]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6f80eaa8  // udot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x6f81eaac  // udot v12.4s, v21.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xb8]\n"
      ".inst 0x6f82eab0  // udot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr d21, [x16, #0xa0]\n"
      ".inst 0x6f80ea89  // udot v9.4s, v20.16b, v0.4b[2]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6f81ea8d  // udot v13.4s, v20.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xc8]\n"
      ".inst 0x6f82ea91  // udot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr d20, [x16, #0xb0]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6f80eaaa  // udot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x6f81eaae  // udot v14.4s, v21.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xd8]\n"
      ".inst 0x6f82eab2  // udot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr d21, [x16, #0xc0]\n"
      ".inst 0x6f80ea8b  // udot v11.4s, v20.16b, v0.4b[2]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6f81ea8f  // udot v15.4s, v20.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xe8]\n"
      ".inst 0x6f82ea93  // udot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr d20, [x16, #0xd0]\n"
      "mov v20.d[1], x20\n"
      ".inst 0x6fa0eaa8  // udot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x6fa1eaac  // udot v12.4s, v21.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0xf8]\n"
      ".inst 0x6fa2eab0  // udot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr d21, [x16, #0xe0]\n"
      ".inst 0x6fa0ea89  // udot v9.4s, v20.16b, v0.4b[3]\n"
      "mov v21.d[1], x21\n"
      ".inst 0x6fa1ea8d  // udot v13.4s, v20.16b, v1.4b[3]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6fa2ea91  // udot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr d20, [x16, #0xf0]\n"
      "mov v20.d[1], x20\n"
      "add x12, x12, #0x10\n"
      "add x11, x11, #0x10\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0eaaa  // udot v10.4s, v21.16b, v0.4b[3]\n"
      "ldr x20, [x16, #0x8]\n"
      ".inst 0x6fa1eaae  // udot v14.4s, v21.16b, v1.4b[3]\n"
      "ldr x23, [x13, #0x8]\n"
      ".inst 0x6fa2eab2  // udot v18.4s, v21.16b, v2.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      ".inst 0x6fa0ea8b  // udot v11.4s, v20.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      ".inst 0x6fa1ea8f  // udot v15.4s, v20.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0x0]\n"
      "ldr x22, [x12, #0x8]\n"
      ".inst 0x6fa2ea93  // udot v19.4s, v20.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0x0]\n"
      "sub x14, x14, #0x10\n"
      "ldr d7, [x16, #0x10]\n"
      "cmp x14, #0x20\n"
      "ldr x21, [x11, #0x8]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x16, #0x18]\n"
      "mov v0.d[1], x23\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "mov v1.d[1], x22\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      "mov v2.d[1], x21\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "mov v7.d[1], x20\n"
      "bge 85b\n"
      "86:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q21, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q20, [x16, #0x30]\n"
      ".inst 0x6f80e2aa  // udot v10.4s, v21.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f81e2ae  // udot v14.4s, v21.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6f82e2b2  // udot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x16, #0x40]\n"
      ".inst 0x6f80e28b  // udot v11.4s, v20.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6f81e28f  // udot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x6f82e293  // udot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x16, #0x50]\n"
      ".inst 0x6fa0e2a8  // udot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x6fa1e2ac  // udot v12.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x6fa2e2b0  // udot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x16, #0x60]\n"
      ".inst 0x6fa0e289  // udot v9.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x6fa1e28d  // udot v13.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x6fa2e291  // udot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x16, #0x70]\n"
      ".inst 0x6fa0e2aa  // udot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x6fa1e2ae  // udot v14.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x6fa2e2b2  // udot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x16, #0x80]\n"
      ".inst 0x6fa0e28b  // udot v11.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x6fa1e28f  // udot v15.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x6fa2e293  // udot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x16, #0x90]\n"
      ".inst 0x6f80eaa8  // udot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x6f81eaac  // udot v12.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x6f82eab0  // udot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x16, #0xa0]\n"
      ".inst 0x6f80ea89  // udot v9.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x6f81ea8d  // udot v13.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x6f82ea91  // udot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x16, #0xb0]\n"
      ".inst 0x6f80eaaa  // udot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x6f81eaae  // udot v14.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x6f82eab2  // udot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x16, #0xc0]\n"
      ".inst 0x6f80ea8b  // udot v11.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x6f81ea8f  // udot v15.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x6f82ea93  // udot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x16, #0xd0]\n"
      ".inst 0x6fa0eaa8  // udot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x6fa1eaac  // udot v12.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x6fa2eab0  // udot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr q21, [x16, #0xe0]\n"
      ".inst 0x6fa0ea89  // udot v9.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea8d  // udot v13.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x6fa2ea91  // udot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr q20, [x16, #0xf0]\n"
      ".inst 0x6fa0eaaa  // udot v10.4s, v21.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa1eaae  // udot v14.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x6fa2eab2  // udot v18.4s, v21.16b, v2.4b[3]\n"
      ".inst 0x6fa0ea8b  // udot v11.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x6fa1ea8f  // udot v15.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x6fa2ea93  // udot v19.4s, v20.16b, v2.4b[3]\n"
      "87:"  // Height 3: Multiply loop: Main loop skip
      "cbz x14, 92f\n"
      "cmp x14, #0x4\n"
      "blt 89f\n"
      "88:"  // Height 3: Multiply loop: Odd block loop
      "ldr s24, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s23, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s22, [x11], #0x4\n"
      "ldr q21, [x16, #0x0]\n"
      ".inst 0x6f98e2a8  // udot v8.4s, v21.16b, v24.4b[0]\n"
      "ldr q20, [x16, #0x10]\n"
      ".inst 0x6f97e2ac  // udot v12.4s, v21.16b, v23.4b[0]\n"
      ".inst 0x6f96e2b0  // udot v16.4s, v21.16b, v22.4b[0]\n"
      "ldr q21, [x16, #0x20]\n"
      ".inst 0x6f98e289  // udot v9.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x6f97e28d  // udot v13.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x6f96e291  // udot v17.4s, v20.16b, v22.4b[0]\n"
      "ldr q20, [x16, #0x30]\n"
      ".inst 0x6f98e2aa  // udot v10.4s, v21.16b, v24.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f97e2ae  // udot v14.4s, v21.16b, v23.4b[0]\n"
      ".inst 0x6f96e2b2  // udot v18.4s, v21.16b, v22.4b[0]\n"
      ".inst 0x6f98e28b  // udot v11.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x6f97e28f  // udot v15.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x6f96e293  // udot v19.4s, v20.16b, v22.4b[0]\n"
      "bge 88b\n"
      "89:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x14, 92f\n"
      "tbz x14, #1, 90f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "tbz x14, #0, 91f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "b 91f\n"
      "90:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "ldr b1, [x12, #0x0]\n"
      "ldr b2, [x11, #0x0]\n"
      "91:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q21, [x16, #0x0]\n"
      ".inst 0x6f80e2a8  // udot v8.4s, v21.16b, v0.4b[0]\n"
      "ldr q20, [x16, #0x10]\n"
      ".inst 0x6f81e2ac  // udot v12.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x6f82e2b0  // udot v16.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x16, #0x20]\n"
      ".inst 0x6f80e289  // udot v9.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x6f81e28d  // udot v13.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x6f82e291  // udot v17.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x16, #0x30]\n"
      ".inst 0x6f80e2aa  // udot v10.4s, v21.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e2ae  // udot v14.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x6f82e2b2  // udot v18.4s, v21.16b, v2.4b[0]\n"
      ".inst 0x6f80e28b  // udot v11.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x6f81e28f  // udot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x6f82e293  // udot v19.4s, v20.16b, v2.4b[0]\n"
      "92:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 82b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "bge 101f\n"
      "tbz x8, #3, 96f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x8, #2, 94f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x8, #1, 93f\n"
      "str d11, [x17], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x8, #0, 100f\n"
      "st1 { v11.s }[2], [x17]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 100f\n"
      "93:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x8, #0, 100f\n"
      "str s11, [x17, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "b 100f\n"
      "94:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x8, #1, 95f\n"
      "str d10, [x17], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x8, #0, 100f\n"
      "st1 { v10.s }[2], [x17]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 100f\n"
      "95:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x8, #0, 100f\n"
      "str s10, [x17, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "b 100f\n"
      "96:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x8, #2, 98f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x8, #1, 97f\n"
      "str d9, [x17], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x8, #0, 100f\n"
      "st1 { v9.s }[2], [x17]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 100f\n"
      "97:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x8, #0, 100f\n"
      "str s9, [x17, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "b 100f\n"
      "98:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x8, #1, 99f\n"
      "str d8, [x17], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x8, #0, 100f\n"
      "st1 { v8.s }[2], [x17]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 100f\n"
      "99:"  // Height 3: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "100:"  // Height 3: Partial direct writeback: Done
      "b 102f\n"
      "101:"  // Height 3: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "102:"  // Height 3: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 70b\n"
      "b 206f\n"
      "103:"  // Height 4
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "104:"  // Height 4: Column loop
      "tbz %x[flags], #0, 114f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 113f\n"
      "tbz x8, #3, 108f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x8, #2, 106f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x8, #1, 105f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x8, #0, 112f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 112f\n"
      "105:"  // Height 4: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 112f\n"
      "ldr s11, [x17, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "b 112f\n"
      "106:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x8, #1, 107f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x8, #0, 112f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 112f\n"
      "107:"  // Height 4: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 112f\n"
      "ldr s10, [x17, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "b 112f\n"
      "108:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x8, #2, 110f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x8, #1, 109f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x8, #0, 112f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 112f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 112f\n"
      "ldr s9, [x17, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "b 112f\n"
      "110:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x8, #1, 111f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x8, #0, 112f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 112f\n"
      "111:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s12, [x24, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "112:"  // Height 4: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 115f\n"
      "113:"  // Height 4: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "b 115f\n"
      "114:"  // Height 4: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "115:"  // Height 4: setup done
      "mov x15, #0x0\n"
      "116:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 117f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "cbnz x15, 118f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "b 118f\n"
      "117:"  // Height 4: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "118:"  // Height 4: input setup done
      "cmp x14, #0x10\n"
      "blt 121f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q2, [x11, #0x0]\n"
      "ldr q3, [x10, #0x0]\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 120f\n"
      "119:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x16, #0x28]\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x38]\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr d25, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x48]\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr d24, [x16, #0x30]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6f80e32a  // udot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x6f81e32e  // udot v14.4s, v25.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x58]\n"
      ".inst 0x6f82e332  // udot v18.4s, v25.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f83e336  // udot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr d25, [x16, #0x40]\n"
      ".inst 0x6f80e30b  // udot v11.4s, v24.16b, v0.4b[0]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6f81e30f  // udot v15.4s, v24.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x68]\n"
      ".inst 0x6f82e313  // udot v19.4s, v24.16b, v2.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f83e317  // udot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr d24, [x16, #0x50]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6fa0e328  // udot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x6fa1e32c  // udot v12.4s, v25.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x78]\n"
      ".inst 0x6fa2e330  // udot v16.4s, v25.16b, v2.4b[1]\n"
      "ldr x25, [x13, #0x8]\n"
      ".inst 0x6fa3e334  // udot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr d25, [x16, #0x60]\n"
      ".inst 0x6fa0e309  // udot v9.4s, v24.16b, v0.4b[1]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6fa1e30d  // udot v13.4s, v24.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0x88]\n"
      ".inst 0x6fa2e311  // udot v17.4s, v24.16b, v2.4b[1]\n"
      "ldr x24, [x12, #0x8]\n"
      ".inst 0x6fa3e315  // udot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr d24, [x16, #0x70]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6fa0e32a  // udot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x6fa1e32e  // udot v14.4s, v25.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x98]\n"
      ".inst 0x6fa2e332  // udot v18.4s, v25.16b, v2.4b[1]\n"
      "ldr x23, [x11, #0x8]\n"
      ".inst 0x6fa3e336  // udot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr d25, [x16, #0x80]\n"
      ".inst 0x6fa0e30b  // udot v11.4s, v24.16b, v0.4b[1]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6fa1e30f  // udot v15.4s, v24.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0xa8]\n"
      ".inst 0x6fa2e313  // udot v19.4s, v24.16b, v2.4b[1]\n"
      "ldr x22, [x10, #0x8]\n"
      ".inst 0x6fa3e317  // udot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr d24, [x16, #0x90]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6f80eb28  // udot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x6f81eb2c  // udot v12.4s, v25.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xb8]\n"
      ".inst 0x6f82eb30  // udot v16.4s, v25.16b, v2.4b[2]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f83eb34  // udot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr d25, [x16, #0xa0]\n"
      ".inst 0x6f80eb09  // udot v9.4s, v24.16b, v0.4b[2]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6f81eb0d  // udot v13.4s, v24.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xc8]\n"
      ".inst 0x6f82eb11  // udot v17.4s, v24.16b, v2.4b[2]\n"
      "cmp x14, #0x20\n"
      ".inst 0x6f83eb15  // udot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr d24, [x16, #0xb0]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6f80eb2a  // udot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x6f81eb2e  // udot v14.4s, v25.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xd8]\n"
      ".inst 0x6f82eb32  // udot v18.4s, v25.16b, v2.4b[2]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f83eb36  // udot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr d25, [x16, #0xc0]\n"
      ".inst 0x6f80eb0b  // udot v11.4s, v24.16b, v0.4b[2]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6f81eb0f  // udot v15.4s, v24.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xe8]\n"
      ".inst 0x6f82eb13  // udot v19.4s, v24.16b, v2.4b[2]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6f83eb17  // udot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr d24, [x16, #0xd0]\n"
      "mov v24.d[1], x20\n"
      ".inst 0x6fa0eb28  // udot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb2c  // udot v12.4s, v25.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0xf8]\n"
      ".inst 0x6fa2eb30  // udot v16.4s, v25.16b, v2.4b[3]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6fa3eb34  // udot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr d25, [x16, #0xe0]\n"
      ".inst 0x6fa0eb09  // udot v9.4s, v24.16b, v0.4b[3]\n"
      "mov v25.d[1], x21\n"
      ".inst 0x6fa1eb0d  // udot v13.4s, v24.16b, v1.4b[3]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6fa2eb11  // udot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb15  // udot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr d24, [x16, #0xf0]\n"
      "mov v24.d[1], x20\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0eb2a  // udot v10.4s, v25.16b, v0.4b[3]\n"
      "ldr x21, [x16, #0x8]\n"
      ".inst 0x6fa1eb2e  // udot v14.4s, v25.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0x18]\n"
      ".inst 0x6fa2eb32  // udot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb36  // udot v22.4s, v25.16b, v3.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      ".inst 0x6fa0eb0b  // udot v11.4s, v24.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      ".inst 0x6fa1eb0f  // udot v15.4s, v24.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0x0]\n"
      ".inst 0x6fa2eb13  // udot v19.4s, v24.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0x0]\n"
      ".inst 0x6fa3eb17  // udot v23.4s, v24.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0x0]\n"
      "ldr d7, [x16, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x25\n"
      "mov v1.d[1], x24\n"
      "mov v2.d[1], x23\n"
      "mov v3.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 119b\n"
      "120:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q25, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q24, [x16, #0x30]\n"
      ".inst 0x6f80e32a  // udot v10.4s, v25.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6f81e32e  // udot v14.4s, v25.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6f82e332  // udot v18.4s, v25.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6f83e336  // udot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x16, #0x40]\n"
      ".inst 0x6f80e30b  // udot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x6f81e30f  // udot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x6f82e313  // udot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x6f83e317  // udot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x16, #0x50]\n"
      ".inst 0x6fa0e328  // udot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x6fa1e32c  // udot v12.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x6fa2e330  // udot v16.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x6fa3e334  // udot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x16, #0x60]\n"
      ".inst 0x6fa0e309  // udot v9.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x6fa1e30d  // udot v13.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x6fa2e311  // udot v17.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x6fa3e315  // udot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x16, #0x70]\n"
      ".inst 0x6fa0e32a  // udot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x6fa1e32e  // udot v14.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x6fa2e332  // udot v18.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x6fa3e336  // udot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x16, #0x80]\n"
      ".inst 0x6fa0e30b  // udot v11.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x6fa1e30f  // udot v15.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x6fa2e313  // udot v19.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x6fa3e317  // udot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x16, #0x90]\n"
      ".inst 0x6f80eb28  // udot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x6f81eb2c  // udot v12.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x6f82eb30  // udot v16.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x6f83eb34  // udot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x16, #0xa0]\n"
      ".inst 0x6f80eb09  // udot v9.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x6f81eb0d  // udot v13.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x6f82eb11  // udot v17.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x6f83eb15  // udot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x16, #0xb0]\n"
      ".inst 0x6f80eb2a  // udot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x6f81eb2e  // udot v14.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x6f82eb32  // udot v18.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x6f83eb36  // udot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x16, #0xc0]\n"
      ".inst 0x6f80eb0b  // udot v11.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x6f81eb0f  // udot v15.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x6f82eb13  // udot v19.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x6f83eb17  // udot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x16, #0xd0]\n"
      ".inst 0x6fa0eb28  // udot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb2c  // udot v12.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb30  // udot v16.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb34  // udot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr q25, [x16, #0xe0]\n"
      ".inst 0x6fa0eb09  // udot v9.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb0d  // udot v13.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb11  // udot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb15  // udot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr q24, [x16, #0xf0]\n"
      ".inst 0x6fa0eb2a  // udot v10.4s, v25.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa1eb2e  // udot v14.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb32  // udot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb36  // udot v22.4s, v25.16b, v3.4b[3]\n"
      ".inst 0x6fa0eb0b  // udot v11.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb0f  // udot v15.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb13  // udot v19.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb17  // udot v23.4s, v24.16b, v3.4b[3]\n"
      "121:"  // Height 4: Multiply loop: Main loop skip
      "cbz x14, 126f\n"
      "cmp x14, #0x4\n"
      "blt 123f\n"
      "122:"  // Height 4: Multiply loop: Odd block loop
      "ldr s29, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s28, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s27, [x11], #0x4\n"
      "ldr s26, [x10], #0x4\n"
      "ldr q25, [x16, #0x0]\n"
      ".inst 0x6f9de328  // udot v8.4s, v25.16b, v29.4b[0]\n"
      "ldr q24, [x16, #0x10]\n"
      ".inst 0x6f9ce32c  // udot v12.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x6f9be330  // udot v16.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x6f9ae334  // udot v20.4s, v25.16b, v26.4b[0]\n"
      "ldr q25, [x16, #0x20]\n"
      ".inst 0x6f9de309  // udot v9.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x6f9ce30d  // udot v13.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x6f9be311  // udot v17.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x6f9ae315  // udot v21.4s, v24.16b, v26.4b[0]\n"
      "ldr q24, [x16, #0x30]\n"
      ".inst 0x6f9de32a  // udot v10.4s, v25.16b, v29.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f9ce32e  // udot v14.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x6f9be332  // udot v18.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x6f9ae336  // udot v22.4s, v25.16b, v26.4b[0]\n"
      ".inst 0x6f9de30b  // udot v11.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x6f9ce30f  // udot v15.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x6f9be313  // udot v19.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x6f9ae317  // udot v23.4s, v24.16b, v26.4b[0]\n"
      "bge 122b\n"
      "123:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x14, 126f\n"
      "tbz x14, #1, 124f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "tbz x14, #0, 125f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "b 125f\n"
      "124:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "ldr b1, [x12, #0x0]\n"
      "ldr b2, [x11, #0x0]\n"
      "ldr b3, [x10, #0x0]\n"
      "125:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q25, [x16, #0x0]\n"
      ".inst 0x6f80e328  // udot v8.4s, v25.16b, v0.4b[0]\n"
      "ldr q24, [x16, #0x10]\n"
      ".inst 0x6f81e32c  // udot v12.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x6f82e330  // udot v16.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x6f83e334  // udot v20.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x16, #0x20]\n"
      ".inst 0x6f80e309  // udot v9.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x6f81e30d  // udot v13.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x6f82e311  // udot v17.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x6f83e315  // udot v21.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x16, #0x30]\n"
      ".inst 0x6f80e32a  // udot v10.4s, v25.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e32e  // udot v14.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x6f82e332  // udot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x6f83e336  // udot v22.4s, v25.16b, v3.4b[0]\n"
      ".inst 0x6f80e30b  // udot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x6f81e30f  // udot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x6f82e313  // udot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x6f83e317  // udot v23.4s, v24.16b, v3.4b[0]\n"
      "126:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 116b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "bge 135f\n"
      "tbz x8, #3, 130f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "tbz x8, #2, 128f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "tbz x8, #1, 127f\n"
      "str d11, [x17], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "tbz x8, #0, 134f\n"
      "st1 { v11.s }[2], [x17]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "b 134f\n"
      "127:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x8, #0, 134f\n"
      "str s11, [x17, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "b 134f\n"
      "128:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x8, #1, 129f\n"
      "str d10, [x17], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "tbz x8, #0, 134f\n"
      "st1 { v10.s }[2], [x17]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "b 134f\n"
      "129:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x8, #0, 134f\n"
      "str s10, [x17, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "b 134f\n"
      "130:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x8, #2, 132f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "tbz x8, #1, 131f\n"
      "str d9, [x17], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "tbz x8, #0, 134f\n"
      "st1 { v9.s }[2], [x17]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "b 134f\n"
      "131:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x8, #0, 134f\n"
      "str s9, [x17, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "b 134f\n"
      "132:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x8, #1, 133f\n"
      "str d8, [x17], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "tbz x8, #0, 134f\n"
      "st1 { v8.s }[2], [x17]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "b 134f\n"
      "133:"  // Height 4: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "134:"  // Height 4: Partial direct writeback: Done
      "b 136f\n"
      "135:"  // Height 4: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "136:"  // Height 4: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 104b\n"
      "b 206f\n"
      "137:"  // Height 5
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "138:"  // Height 5: Column loop
      "tbz %x[flags], #0, 148f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 147f\n"
      "tbz x8, #3, 142f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "tbz x8, #2, 140f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x8, #1, 139f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x8, #0, 146f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 146f\n"
      "139:"  // Height 5: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 146f\n"
      "ldr s11, [x17, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "b 146f\n"
      "140:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x8, #1, 141f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x8, #0, 146f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 146f\n"
      "141:"  // Height 5: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 146f\n"
      "ldr s10, [x17, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "b 146f\n"
      "142:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x8, #2, 144f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "tbz x8, #1, 143f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "tbz x8, #0, 146f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "b 146f\n"
      "143:"  // Height 5: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 146f\n"
      "ldr s9, [x17, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "b 146f\n"
      "144:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x8, #1, 145f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "tbz x8, #0, 146f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "b 146f\n"
      "145:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s12, [x24, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s24, [x21, #0x0]\n"
      "146:"  // Height 5: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 149f\n"
      "147:"  // Height 5: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0x0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "b 149f\n"
      "148:"  // Height 5: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "149:"  // Height 5: setup done
      "mov x15, #0x0\n"
      "150:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 151f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "ldr x9, [x20, #0x20]\n"
      "cbnz x15, 152f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "b 152f\n"
      "151:"  // Height 5: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "add x9, x10, x21\n"
      "152:"  // Height 5: input setup done
      "cmp x14, #0x10\n"
      "blt 155f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q2, [x11, #0x0]\n"
      "ldr q3, [x10, #0x0]\n"
      "ldr q4, [x9, #0x0]\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 154f\n"
      "153:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x16, #0x28]\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x38]\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f84e0d8  // udot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr d29, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x48]\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f84e0f9  // udot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr d28, [x16, #0x30]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6f80e3aa  // udot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x6f81e3ae  // udot v14.4s, v29.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x58]\n"
      ".inst 0x6f82e3b2  // udot v18.4s, v29.16b, v2.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x6f83e3b6  // udot v22.4s, v29.16b, v3.4b[0]\n"
      "ldr x26, [x13, #0x8]\n"
      ".inst 0x6f84e3ba  // udot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr d29, [x16, #0x40]\n"
      ".inst 0x6f80e38b  // udot v11.4s, v28.16b, v0.4b[0]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6f81e38f  // udot v15.4s, v28.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x68]\n"
      ".inst 0x6f82e393  // udot v19.4s, v28.16b, v2.4b[0]\n"
      "ldr x25, [x12, #0x8]\n"
      ".inst 0x6f83e397  // udot v23.4s, v28.16b, v3.4b[0]\n"
      "ldr x24, [x11, #0x8]\n"
      ".inst 0x6f84e39b  // udot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr d28, [x16, #0x50]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6fa0e3a8  // udot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x6fa1e3ac  // udot v12.4s, v29.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x78]\n"
      ".inst 0x6fa2e3b0  // udot v16.4s, v29.16b, v2.4b[1]\n"
      "ldr x23, [x10, #0x8]\n"
      ".inst 0x6fa3e3b4  // udot v20.4s, v29.16b, v3.4b[1]\n"
      "ldr x22, [x9, #0x8]\n"
      ".inst 0x6fa4e3b8  // udot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr d29, [x16, #0x60]\n"
      ".inst 0x6fa0e389  // udot v9.4s, v28.16b, v0.4b[1]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6fa1e38d  // udot v13.4s, v28.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0x88]\n"
      ".inst 0x6fa2e391  // udot v17.4s, v28.16b, v2.4b[1]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6fa3e395  // udot v21.4s, v28.16b, v3.4b[1]\n"
      "cmp x14, #0x20\n"
      ".inst 0x6fa4e399  // udot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr d28, [x16, #0x70]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6fa0e3aa  // udot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x6fa1e3ae  // udot v14.4s, v29.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x98]\n"
      ".inst 0x6fa2e3b2  // udot v18.4s, v29.16b, v2.4b[1]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6fa3e3b6  // udot v22.4s, v29.16b, v3.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6fa4e3ba  // udot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr d29, [x16, #0x80]\n"
      ".inst 0x6fa0e38b  // udot v11.4s, v28.16b, v0.4b[1]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6fa1e38f  // udot v15.4s, v28.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0xa8]\n"
      ".inst 0x6fa2e393  // udot v19.4s, v28.16b, v2.4b[1]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6fa3e397  // udot v23.4s, v28.16b, v3.4b[1]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6fa4e39b  // udot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr d28, [x16, #0x90]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6f80eba8  // udot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x6f81ebac  // udot v12.4s, v29.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xb8]\n"
      ".inst 0x6f82ebb0  // udot v16.4s, v29.16b, v2.4b[2]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x6f83ebb4  // udot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x6f84ebb8  // udot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr d29, [x16, #0xa0]\n"
      ".inst 0x6f80eb89  // udot v9.4s, v28.16b, v0.4b[2]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6f81eb8d  // udot v13.4s, v28.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xc8]\n"
      ".inst 0x6f82eb91  // udot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x6f83eb95  // udot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x6f84eb99  // udot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr d28, [x16, #0xb0]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6f80ebaa  // udot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x6f81ebae  // udot v14.4s, v29.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xd8]\n"
      ".inst 0x6f82ebb2  // udot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x6f83ebb6  // udot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x6f84ebba  // udot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr d29, [x16, #0xc0]\n"
      ".inst 0x6f80eb8b  // udot v11.4s, v28.16b, v0.4b[2]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6f81eb8f  // udot v15.4s, v28.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xe8]\n"
      ".inst 0x6f82eb93  // udot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x6f83eb97  // udot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x6f84eb9b  // udot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr d28, [x16, #0xd0]\n"
      "mov v28.d[1], x20\n"
      ".inst 0x6fa0eba8  // udot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x6fa1ebac  // udot v12.4s, v29.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0xf8]\n"
      ".inst 0x6fa2ebb0  // udot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x6fa3ebb4  // udot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x6fa4ebb8  // udot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr d29, [x16, #0xe0]\n"
      ".inst 0x6fa0eb89  // udot v9.4s, v28.16b, v0.4b[3]\n"
      "mov v29.d[1], x21\n"
      ".inst 0x6fa1eb8d  // udot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb91  // udot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb95  // udot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x6fa4eb99  // udot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr d28, [x16, #0xf0]\n"
      "mov v28.d[1], x20\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0ebaa  // udot v10.4s, v29.16b, v0.4b[3]\n"
      "ldr x21, [x16, #0x8]\n"
      ".inst 0x6fa1ebae  // udot v14.4s, v29.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0x18]\n"
      ".inst 0x6fa2ebb2  // udot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x6fa3ebb6  // udot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x6fa4ebba  // udot v26.4s, v29.16b, v4.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      ".inst 0x6fa0eb8b  // udot v11.4s, v28.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      ".inst 0x6fa1eb8f  // udot v15.4s, v28.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0x0]\n"
      ".inst 0x6fa2eb93  // udot v19.4s, v28.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0x0]\n"
      ".inst 0x6fa3eb97  // udot v23.4s, v28.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0x0]\n"
      ".inst 0x6fa4eb9b  // udot v27.4s, v28.16b, v4.4b[3]\n"
      "ldr d4, [x9, #0x0]\n"
      "ldr d7, [x16, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x26\n"
      "mov v1.d[1], x25\n"
      "mov v2.d[1], x24\n"
      "mov v3.d[1], x23\n"
      "mov v4.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 153b\n"
      "154:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f84e0d8  // udot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q29, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6f84e0f9  // udot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q28, [x16, #0x30]\n"
      ".inst 0x6f80e3aa  // udot v10.4s, v29.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6f81e3ae  // udot v14.4s, v29.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6f82e3b2  // udot v18.4s, v29.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x6f83e3b6  // udot v22.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x6f84e3ba  // udot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x16, #0x40]\n"
      ".inst 0x6f80e38b  // udot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x6f81e38f  // udot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x6f82e393  // udot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x6f83e397  // udot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x6f84e39b  // udot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x16, #0x50]\n"
      ".inst 0x6fa0e3a8  // udot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x6fa1e3ac  // udot v12.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x6fa2e3b0  // udot v16.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x6fa3e3b4  // udot v20.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x6fa4e3b8  // udot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x16, #0x60]\n"
      ".inst 0x6fa0e389  // udot v9.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x6fa1e38d  // udot v13.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x6fa2e391  // udot v17.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x6fa3e395  // udot v21.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x6fa4e399  // udot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x16, #0x70]\n"
      ".inst 0x6fa0e3aa  // udot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x6fa1e3ae  // udot v14.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x6fa2e3b2  // udot v18.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x6fa3e3b6  // udot v22.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x6fa4e3ba  // udot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x16, #0x80]\n"
      ".inst 0x6fa0e38b  // udot v11.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x6fa1e38f  // udot v15.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x6fa2e393  // udot v19.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x6fa3e397  // udot v23.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x6fa4e39b  // udot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x16, #0x90]\n"
      ".inst 0x6f80eba8  // udot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x6f81ebac  // udot v12.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x6f82ebb0  // udot v16.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x6f83ebb4  // udot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x6f84ebb8  // udot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x16, #0xa0]\n"
      ".inst 0x6f80eb89  // udot v9.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x6f81eb8d  // udot v13.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x6f82eb91  // udot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x6f83eb95  // udot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x6f84eb99  // udot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x16, #0xb0]\n"
      ".inst 0x6f80ebaa  // udot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x6f81ebae  // udot v14.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x6f82ebb2  // udot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x6f83ebb6  // udot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x6f84ebba  // udot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x16, #0xc0]\n"
      ".inst 0x6f80eb8b  // udot v11.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x6f81eb8f  // udot v15.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x6f82eb93  // udot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x6f83eb97  // udot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x6f84eb9b  // udot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x16, #0xd0]\n"
      ".inst 0x6fa0eba8  // udot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x6fa1ebac  // udot v12.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x6fa2ebb0  // udot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x6fa3ebb4  // udot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x6fa4ebb8  // udot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr q29, [x16, #0xe0]\n"
      ".inst 0x6fa0eb89  // udot v9.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb8d  // udot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb91  // udot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb95  // udot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x6fa4eb99  // udot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr q28, [x16, #0xf0]\n"
      ".inst 0x6fa0ebaa  // udot v10.4s, v29.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa1ebae  // udot v14.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x6fa2ebb2  // udot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x6fa3ebb6  // udot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x6fa4ebba  // udot v26.4s, v29.16b, v4.4b[3]\n"
      ".inst 0x6fa0eb8b  // udot v11.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x6fa1eb8f  // udot v15.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x6fa2eb93  // udot v19.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x6fa3eb97  // udot v23.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x6fa4eb9b  // udot v27.4s, v28.16b, v4.4b[3]\n"
      "155:"  // Height 5: Multiply loop: Main loop skip
      "cbz x14, 160f\n"
      "cmp x14, #0x4\n"
      "blt 157f\n"
      "156:"  // Height 5: Multiply loop: Odd block loop
      "ldr s2, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s0, [x11], #0x4\n"
      "ldr s31, [x10], #0x4\n"
      "ldr s30, [x9], #0x4\n"
      "ldr q29, [x16, #0x0]\n"
      ".inst 0x6f82e3a8  // udot v8.4s, v29.16b, v2.4b[0]\n"
      "ldr q28, [x16, #0x10]\n"
      ".inst 0x6f81e3ac  // udot v12.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x6f80e3b0  // udot v16.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x6f9fe3b4  // udot v20.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x6f9ee3b8  // udot v24.4s, v29.16b, v30.4b[0]\n"
      "ldr q29, [x16, #0x20]\n"
      ".inst 0x6f82e389  // udot v9.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x6f81e38d  // udot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x6f80e391  // udot v17.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x6f9fe395  // udot v21.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x6f9ee399  // udot v25.4s, v28.16b, v30.4b[0]\n"
      "ldr q28, [x16, #0x30]\n"
      ".inst 0x6f82e3aa  // udot v10.4s, v29.16b, v2.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e3ae  // udot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x6f80e3b2  // udot v18.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x6f9fe3b6  // udot v22.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x6f9ee3ba  // udot v26.4s, v29.16b, v30.4b[0]\n"
      ".inst 0x6f82e38b  // udot v11.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x6f81e38f  // udot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x6f80e393  // udot v19.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x6f9fe397  // udot v23.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x6f9ee39b  // udot v27.4s, v28.16b, v30.4b[0]\n"
      "bge 156b\n"
      "157:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x14, 160f\n"
      "tbz x14, #1, 158f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "ldr h4, [x9], #0x2\n"
      "tbz x14, #0, 159f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "ld1 { v4.b }[2], [x9]\n"
      "b 159f\n"
      "158:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "ldr b1, [x12, #0x0]\n"
      "ldr b2, [x11, #0x0]\n"
      "ldr b3, [x10, #0x0]\n"
      "ldr b4, [x9, #0x0]\n"
      "159:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q29, [x16, #0x0]\n"
      ".inst 0x6f80e3a8  // udot v8.4s, v29.16b, v0.4b[0]\n"
      "ldr q28, [x16, #0x10]\n"
      ".inst 0x6f81e3ac  // udot v12.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x6f82e3b0  // udot v16.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x6f83e3b4  // udot v20.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x6f84e3b8  // udot v24.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x16, #0x20]\n"
      ".inst 0x6f80e389  // udot v9.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x6f81e38d  // udot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x6f82e391  // udot v17.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x6f83e395  // udot v21.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x6f84e399  // udot v25.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x16, #0x30]\n"
      ".inst 0x6f80e3aa  // udot v10.4s, v29.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e3ae  // udot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x6f82e3b2  // udot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x6f83e3b6  // udot v22.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x6f84e3ba  // udot v26.4s, v29.16b, v4.4b[0]\n"
      ".inst 0x6f80e38b  // udot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x6f81e38f  // udot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x6f82e393  // udot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x6f83e397  // udot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x6f84e39b  // udot v27.4s, v28.16b, v4.4b[0]\n"
      "160:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 150b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "bge 169f\n"
      "tbz x8, #3, 164f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "tbz x8, #2, 162f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x8, #1, 161f\n"
      "str d11, [x17], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x8, #0, 168f\n"
      "st1 { v11.s }[2], [x17]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 168f\n"
      "161:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x8, #0, 168f\n"
      "str s11, [x17, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "b 168f\n"
      "162:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x8, #1, 163f\n"
      "str d10, [x17], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x8, #0, 168f\n"
      "st1 { v10.s }[2], [x17]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 168f\n"
      "163:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x8, #0, 168f\n"
      "str s10, [x17, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "b 168f\n"
      "164:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x8, #2, 166f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x8, #1, 165f\n"
      "str d9, [x17], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x8, #0, 168f\n"
      "st1 { v9.s }[2], [x17]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 168f\n"
      "165:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x8, #0, 168f\n"
      "str s9, [x17, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "b 168f\n"
      "166:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x8, #1, 167f\n"
      "str d8, [x17], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x8, #0, 168f\n"
      "st1 { v8.s }[2], [x17]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 168f\n"
      "167:"  // Height 5: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "168:"  // Height 5: Partial direct writeback: Done
      "b 170f\n"
      "169:"  // Height 5: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0x0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "170:"  // Height 5: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 138b\n"
      "b 206f\n"
      "171:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x18\n"
      "ldr x8, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x17, %x[output_ptr]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "172:"  // Height 6: Column loop
      "tbz %x[flags], #0, 182f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "add x20, x21, x20, LSL #2\n"
      "bge 181f\n"
      "tbz x8, #3, 176f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "ld1 { v9.4s }, [x17], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v29.4s }, [x20], #0x10\n"
      "tbz x8, #2, 174f\n"
      "ld1 { v10.4s }, [x17], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v30.4s }, [x20], #0x10\n"
      "tbz x8, #1, 173f\n"
      "ldr d11, [x17], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "ldr d31, [x20], #0x8\n"
      "tbz x8, #0, 180f\n"
      "ld1 { v11.s }[2], [x17]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "ld1 { v31.s }[2], [x20]\n"
      "b 180f\n"
      "173:"  // Height 6: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x8, #0, 180f\n"
      "ldr s11, [x17, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "ldr s31, [x20, #0x0]\n"
      "b 180f\n"
      "174:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x8, #1, 175f\n"
      "ldr d10, [x17], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "ldr d30, [x20], #0x8\n"
      "tbz x8, #0, 180f\n"
      "ld1 { v10.s }[2], [x17]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "ld1 { v30.s }[2], [x20]\n"
      "b 180f\n"
      "175:"  // Height 6: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x8, #0, 180f\n"
      "ldr s10, [x17, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "ldr s30, [x20, #0x0]\n"
      "b 180f\n"
      "176:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x8, #2, 178f\n"
      "ld1 { v8.4s }, [x17], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "tbz x8, #1, 177f\n"
      "ldr d9, [x17], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "ldr d29, [x20], #0x8\n"
      "tbz x8, #0, 180f\n"
      "ld1 { v9.s }[2], [x17]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "ld1 { v29.s }[2], [x20]\n"
      "b 180f\n"
      "177:"  // Height 6: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x8, #0, 180f\n"
      "ldr s9, [x17, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "ldr s29, [x20, #0x0]\n"
      "b 180f\n"
      "178:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x8, #1, 179f\n"
      "ldr d8, [x17], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "ldr d28, [x20], #0x8\n"
      "tbz x8, #0, 180f\n"
      "ld1 { v8.s }[2], [x17]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "ld1 { v28.s }[2], [x20]\n"
      "b 180f\n"
      "179:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s8, [x17, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s12, [x24, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s24, [x21, #0x0]\n"
      "ldr s28, [x20, #0x0]\n"
      "180:"  // Height 6: Partial accumulate: Done
      "sub x17, x17, x25\n"
      "b 183f\n"
      "181:"  // Height 6: full accumulate
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0x0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "ldr q28, [x20, #0x0]\n"
      "ldr q29, [x20, #0x10]\n"
      "ldr q30, [x20, #0x20]\n"
      "ldr q31, [x20, #0x30]\n"
      "b 183f\n"
      "182:"  // Height 6: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "183:"  // Height 6: setup done
      "mov x15, #0x0\n"
      "184:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 185f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0x0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "ldr x9, [x20, #0x20]\n"
      "ldr x28, [x20, #0x28]\n"
      "cbnz x15, 186f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "add x28, x28, x20\n"
      "b 186f\n"
      "185:"  // Height 6: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "add x9, x10, x21\n"
      "add x28, x9, x21\n"
      "186:"  // Height 6: input setup done
      "cmp x14, #0x10\n"
      "blt 189f\n"
      "ldr q0, [x13, #0x0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q2, [x11, #0x0]\n"
      "ldr q3, [x10, #0x0]\n"
      "ldr q4, [x9, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x16, #0x0]\n"
      "ldr q7, [x16, #0x10]\n"
      "blt 188f\n"
      "187:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x16, #0x28]\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x38]\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f84e0d8  // udot v24.4s, v6.16b, v4.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f85e0dc  // udot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr d6, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x48]\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x6f84e0f9  // udot v25.4s, v7.16b, v4.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x6f85e0fd  // udot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr d7, [x16, #0x30]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6f80e0ca  // udot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x6f81e0ce  // udot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x16, #0x58]\n"
      ".inst 0x6f82e0d2  // udot v18.4s, v6.16b, v2.4b[0]\n"
      "ldr x27, [x13, #0x8]\n"
      ".inst 0x6f83e0d6  // udot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr x26, [x12, #0x8]\n"
      ".inst 0x6f84e0da  // udot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr x25, [x11, #0x8]\n"
      ".inst 0x6f85e0de  // udot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr d6, [x16, #0x40]\n"
      ".inst 0x6f80e0eb  // udot v11.4s, v7.16b, v0.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6f81e0ef  // udot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr x21, [x16, #0x68]\n"
      ".inst 0x6f82e0f3  // udot v19.4s, v7.16b, v2.4b[0]\n"
      "ldr x24, [x10, #0x8]\n"
      ".inst 0x6f83e0f7  // udot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr x23, [x9, #0x8]\n"
      ".inst 0x6f84e0fb  // udot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr x22, [x28, #0x8]\n"
      ".inst 0x6f85e0ff  // udot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr d7, [x16, #0x50]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6fa0e0c8  // udot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0cc  // udot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x78]\n"
      ".inst 0x6fa2e0d0  // udot v16.4s, v6.16b, v2.4b[1]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6fa3e0d4  // udot v20.4s, v6.16b, v3.4b[1]\n"
      "cmp x14, #0x20\n"
      ".inst 0x6fa4e0d8  // udot v24.4s, v6.16b, v4.4b[1]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6fa5e0dc  // udot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr d6, [x16, #0x60]\n"
      ".inst 0x6fa0e0e9  // udot v9.4s, v7.16b, v0.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6fa1e0ed  // udot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0x88]\n"
      ".inst 0x6fa2e0f1  // udot v17.4s, v7.16b, v2.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6fa3e0f5  // udot v21.4s, v7.16b, v3.4b[1]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6fa4e0f9  // udot v25.4s, v7.16b, v4.4b[1]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6fa5e0fd  // udot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr d7, [x16, #0x70]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6fa0e0ca  // udot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0ce  // udot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr x20, [x16, #0x98]\n"
      ".inst 0x6fa2e0d2  // udot v18.4s, v6.16b, v2.4b[1]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x6fa3e0d6  // udot v22.4s, v6.16b, v3.4b[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      ".inst 0x6fa4e0da  // udot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0de  // udot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr d6, [x16, #0x80]\n"
      ".inst 0x6fa0e0eb  // udot v11.4s, v7.16b, v0.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6fa1e0ef  // udot v15.4s, v7.16b, v1.4b[1]\n"
      "ldr x21, [x16, #0xa8]\n"
      ".inst 0x6fa2e0f3  // udot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x6fa3e0f7  // udot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x6fa4e0fb  // udot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0ff  // udot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr d7, [x16, #0x90]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6f80e8c8  // udot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x6f81e8cc  // udot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xb8]\n"
      ".inst 0x6f82e8d0  // udot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x6f83e8d4  // udot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x6f84e8d8  // udot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x6f85e8dc  // udot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr d6, [x16, #0xa0]\n"
      ".inst 0x6f80e8e9  // udot v9.4s, v7.16b, v0.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6f81e8ed  // udot v13.4s, v7.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xc8]\n"
      ".inst 0x6f82e8f1  // udot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x6f83e8f5  // udot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x6f84e8f9  // udot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x6f85e8fd  // udot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr d7, [x16, #0xb0]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6f80e8ca  // udot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x6f81e8ce  // udot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr x20, [x16, #0xd8]\n"
      ".inst 0x6f82e8d2  // udot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x6f83e8d6  // udot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x6f84e8da  // udot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x6f85e8de  // udot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr d6, [x16, #0xc0]\n"
      ".inst 0x6f80e8eb  // udot v11.4s, v7.16b, v0.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6f81e8ef  // udot v15.4s, v7.16b, v1.4b[2]\n"
      "ldr x21, [x16, #0xe8]\n"
      ".inst 0x6f82e8f3  // udot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x6f83e8f7  // udot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x6f84e8fb  // udot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x6f85e8ff  // udot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr d7, [x16, #0xd0]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x6fa0e8c8  // udot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x6fa1e8cc  // udot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0xf8]\n"
      ".inst 0x6fa2e8d0  // udot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8d4  // udot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8d8  // udot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8dc  // udot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr d6, [x16, #0xe0]\n"
      ".inst 0x6fa0e8e9  // udot v9.4s, v7.16b, v0.4b[3]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x6fa1e8ed  // udot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x6fa2e8f1  // udot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8f5  // udot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8f9  // udot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8fd  // udot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr d7, [x16, #0xf0]\n"
      "mov v7.d[1], x20\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa0e8ca  // udot v10.4s, v6.16b, v0.4b[3]\n"
      "ldr x21, [x16, #0x8]\n"
      ".inst 0x6fa1e8ce  // udot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr x20, [x16, #0x18]\n"
      ".inst 0x6fa2e8d2  // udot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8d6  // udot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8da  // udot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8de  // udot v30.4s, v6.16b, v5.4b[3]\n"
      "ldr d6, [x16, #0x0]\n"
      ".inst 0x6fa0e8eb  // udot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0x0]\n"
      ".inst 0x6fa1e8ef  // udot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0x0]\n"
      ".inst 0x6fa2e8f3  // udot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0x0]\n"
      ".inst 0x6fa3e8f7  // udot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0x0]\n"
      ".inst 0x6fa4e8fb  // udot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr d4, [x9, #0x0]\n"
      ".inst 0x6fa5e8ff  // udot v31.4s, v7.16b, v5.4b[3]\n"
      "ldr d5, [x28, #0x0]\n"
      "ldr d7, [x16, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x27\n"
      "mov v1.d[1], x26\n"
      "mov v2.d[1], x25\n"
      "mov v3.d[1], x24\n"
      "mov v4.d[1], x23\n"
      "mov v5.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 187b\n"
      "188:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x6f80e0c8  // udot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x6f81e0cc  // udot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x6f82e0d0  // udot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x6f83e0d4  // udot v20.4s, v6.16b, v3.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x6f84e0d8  // udot v24.4s, v6.16b, v4.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x6f85e0dc  // udot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x16, #0x20]\n"
      ".inst 0x6f80e0e9  // udot v9.4s, v7.16b, v0.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x6f81e0ed  // udot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x6f82e0f1  // udot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x6f83e0f5  // udot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x6f84e0f9  // udot v25.4s, v7.16b, v4.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x6f85e0fd  // udot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x16, #0x30]\n"
      ".inst 0x6f80e0ca  // udot v10.4s, v6.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x6f81e0ce  // udot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x6f82e0d2  // udot v18.4s, v6.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      ".inst 0x6f83e0d6  // udot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x6f84e0da  // udot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x6f85e0de  // udot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x16, #0x40]\n"
      ".inst 0x6f80e0eb  // udot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x6f81e0ef  // udot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x6f82e0f3  // udot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x6f83e0f7  // udot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x6f84e0fb  // udot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x6f85e0ff  // udot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x16, #0x50]\n"
      ".inst 0x6fa0e0c8  // udot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0cc  // udot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x6fa2e0d0  // udot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x6fa3e0d4  // udot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x6fa4e0d8  // udot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0dc  // udot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x16, #0x60]\n"
      ".inst 0x6fa0e0e9  // udot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0ed  // udot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x6fa2e0f1  // udot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x6fa3e0f5  // udot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x6fa4e0f9  // udot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0fd  // udot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x16, #0x70]\n"
      ".inst 0x6fa0e0ca  // udot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0ce  // udot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x6fa2e0d2  // udot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x6fa3e0d6  // udot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x6fa4e0da  // udot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0de  // udot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x16, #0x80]\n"
      ".inst 0x6fa0e0eb  // udot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x6fa1e0ef  // udot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x6fa2e0f3  // udot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x6fa3e0f7  // udot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x6fa4e0fb  // udot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x6fa5e0ff  // udot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x16, #0x90]\n"
      ".inst 0x6f80e8c8  // udot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x6f81e8cc  // udot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x6f82e8d0  // udot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x6f83e8d4  // udot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x6f84e8d8  // udot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x6f85e8dc  // udot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x16, #0xa0]\n"
      ".inst 0x6f80e8e9  // udot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x6f81e8ed  // udot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x6f82e8f1  // udot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x6f83e8f5  // udot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x6f84e8f9  // udot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x6f85e8fd  // udot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x16, #0xb0]\n"
      ".inst 0x6f80e8ca  // udot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x6f81e8ce  // udot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x6f82e8d2  // udot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x6f83e8d6  // udot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x6f84e8da  // udot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x6f85e8de  // udot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x16, #0xc0]\n"
      ".inst 0x6f80e8eb  // udot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x6f81e8ef  // udot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x6f82e8f3  // udot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x6f83e8f7  // udot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x6f84e8fb  // udot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x6f85e8ff  // udot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x16, #0xd0]\n"
      ".inst 0x6fa0e8c8  // udot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x6fa1e8cc  // udot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x6fa2e8d0  // udot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8d4  // udot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8d8  // udot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8dc  // udot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x16, #0xe0]\n"
      ".inst 0x6fa0e8e9  // udot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x6fa1e8ed  // udot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x6fa2e8f1  // udot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8f5  // udot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8f9  // udot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8fd  // udot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x16, #0xf0]\n"
      ".inst 0x6fa0e8ca  // udot v10.4s, v6.16b, v0.4b[3]\n"
      "add x16, x16, #0x100\n"
      ".inst 0x6fa1e8ce  // udot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x6fa2e8d2  // udot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8d6  // udot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8da  // udot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8de  // udot v30.4s, v6.16b, v5.4b[3]\n"
      ".inst 0x6fa0e8eb  // udot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x6fa1e8ef  // udot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x6fa2e8f3  // udot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x6fa3e8f7  // udot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x6fa4e8fb  // udot v27.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x6fa5e8ff  // udot v31.4s, v7.16b, v5.4b[3]\n"
      "189:"  // Height 6: Multiply loop: Main loop skip
      "cbz x14, 194f\n"
      "cmp x14, #0x4\n"
      "blt 191f\n"
      "190:"  // Height 6: Multiply loop: Odd block loop
      "ldr s7, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s6, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s5, [x11], #0x4\n"
      "ldr s4, [x10], #0x4\n"
      "ldr s3, [x9], #0x4\n"
      "ldr s2, [x28], #0x4\n"
      "ldr q1, [x16, #0x0]\n"
      ".inst 0x6f87e028  // udot v8.4s, v1.16b, v7.4b[0]\n"
      "ldr q0, [x16, #0x10]\n"
      ".inst 0x6f86e02c  // udot v12.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x6f85e030  // udot v16.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x6f84e034  // udot v20.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x6f83e038  // udot v24.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x6f82e03c  // udot v28.4s, v1.16b, v2.4b[0]\n"
      "ldr q1, [x16, #0x20]\n"
      ".inst 0x6f87e009  // udot v9.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x6f86e00d  // udot v13.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x6f85e011  // udot v17.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x6f84e015  // udot v21.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x6f83e019  // udot v25.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x6f82e01d  // udot v29.4s, v0.16b, v2.4b[0]\n"
      "ldr q0, [x16, #0x30]\n"
      ".inst 0x6f87e02a  // udot v10.4s, v1.16b, v7.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f86e02e  // udot v14.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x6f85e032  // udot v18.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x6f84e036  // udot v22.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x6f83e03a  // udot v26.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x6f82e03e  // udot v30.4s, v1.16b, v2.4b[0]\n"
      ".inst 0x6f87e00b  // udot v11.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x6f86e00f  // udot v15.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x6f85e013  // udot v19.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x6f84e017  // udot v23.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x6f83e01b  // udot v27.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x6f82e01f  // udot v31.4s, v0.16b, v2.4b[0]\n"
      "bge 190b\n"
      "191:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x14, 194f\n"
      "tbz x14, #1, 192f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "ldr h4, [x9], #0x2\n"
      "ldr h5, [x28], #0x2\n"
      "tbz x14, #0, 193f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "ld1 { v4.b }[2], [x9]\n"
      "ld1 { v5.b }[2], [x28]\n"
      "b 193f\n"
      "192:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0x0]\n"
      "ldr b1, [x12, #0x0]\n"
      "ldr b2, [x11, #0x0]\n"
      "ldr b3, [x10, #0x0]\n"
      "ldr b4, [x9, #0x0]\n"
      "ldr b5, [x28, #0x0]\n"
      "193:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x16, #0x0]\n"
      ".inst 0x6f80e0e8  // udot v8.4s, v7.16b, v0.4b[0]\n"
      "ldr q6, [x16, #0x10]\n"
      ".inst 0x6f81e0ec  // udot v12.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x6f82e0f0  // udot v16.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x6f83e0f4  // udot v20.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x6f84e0f8  // udot v24.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x6f85e0fc  // udot v28.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x16, #0x20]\n"
      ".inst 0x6f80e0c9  // udot v9.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x6f81e0cd  // udot v13.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x6f82e0d1  // udot v17.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x6f83e0d5  // udot v21.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x6f84e0d9  // udot v25.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x6f85e0dd  // udot v29.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x16, #0x30]\n"
      ".inst 0x6f80e0ea  // udot v10.4s, v7.16b, v0.4b[0]\n"
      "add x16, x16, #0x40\n"
      ".inst 0x6f81e0ee  // udot v14.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x6f82e0f2  // udot v18.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x6f83e0f6  // udot v22.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x6f84e0fa  // udot v26.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x6f85e0fe  // udot v30.4s, v7.16b, v5.4b[0]\n"
      ".inst 0x6f80e0cb  // udot v11.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x6f81e0cf  // udot v15.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x6f82e0d3  // udot v19.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x6f83e0d7  // udot v23.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x6f84e0db  // udot v27.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x6f85e0df  // udot v31.4s, v6.16b, v5.4b[0]\n"
      "194:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 184b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x17, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "cmp x8, #0x10\n"
      "prfm pstl1keep, [x17, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "prfm pstl1keep, [x20, #0x0]\n"
      "bge 203f\n"
      "tbz x8, #3, 198f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v9.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "st1 { v29.4s }, [x20], #0x10\n"
      "tbz x8, #2, 196f\n"
      "st1 { v10.4s }, [x17], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "st1 { v30.4s }, [x20], #0x10\n"
      "tbz x8, #1, 195f\n"
      "str d11, [x17], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "str d31, [x20], #0x8\n"
      "tbz x8, #0, 202f\n"
      "st1 { v11.s }[2], [x17]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "st1 { v31.s }[2], [x20]\n"
      "b 202f\n"
      "195:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x8, #0, 202f\n"
      "str s11, [x17, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "str s31, [x20, #0x0]\n"
      "b 202f\n"
      "196:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x8, #1, 197f\n"
      "str d10, [x17], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "str d30, [x20], #0x8\n"
      "tbz x8, #0, 202f\n"
      "st1 { v10.s }[2], [x17]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "st1 { v30.s }[2], [x20]\n"
      "b 202f\n"
      "197:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x8, #0, 202f\n"
      "str s10, [x17, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "str s30, [x20, #0x0]\n"
      "b 202f\n"
      "198:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x8, #2, 200f\n"
      "st1 { v8.4s }, [x17], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "tbz x8, #1, 199f\n"
      "str d9, [x17], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "str d29, [x20], #0x8\n"
      "tbz x8, #0, 202f\n"
      "st1 { v9.s }[2], [x17]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "st1 { v29.s }[2], [x20]\n"
      "b 202f\n"
      "199:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x8, #0, 202f\n"
      "str s9, [x17, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "str s29, [x20, #0x0]\n"
      "b 202f\n"
      "200:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x8, #1, 201f\n"
      "str d8, [x17], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "str d28, [x20], #0x8\n"
      "tbz x8, #0, 202f\n"
      "st1 { v8.s }[2], [x17]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "st1 { v28.s }[2], [x20]\n"
      "b 202f\n"
      "201:"  // Height 6: Partial direct writeback: partial_1_0
      "str s8, [x17, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "str s28, [x20, #0x0]\n"
      "202:"  // Height 6: Partial direct writeback: Done
      "b 204f\n"
      "203:"  // Height 6: Full writeback
      "str q8, [x17, #0x0]\n"
      "str q9, [x17, #0x10]\n"
      "str q10, [x17, #0x20]\n"
      "str q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0x0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "str q28, [x20, #0x0]\n"
      "str q29, [x20, #0x10]\n"
      "str q30, [x20, #0x20]\n"
      "str q31, [x20, #0x30]\n"
      "204:"  // Height 6: Writeback done
      "subs x8, x8, #0x10\n"
      "bgt 172b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 206f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 205f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "205:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "206:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
