/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.translation;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.annotation.SystemApi;
import android.app.assist.ActivityId;
import android.os.Parcel;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

import java.util.concurrent.Executor;
import java.util.function.Consumer;

/**
 * Info class holding information for {@link Translator}s and used by
 * {@link TranslationManager#createOnDeviceTranslator(TranslationContext, Executor, Consumer)}.
 */
@DataClass(genHiddenConstDefs = true, genToString = true, genBuilder = true)
public final class TranslationContext implements Parcelable {

    /**
     * This context will perform translations in low latency mode.
     */
    public static final @TranslationFlag int FLAG_LOW_LATENCY = 0x1;
    /**
     * This context will enable the {@link Translator} to return transliteration results.
     */
    public static final @TranslationFlag int FLAG_TRANSLITERATION = 0x2;
    /**
     * This context will enable the {@link Translator} to return dictionary definitions.
     */
    public static final @TranslationFlag int FLAG_DEFINITIONS = 0x4;

    /**
     * {@link TranslationSpec} describing the source data to be translated.
     */
    @NonNull
    private final TranslationSpec mSourceSpec;

    /**
     * {@link TranslationSpec} describing the target translated data.
     */
    @NonNull
    private final TranslationSpec mTargetSpec;

    /**
     * Translation flags to be used by the {@link Translator}
     */
    private final @TranslationFlag int mTranslationFlags;

    private static int defaultTranslationFlags() {
        return 0;
    }

    /**
     * The identifier for the Activity which needs UI translation.
     *
     * @hide
     */
    @Nullable
    private final ActivityId mActivityId;

    private static ActivityId defaultActivityId() {
        return null;
    }

    private void parcelActivityId(@NonNull Parcel dest, int flags) {
        dest.writeBoolean(mActivityId != null);
        if (mActivityId != null) {
            mActivityId.writeToParcel(dest, flags);
        }
    }

    @Nullable
    private ActivityId unparcelActivityId(@NonNull Parcel in) {
        final boolean hasActivityId = in.readBoolean();
        return hasActivityId ? new ActivityId(in) : null;
    }

    /**
     * Returns the identifier for the Activity which needs UI translation or {@code null}
     * if it is a non-UI translation request.
     *
     * NOTE: If the application receiving this ActivityId also provides a ContentCaptureService, it
     * can be used to associate a TranslationRequest with a particular ContentCaptureSession.
     *
     * @hide
     */
    @SystemApi
    @Nullable
    public ActivityId getActivityId() {
        return mActivityId;
    }

    @DataClass.Suppress({"setSourceSpec", "setTargetSpec"})
    abstract static class BaseBuilder {

    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/translation/TranslationContext.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /** @hide */
    @android.annotation.IntDef(flag = true, prefix = "FLAG_", value = {
        FLAG_LOW_LATENCY,
        FLAG_TRANSLITERATION,
        FLAG_DEFINITIONS
    })
    @java.lang.annotation.Retention(java.lang.annotation.RetentionPolicy.SOURCE)
    @DataClass.Generated.Member
    public @interface TranslationFlag {}

    /** @hide */
    @DataClass.Generated.Member
    public static String translationFlagToString(@TranslationFlag int value) {
        return com.android.internal.util.BitUtils.flagsToString(
                value, TranslationContext::singleTranslationFlagToString);
    }

    @DataClass.Generated.Member
    static String singleTranslationFlagToString(@TranslationFlag int value) {
        switch (value) {
            case FLAG_LOW_LATENCY:
                    return "FLAG_LOW_LATENCY";
            case FLAG_TRANSLITERATION:
                    return "FLAG_TRANSLITERATION";
            case FLAG_DEFINITIONS:
                    return "FLAG_DEFINITIONS";
            default: return Integer.toHexString(value);
        }
    }

    @DataClass.Generated.Member
    /* package-private */ TranslationContext(
            @NonNull TranslationSpec sourceSpec,
            @NonNull TranslationSpec targetSpec,
            @TranslationFlag int translationFlags,
            @Nullable ActivityId activityId) {
        this.mSourceSpec = sourceSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mSourceSpec);
        this.mTargetSpec = targetSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTargetSpec);
        this.mTranslationFlags = translationFlags;

        com.android.internal.util.Preconditions.checkFlagsArgument(
                mTranslationFlags,
                FLAG_LOW_LATENCY
                        | FLAG_TRANSLITERATION
                        | FLAG_DEFINITIONS);
        this.mActivityId = activityId;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * {@link TranslationSpec} describing the source data to be translated.
     */
    @DataClass.Generated.Member
    public @NonNull TranslationSpec getSourceSpec() {
        return mSourceSpec;
    }

    /**
     * {@link TranslationSpec} describing the target translated data.
     */
    @DataClass.Generated.Member
    public @NonNull TranslationSpec getTargetSpec() {
        return mTargetSpec;
    }

    /**
     * Translation flags to be used by the {@link Translator}
     */
    @DataClass.Generated.Member
    public @TranslationFlag int getTranslationFlags() {
        return mTranslationFlags;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "TranslationContext { " +
                "sourceSpec = " + mSourceSpec + ", " +
                "targetSpec = " + mTargetSpec + ", " +
                "translationFlags = " + translationFlagToString(mTranslationFlags) + ", " +
                "activityId = " + mActivityId +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mActivityId != null) flg |= 0x8;
        dest.writeByte(flg);
        dest.writeTypedObject(mSourceSpec, flags);
        dest.writeTypedObject(mTargetSpec, flags);
        dest.writeInt(mTranslationFlags);
        parcelActivityId(dest, flags);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ TranslationContext(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        TranslationSpec sourceSpec = (TranslationSpec) in.readTypedObject(TranslationSpec.CREATOR);
        TranslationSpec targetSpec = (TranslationSpec) in.readTypedObject(TranslationSpec.CREATOR);
        int translationFlags = in.readInt();
        ActivityId activityId = unparcelActivityId(in);

        this.mSourceSpec = sourceSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mSourceSpec);
        this.mTargetSpec = targetSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTargetSpec);
        this.mTranslationFlags = translationFlags;

        com.android.internal.util.Preconditions.checkFlagsArgument(
                mTranslationFlags,
                FLAG_LOW_LATENCY
                        | FLAG_TRANSLITERATION
                        | FLAG_DEFINITIONS);
        this.mActivityId = activityId;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<TranslationContext> CREATOR
            = new Parcelable.Creator<TranslationContext>() {
        @Override
        public TranslationContext[] newArray(int size) {
            return new TranslationContext[size];
        }

        @Override
        public TranslationContext createFromParcel(@NonNull Parcel in) {
            return new TranslationContext(in);
        }
    };

    /**
     * A builder for {@link TranslationContext}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static final class Builder extends BaseBuilder {

        private @NonNull TranslationSpec mSourceSpec;
        private @NonNull TranslationSpec mTargetSpec;
        private @TranslationFlag int mTranslationFlags;
        private @Nullable ActivityId mActivityId;

        private long mBuilderFieldsSet = 0L;

        /**
         * Creates a new Builder.
         *
         * @param sourceSpec
         *   {@link TranslationSpec} describing the source data to be translated.
         * @param targetSpec
         *   {@link TranslationSpec} describing the target translated data.
         */
        public Builder(
                @NonNull TranslationSpec sourceSpec,
                @NonNull TranslationSpec targetSpec) {
            mSourceSpec = sourceSpec;
            com.android.internal.util.AnnotationValidations.validate(
                    NonNull.class, null, mSourceSpec);
            mTargetSpec = targetSpec;
            com.android.internal.util.AnnotationValidations.validate(
                    NonNull.class, null, mTargetSpec);
        }

        /**
         * Translation flags to be used by the {@link Translator}
         */
        @DataClass.Generated.Member
        public @NonNull Builder setTranslationFlags(@TranslationFlag int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4;
            mTranslationFlags = value;
            return this;
        }

        /**
         * The identifier for the Activity which needs UI translation.
         *
         * @hide
         */
        @DataClass.Generated.Member
        public @NonNull Builder setActivityId(@NonNull ActivityId value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x8;
            mActivityId = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull TranslationContext build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x10; // Mark builder used

            if ((mBuilderFieldsSet & 0x4) == 0) {
                mTranslationFlags = defaultTranslationFlags();
            }
            if ((mBuilderFieldsSet & 0x8) == 0) {
                mActivityId = defaultActivityId();
            }
            TranslationContext o = new TranslationContext(
                    mSourceSpec,
                    mTargetSpec,
                    mTranslationFlags,
                    mActivityId);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x10) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1638348645427L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/view/translation/TranslationContext.java",
            inputSignatures = "public static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_LOW_LATENCY\npublic static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_TRANSLITERATION\npublic static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_DEFINITIONS\nprivate final @android.annotation.NonNull android.view.translation.TranslationSpec mSourceSpec\nprivate final @android.annotation.NonNull android.view.translation.TranslationSpec mTargetSpec\nprivate final @android.view.translation.TranslationContext.TranslationFlag int mTranslationFlags\nprivate final @android.annotation.Nullable android.app.assist.ActivityId mActivityId\nprivate static  int defaultTranslationFlags()\nprivate static  android.app.assist.ActivityId defaultActivityId()\nprivate  void parcelActivityId(android.os.Parcel,int)\nprivate @android.annotation.Nullable android.app.assist.ActivityId unparcelActivityId(android.os.Parcel)\npublic @android.annotation.SystemApi @android.annotation.Nullable android.app.assist.ActivityId getActivityId()\nclass TranslationContext extends java.lang.Object implements [android.os.Parcelable]\nclass BaseBuilder extends java.lang.Object implements []\n@com.android.internal.util.DataClass(genHiddenConstDefs=true, genToString=true, genBuilder=true)\nclass BaseBuilder extends java.lang.Object implements []")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
