/* player.c - simple audio file player implementation
 *
 * This version uses PortAudio (http://www.portaudio.com) for audio
 * output.
 *
 * Copyright 2010 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "player.h"
#include "soundfile.h"
#include <assert.h>
#include <glib.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Ring buffer size, in kiB. */
static const int BUFFER_SIZE = 512;
/* Fill this portion of buffer before playback.  This value must be
 * between 0 and 1. */
static const float PREFILL = 0.2;
/* Begin to fill buffer after this portion of buffer has been
 * consumed.  This is intended to keep PortAudio stream callback from
 * signalling spaces semaphore all the time.  SLACK value must be
 * between 0 and 1, though not too close to zero to prevent buffer
 * underruns. */
static const float SLACK = 0.5;

pthread_mutex_t audio_init_count_mutex = PTHREAD_MUTEX_INITIALIZER;
static unsigned int audio_init_count;

/*
 * Message structure for passing requests to the controller thread.
 * These should not be built in user code; API functions (see
 * player.h) should be used instead.
 */
typedef struct {
    enum {
        CMD_PAUSE,
        CMD_PLAY,
        CMD_SEEK,
        CMD_STOP,
        CMD_TERM
    } command;
    double offset;
    int whence;
    sem_t *sem;
} Message;

/*
 * Player handle type.  This struct should not be accessed directly
 * from user code.  See init_player.
 */
struct Player {
    /* Ring buffer.  We always keep one vacant slot, so that it is
     * relatively easy to distinguish between empty (in == out) and
     * full (in + 1 == out) buffer. */
    struct {
        float *begin;
        float *end;
        float *in;
        float *out;
        int bufsize;
        int nframes;
        sem_t prefill;
        sem_t space;
        int prefill_sem_ok;
        int space_sem_ok;
    } buffer;

    /* reader thread */
    struct {
        /* thread status/control values */
        enum {
            READER_RESUME,
            READER_RUNNING,
            READER_STOP,
            READER_STOPPED,
            READER_TERM
        } control;
        /* condition variable and mutex for thread control */
        pthread_cond_t control_cond;
        pthread_mutex_t control_cond_mutex;

        pthread_t thread_id;
        int thread_ok;
        sem_t sem;
        sem_t notify;
        int sem_ok;
        int notify_ok;

        Sound_file *sndfile;
        Metadata metadata;
        char is_eof;
    } reader;

    /* controller thread */
    struct {
        pthread_t thread_id;
        pthread_t thread_ok;
        GAsyncQueue *messages;
    } controller;

    /* current playback status */
    Player_state state;    
    /* location at the most recent seek, in secods */
    double origin;              
    /* frames played after the most recent seek */
    int nplayed;         
    /* PortAudio output stream */
    PaStream *stream;           
};

/*
 * Reader thread.
 *
 * This thread implements the producer part of the producer-consumer
 * model by reading blocks of audio data from the sound file into the
 * ring buffer.  Some notes:
 *
 * - Thread blocks on space semaphore, and expects that the semaphore
 *   will be signaled at some point.
 *
 * - Thread reads the sound file until EOF is encountered, in which
 *   case it sets is_eof to nonzero and waits on control_cond.
 *
 * - Thread signals sem before waiting.
 *
 * - Thread signals sem and resets is_eof to zero when awakened from
 *   wait on control_cond.
 *
 * - After waking up from wait on control_cond, and filling PREFILL *
 *   BUFFER_SIZE portion of the buffer (or hitting EOF), the thread
 *   signals notify.
 */
static void *
reader_main(void *arg)
{
    Player *player;
    int nchannels;
    int notify;

    player = (Player *) arg;
    nchannels = player->reader.metadata.channels; /* for prettier
                                                   * offsetting */

    /* g_debug("starting reader thread, reader.control: %d",  */
    /*         player->reader.control); */

    for (;;) {
        pthread_mutex_lock(&player->reader.control_cond_mutex);
        while (player->reader.control == READER_STOP
               || player->reader.control == READER_TERM
               || player->reader.is_eof) {
            if (player->reader.control == READER_TERM) {
                /* g_debug("terminating reader thread"); */
                pthread_exit(NULL);
            }

            if (player->reader.control == READER_STOP) {
                sem_post(&player->reader.sem);
            }

            player->reader.control = READER_STOPPED;
            /* g_debug("reader stopped"); */
            pthread_cond_wait(&player->reader.control_cond,
                              &player->reader.control_cond_mutex);
            /* g_debug("reader resumed"); */
            assert(player->reader.control == READER_RESUME
                   || player->reader.control == READER_TERM);
            sem_post(&player->reader.sem);
            player->reader.is_eof = 0;
        }

        if (player->reader.control == READER_RESUME) notify = 1;
        else notify = 0;

        player->reader.control = READER_RUNNING;
        pthread_mutex_unlock(&player->reader.control_cond_mutex);
        /* g_debug("reader waiting for space"); */
        sem_wait(&player->buffer.space);

        /* Reader stop/termination request could occur while waiting
         * for buffer space. */
        if (player->reader.control == READER_STOP
            || player->reader.control == READER_TERM) continue;

        /* read in data */
        for (;;) {
            int nitems;
            int nread;
            if ((player->buffer.in == player->buffer.end 
                 && player->buffer.out == player->buffer.begin)
                || player->buffer.in + nchannels == player->buffer.out) {
                /* full buffer */
                /* g_debug("full buffer"); */
                if (notify) {
                    /* g_debug("notifying prefill"); */
                    sem_post(&player->reader.notify);
                }
                break;
            }

            if (player->buffer.in == player->buffer.end) {
                /* rewind the ring buffer */
                /* g_debug("rewound the ring buffer"); */
                player->buffer.in = player->buffer.begin;
            }

            if (player->buffer.in >= player->buffer.out) {
                nitems = ((player->buffer.end 
                           - player->buffer.in) / nchannels);
            } else {
                nitems = ((player->buffer.out - 
                           player->buffer.in) / nchannels) - 1;
            }

            assert(nitems > 0);

            /* g_debug("reading %d frames, %d frames in buffer, " */
            /*         "buffer begin = %p, in = %p, out = %p, end = %p", */
            /*         nitems, player->buffer.nframes, */
            /*         (void *) player->buffer.begin, */
            /*         (void *) player->buffer.in, */
            /*         (void *) player->buffer.out, */
            /*         (void *) player->buffer.end); */

            nread = read_pcm_data(player->reader.sndfile,
                                  player->buffer.in, nitems);

            /* g_debug("read %d frames", nread); */

            player->buffer.nframes += nread;
            player->buffer.in += (nread * nchannels);
            assert(player->buffer.nframes <= player->buffer.bufsize);

            if (nread != nitems) {
                /* EOF */
                /* g_debug("eof"); */
                player->reader.is_eof = 1;
                if (notify) {
                    sem_post(&player->reader.notify);
                    notify = 0;
                }
                break;
            }

            if (notify && player->buffer.nframes 
                > PREFILL * player->buffer.bufsize) {
                /* g_debug("read %d frames, notifying prefill", nframes); */
                sem_post(&player->reader.notify);
                notify = 0;
            }
        }
    }
}

/*
 * Stop reader thread.  Playback must be stopped, since this function
 * can garble the input buffer.
 *
 * Return 0 on success, or 1 if reader was already stopped.
 */
static int
stop_reader_thread(Player *player)
{
    /* g_debug("stopping reader"); */
    /* g_debug("locking reader mutex"); */
    pthread_mutex_lock(&player->reader.control_cond_mutex);
    assert(player->reader.control != READER_STOP);
    if (player->reader.control == READER_STOPPED) {
        /* g_debug("reader is already waiting"); */
        pthread_mutex_unlock(&player->reader.control_cond_mutex);
        /* g_debug("released reader mutex"); */
        return 1;
    } else {
        int space;
        /* g_debug("telling reader to stop"); */
        player->reader.control = READER_STOP;
        sem_getvalue(&player->buffer.space, &space);
        assert(space == 0 || space == 1);
        if (space == 0) {
            /* g_debug("signaling spaces in case reader " */
            /*         "is waiting"); */
            sem_post(&player->buffer.space);
        }
        pthread_mutex_unlock(&player->reader.control_cond_mutex);
        /* g_debug("released reader mutex"); */
        /* g_debug("waiting for reader to stop"); */
        sem_wait(&player->reader.sem);
        /* g_debug("reader is now sleeping"); */
        return 0;
    }
}

/*
 * Resume reader thread.  Block until the input buffer has been filled
 * with at least PREFILL * BUFFER_SIZE new frames, or EOF has been
 * encountered.
 *
 * Return 0 on success, or 1 if reader was already running.
 */
static int
resume_reader_thread(Player *player)
{
    /* g_debug("resuming reader"); */
    /* g_debug("locking reader mutex"); */
    pthread_mutex_lock(&player->reader.control_cond_mutex);
    assert(player->reader.control != READER_RESUME);
    if (player->reader.control == READER_RUNNING) {
        /* g_debug("reader is already running"); */
        pthread_mutex_unlock(&player->reader.control_cond_mutex);
        /* g_debug("released reader mutex"); */
        return 1;
    } else {
        /* g_debug("telling reader to resume"); */
        player->reader.control = READER_RESUME;
        pthread_cond_signal(&player->reader.control_cond);
        pthread_mutex_unlock(&player->reader.control_cond_mutex);
        /* g_debug("released reader mutex"); */
        /* g_debug("waiting for reader to wake up"); */
        sem_wait(&player->reader.sem);
        /* g_debug("reader is now running"); */
        /* g_debug("waiting for prefill"); */
        sem_wait(&player->reader.notify);
        return 0;
    }
}

/*
 * Terminate reader thread.  Playback must be stopped, since this
 * function can garble the input buffer.
 */
static void
terminate_reader_thread(Player *player)
{
    int space;
    assert(player->reader.control != READER_TERM);
    /* g_debug("terminating reader"); */
    /* g_debug("locking reader mutex"); */
    pthread_mutex_lock(&player->reader.control_cond_mutex);
    /* g_debug("telling reader to terminate"); */
    player->reader.control = READER_TERM;
    sem_getvalue(&player->buffer.space, &space);
    assert(space == 0 || space == 1);
    if (space == 0) {
        /* g_debug("signaling spaces in case reader is waiting"); */
        sem_post(&player->buffer.space);
    }
    pthread_cond_signal(&player->reader.control_cond);
    pthread_mutex_unlock(&player->reader.control_cond_mutex);
    /* g_debug("waiting for reader to terminate"); */
    pthread_join(player->reader.thread_id, NULL);
    /* g_debug("reader has been terminated"); */
}

/*
 * Seek player.  Semantics are as in fseek, offset is in seconds.  If
 * playback is in progress, it is continued from the new location.
 * Return the new location, or -1 on error.
 */
static double
seek(Player *player, double offset, int whence)
{
    int space;
    int state;
    double loc;

    /* g_debug("seeking to %f, whence %d", offset, whence); */

    /* stop playback */
    state = player->state.playback;
    Pa_StopStream(player->stream);
    stop_reader_thread(player);

    /* reset state */
    switch (whence) {
    case SEEK_SET:
        loc = offset;
        break;

    case SEEK_CUR:
        loc = player->state.location + offset;
        break;

    case SEEK_END:
        loc = player->reader.metadata.duration + offset;
        break;
    }

    if (loc < 0 || loc > player->reader.metadata.duration) {
        return -1;
    }

    player->state.location = loc;
    player->origin = loc;
    player->nplayed = 0;
    player->buffer.in = player->buffer.out = player->buffer.begin;
    player->buffer.nframes = 0;
    if (seek_sound_file(player->reader.sndfile, loc, SEEK_SET) == -1) {
        return -1;
    }

    sem_getvalue(&player->buffer.space, &space);
    assert(space == 0 || space == 1);
    if (space == 0) sem_post(&player->buffer.space);

    resume_reader_thread(player);
    /* resume playback if it was in progress */
    if (state == PLAYING) {
        Pa_StartStream(player->stream);
        player->state.playback = PLAYING;
    }

    return loc;
}

/*
 * Controller thread.
 *
 * This thread polls an asynchronous message queue for command
 * requests (stop playback, start playback, seek, etc.) and services
 * them in the order or arrival.  This way multiple threads can access
 * the same player in a serialized manner by calling the "API"
 * functions.  A message can contain a semaphore reference, which is
 * signalled after the corresponding command has been executed by the
 * controller (in case the calling code wants to synchronize with the
 * player).
 */
static void *
controller_main(void *arg)
{
    Player *player = (Player *) arg;
    Message *msg;
    int term = 0;

    for (;;) {
        msg = (Message *) g_async_queue_pop(player->controller.messages);
        /* g_debug("servicing new request %d", msg->command); */
        switch (msg->command) {
        case CMD_PAUSE:
            switch (player->state.playback) {
            case PLAYING:
                Pa_StopStream(player->stream);
                player->state.playback = PAUSED;
                break;
            case PAUSED:
                Pa_StartStream(player->stream);
                player->state.playback = PLAYING;
                break;
            default:
                break;
            }
            break;
        case CMD_PLAY:
            Pa_StartStream(player->stream);
            player->state.playback = PLAYING;
            break;
        case CMD_SEEK:
            seek(player, msg->offset, msg->whence);
            break;
        case CMD_STOP:
            Pa_StopStream(player->stream);
            player->state.playback = STOPPED;
            break;
        case CMD_TERM:
            if (player->stream) {
                Pa_StopStream(player->stream);
                if (Pa_CloseStream(player->stream) != paNoError) {
                    g_warning("failed to close playback stream");
                }
                player->stream = NULL;
            }
            if (player->reader.thread_ok) {
                terminate_reader_thread(player);
            }
            term = 1;
            break;
        }
        /* g_debug("request handled"); */
        if (msg->sem) sem_post(msg->sem);
        g_free(msg);
        if (term) {
            /* g_debug("terminating controller thread"); */
            pthread_exit(NULL);
        }
    }

    return NULL;
}

/* 
 * Stream callback for PortAudio.  Fill the audio output buffer from
 * the ring buffer, and signal space so that the reader thread can
 * fill the ring buffer.
 *
 * Return PaContinue so that PortAudio will keep on playback, or
 * PaComplete if EOF has been reached.
 */
static int
stream_callback(const void *input, void *output, 
                unsigned long nframes, 
                const PaStreamCallbackTimeInfo *timeinfo, 
                PaStreamCallbackFlags statusflags, 
                void *userdata)
{
    Player *player = (Player *) userdata;
    char eof = 0;
    int nchannels = player->reader.metadata.channels;
    float *outbuf = (float *) output;

    player->nplayed += nframes;
    while (nframes > 0) {
        size_t i;
        if (player->buffer.out == player->buffer.end) {
            /* rewind the ring buffer */
            /* g_debug("rewound the ring buffer"); */
            player->buffer.out = player->buffer.begin;
        }

        if (player->buffer.out == player->buffer.in) {
            /* buffer underflow or eof, output silence */
            if (!player->reader.is_eof) {
                g_warning("buffer underflow");
            } else {
                /* g_debug("eof"); */
                eof = 1;
            }
            while (nframes > 0) {
                for (i = 0; i < nchannels; i++) {
                    *outbuf++ = -1.0f;
                }
                nframes--;
            }
        } else {
            for (i = 0; i < nchannels; i++) {
                *outbuf++ = *player->buffer.out++;
            }
            player->buffer.nframes--;
            nframes--;
        }
    }

    /* update current playback location */
    player->state.location = (player->origin 
                              + (1.0 * player->nplayed 
                                 / player->reader.metadata.rate));
    /* g_debug("location %f", player->state.location); */

    if (player->buffer.nframes < SLACK * player->buffer.bufsize) {
        int space;
        sem_getvalue(&player->buffer.space, &space);
        assert(space == 0 || space == 1);
        if (space == 0) {
            sem_post(&player->buffer.space);
        }
    }

    if (!eof) return paContinue;
    else return paComplete;
}

/*
 * Stream finished callback for PortAudio.
 */
static void
stream_finished_callback(void *userdata)
{
    Player *player = (Player *) userdata;
    player->state.playback = STOPPED;
}

/*
 * Scan for default PortAudio host API output device.  Return the
 * device index, or -1 if no device was found.
 */
static PaDeviceIndex
scan_audio_output(void)
{
    const PaDeviceInfo *devinfo;
    PaDeviceIndex i;
    PaDeviceIndex outdev = -1;
    PaDeviceIndex ndevices;

    ndevices = Pa_GetDeviceCount();
    if (ndevices < 0) {
        g_warning("can't find available audio devices: %s",
                  Pa_GetErrorText(ndevices));
    }
    for (i = 0; i < ndevices; i++) {
        devinfo = Pa_GetDeviceInfo(i);
        if (!devinfo) continue;
        if (i == Pa_GetHostApiInfo
            (devinfo->hostApi)->defaultOutputDevice) {
            outdev = i;
            /* g_debug("found default host api output device index %d", */
            /*         outdev); */
            break;
        }
    }
    return outdev;
}

/*
 * Initialize a new player for given sound file.  Use the given
 * PortAudio device for audio output, or scan for default host API
 * output device if outdev = -1.
 *
 * Return handle to the new player, or NULL on error.
 */
Player *
init_player(const char *filename, PaDeviceIndex outdev)
{
    Player *player;
    PaError pa_rval;
    size_t nbytes;
    PaStreamParameters strparams;

    /* init player structure */
    player = g_malloc(sizeof(Player));
    player->stream = NULL;
    player->reader.sndfile = NULL;
    player->reader.sem_ok = player->reader.thread_ok = 0;
    player->reader.is_eof = 0;
    pthread_mutex_init(&player->reader.control_cond_mutex, NULL);
    pthread_cond_init(&player->reader.control_cond, NULL);
    player->controller.messages = NULL;
    player->controller.thread_ok = 0;
    player->buffer.begin = player->buffer.end = NULL;
    player->buffer.in = player->buffer.out = NULL;
    player->buffer.space_sem_ok = 0;
    player->state.location = 0;
    player->origin = 0;

    /* initialize audio */
    pthread_mutex_lock(&audio_init_count_mutex);
    if (audio_init_count == 0) {
        pa_rval = Pa_Initialize();
        /* scan for default device */

        if (pa_rval != paNoError) {
            g_warning("can't initialize audio subsystem: %s",
                      Pa_GetErrorText(pa_rval));
            close_player(player);
            pthread_mutex_unlock(&audio_init_count_mutex);
            return NULL;
        }

    }
    audio_init_count++;
    pthread_mutex_unlock(&audio_init_count_mutex);

    /* scan for default output device, if needed */
    if (outdev == -1) outdev = scan_audio_output();
    if (outdev == -1) {
        g_warning("can't find default output audio device");
        close_player(player);
        return NULL;
    }

    /* open sound file */
    if ((player->reader.sndfile = open_sound_file(filename))) {
        player->reader.metadata = get_metadata(player->reader.sndfile);
    } else {
        g_warning("can't open file '%s'", filename);
        close_player(player);
        return NULL;
    }

    /* open playback stream */
    memset(&strparams, 0, sizeof(PaStreamParameters));
    strparams.channelCount = player->reader.metadata.channels;
    strparams.device = outdev;
    strparams.sampleFormat = paFloat32;
    strparams.suggestedLatency = 
        Pa_GetDeviceInfo(outdev)->defaultHighOutputLatency;
    strparams.hostApiSpecificStreamInfo = NULL;
    pa_rval = Pa_OpenStream(&player->stream, NULL, &strparams,
                            player->reader.metadata.rate, 0,
                            paNoFlag, stream_callback,
                            player);
    if (pa_rval != paNoError) {
        g_warning("can't open audio stream for playback: %s",
                  Pa_GetErrorText(pa_rval));
        player->stream = NULL;
        close_player(player);
        return NULL;
    }
    Pa_SetStreamFinishedCallback(player->stream, stream_finished_callback);

    /* calculate buffer size */
    nbytes = 1024 * BUFFER_SIZE;
    nbytes -= nbytes % (player->reader.metadata.channels 
                        * sizeof(float));
    player->buffer.bufsize = nbytes / sizeof(float);
    player->buffer.begin = (float *) g_malloc(nbytes);
    player->buffer.end = player->buffer.begin + player->buffer.bufsize;
    player->buffer.in = player->buffer.out = player->buffer.begin;
    player->buffer.nframes = 0;
    /* g_debug("allocated %d bytes of memory for sample buffer", nbytes); */

    /* initialize semaphores */
    player->buffer.space_sem_ok =
        (sem_init(&player->buffer.space, 0, 1) == 0);
    player->reader.sem_ok =
        (sem_init(&player->reader.sem, 0, 0) == 0);
    player->reader.notify_ok =
        (sem_init(&player->reader.notify, 0, 0) == 0);
    if (!player->buffer.space_sem_ok
        || !player->reader.sem_ok
        || !player->reader.notify_ok) {
        close_player(player);
        return NULL;
    }

    /* create threads */
    if (!g_thread_supported()) g_thread_init(NULL);

    player->controller.messages = g_async_queue_new();
    player->controller.thread_ok =
        (pthread_create(&player->controller.thread_id, NULL,
                        controller_main, player) == 0);
    if (!player->controller.thread_ok) {
        g_warning("can't create controller thread");
        close_player(player);
        return NULL;
    }

    player->reader.control = READER_STOP;
    player->reader.thread_ok =
        (pthread_create(&player->reader.thread_id, NULL,
                        reader_main, player) == 0);
    if (!player->reader.thread_ok) {
        g_warning("can't create reader thread");
        close_player(player);
        return NULL;
    }
    /* g_debug("waiting for reader"); */
    sem_wait(&player->reader.sem);

    /* g_debug("player %p ok", (void *) player); */
    return player;
}

/*
 * Close player and free allocated resources.
 */
void
close_player(Player *player)
{
    Message *msg;
    assert(player);

    /* g_debug("closing player %p", (void *) player); */

    if (player->controller.thread_ok) {
        /* terminate threads */
        msg = g_malloc(sizeof(Message));
        msg->command = CMD_TERM;
        msg->sem = NULL;
        g_async_queue_push(player->controller.messages, msg);
        pthread_join(player->controller.thread_id, NULL);
    }

    if (player->reader.sndfile &&
        close_sound_file(player->reader.sndfile) != 0) {
        g_warning("failed to close sound file");
    }

    pthread_mutex_lock(&audio_init_count_mutex);
    audio_init_count--;
    if (audio_init_count == 0) {
        if (Pa_Terminate() != paNoError) {
            g_warning("failed to shut down audio subsystem");
        }
    }
    pthread_mutex_unlock(&audio_init_count_mutex);

    if (player->buffer.begin) g_free(player->buffer.begin);
    pthread_mutex_destroy(&player->reader.control_cond_mutex);
    pthread_cond_destroy(&player->reader.control_cond);
    if (player->buffer.space_sem_ok) sem_destroy(&player->buffer.space);
    if (player->reader.sem_ok) sem_destroy(&player->reader.sem);

    g_free(player);
    /* g_debug("closed player %p", (void *) player); */
}

/*
 * Return sound file metadata from player.
 */
Metadata
get_player_metadata(Player *player)
{
    assert(player);
    return player->reader.metadata;
}

/*
 * Return the current playback state.
 */
Player_state
get_player_state(Player *player)
{
    assert(player);
    return player->state;
}

/*
 * Start playback.  If sem is non-NULL, it will be signalled after
 * playback has been started.
 */
void
start_player(Player *player, sem_t *sem)
{
    Message *msg;
    assert(player);
    msg = g_malloc(sizeof(Message));
    msg->command = CMD_PLAY;
    msg->sem = sem;
    g_async_queue_push(player->controller.messages, msg);
}

/*
 * Stop playback.  If sem is non-NULL, it will be signalled after
 * playback has been stopped.
 */
void
stop_player(Player *player, sem_t *sem)
{
    Message *msg;
    assert(player);
    msg = g_malloc(sizeof(Message));
    msg->command = CMD_STOP;
    msg->sem = sem;
    g_async_queue_push(player->controller.messages, msg);
}

/*
 * Pause or resume playback.  If sem is not NULL,
 * it will be signalled after seeking.
 */
void
pause_or_resume_player(Player *player, sem_t *sem)
{
    Message *msg;
    assert(player);
    msg = g_malloc(sizeof(Message));
    msg->command = CMD_PAUSE;
    msg->sem = sem;
    g_async_queue_push(player->controller.messages, msg);
}

/*
 * Seek player using seek function (which see).  If sem is not NULL,
 * it will be signalled after seeking.
 */
void
seek_player(Player *player, double offset, int whence, sem_t *sem)
{
    Message *msg;
    assert(player);
    msg = g_malloc(sizeof(Message));
    msg->command = CMD_SEEK;
    msg->offset = offset;
    msg->whence = whence;
    msg->sem = sem;
    g_async_queue_push(player->controller.messages, msg);
}
