﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2Request.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/BehaviorOnMxFailure.h>

#include <utility>

namespace Aws {
namespace SESV2 {
namespace Model {

/**
 * <p>A request to configure the custom MAIL FROM domain for a verified
 * identity.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutEmailIdentityMailFromAttributesRequest">AWS
 * API Reference</a></p>
 */
class PutEmailIdentityMailFromAttributesRequest : public SESV2Request {
 public:
  AWS_SESV2_API PutEmailIdentityMailFromAttributesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutEmailIdentityMailFromAttributes"; }

  AWS_SESV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The verified email identity.</p>
   */
  inline const Aws::String& GetEmailIdentity() const { return m_emailIdentity; }
  inline bool EmailIdentityHasBeenSet() const { return m_emailIdentityHasBeenSet; }
  template <typename EmailIdentityT = Aws::String>
  void SetEmailIdentity(EmailIdentityT&& value) {
    m_emailIdentityHasBeenSet = true;
    m_emailIdentity = std::forward<EmailIdentityT>(value);
  }
  template <typename EmailIdentityT = Aws::String>
  PutEmailIdentityMailFromAttributesRequest& WithEmailIdentity(EmailIdentityT&& value) {
    SetEmailIdentity(std::forward<EmailIdentityT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The custom MAIL FROM domain that you want the verified identity to use. The
   * MAIL FROM domain must meet the following criteria:</p> <ul> <li> <p>It has to be
   * a subdomain of the verified identity.</p> </li> <li> <p>It can't be used to
   * receive email.</p> </li> <li> <p>It can't be used in a "From" address if the
   * MAIL FROM domain is a destination for feedback forwarding emails.</p> </li>
   * </ul>
   */
  inline const Aws::String& GetMailFromDomain() const { return m_mailFromDomain; }
  inline bool MailFromDomainHasBeenSet() const { return m_mailFromDomainHasBeenSet; }
  template <typename MailFromDomainT = Aws::String>
  void SetMailFromDomain(MailFromDomainT&& value) {
    m_mailFromDomainHasBeenSet = true;
    m_mailFromDomain = std::forward<MailFromDomainT>(value);
  }
  template <typename MailFromDomainT = Aws::String>
  PutEmailIdentityMailFromAttributesRequest& WithMailFromDomain(MailFromDomainT&& value) {
    SetMailFromDomain(std::forward<MailFromDomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action to take if the required MX record isn't found when you send an
   * email. When you set this value to <code>UseDefaultValue</code>, the mail is sent
   * using <i>amazonses.com</i> as the MAIL FROM domain. When you set this value to
   * <code>RejectMessage</code>, the Amazon SES API v2 returns a
   * <code>MailFromDomainNotVerified</code> error, and doesn't attempt to deliver the
   * email.</p> <p>These behaviors are taken when the custom MAIL FROM domain
   * configuration is in the <code>Pending</code>, <code>Failed</code>, and
   * <code>TemporaryFailure</code> states.</p>
   */
  inline BehaviorOnMxFailure GetBehaviorOnMxFailure() const { return m_behaviorOnMxFailure; }
  inline bool BehaviorOnMxFailureHasBeenSet() const { return m_behaviorOnMxFailureHasBeenSet; }
  inline void SetBehaviorOnMxFailure(BehaviorOnMxFailure value) {
    m_behaviorOnMxFailureHasBeenSet = true;
    m_behaviorOnMxFailure = value;
  }
  inline PutEmailIdentityMailFromAttributesRequest& WithBehaviorOnMxFailure(BehaviorOnMxFailure value) {
    SetBehaviorOnMxFailure(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_emailIdentity;

  Aws::String m_mailFromDomain;

  BehaviorOnMxFailure m_behaviorOnMxFailure{BehaviorOnMxFailure::NOT_SET};
  bool m_emailIdentityHasBeenSet = false;
  bool m_mailFromDomainHasBeenSet = false;
  bool m_behaviorOnMxFailureHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
