﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glacier/GlacierRequest.h>
#include <aws/glacier/Glacier_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glacier {
namespace Model {

/**
 * <p>The input values for <code>AbortVaultLock</code>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glacier-2012-06-01/AbortVaultLockInput">AWS
 * API Reference</a></p>
 */
class AbortVaultLockRequest : public GlacierRequest {
 public:
  AWS_GLACIER_API AbortVaultLockRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AbortVaultLock"; }

  AWS_GLACIER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <code>AccountId</code> value is the AWS account ID. This value must match
   * the AWS account ID associated with the credentials used to sign the request. You
   * can either specify an AWS account ID or optionally a single '<code>-</code>'
   * (hyphen), in which case Amazon Glacier uses the AWS account ID associated with
   * the credentials used to sign the request. If you specify your account ID, do not
   * include any hyphens ('-') in the ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  AbortVaultLockRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the vault.</p>
   */
  inline const Aws::String& GetVaultName() const { return m_vaultName; }
  inline bool VaultNameHasBeenSet() const { return m_vaultNameHasBeenSet; }
  template <typename VaultNameT = Aws::String>
  void SetVaultName(VaultNameT&& value) {
    m_vaultNameHasBeenSet = true;
    m_vaultName = std::forward<VaultNameT>(value);
  }
  template <typename VaultNameT = Aws::String>
  AbortVaultLockRequest& WithVaultName(VaultNameT&& value) {
    SetVaultName(std::forward<VaultNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_vaultName;
  bool m_accountIdHasBeenSet = false;
  bool m_vaultNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
