// Copyright (C) 2025 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only

#ifndef QCOFFSCREENCANVAS_H
#define QCOFFSCREENCANVAS_H

#include <QtCanvasPainter/qtcanvaspainterglobal.h>
#include <QtCore/qshareddata.h>
#include <QtCore/qobject.h>
#include <QtGui/qcolor.h>

QT_BEGIN_NAMESPACE

class QCOffscreenCanvasPrivate;
class QRhiTexture;

QT_DECLARE_QESDP_SPECIALIZATION_DTOR(QCOffscreenCanvasPrivate)

class Q_CANVASPAINTER_EXPORT QCOffscreenCanvas
{
    Q_GADGET

public:
    enum class Flag {
        // Requests preserving (no clearing) of the color texture.
        // There's a catch: this will not always work with MSAA, depending on underlying details (on GLES, with certain extensions present) .
        // So probably going to be limited to non-multisample canvases.
        PreserveContents = 0x01
    };
    Q_DECLARE_FLAGS(Flags, Flag)

    QCOffscreenCanvas();
    QCOffscreenCanvas(const QCOffscreenCanvas &canvas) noexcept;
    ~QCOffscreenCanvas();
    QCOffscreenCanvas &operator=(const QCOffscreenCanvas &canvas) noexcept;
    QCOffscreenCanvas(QCOffscreenCanvas &&other) noexcept = default;
    QT_MOVE_ASSIGNMENT_OPERATOR_IMPL_VIA_PURE_SWAP(QCOffscreenCanvas)
    void swap(QCOffscreenCanvas &other) noexcept { d.swap(other.d); }

    bool operator==(const QCOffscreenCanvas &canvas) const;
    inline bool operator!=(const QCOffscreenCanvas &canvas) const { return !(operator==(canvas)); }

    void detach();

    bool isNull() const;

    // flags and settings such as multisample count must be controlled by the
    // canvas factory function, we cannot have simple setters here, since under
    // the hood everything is immutable...
    Flags flags() const;

    QColor fillColor() const;
    void setFillColor(const QColor &color);

    QRhiTexture *texture() const;

private:
    QExplicitlySharedDataPointer<QCOffscreenCanvasPrivate> d;
    friend class QCOffscreenCanvasPrivate;
    friend class QCPainterRhiRenderer;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(QCOffscreenCanvas::Flags)
Q_DECLARE_SHARED(QCOffscreenCanvas)

QT_END_NAMESPACE

#endif // QCOFFSCREENCANVAS_H
