/*
 * TemplateRunner.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2003 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, suhler.
 *
 * Version:  2.3
 * Created by suhler on 99/05/06
 * Last modified by suhler on 03/08/01 16:19:00
 */

package sunlabs.brazil.template;

import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.session.SessionManager;
import sunlabs.brazil.util.regexp.Regexp;
import sunlabs.brazil.util.regexp.Regsub;
import sunlabs.brazil.util.LexHTML;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;

/**
 * Class for processing html templates.
 * An html template is an ordinary html string, with additional
 * application specific tags added (sort of like XML).  Each
 * tag is mapped to a java method of a template class, that rewrites its
 * tag into normal html.
 * <p>
 * The mechanism used to map templates into sessions is inadaquate, and
 * should be fixed in a future version.  In the current implementation,
 * Each session maintains its own set of template instances.  Instance
 * variables in template classes may be used to hold session specific
 * state.  Calls to a template are synchronized on the session id;
 * only one request per session is dealt with simultaneously.
 *
 * @author Colin Stevens
 * @author Stephen Uhler
 * @version %W
 */

public class TemplateRunner
{
    static final String TAG_PREFIX = "tagPrefix"; 
    static final String TEMPLATE = "template";

    private Server server;
    private String prefix;		// prefix of the invoker

    private int tagsProcessed = 0;
    private int tagsSeen = 0;
    private String error = null; 	// The last error

    private Class[] types;		// Types of templates.
    private String[] prefixes;		// Template specific prefixes
    private Hashtable dispatch;		// Map: tag -> class index, method.

    private static final Regexp hex = new Regexp("_x[0-9a-zA-Z][0-9a-zA-Z]");
    private static final Regexp.Filter hexFilter = new Regexp.Filter() {
	public boolean filter(Regsub rs, StringBuffer sb) {
	    String match = rs.matched();
	    int hi = Character.digit(match.charAt(2), 16);
	    int lo = Character.digit(match.charAt(3), 16);
	    sb.append((char) ((hi << 4) | lo));

	    return true;
	}
    };

    private static class Dispatch {
	int index;
	Method method;
	String prefix;

	public Dispatch(int index, Method method, String prefix) {
	    this.index = index;
	    this.method = method;
	    this.prefix = prefix;
	}
    }

    /**
     * Process an HTML template with a template processing class.
     * We peruse the template class, looking at all of its methods.
     * When when we process a template, we match the html tags against
     * the declared methods in the template class.  Each
     * method name of the form <code>tag_<i>xxx</i></code> or
     * <code>tag_slash_<i>xxx</i></code> is invoked when the corrosponding
     * <i>&lt;xxx&gt;</i> or <i>&lt;/xxx&gt;</i> tag is found.
     * <p>
     * Each instance of <code>_x<i>nn</i></code> in the method name
     * is replaced by the corrosponding hex code for the character.
     * This allows non-printable tags to to be processed with templates.
     * <p>
     * The methods <code>init</code> and <code>done</code> are each called
     * once, at the beginning and end of the page respectively.  These methods
     * are called for all templates, in the order they are specified in
     * the <i>templates</i> parameter.
     * <p>
     * There are three methods taht may be defined that don't follow the naming
     * convension described above. They are:
     * <ul>
     * <li><code>comment</code><br>
     * is called for each html/XML comment.
     * <li><code>string</code><br>
     * is called for all text between any tags.
     * <li><code>defaultTag</code><br>
     * is called for every tag that does not specifically have a tag method.
     * If more than one template defines one of these methods, only the
     * first template's method will be called.
     * </ul>
     * <p>
     * If the server property "tagPrefix" associated with each template's
     * properties prefix exists, it is used to prefix each tag name
     * (this feature is for experimental support of XML namespaces,
     * and probably doesn't belong here).
     * <p>
     * @param	server
     *		The HTTP server that created the <code>Handler</code> or
     *		<code>Filter</code> that invoked this
     *		<code>TemplateRunner</code>
     * @param	prefix
     *		The prefix associated with the parent <code>Handler</code>
     *		or <code>Filter</code>
     * @param	names
     *		The names of the Template classes or property prefixes (i.e.
     *          tokens) that, when concatenated with ".class" define a property
     *          that names a Template class.  This TemplateRunner will
     *		dispatch to the methods described by the union of all the
     *		tag methods in the given Template classes.
     *		<p>
     *		The <code>init</code> and <code>done</code> methods for each
     *		template specified will be called in order.  If any of
     *		the calls returns <code>false</code>, this handler terminates
     *		and no output is generated.
     *		<p>
     *		The names "comment", "string",  and "defaultTag" are
     *		handled specially.
     */
    public
    TemplateRunner(Server server, String prefix, String names)
	throws ClassNotFoundException, ClassCastException
    {
	this.server = server;
	this.prefix = prefix;

	dispatch = new Hashtable();

	Vector types = new Vector();
	Vector prefixes = new Vector();
	int count = 0;

	StringTokenizer st = new StringTokenizer(names);
	while (st.hasMoreTokens()) {
	    String temPrefix = null;
	    String temName = st.nextToken();
	    // System.out.println("Processing template: " + temName);
	    String className = server.props.getProperty(temName + ".class");
	    if (className == null) {
		className = temName;
		temPrefix = prefix;
	    } else {
		temPrefix = temName + ".";
	    }

	    Class temType = Class.forName(className.trim());
            if (TemplateInterface.class.isAssignableFrom(temType) == false) {
		throw new ClassCastException(temType.getName());
	    }

	    types.addElement(temType);
	    prefixes.addElement(temPrefix);

	    Method[] methods = temType.getMethods();
	    String tagPrefix = server.props.getProperty(temPrefix + TAG_PREFIX,
		    server.props.getProperty(prefix + TAG_PREFIX, ""));
	    for (int i = 0; i < methods.length; i++) {
		Method method = methods[i];
		String name = method.getName();
		if (name.equals("comment") && dispatch.get(name) == null) {
		    dispatch.put(name, new Dispatch(count, method, temPrefix));
		    // System.out.println("Found comment in: " + temName);
		    continue;
		}
		if (name.equals("string") && dispatch.get(name) == null) {
		    dispatch.put(name, new Dispatch(count, method, temPrefix));
		    // System.out.println("Found string in: " + temName);
		    continue;
		}
		if (name.equals("defaultTag") && dispatch.get(name) == null) {
		    dispatch.put(name, new Dispatch(count, method, temPrefix));
		    // System.out.println("Found defaultTag in: " + temName);
		    continue;
		}
		if (name.startsWith("tag_") == false) {
		    continue;
		}
		name = name.substring(4);
		if (name.startsWith("slash_")) {
		    name = "/" + tagPrefix + name.substring(6);
		} else {
		    name = tagPrefix + name;
		}
		name = hex.sub(name, hexFilter);

		if (dispatch.get(name) == null) {
		    dispatch.put(name, new Dispatch(count, method, temPrefix));
		}
	    }
	    count++;
	}

	types.copyInto(this.types = new Class[count]);
	prefixes.copyInto(this.prefixes = new String[count]);
    }

    /**
     * Process an html template file, using the supplied template processing
     * Return the content of the template just processed, or null if
     * there was no template processed.
     *
     * @param   content
     *		The template.
     *
     * @param   sessionId
     *		An arbitrary identifier used to locate the correct instance
     *		of the template class for processing this template.  The
     *		first time an identifier is used, a new instance is created.
     *
     * @param   args
     *		The arguments passed to the templates init method.
     *
     * @return 	content or null
     */
    public String
    process(Request request, String content, String sessionId)
    {
	tagsProcessed = 0;
	tagsSeen = 0;
	// error = null;

	/*
	 * Now look up the session object.  If this is a new session, then we
	 * get a new object, otherwise we get the last one we used for this
	 * session
	 */

	Vector templates = (Vector) SessionManager.getSession(sessionId,
		prefix + TEMPLATE, Vector.class);

	synchronized (templates) {
	    if (templates.size() != types.length) {
		templates.setSize(0);
		for (int i = 0; i < types.length; i++) {
		    try {
			templates.addElement(types[i].newInstance());
		    } catch (Exception e) {
			throw new ClassCastException("Missing constructor " +
				types[i].getName() + "()");
		    }
		}
	    }
	    RewriteContext hr = new RewriteContext(server, prefix, request,
		    content, sessionId, this, templates);

	    /*
	     * Call the init() method of all the Templates.
	     */
	    for (int i = 0; i < types.length; i++) {
		Template obj = (Template) templates.elementAt(i);
		hr.prefix = prefixes[i];  // see discussion in RewriteContext
		if (obj.init(hr) == false) {
		    error = types[i] + " " + request.url +
		            ": init Rejecting request";
		    return null;
		}
	    }

	    /*
	     * Process the document.
	     */
	    while (hr.nextToken()) {
		process(hr);
	    }

	    /*
	     * Call the done() method of all the Templates.
	     */
	    for (int i = 0; i < templates.size(); i++) {
		Template obj = (Template) templates.elementAt(i);
		hr.prefix = prefixes[i];  // see discussion in RewriteContext
		if (obj.done(hr) == false) {
		    error = types[i] + " " + request.url +
		       	    ": done rejecting request";
		    return null;
		}
	    }
	    return hr.toString();
	}
    }

    /**
     * Processes the next token in the HTML document represented by the
     * given <code>RewriteContext</code>.  Processing a token involves either
     * dispatching to a tag-handling method in one of the
     * <code>Template</code> objects, or just advancing to the next token
     * if no <code>Template</code> was interested.
     *
     * @param	hr
     *		The RewriteContext holding the state of the HTML document.
     */
    public void
    process(RewriteContext hr)
    {
	switch (hr.getType()) {
	    case LexHTML.COMMENT:
	    case LexHTML.STRING:
	    case LexHTML.TAG: {
		String tag;
		if (hr.getType() == LexHTML.COMMENT) {
		    tag = "comment";
		    // System.out.println("(comment)");
		} else if (hr.getType() == LexHTML.STRING) {
		    tag = "string";
		} else {
		    tag = hr.getTag();
		    tagsSeen++;
		}
		Dispatch d = (Dispatch) dispatch.get(tag);
		if (hr.getType() == LexHTML.TAG && d == null) {
		    d = (Dispatch) dispatch.get("defaultTag");
		}
		if (d != null) {
		    Template obj = (Template) hr.templates.elementAt(d.index);
		    // see discussion in RewriteContext regarding prefix
		    hr.prefix = d.prefix;
		    try {
			d.method.invoke(obj, hr.args);
			tagsProcessed++;
		    } catch (InvocationTargetException e) {
			hr.append("<!-- " + tag + ":  " +
				e.getTargetException() + " -->");
			e.getTargetException().printStackTrace();
		    } catch (Exception e) {
			hr.append("<!-- " + tag + ":  " + e + " -->");
			e.printStackTrace();
		    }
		}
	    }
	}
    }

    /**
     * Return the last error message generated, or null of no
     * errors have occurred since the last call to "process".
     * XXX not thread safe between calls to process() and getError().
     */

    public String
    getError() {
	return error;
    }

    /**
     * Return the # of HTML tags seen in the previous call to "process".
     */

    public int
    tagsSeen() {
	return tagsSeen;
    }
    /**
     * Return the # of tags replaced in the previous call to "process".
     */

    public int
    tagsProcessed() {
	return tagsProcessed;
    }
}
