/*
 * Decompiled with CFR 0.152.
 */
package tachyon.client.file;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tachyon.Constants;
import tachyon.client.BoundedStream;
import tachyon.client.ClientContext;
import tachyon.client.Seekable;
import tachyon.client.TachyonStorageType;
import tachyon.client.block.BlockInStream;
import tachyon.client.block.BufferedBlockOutStream;
import tachyon.client.block.LocalBlockInStream;
import tachyon.client.file.FileSystemContext;
import tachyon.client.file.UnderStoreFileInStream;
import tachyon.client.file.options.InStreamOptions;
import tachyon.master.block.BlockId;
import tachyon.thrift.FileInfo;
import tachyon.util.network.NetworkAddressUtils;

public final class FileInStream
extends InputStream
implements BoundedStream,
Seekable {
    private static final Logger LOG = LoggerFactory.getLogger((String)Constants.LOGGER_TYPE);
    private static final String ERR_BLOCK_INDEX = "Current block index exceeds max index.";
    private static final String ERR_BUFFER_NULL = "Cannot read with a null buffer.";
    private static final String ERR_BUFFER_STATE = "Buffer length: %s, offset: %s, len: %s";
    private static final String ERR_SEEK_PAST_END_OF_FILE = "Seek position past end of file: %s";
    private static final String ERR_SEEK_NEGATIVE = "Seek position is negative: %s";
    private final TachyonStorageType mTachyonStorageType;
    private final long mBlockSize;
    private final long mFileLength;
    private final FileSystemContext mContext;
    private final FileInfo mFileInfo;
    private boolean mClosed;
    private boolean mShouldCacheCurrentBlock;
    private long mPos;
    private BlockInStream mCurrentBlockInStream;
    private BufferedBlockOutStream mCurrentCacheStream;

    public FileInStream(FileInfo info, InStreamOptions options) {
        this.mFileInfo = info;
        this.mBlockSize = info.getBlockSizeBytes();
        this.mFileLength = info.getLength();
        this.mContext = FileSystemContext.INSTANCE;
        this.mTachyonStorageType = options.getTachyonStorageType();
        this.mShouldCacheCurrentBlock = this.mTachyonStorageType.isStore();
        this.mClosed = false;
    }

    @Override
    public void close() throws IOException {
        if (this.mClosed) {
            return;
        }
        if (this.mCurrentBlockInStream != null) {
            this.mCurrentBlockInStream.close();
        }
        this.closeCacheStream();
        this.mClosed = true;
    }

    @Override
    public int read() throws IOException {
        if (this.mPos >= this.mFileLength) {
            return -1;
        }
        this.checkAndAdvanceBlockInStream();
        int data = this.mCurrentBlockInStream.read();
        ++this.mPos;
        if (this.mShouldCacheCurrentBlock) {
            try {
                this.mCurrentCacheStream.write(data);
            }
            catch (IOException ioe) {
                LOG.warn("Block of ID " + this.getCurrentBlockId() + " could not be cached into Tachyon. Exception:" + ioe.getMessage());
                this.mShouldCacheCurrentBlock = false;
            }
        }
        return data;
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        Preconditions.checkArgument((b != null ? 1 : 0) != 0, (Object)ERR_BUFFER_NULL);
        Preconditions.checkArgument((off >= 0 && len >= 0 && len + off <= b.length ? 1 : 0) != 0, (String)ERR_BUFFER_STATE, (Object[])new Object[]{b.length, off, len});
        if (len == 0) {
            return 0;
        }
        if (this.mPos >= this.mFileLength) {
            return -1;
        }
        int currentOffset = off;
        int bytesLeftToRead = len;
        while (bytesLeftToRead > 0 && this.mPos < this.mFileLength) {
            this.checkAndAdvanceBlockInStream();
            int bytesToRead = (int)Math.min((long)bytesLeftToRead, this.mCurrentBlockInStream.remaining());
            int bytesRead = this.mCurrentBlockInStream.read(b, currentOffset, bytesToRead);
            if (bytesRead > 0 && this.mShouldCacheCurrentBlock) {
                try {
                    this.mCurrentCacheStream.write(b, currentOffset, bytesRead);
                }
                catch (IOException ioe) {
                    LOG.warn("Failed to write into TachyonStorage, the block " + this.getCurrentBlockId() + " will not be in TachyonStorage. Exception:" + ioe.getMessage());
                    this.mShouldCacheCurrentBlock = false;
                }
            }
            if (bytesRead == -1) continue;
            this.mPos += (long)bytesRead;
            bytesLeftToRead -= bytesRead;
            currentOffset += bytesRead;
        }
        return len - bytesLeftToRead;
    }

    @Override
    public long remaining() {
        return this.mFileLength - this.mPos;
    }

    @Override
    public void seek(long pos) throws IOException {
        if (this.mPos == pos) {
            return;
        }
        Preconditions.checkArgument((pos >= 0L ? 1 : 0) != 0, (String)ERR_SEEK_NEGATIVE, (Object[])new Object[]{pos});
        Preconditions.checkArgument((pos < this.mFileLength ? 1 : 0) != 0, (String)ERR_SEEK_PAST_END_OF_FILE, (Object[])new Object[]{pos});
        this.seekBlockInStream(pos);
        this.checkAndAdvanceBlockInStream();
        this.mCurrentBlockInStream.seek(this.mPos % this.mBlockSize);
    }

    @Override
    public long skip(long n) throws IOException {
        if (n <= 0L) {
            return 0L;
        }
        long toSkip = Math.min(n, this.mFileLength - this.mPos);
        long newPos = this.mPos + toSkip;
        long toSkipInBlock = newPos / this.mBlockSize > this.mPos / this.mBlockSize ? newPos % this.mBlockSize : toSkip;
        this.seekBlockInStream(newPos);
        this.checkAndAdvanceBlockInStream();
        if (toSkipInBlock != this.mCurrentBlockInStream.skip(toSkipInBlock)) {
            throw new IOException("The underlying BlockInStream could not skip " + toSkip);
        }
        return toSkip;
    }

    private void checkAndAdvanceBlockInStream() throws IOException {
        long currentBlockId = this.getCurrentBlockId();
        if (this.mCurrentBlockInStream == null || this.mCurrentBlockInStream.remaining() == 0L) {
            this.closeCacheStream();
            this.updateBlockInStream(currentBlockId);
            if (this.mShouldCacheCurrentBlock) {
                try {
                    this.mCurrentCacheStream = this.mContext.getTachyonBlockStore().getOutStream(currentBlockId, -1L, NetworkAddressUtils.getLocalHostName(ClientContext.getConf()));
                }
                catch (IOException ioe) {
                    LOG.warn("Failed to get TachyonStore stream, the block " + currentBlockId + " will not be in TachyonStorage. Exception:" + ioe.getMessage());
                    this.mShouldCacheCurrentBlock = false;
                }
            }
        }
    }

    private void closeCacheStream() throws IOException {
        if (this.mCurrentCacheStream == null) {
            return;
        }
        if (this.mCurrentCacheStream.remaining() == 0L) {
            this.mCurrentCacheStream.close();
        } else {
            this.mCurrentCacheStream.cancel();
        }
        this.mShouldCacheCurrentBlock = false;
    }

    private long getCurrentBlockId() {
        if (this.mPos == this.mFileLength) {
            return -1L;
        }
        int index = (int)(this.mPos / this.mBlockSize);
        Preconditions.checkState((index < this.mFileInfo.blockIds.size() ? 1 : 0) != 0, (Object)ERR_BLOCK_INDEX);
        return this.mFileInfo.blockIds.get(index);
    }

    private void seekBlockInStream(long newPos) throws IOException {
        long oldBlockId = this.getCurrentBlockId();
        this.mPos = newPos;
        this.closeCacheStream();
        long currentBlockId = this.getCurrentBlockId();
        if (oldBlockId != currentBlockId) {
            this.updateBlockInStream(currentBlockId);
            if (this.mPos % this.mBlockSize == 0L && this.mShouldCacheCurrentBlock) {
                try {
                    this.mCurrentCacheStream = this.mContext.getTachyonBlockStore().getOutStream(currentBlockId, -1L, NetworkAddressUtils.getLocalHostName(ClientContext.getConf()));
                }
                catch (IOException ioe) {
                    LOG.warn("Failed to write to TachyonStore stream, block " + this.getCurrentBlockId() + " will not be in TachyonStorage. Exception:" + ioe.getMessage());
                    this.mShouldCacheCurrentBlock = false;
                }
            } else {
                this.mShouldCacheCurrentBlock = false;
            }
        }
    }

    private void updateBlockInStream(long blockId) throws IOException {
        if (this.mCurrentBlockInStream != null) {
            this.mCurrentBlockInStream.close();
        }
        try {
            if (this.mTachyonStorageType.isPromote()) {
                try {
                    this.mContext.getTachyonBlockStore().promote(blockId);
                }
                catch (IOException ioe) {
                    LOG.warn("Promotion of block " + blockId + " failed.");
                }
            }
            this.mCurrentBlockInStream = this.mContext.getTachyonBlockStore().getInStream(blockId);
            this.mShouldCacheCurrentBlock = !(this.mCurrentBlockInStream instanceof LocalBlockInStream) && this.mTachyonStorageType.isStore();
        }
        catch (IOException ioe) {
            LOG.debug("Failed to get BlockInStream for " + blockId + ", using ufs instead. Exception:" + ioe.getMessage());
            if (!this.mFileInfo.isPersisted) {
                LOG.error("Could not obtain data for " + blockId + " from Tachyon and data is not persisted in under storage.");
                throw ioe;
            }
            long blockStart = BlockId.getSequenceNumber(blockId) * this.mBlockSize;
            this.mCurrentBlockInStream = new UnderStoreFileInStream(blockStart, this.mBlockSize, this.mFileInfo.getUfsPath());
            this.mShouldCacheCurrentBlock = this.mTachyonStorageType.isStore();
        }
    }
}

