/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.pig.builtin;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.pig.EvalFunc;
import org.apache.pig.FuncSpec;
import org.apache.pig.PigWarning;
import org.apache.pig.data.DataType;
import org.apache.pig.data.Tuple;
import org.apache.pig.impl.logicalLayer.FrontendException;
import org.apache.pig.impl.logicalLayer.schema.Schema;

/**
 * REPLACE implements eval function to replace part of a string.
 * Example:<code>
 *      A = load 'mydata' as (name);
 *      B = foreach A generate REPLACE(name, 'blabla', 'bla');
 * The first argument is a string on which to perform the operation. The second argument
 * is treated as a regular expression. The third argument is the replacement string.
 * This is a wrapper around Java's String.replaceAll(String, String);
 *
 */
public class REPLACE extends EvalFunc<String>
{
    private Pattern mPattern = null;

    /**
     * Method invoked on every tuple during foreach evaluation
     * @param input tuple; first column is assumed to have the column to convert
     * @exception java.io.IOException
     */
    @Override
    public String exec(Tuple input) throws IOException {
        if (input == null || input.size() < 3)
            return null;

        String source = (String)input.get(0);
        String target = (String)input.get(1);

        if (target == null) {
            warn("Replace : Regular expression is null", PigWarning.UDF_WARNING_1);
            return null;
        }
        
        if (mPattern == null || ! target.equals(mPattern.pattern())) {
            try {
                mPattern = Pattern.compile(target);
            } catch (Exception e) {
                warn("Replace : Mal-Formed Regular expression : " + target, PigWarning.UDF_WARNING_1);
                return null;
            }
        }            
        
        String replacewith = (String)input.get(2);
        
        try {    
           return mPattern.matcher(source).replaceAll(replacewith);
        }catch(Exception e){
            warn("Replace : Failed to process input; error - " + e.getMessage(), PigWarning.UDF_WARNING_1);
            return null;
        }
    }

    @Override
    public Schema outputSchema(Schema input) {
        return new Schema(new Schema.FieldSchema(null, DataType.CHARARRAY));
    }

    /* (non-Javadoc)
     * @see org.apache.pig.EvalFunc#getArgToFuncMapping()
     */
    @Override
    public List<FuncSpec> getArgToFuncMapping() throws FrontendException {
        List<FuncSpec> funcList = new ArrayList<FuncSpec>();
        Schema s = new Schema();
        s.add(new Schema.FieldSchema(null, DataType.CHARARRAY));
        s.add(new Schema.FieldSchema(null, DataType.CHARARRAY));
        s.add(new Schema.FieldSchema(null, DataType.CHARARRAY));
        funcList.add(new FuncSpec(this.getClass().getName(), s));
        return funcList;
    }

    @Override
    public boolean allowCompileTimeCalculation() {
        return true;
    }
}
