/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.client.RatisReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.SCMCommonPlacementPolicy;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PipelinePlacementPolicy
extends SCMCommonPlacementPolicy {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(PipelinePlacementPolicy.class);
    private final NodeManager nodeManager;
    private final PipelineStateManager stateManager;
    private final int heavyNodeCriteria;
    private static final int REQUIRED_RACKS = 2;
    public static final String MULTIPLE_RACK_PIPELINE_MSG = "The cluster has multiple racks, but all nodes with available pipeline capacity are on a single rack. There are insufficient cross rack nodes available to create a pipeline";

    public PipelinePlacementPolicy(NodeManager nodeManager, PipelineStateManager stateManager, ConfigurationSource conf) {
        super(nodeManager, conf);
        this.nodeManager = nodeManager;
        this.stateManager = stateManager;
        String dnLimit = conf.get("ozone.scm.datanode.pipeline.limit");
        this.heavyNodeCriteria = dnLimit == null ? 0 : Integer.parseInt(dnLimit);
    }

    public static int currentRatisThreePipelineCount(NodeManager nodeManager, PipelineStateManager stateManager, DatanodeDetails datanodeDetails) {
        return (int)nodeManager.getPipelines(datanodeDetails).stream().map(id -> {
            try {
                return stateManager.getPipeline((PipelineID)id);
            }
            catch (PipelineNotFoundException e) {
                LOG.debug("Pipeline not found in pipeline state manager during pipeline creation. PipelineID: {}", id, (Object)e);
                return null;
            }
        }).filter(PipelinePlacementPolicy::isNonClosedRatisThreePipeline).count();
    }

    private static boolean isNonClosedRatisThreePipeline(Pipeline p) {
        return p != null && p.getReplicationConfig().equals((Object)RatisReplicationConfig.getInstance((HddsProtos.ReplicationFactor)HddsProtos.ReplicationFactor.THREE)) && !p.isClosed();
    }

    @Override
    protected int getMaxReplicasPerRack(int numReplicas, int numberOfRacks) {
        if (numberOfRacks == 1) {
            return numReplicas;
        }
        return Math.max(numReplicas - 1, 1);
    }

    List<DatanodeDetails> filterViableNodes(List<DatanodeDetails> excludedNodes, List<DatanodeDetails> usedNodes, int nodesRequired, long metadataSizeRequired, long dataSizeRequired) throws SCMException {
        int initialHealthyNodesCount;
        List<DatanodeDetails> healthyNodes = this.nodeManager.getNodes(NodeStatus.inServiceHealthy());
        if (healthyNodes.isEmpty()) {
            String msg = "No healthy node found to allocate container.";
            LOG.error(msg);
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_HEALTHY_NODES);
        }
        healthyNodes = this.filterNodesWithSpace(healthyNodes, nodesRequired, metadataSizeRequired, dataSizeRequired);
        boolean multipleRacks = this.multipleRacksAvailable(healthyNodes);
        int excludedNodesSize = 0;
        if (excludedNodes != null) {
            excludedNodesSize = excludedNodes.size();
            healthyNodes.removeAll(excludedNodes);
        }
        if (usedNodes != null) {
            excludedNodesSize += usedNodes.size();
            healthyNodes.removeAll(usedNodes);
        }
        if ((initialHealthyNodesCount = healthyNodes.size()) < nodesRequired) {
            String msg = String.format("Pipeline creation failed due to no sufficient healthy datanodes. Required %d. Found %d. Excluded %d.", nodesRequired, initialHealthyNodesCount, excludedNodesSize);
            LOG.debug(msg);
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        List<DatanodeDetails> healthyList = healthyNodes.stream().map(d -> new DnWithPipelines((DatanodeDetails)d, PipelinePlacementPolicy.currentRatisThreePipelineCount(this.nodeManager, this.stateManager, d))).filter(d -> d.getPipelines() < this.nodeManager.pipelineLimit(d.getDn())).sorted(Comparator.comparingInt(DnWithPipelines::getPipelines)).map(d -> d.getDn()).collect(Collectors.toList());
        if (healthyList.size() < nodesRequired) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Unable to find enough nodes that meet the criteria that cannot engage in more than" + this.heavyNodeCriteria + " pipelines. Nodes required: " + nodesRequired + " Excluded: " + excludedNodesSize + " Found:" + healthyList.size() + " healthy nodes count in NodeManager: " + initialHealthyNodesCount);
            }
            String msg = String.format("Pipeline creation failed because nodes are engaged in other pipelines and every node can only be engaged in max %d pipelines. Required %d. Found %d. Excluded: %d.", this.heavyNodeCriteria, nodesRequired, healthyList.size(), excludedNodesSize);
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        if (!this.checkAllNodesAreEqual(this.nodeManager.getClusterNetworkTopologyMap())) {
            boolean multipleRacksAfterFilter = this.multipleRacksAvailable(healthyList);
            if (multipleRacks && !multipleRacksAfterFilter) {
                LOG.debug(MULTIPLE_RACK_PIPELINE_MSG);
                throw new SCMException(MULTIPLE_RACK_PIPELINE_MSG, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
            }
        }
        return healthyList;
    }

    private boolean multipleRacksAvailable(List<DatanodeDetails> dns) {
        if (dns.size() <= 1) {
            return false;
        }
        String initialRack = dns.get(0).getNetworkLocation();
        for (DatanodeDetails dn : dns) {
            if (dn.getNetworkLocation().equals(initialRack)) continue;
            return true;
        }
        return false;
    }

    @Override
    protected List<DatanodeDetails> chooseDatanodesInternal(List<DatanodeDetails> usedNodes, List<DatanodeDetails> excludedNodes, List<DatanodeDetails> favoredNodes, int nodesRequired, long metadataSizeRequired, long dataSizeRequired) throws SCMException {
        List<DatanodeDetails> healthyNodes = this.filterViableNodes(excludedNodes, usedNodes, nodesRequired, metadataSizeRequired, dataSizeRequired);
        if (this.checkAllNodesAreEqual(this.nodeManager.getClusterNetworkTopologyMap())) {
            return super.getResultSet(nodesRequired, healthyNodes);
        }
        return this.getResultSetWithTopology(nodesRequired, healthyNodes, usedNodes);
    }

    DatanodeDetails fallBackPickNodes(List<DatanodeDetails> nodeSet, List<DatanodeDetails> excludedNodes) {
        DatanodeDetails node;
        if (excludedNodes == null || excludedNodes.isEmpty()) {
            node = this.chooseNode(nodeSet);
        } else {
            List<DatanodeDetails> inputNodes = nodeSet.stream().filter(p -> !excludedNodes.contains(p)).collect(Collectors.toList());
            node = this.chooseNode(inputNodes);
        }
        return node;
    }

    private List<DatanodeDetails> getResultSetWithTopology(int nodesRequired, List<DatanodeDetails> healthyNodes, List<DatanodeDetails> usedNodes) throws SCMException {
        Preconditions.checkNotNull(usedNodes);
        Preconditions.checkNotNull(healthyNodes);
        Preconditions.checkState((nodesRequired >= 1 ? 1 : 0) != 0);
        if (nodesRequired + usedNodes.size() != HddsProtos.ReplicationFactor.THREE.getNumber()) {
            throw new SCMException("Nodes required number is not supported: " + nodesRequired, SCMException.ResultCodes.INVALID_CAPACITY);
        }
        ArrayList<DatanodeDetails> results = new ArrayList<DatanodeDetails>(nodesRequired);
        ArrayList<DatanodeDetails> mutableLstNodes = new ArrayList<DatanodeDetails>();
        ArrayList<DatanodeDetails> mutableExclude = new ArrayList<DatanodeDetails>();
        boolean rackAwareness = this.getAnchorAndNextNode(healthyNodes, usedNodes, results, mutableLstNodes, mutableExclude);
        if (mutableLstNodes.isEmpty()) {
            LOG.warn("Unable to find healthy node for anchor(first) node.");
            throw new SCMException("Unable to find anchor node.", SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        DatanodeDetails anchor = (DatanodeDetails)mutableLstNodes.get(0);
        DatanodeDetails nextNode = null;
        if (mutableLstNodes.size() == 2) {
            nextNode = (DatanodeDetails)mutableLstNodes.get(1);
        }
        int nodesToFind = nodesRequired - results.size();
        boolean bCheckNodeInAnchorRack = true;
        for (int x = 0; x < nodesToFind; ++x) {
            DatanodeDetails pick = null;
            if (rackAwareness && bCheckNodeInAnchorRack && (pick = this.chooseNodeBasedOnSameRack(healthyNodes, mutableExclude, this.nodeManager.getClusterNetworkTopologyMap(), anchor)) == null) {
                anchor = nextNode;
                pick = this.chooseNodeBasedOnSameRack(healthyNodes, mutableExclude, this.nodeManager.getClusterNetworkTopologyMap(), anchor);
            }
            if (pick == null) {
                bCheckNodeInAnchorRack = false;
                pick = this.fallBackPickNodes(healthyNodes, mutableExclude);
                if (rackAwareness) {
                    LOG.debug("Failed to choose node based on topology. Fallback picks node as: {}", (Object)pick);
                }
            }
            if (pick == null) {
                String msg = String.format("Unable to find suitable node in pipeline allocation. healthyNodes size: %d, excludeNodes size: %d", healthyNodes.size(), mutableExclude.size());
                LOG.debug(msg);
                throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
            }
            results.add(pick);
            this.removePeers(pick, healthyNodes);
            mutableExclude.add(pick);
            LOG.debug("Remaining node chosen: {}", (Object)pick);
        }
        if (results.size() < nodesRequired) {
            LOG.debug("Unable to find the required number of healthy nodes that  meet the criteria. Required nodes: {}, Found nodes: {}", (Object)nodesRequired, (Object)results.size());
            throw new SCMException("Unable to find required number of nodes.", SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        return results;
    }

    private boolean getAnchorAndNextNode(List<DatanodeDetails> healthyNodes, List<DatanodeDetails> usedNodes, List<DatanodeDetails> results, List<DatanodeDetails> mutableLstNodes, List<DatanodeDetails> mutableExclude) throws SCMException {
        DatanodeDetails anchor;
        boolean rackAwareness = false;
        DatanodeDetails nextNode = null;
        if (usedNodes.isEmpty()) {
            anchor = this.chooseFirstNode(healthyNodes);
            if (anchor == null) {
                LOG.debug("Unable to find healthy node for anchor(first) node.");
                throw new SCMException("Unable to find anchor node.", SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
            }
            results.add(anchor);
            this.removePeers(anchor, healthyNodes);
            mutableExclude.add(anchor);
            LOG.debug("First node chosen: {}", (Object)anchor);
        } else if (usedNodes.size() == 1) {
            anchor = usedNodes.get(0);
            this.removePeers(anchor, healthyNodes);
            mutableExclude.add(anchor);
        } else if (usedNodes.size() == 2) {
            anchor = usedNodes.get(0);
            this.removePeers(anchor, healthyNodes);
            mutableExclude.add(anchor);
            if (usedNodes.get(0).getParent() != usedNodes.get(1).getParent()) {
                nextNode = usedNodes.get(1);
                rackAwareness = true;
            }
            mutableExclude.add(usedNodes.get(1));
            this.removePeers(nextNode, healthyNodes);
        } else {
            LOG.warn("More than 2 used nodes, unable to choose anchor node.");
            throw new SCMException("Used Nodes required number is not supported: " + usedNodes.size(), SCMException.ResultCodes.INVALID_CAPACITY);
        }
        if (nextNode == null) {
            nextNode = this.chooseNodeBasedOnRackAwareness(healthyNodes, mutableExclude, this.nodeManager.getClusterNetworkTopologyMap(), anchor);
            if (nextNode != null) {
                rackAwareness = true;
                results.add(nextNode);
                this.removePeers(nextNode, healthyNodes);
                mutableExclude.add(nextNode);
                LOG.debug("Second node chosen: {}", (Object)nextNode);
            } else {
                LOG.debug("Pipeline Placement: Unable to find 2nd node on different rack based on rack awareness. anchor: {}", (Object)anchor);
            }
        }
        mutableLstNodes.add(anchor);
        if (nextNode != null) {
            mutableLstNodes.add(nextNode);
        }
        return rackAwareness;
    }

    @Override
    public DatanodeDetails chooseNode(List<DatanodeDetails> healthyNodes) {
        if (healthyNodes == null || healthyNodes.isEmpty()) {
            return null;
        }
        DatanodeDetails selectedNode = healthyNodes.get(this.getRand().nextInt(healthyNodes.size()));
        healthyNodes.remove(selectedNode);
        if (selectedNode != null) {
            this.removePeers(selectedNode, healthyNodes);
        }
        return selectedNode;
    }

    private DatanodeDetails chooseFirstNode(List<DatanodeDetails> healthyNodes) {
        if (healthyNodes == null || healthyNodes.isEmpty()) {
            return null;
        }
        DatanodeDetails selectedNode = healthyNodes.get(0);
        healthyNodes.remove(selectedNode);
        if (selectedNode != null) {
            this.removePeers(selectedNode, healthyNodes);
        }
        return selectedNode;
    }

    @VisibleForTesting
    protected DatanodeDetails chooseNodeBasedOnRackAwareness(List<DatanodeDetails> healthyNodes, List<DatanodeDetails> excludedNodes, NetworkTopology networkTopology, DatanodeDetails anchor) {
        Preconditions.checkArgument((networkTopology != null ? 1 : 0) != 0);
        if (this.checkAllNodesAreEqual(networkTopology)) {
            return null;
        }
        List nodesOnOtherRack = healthyNodes.stream().filter(p -> !excludedNodes.contains(p) && !anchor.getNetworkLocation().equals(p.getNetworkLocation())).collect(Collectors.toList());
        if (!nodesOnOtherRack.isEmpty()) {
            return (DatanodeDetails)nodesOnOtherRack.get(0);
        }
        return null;
    }

    @VisibleForTesting
    protected DatanodeDetails chooseNodeBasedOnSameRack(List<DatanodeDetails> healthyNodes, List<DatanodeDetails> excludedNodes, NetworkTopology networkTopology, DatanodeDetails anchor) {
        Preconditions.checkArgument((networkTopology != null ? 1 : 0) != 0);
        if (this.checkAllNodesAreEqual(networkTopology)) {
            return null;
        }
        List nodesOnSameRack = healthyNodes.stream().filter(p -> !excludedNodes.contains(p) && anchor.getNetworkLocation().equals(p.getNetworkLocation())).collect(Collectors.toList());
        if (!nodesOnSameRack.isEmpty()) {
            return (DatanodeDetails)nodesOnSameRack.get(0);
        }
        return null;
    }

    private boolean checkAllNodesAreEqual(NetworkTopology topology) {
        if (topology == null) {
            return true;
        }
        return topology.getNumOfNodes(topology.getMaxLevel() - 1) == 1;
    }

    @Override
    protected int getRequiredRackCount(int numReplicas, int excludedRackCount) {
        return 2;
    }

    public static class DnWithPipelines {
        private DatanodeDetails dn;
        private int pipelines;

        DnWithPipelines(DatanodeDetails dn, int pipelines) {
            this.dn = dn;
            this.pipelines = pipelines;
        }

        public int getPipelines() {
            return this.pipelines;
        }

        public DatanodeDetails getDn() {
            return this.dn;
        }
    }
}

