/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry.provider.flow.git;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.registry.flow.FlowPersistenceException;
import org.apache.nifi.registry.flow.FlowSnapshotContext;
import org.apache.nifi.registry.flow.MetadataAwareFlowPersistenceProvider;
import org.apache.nifi.registry.metadata.BucketMetadata;
import org.apache.nifi.registry.metadata.FlowMetadata;
import org.apache.nifi.registry.metadata.FlowSnapshotMetadata;
import org.apache.nifi.registry.provider.ProviderConfigurationContext;
import org.apache.nifi.registry.provider.ProviderCreationException;
import org.apache.nifi.registry.provider.flow.git.Bucket;
import org.apache.nifi.registry.provider.flow.git.Flow;
import org.apache.nifi.registry.provider.flow.git.GitFlowMetaData;
import org.apache.nifi.registry.util.FileUtils;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GitFlowPersistenceProvider
implements MetadataAwareFlowPersistenceProvider {
    private static final Logger logger = LoggerFactory.getLogger(GitFlowMetaData.class);
    static final String FLOW_STORAGE_DIR_PROP = "Flow Storage Directory";
    private static final String REMOTE_TO_PUSH = "Remote To Push";
    private static final String REMOTE_ACCESS_USER = "Remote Access User";
    private static final String REMOTE_ACCESS_PASSWORD = "Remote Access Password";
    private static final String REMOTE_CLONE_REPOSITORY = "Remote Clone Repository";
    static final String SNAPSHOT_EXTENSION = ".snapshot";
    private File flowStorageDir;
    private GitFlowMetaData flowMetaData;

    public void onConfigured(ProviderConfigurationContext configurationContext) throws ProviderCreationException {
        this.flowMetaData = new GitFlowMetaData();
        Map props = configurationContext.getProperties();
        if (!props.containsKey(FLOW_STORAGE_DIR_PROP)) {
            throw new ProviderCreationException("The property Flow Storage Directory must be provided");
        }
        String flowStorageDirValue = (String)props.get(FLOW_STORAGE_DIR_PROP);
        if (StringUtils.isEmpty((CharSequence)flowStorageDirValue)) {
            throw new ProviderCreationException("The property Flow Storage Directory cannot be null or blank");
        }
        this.flowMetaData.setRemoteToPush((String)props.get(REMOTE_TO_PUSH));
        String remoteUser = (String)props.get(REMOTE_ACCESS_USER);
        String remotePassword = (String)props.get(REMOTE_ACCESS_PASSWORD);
        String remoteRepo = (String)props.get(REMOTE_CLONE_REPOSITORY);
        if (!StringUtils.isEmpty((CharSequence)remoteRepo) && (StringUtils.isEmpty((CharSequence)remoteUser) || StringUtils.isEmpty((CharSequence)remotePassword))) {
            throw new ProviderCreationException(String.format("The property %s needs remote username and remote password", REMOTE_CLONE_REPOSITORY));
        }
        if (!StringUtils.isEmpty((CharSequence)remoteUser) && StringUtils.isEmpty((CharSequence)remotePassword)) {
            throw new ProviderCreationException(String.format("The property %s is specified but %s is not. %s is required for username password authentication.", REMOTE_ACCESS_USER, REMOTE_ACCESS_PASSWORD, REMOTE_ACCESS_PASSWORD));
        }
        if (!StringUtils.isEmpty((CharSequence)remotePassword)) {
            this.flowMetaData.setRemoteCredential(remoteUser, remotePassword);
        }
        try {
            this.flowStorageDir = new File(flowStorageDirValue);
            boolean localRepoExists = this.flowMetaData.localRepoExists(this.flowStorageDir);
            if (remoteRepo != null && !remoteRepo.isEmpty() && !localRepoExists) {
                logger.info("Validating remote repository [{}]", (Object)remoteRepo);
                this.flowMetaData.remoteRepoExists(remoteRepo);
                logger.info("Cloning remote repository [{}] to [{}]", (Object)remoteRepo, (Object)flowStorageDirValue);
                this.flowMetaData.cloneRepository(this.flowStorageDir, remoteRepo);
            }
            logger.info("Loading remote repository [{}]", (Object)remoteRepo);
            this.flowMetaData.loadGitRepository(this.flowStorageDir);
            this.flowMetaData.startPushThread();
            logger.info("Configured GitFlowPersistenceProvider with Flow Storage Directory {}", (Object)this.flowStorageDir.getAbsolutePath());
        }
        catch (IOException | GitAPIException e) {
            throw new ProviderCreationException("Failed to load a git repository " + this.flowStorageDir, e);
        }
    }

    public void saveFlowContent(FlowSnapshotContext context, byte[] content) throws FlowPersistenceException {
        File currentBucketDir;
        try {
            if (!this.flowMetaData.isGitDirectoryClean()) {
                throw new FlowPersistenceException(String.format("Git directory %s is not clean or has uncommitted changes, resolve those changes first to save flow contents.", this.flowStorageDir));
            }
        }
        catch (GitAPIException e) {
            throw new FlowPersistenceException(String.format("Failed to get Git status for directory %s due to %s", new Object[]{this.flowStorageDir, e}));
        }
        String bucketId = context.getBucketId();
        Bucket bucket = this.flowMetaData.getBucketOrCreate(bucketId);
        String currentBucketDirName = bucket.getBucketDirName();
        String bucketDirName = FileUtils.sanitizeFilename((String)context.getBucketName());
        boolean isBucketNameChanged = !bucketDirName.equals(currentBucketDirName);
        bucket.setBucketDirName(bucketDirName);
        Flow flow = bucket.getFlowOrCreate(context.getFlowId());
        String flowSnapshotFilename = FileUtils.sanitizeFilename((String)context.getFlowName()) + SNAPSHOT_EXTENSION;
        Optional<String> currentFlowSnapshotFilename = flow.getLatestVersion().map(flow::getFlowVersion).map(Flow.FlowPointer::getFileName);
        Flow.FlowPointer flowPointer = new Flow.FlowPointer(flowSnapshotFilename);
        flowPointer.setFlowName(context.getFlowName());
        flowPointer.setFlowDescription(context.getFlowDescription());
        flowPointer.setAuthor(context.getAuthor());
        flowPointer.setComment(context.getComments());
        flowPointer.setCreated(context.getSnapshotTimestamp());
        flow.putVersion(context.getVersion(), flowPointer);
        File bucketDir = new File(this.flowStorageDir, bucketDirName);
        File flowSnippetFile = new File(bucketDir, flowSnapshotFilename);
        File file = currentBucketDir = StringUtils.isEmpty((CharSequence)currentBucketDirName) ? null : new File(this.flowStorageDir, currentBucketDirName);
        if (currentBucketDir != null && currentBucketDir.isDirectory()) {
            if (isBucketNameChanged) {
                logger.debug("Detected bucket name change from {} to {}, moving it.", (Object)currentBucketDirName, (Object)bucketDirName);
                if (!currentBucketDir.renameTo(bucketDir)) {
                    throw new FlowPersistenceException(String.format("Failed to move existing bucket %s to %s.", currentBucketDir, bucketDir));
                }
            }
        } else if (!bucketDir.mkdirs()) {
            throw new FlowPersistenceException(String.format("Failed to create new bucket dir %s.", bucketDir));
        }
        try {
            if (currentFlowSnapshotFilename.isPresent() && !flowSnapshotFilename.equals(currentFlowSnapshotFilename.get())) {
                File latestFlowSnapshotFile = new File(bucketDir, currentFlowSnapshotFilename.get());
                logger.debug("Detected flow name change from {} to {}, deleting the old snapshot file.", (Object)currentFlowSnapshotFilename.get(), (Object)flowSnapshotFilename);
                latestFlowSnapshotFile.delete();
            }
            try (FileOutputStream os = new FileOutputStream(flowSnippetFile);){
                ((OutputStream)os).write(content);
                os.flush();
            }
            this.flowMetaData.saveBucket(bucket, bucketDir);
            this.flowMetaData.commit(context.getAuthor(), context.getComments(), bucket, flowPointer);
        }
        catch (IOException | GitAPIException e) {
            throw new FlowPersistenceException("Failed to persist flow.", e);
        }
    }

    public byte[] getFlowContent(String bucketId, String flowId, int version) throws FlowPersistenceException {
        Bucket bucket = this.getBucketOrFail(bucketId);
        Flow flow = this.getFlowOrFail(bucket, flowId);
        if (!flow.hasVersion(version)) {
            throw new FlowPersistenceException(String.format("Flow ID %s version %d was not found in bucket %s:%s.", flowId, version, bucket.getBucketDirName(), bucketId));
        }
        Flow.FlowPointer flowPointer = flow.getFlowVersion(version);
        try {
            return this.flowMetaData.getContent(flowPointer.getObjectId());
        }
        catch (IOException e) {
            throw new FlowPersistenceException(String.format("Failed to get content of Flow ID %s version %d in bucket %s:%s due to %s.", flowId, version, bucket.getBucketDirName(), bucketId, e), (Throwable)e);
        }
    }

    public void deleteAllFlowContent(String bucketId, String flowId) throws FlowPersistenceException {
        Bucket bucket = this.getBucketOrFail(bucketId);
        Optional<Flow> flowOpt = bucket.getFlow(flowId);
        if (!flowOpt.isPresent()) {
            logger.debug(String.format("Tried deleting all versions, but the Flow ID %s was not found in bucket %s:%s.", flowId, bucket.getBucketDirName(), bucket.getBucketId()));
            return;
        }
        Flow flow = flowOpt.get();
        Optional<Integer> latestVersionOpt = flow.getLatestVersion();
        if (!latestVersionOpt.isPresent()) {
            throw new IllegalStateException("Flow version is not added yet, can not be deleted.");
        }
        Integer latestVersion = latestVersionOpt.get();
        Flow.FlowPointer flowPointer = flow.getFlowVersion(latestVersion);
        File bucketDir = new File(this.flowStorageDir, bucket.getBucketDirName());
        File flowSnapshotFile = new File(bucketDir, flowPointer.getFileName());
        if (flowSnapshotFile.exists() && !flowSnapshotFile.delete()) {
            throw new FlowPersistenceException(String.format("Failed to delete flow content for %s:%s in bucket %s:%s", flowPointer.getFileName(), flowId, bucket.getBucketDirName(), bucketId));
        }
        bucket.removeFlow(flowId);
        try {
            if (bucket.isEmpty()) {
                FileUtils.deleteFile((File)bucketDir, (boolean)true);
            } else {
                this.flowMetaData.saveBucket(bucket, bucketDir);
            }
            String commitMessage = String.format("Deleted flow %s:%s in bucket %s:%s.", flowPointer.getFileName(), flowId, bucket.getBucketDirName(), bucketId);
            this.flowMetaData.commit(null, commitMessage, bucket, null);
        }
        catch (IOException | GitAPIException e) {
            throw new FlowPersistenceException(String.format("Failed to delete flow %s:%s in bucket %s:%s due to %s", flowPointer.getFileName(), flowId, bucket.getBucketDirName(), bucketId, e), e);
        }
    }

    private Bucket getBucketOrFail(String bucketId) throws FlowPersistenceException {
        Optional<Bucket> bucketOpt = this.flowMetaData.getBucket(bucketId);
        if (!bucketOpt.isPresent()) {
            throw new FlowPersistenceException(String.format("Bucket ID %s was not found.", bucketId));
        }
        return bucketOpt.get();
    }

    private Flow getFlowOrFail(Bucket bucket, String flowId) throws FlowPersistenceException {
        Optional<Flow> flowOpt = bucket.getFlow(flowId);
        if (!flowOpt.isPresent()) {
            throw new FlowPersistenceException(String.format("Flow ID %s was not found in bucket %s:%s.", flowId, bucket.getBucketDirName(), bucket.getBucketId()));
        }
        return flowOpt.get();
    }

    public void deleteFlowContent(String bucketId, String flowId, int version) throws FlowPersistenceException {
    }

    public List<BucketMetadata> getMetadata() {
        Map<String, Bucket> gitBuckets = this.flowMetaData.getBuckets();
        if (gitBuckets == null || gitBuckets.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<BucketMetadata> bucketMetadataList = new ArrayList<BucketMetadata>();
        for (Map.Entry<String, Bucket> bucketEntry : gitBuckets.entrySet()) {
            String bucketId = bucketEntry.getKey();
            Bucket gitBucket = bucketEntry.getValue();
            BucketMetadata bucketMetadata = new BucketMetadata();
            bucketMetadata.setIdentifier(bucketId);
            bucketMetadata.setName(gitBucket.getBucketDirName());
            bucketMetadata.setFlowMetadata(this.createFlowMetadata(gitBucket));
            bucketMetadataList.add(bucketMetadata);
        }
        return bucketMetadataList;
    }

    private List<FlowMetadata> createFlowMetadata(Bucket bucket) {
        if (bucket.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<FlowMetadata> flowMetadataList = new ArrayList<FlowMetadata>();
        for (Map.Entry<String, Flow> flowEntry : bucket.getFlows().entrySet()) {
            String flowId = flowEntry.getKey();
            Flow flow = flowEntry.getValue();
            Optional<Integer> latestVersion = flow.getLatestVersion();
            if (!latestVersion.isPresent()) continue;
            Flow.FlowPointer latestFlowPointer = flow.getFlowVersion(latestVersion.get());
            String flowName = latestFlowPointer.getFlowName();
            if (flowName == null && (flowName = latestFlowPointer.getFileName()).endsWith(SNAPSHOT_EXTENSION)) {
                flowName = flowName.substring(0, flowName.lastIndexOf("."));
            }
            FlowMetadata flowMetadata = new FlowMetadata();
            flowMetadata.setIdentifier(flowId);
            flowMetadata.setName(flowName);
            flowMetadata.setDescription(latestFlowPointer.getFlowDescription());
            flowMetadata.setFlowSnapshotMetadata(this.createFlowSnapshotMetdata(flow));
            flowMetadataList.add(flowMetadata);
        }
        return flowMetadataList;
    }

    private List<FlowSnapshotMetadata> createFlowSnapshotMetdata(Flow flow) {
        ArrayList<FlowSnapshotMetadata> flowSnapshotMetadataList = new ArrayList<FlowSnapshotMetadata>();
        Map<Integer, Flow.FlowPointer> versions = flow.getVersions();
        for (Map.Entry<Integer, Flow.FlowPointer> entry : versions.entrySet()) {
            Integer version = entry.getKey();
            Flow.FlowPointer flowPointer = entry.getValue();
            FlowSnapshotMetadata snapshotMetadata = new FlowSnapshotMetadata();
            snapshotMetadata.setVersion(version);
            snapshotMetadata.setAuthor(flowPointer.getAuthor());
            snapshotMetadata.setComments(flowPointer.getComment());
            snapshotMetadata.setCreated(flowPointer.getCreated());
            flowSnapshotMetadataList.add(snapshotMetadata);
        }
        return flowSnapshotMetadataList;
    }
}

