/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.stream.coordinator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.utils.ZKPaths;
import org.apache.kylin.common.util.Bytes;
import org.apache.kylin.common.util.JsonUtil;
import org.apache.kylin.shaded.com.google.common.base.Function;
import org.apache.kylin.shaded.com.google.common.collect.Lists;
import org.apache.kylin.shaded.com.google.common.collect.Maps;
import org.apache.kylin.stream.coordinator.StreamMetadataStore;
import org.apache.kylin.stream.coordinator.StreamingUtils;
import org.apache.kylin.stream.coordinator.assign.AssignmentUtil;
import org.apache.kylin.stream.coordinator.exception.StoreException;
import org.apache.kylin.stream.core.model.CubeAssignment;
import org.apache.kylin.stream.core.model.Node;
import org.apache.kylin.stream.core.model.ReplicaSet;
import org.apache.kylin.stream.core.model.SegmentBuildState;
import org.apache.kylin.stream.core.model.StreamingCubeConsumeState;
import org.apache.kylin.stream.core.source.Partition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZookeeperStreamMetadataStore
implements StreamMetadataStore {
    public static final String REPLICA_SET_ROOT = "/replica_sets";
    public static final String RECEIVER_ROOT = "/receivers";
    public static final String CUBE_ROOT = "/cubes";
    public static final String COORDINATOR_NODE = "/coordinator";
    public static final String CUBE_BUILD_STATE = "build_state";
    public static final String CUBE_CONSUME_STATE = "consume_state";
    public static final String CUBE_ASSIGNMENT = "assignment";
    public static final String CUBE_CONSUME_SRC_STATE = "consume_source_state";
    public static final String CUBE_SRC_CHECKPOINT = "source_checkpoint";
    private static final Logger logger = LoggerFactory.getLogger(ZookeeperStreamMetadataStore.class);
    private CuratorFramework client;
    private String zkRoot = "/stream";
    private String replicaSetRoot;
    private String receiverRoot;
    private String cubeRoot;
    private String coordinatorRoot;
    private AtomicLong readSuccess = new AtomicLong();
    private AtomicLong readFail = new AtomicLong();
    private AtomicLong writeSuccess = new AtomicLong();
    private AtomicLong writeFail = new AtomicLong();
    String reportTemplate = "[StreamMetadataStoreStats]  read : {} ; write: {} ; read failed: {} ; write failed: {} .";
    private AtomicLong lastReport = new AtomicLong();
    private static final long REPORT_DURATION = 300000L;

    public ZookeeperStreamMetadataStore() {
        this.client = StreamingUtils.getZookeeperClient();
        this.init();
    }

    private void init() {
        try {
            this.replicaSetRoot = this.zkRoot + REPLICA_SET_ROOT;
            this.receiverRoot = this.zkRoot + RECEIVER_ROOT;
            this.cubeRoot = this.zkRoot + CUBE_ROOT;
            this.coordinatorRoot = this.zkRoot + COORDINATOR_NODE;
            this.createZKNodeIfNotExist(this.zkRoot);
            this.createZKNodeIfNotExist(this.replicaSetRoot);
            this.createZKNodeIfNotExist(this.receiverRoot);
            this.createZKNodeIfNotExist(this.cubeRoot);
            this.createZKNodeIfNotExist(this.coordinatorRoot);
        }
        catch (Exception e) {
            logger.error("error when create zk nodes", e);
            throw new StoreException(e);
        }
    }

    private void createZKNodeIfNotExist(String path) throws Exception {
        if (this.client.checkExists().forPath(path) == null) {
            this.client.create().creatingParentsIfNeeded().forPath(path);
        }
    }

    @Override
    public void removeCubeAssignment(String cubeName) {
        logger.trace("Remove cube assignment {}.", (Object)cubeName);
        this.checkPath(cubeName);
        try {
            this.client.delete().forPath(ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_ASSIGNMENT));
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when remove cube assignment " + cubeName, e);
            throw new StoreException(e);
        }
    }

    @Override
    public List<CubeAssignment> getAllCubeAssignments() {
        try {
            ArrayList<CubeAssignment> cubeAssignmentList = Lists.newArrayList();
            List cubes = (List)this.client.getChildren().forPath(this.cubeRoot);
            for (String cube : cubes) {
                String cubeAssignmentPath = this.getCubeAssignmentPath(cube);
                if (this.client.checkExists().forPath(cubeAssignmentPath) == null) continue;
                byte[] data = (byte[])this.client.getData().forPath(cubeAssignmentPath);
                CubeAssignment assignment = CubeAssignment.deserializeCubeAssignment(data);
                cubeAssignmentList.add(assignment);
            }
            this.readSuccess.getAndIncrement();
            return cubeAssignmentList;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get assignments", e);
            throw new StoreException(e);
        }
    }

    @Override
    public Map<Integer, Map<String, List<Partition>>> getAllReplicaSetAssignments() {
        try {
            List<CubeAssignment> cubeAssignmentList = this.getAllCubeAssignments();
            this.readSuccess.getAndIncrement();
            return AssignmentUtil.convertCubeAssign2ReplicaSetAssign(cubeAssignmentList);
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get assignments", e);
            throw new StoreException(e);
        }
    }

    @Override
    public Map<String, List<Partition>> getAssignmentsByReplicaSet(int replicaSetID) {
        try {
            Map<Integer, Map<String, List<Partition>>> replicaSetAssignmentsMap = this.getAllReplicaSetAssignments();
            this.readSuccess.getAndIncrement();
            return replicaSetAssignmentsMap.get(replicaSetID);
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get assignment for replica set " + replicaSetID, e);
            throw new StoreException(e);
        }
    }

    @Override
    public CubeAssignment getAssignmentsByCube(String cubeName) {
        try {
            String cubeAssignmentPath = this.getCubeAssignmentPath(cubeName);
            if (this.client.checkExists().forPath(cubeAssignmentPath) == null) {
                logger.warn("Cannot find content at {}.", (Object)cubeAssignmentPath);
                return null;
            }
            byte[] data = (byte[])this.client.getData().forPath(cubeAssignmentPath);
            this.readSuccess.getAndIncrement();
            CubeAssignment assignment = CubeAssignment.deserializeCubeAssignment(data);
            return assignment;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get cube assignment for " + cubeName, e);
            throw new StoreException(e);
        }
    }

    @Override
    public List<ReplicaSet> getReplicaSets() {
        ArrayList<ReplicaSet> result = Lists.newArrayList();
        try {
            List replicaSetIDs = (List)this.client.getChildren().forPath(this.replicaSetRoot);
            this.readSuccess.getAndIncrement();
            for (String replicaSetID : replicaSetIDs) {
                ReplicaSet replicaSet = this.getReplicaSet(Integer.parseInt(replicaSetID));
                result.add(replicaSet);
            }
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get replica sets", e);
            throw new StoreException(e);
        }
        return result;
    }

    @Override
    public List<Integer> getReplicaSetIDs() {
        try {
            List replicaSetIDs = (List)this.client.getChildren().forPath(this.replicaSetRoot);
            this.readSuccess.getAndIncrement();
            return Lists.transform(replicaSetIDs, new Function<String, Integer>(){

                @Override
                @Nullable
                public Integer apply(@Nullable String input) {
                    return Integer.valueOf(input);
                }
            });
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get replica sets", e);
            throw new StoreException(e);
        }
    }

    @Override
    public int createReplicaSet(ReplicaSet rs) {
        try {
            List rsList = (List)this.client.getChildren().forPath(this.replicaSetRoot);
            List<Integer> rsIDList = Lists.transform(rsList, new Function<String, Integer>(){

                @Override
                @Nullable
                public Integer apply(@Nullable String input) {
                    Integer result;
                    try {
                        result = Integer.valueOf(input);
                    }
                    catch (Exception e) {
                        result = 0;
                    }
                    return result;
                }
            });
            int currMaxID = -1;
            if (rsIDList != null && !rsIDList.isEmpty()) {
                currMaxID = Collections.max(rsIDList);
            }
            int newReplicaSetID = currMaxID + 1;
            logger.trace("Id of new replica set {} is {}.", (Object)rs, (Object)newReplicaSetID);
            rs.setReplicaSetID(newReplicaSetID);
            String replicaSetPath = ZKPaths.makePath(this.replicaSetRoot, String.valueOf(newReplicaSetID));
            this.client.create().creatingParentsIfNeeded().forPath(replicaSetPath, this.serializeReplicaSet(rs));
            this.writeSuccess.getAndIncrement();
            return newReplicaSetID;
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when create replicaSet " + rs, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void updateReplicaSet(ReplicaSet rs) {
        try {
            byte[] replicaSetData = this.serializeReplicaSet(rs);
            this.client.setData().forPath(ZKPaths.makePath(this.replicaSetRoot, String.valueOf(rs.getReplicaSetID())), replicaSetData);
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("error when update replicaSet " + rs, e);
            throw new StoreException(e);
        }
    }

    @Override
    public Node getCoordinatorNode() {
        try {
            byte[] nodeData = (byte[])this.client.getData().forPath(this.coordinatorRoot);
            this.readSuccess.getAndIncrement();
            return JsonUtil.readValue(nodeData, Node.class);
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get coordinator leader", e);
            throw new StoreException(e);
        }
    }

    @Override
    public void setCoordinatorNode(Node coordinator) {
        try {
            byte[] coordinatorBytes = JsonUtil.writeValueAsBytes(coordinator);
            this.client.setData().forPath(this.coordinatorRoot, coordinatorBytes);
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when set coordinator leader to " + coordinator, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void saveSourceCheckpoint(String cubeName, String segmentName, int rsID, String sourceCheckpoint) {
        this.checkPath(cubeName, segmentName);
        logger.trace("Save remote checkpoint {} {} {} with content {}.", cubeName, segmentName, rsID, sourceCheckpoint);
        try {
            String path = ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_SRC_CHECKPOINT, segmentName, String.valueOf(rsID));
            if (this.client.checkExists().forPath(path) == null) {
                this.client.create().creatingParentsIfNeeded().forPath(path);
            } else {
                logger.warn("Checkpoint path already existed under path {}, overwrite with new one.", (Object)path);
            }
            this.client.setData().forPath(path, Bytes.toBytes(sourceCheckpoint));
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when save remote checkpoint for " + cubeName + " " + segmentName, e);
            throw new StoreException(e);
        }
    }

    @Override
    public Map<Integer, String> getSourceCheckpoint(String cubeName, String segmentName) {
        try {
            HashMap<Integer, String> result = Maps.newHashMap();
            String ckRoot = ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_SRC_CHECKPOINT, segmentName);
            if (this.client.checkExists().forPath(ckRoot) == null) {
                return null;
            }
            List children = (List)this.client.getChildren().forPath(ckRoot);
            if (children == null) {
                return null;
            }
            for (String child : children) {
                String rsPath = ZKPaths.makePath(ckRoot, child);
                byte[] checkpointBytes = (byte[])this.client.getData().forPath(rsPath);
                String sourcePos = Bytes.toString(checkpointBytes);
                result.put(Integer.valueOf(child), sourcePos);
            }
            this.readSuccess.getAndIncrement();
            return result;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error to fetch remote checkpoint for " + cubeName + " " + segmentName, e);
            throw new StoreException(e);
        }
    }

    private byte[] serializeReplicaSet(ReplicaSet rs) throws Exception {
        String nodesStr = JsonUtil.writeValueAsString(rs);
        return Bytes.toBytes(nodesStr);
    }

    @Override
    public ReplicaSet getReplicaSet(int rsID) {
        try {
            ReplicaSet result = new ReplicaSet();
            result.setReplicaSetID(rsID);
            byte[] replicaSetData = (byte[])this.client.getData().forPath(ZKPaths.makePath(this.replicaSetRoot, String.valueOf(rsID)));
            if (replicaSetData != null && replicaSetData.length > 0) {
                result = JsonUtil.readValue(Bytes.toString(replicaSetData), ReplicaSet.class);
            }
            this.readSuccess.getAndIncrement();
            return result;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get replica set " + rsID, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void removeReplicaSet(int rsID) {
        try {
            this.client.delete().forPath(ZKPaths.makePath(this.replicaSetRoot, String.valueOf(rsID)));
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when remove replica set " + rsID, e);
            throw new StoreException(e);
        }
    }

    @Override
    public List<Node> getReceivers() {
        ArrayList<Node> result = Lists.newArrayList();
        try {
            List receiverNames = (List)this.client.getChildren().forPath(this.receiverRoot);
            for (String receiverName : receiverNames) {
                Node node = Node.from(receiverName.replace('_', ':'));
                result.add(node);
            }
            this.readSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when fetch receivers", e);
            throw new StoreException(e);
        }
        return result;
    }

    @Override
    public List<String> getCubes() {
        try {
            List res = (List)this.client.getChildren().forPath(this.cubeRoot);
            this.readSuccess.getAndIncrement();
            return res;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when fetch cubes", e);
            throw new StoreException(e);
        }
    }

    @Override
    public void addStreamingCube(String cube) {
        this.checkPath(cube);
        try {
            String path = ZKPaths.makePath(this.cubeRoot, cube);
            if (this.client.checkExists().forPath(path) == null) {
                this.client.create().creatingParentsIfNeeded().forPath(path);
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when add cube " + cube, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void removeStreamingCube(String cube) {
        logger.trace("Remove cube {}", (Object)cube);
        this.checkPath(cube);
        try {
            String path = ZKPaths.makePath(this.cubeRoot, cube);
            if (this.client.checkExists().forPath(path) != null) {
                this.client.delete().deletingChildrenIfNeeded().forPath(ZKPaths.makePath(this.cubeRoot, cube));
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when remove cube " + cube, e);
            throw new StoreException(e);
        }
    }

    @Override
    public StreamingCubeConsumeState getStreamingCubeConsumeState(String cube) {
        try {
            String path = this.getCubeConsumeStatePath(cube);
            if (this.client.checkExists().forPath(path) != null) {
                byte[] cubeInfoData = (byte[])this.client.getData().forPath(path);
                this.readSuccess.getAndIncrement();
                if (cubeInfoData != null && cubeInfoData.length > 0) {
                    return JsonUtil.readValue(cubeInfoData, StreamingCubeConsumeState.class);
                }
                return StreamingCubeConsumeState.RUNNING;
            }
            return StreamingCubeConsumeState.RUNNING;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Error when get streaming cube consume state " + cube, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void saveStreamingCubeConsumeState(String cube, StreamingCubeConsumeState state) {
        this.checkPath(cube);
        try {
            String path = this.getCubeConsumeStatePath(cube);
            if (this.client.checkExists().forPath(path) != null) {
                this.client.setData().forPath(path, JsonUtil.writeValueAsBytes((Object)state));
            } else {
                this.client.create().creatingParentsIfNeeded().forPath(path, JsonUtil.writeValueAsBytes((Object)state));
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when save streaming cube consume state " + cube + " with " + (Object)((Object)state), e);
            throw new StoreException(e);
        }
    }

    @Override
    public void addReceiver(Node receiver) {
        logger.trace("Add {}.", (Object)receiver);
        try {
            String receiverPath = ZKPaths.makePath(this.receiverRoot, receiver.toNormalizeString());
            if (this.client.checkExists().forPath(receiverPath) == null) {
                this.client.create().creatingParentsIfNeeded().forPath(receiverPath);
            } else {
                logger.warn("{} exists.", (Object)receiverPath);
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when add new receiver " + receiver, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void removeReceiver(Node receiver) {
        logger.trace("Remove {}.", (Object)receiver);
        try {
            String receiverPath = ZKPaths.makePath(this.receiverRoot, receiver.toNormalizeString());
            if (this.client.checkExists().forPath(receiverPath) != null) {
                this.client.delete().deletingChildrenIfNeeded().forPath(receiverPath);
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Error when remove receiver " + receiver, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void saveNewCubeAssignment(CubeAssignment newCubeAssignment) {
        logger.trace("Try saving new cube assignment for: {}.", (Object)newCubeAssignment);
        try {
            String path = this.getCubeAssignmentPath(newCubeAssignment.getCubeName());
            if (this.client.checkExists().forPath(path) == null) {
                this.client.create().creatingParentsIfNeeded().forPath(path, CubeAssignment.serializeCubeAssignment(newCubeAssignment));
            } else {
                this.client.setData().forPath(path, CubeAssignment.serializeCubeAssignment(newCubeAssignment));
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Fail to save cube assignment", e);
            throw new StoreException(e);
        }
    }

    public void close() {
        try {
            this.client.close();
        }
        catch (Exception e) {
            logger.error("Exception throws when close assignmentManager", e);
        }
    }

    @Override
    public void addCompleteReplicaSetForSegmentBuild(String cubeName, String segmentName, int rsID) {
        logger.trace("Add completed rs {} to {} {}", rsID, cubeName, segmentName);
        this.checkPath(cubeName, segmentName);
        try {
            String path = ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_BUILD_STATE, segmentName, "replica_sets", String.valueOf(rsID));
            if (this.client.checkExists().forPath(path) == null) {
                this.client.create().creatingParentsIfNeeded().forPath(path);
            } else {
                logger.warn("ReplicaSet id {} existed under path {}", (Object)rsID, (Object)path);
            }
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Fail to add replicaSet Id to segment build state for " + segmentName + " " + rsID, e);
            throw new StoreException(e);
        }
    }

    @Override
    public void updateSegmentBuildState(String cubeName, String segmentName, SegmentBuildState.BuildState state) {
        logger.trace("Update {} {} to state {}", cubeName, segmentName, state);
        this.checkPath(cubeName, segmentName);
        try {
            String stateStr = JsonUtil.writeValueAsString(state);
            String path = ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_BUILD_STATE, segmentName);
            this.client.setData().forPath(path, Bytes.toBytes(stateStr));
            this.writeSuccess.getAndIncrement();
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Fail to update segment build state for " + segmentName + " to " + state, e);
            throw new StoreException(e);
        }
    }

    @Override
    public List<SegmentBuildState> getSegmentBuildStates(String cubeName) {
        try {
            String cubePath = this.getCubeBuildStatePath(cubeName);
            if (this.client.checkExists().forPath(cubePath) == null) {
                return Lists.newArrayList();
            }
            List segments = (List)this.client.getChildren().forPath(cubePath);
            this.readSuccess.getAndIncrement();
            ArrayList<SegmentBuildState> result = Lists.newArrayList();
            for (String segment : segments) {
                SegmentBuildState segmentState = this.doGetSegmentBuildState(cubePath, segment);
                result.add(segmentState);
            }
            return result;
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Fail to get segment build states " + cubeName, e);
            throw new StoreException(e);
        }
    }

    @Override
    public SegmentBuildState getSegmentBuildState(String cubeName, String segmentName) {
        try {
            String cubePath = this.getCubeBuildStatePath(cubeName);
            return this.doGetSegmentBuildState(cubePath, segmentName);
        }
        catch (Exception e) {
            this.readFail.getAndIncrement();
            logger.error("Fail to get segment build state for " + cubeName + " " + segmentName, e);
            throw new StoreException(e);
        }
    }

    private SegmentBuildState doGetSegmentBuildState(String cubePath, String segmentName) throws Exception {
        SegmentBuildState segmentState = new SegmentBuildState(segmentName);
        String segmentPath = ZKPaths.makePath(cubePath, segmentName);
        byte[] stateBytes = (byte[])this.client.getData().forPath(segmentPath);
        this.readSuccess.getAndIncrement();
        if (stateBytes != null && stateBytes.length > 0) {
            String stateStr = Bytes.toString(stateBytes);
            SegmentBuildState.BuildState state = JsonUtil.readValue(stateStr, SegmentBuildState.BuildState.class);
            segmentState.setState(state);
        }
        String replicaSetsPath = ZKPaths.makePath(segmentPath, "replica_sets");
        List replicaSets = (List)this.client.getChildren().forPath(replicaSetsPath);
        for (String replicaSetID : replicaSets) {
            segmentState.addCompleteReplicaSet(Integer.valueOf(replicaSetID));
        }
        return segmentState;
    }

    @Override
    public boolean removeSegmentBuildState(String cubeName, String segmentName) {
        logger.trace("Remove {} {}", (Object)cubeName, (Object)segmentName);
        this.checkPath(cubeName, segmentName);
        try {
            String path = ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_BUILD_STATE, segmentName);
            if (this.client.checkExists().forPath(path) != null) {
                this.client.delete().deletingChildrenIfNeeded().forPath(path);
                this.writeSuccess.getAndIncrement();
                return true;
            }
            logger.warn("Cube segment deep store state does not exisit!, path {} ", (Object)path);
            return false;
        }
        catch (Exception e) {
            this.writeFail.getAndIncrement();
            logger.error("Fail to remove cube segment deep store state " + cubeName + " " + segmentName, e);
            throw new StoreException(e);
        }
    }

    private String getCubeAssignmentPath(String cubeName) {
        return ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_ASSIGNMENT);
    }

    private String getCubeBuildStatePath(String cubeName) {
        return ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_BUILD_STATE);
    }

    private String getCubeConsumeStatePath(String cubeName) {
        return ZKPaths.makePath(this.cubeRoot, cubeName, CUBE_CONSUME_STATE);
    }

    @Override
    public void reportStat() {
        if (this.writeFail.get() > 0L || this.readFail.get() > 0L) {
            logger.warn(this.reportTemplate, this.readSuccess.get(), this.writeSuccess.get(), this.readFail.get(), this.writeFail.get());
        } else if (System.currentTimeMillis() - this.lastReport.get() >= 300000L) {
            logger.debug(this.reportTemplate, this.readSuccess.get(), this.writeSuccess.get(), this.readFail.get(), this.writeFail.get());
        } else {
            return;
        }
        this.lastReport.set(System.currentTimeMillis());
    }

    private void checkPath(String ... paths) {
        for (String path : paths) {
            if (path != null && path.length() != 0) continue;
            throw new IllegalArgumentException("Illegal zookeeper path.");
        }
    }
}

