import Ajv from "ajv";
import * as metaSchemaDraft04 from "ajv/lib/refs/json-schema-draft-04.json";
import { duration } from "moment";
import { DATE_AND_TIME_ENUM_REGEXP, DATE_AND_TIME_FEEL_REGEXP, DATE_ENUM_REGEXP, DATE_FEEL_REGEXP, DURATION_ENUM_REGEXP, DURATION_FEEL_REGEXP, TIME_ENUM_REGEXP, TIME_FEEL_REGEXP, } from "./dmnRegExp";
import { X_DMN_ALLOWED_VALUES_KEYWORD, X_DMN_DESCRIPTIONS_KEYWORD, X_DMN_TYPE_CONSTRAINTS_KEYWORD, X_DMN_TYPE_KEYWORD, } from "./jitExecutorKeywords";
import { RECURSION_KEYWORD, RECURSION_REF_KEYWORD } from "./jsonSchemaConstants";
import { DAYS_AND_TIME_DURATION_FORMAT, DAYS_AND_TIME_DURATION_REGEXP, YEARS_AND_MONTHS_DURATION_FORMAT, YEARS_AND_MONTHS_DURATION_REGEXP, } from "./dmnFormats";
export var DmnAjvSchemaFormat;
(function (DmnAjvSchemaFormat) {
    DmnAjvSchemaFormat["DATE"] = "date";
    DmnAjvSchemaFormat["TIME"] = "time";
    DmnAjvSchemaFormat["DATE_TIME"] = "date-time";
    DmnAjvSchemaFormat["DAYS_TIME_DURATION"] = "days and time duration";
    DmnAjvSchemaFormat["YEARS_MONTHS_DURATION"] = "years and months duration";
})(DmnAjvSchemaFormat || (DmnAjvSchemaFormat = {}));
export class DmnRunnerAjv {
    constructor() {
        this.parseRangeFromConstraints = (constraint, type) => {
            if ((constraint.startsWith("[") || constraint.startsWith("(")) &&
                (constraint.endsWith("]") || constraint.endsWith(")")) &&
                constraint.includes("..")) {
                let feelFunction;
                let regExp;
                switch (type) {
                    case DmnAjvSchemaFormat.DATE:
                        feelFunction = "date";
                        regExp = DATE_FEEL_REGEXP;
                        break;
                    case DmnAjvSchemaFormat.TIME:
                        feelFunction = "time";
                        regExp = TIME_FEEL_REGEXP;
                        break;
                    case DmnAjvSchemaFormat.DATE_TIME:
                        feelFunction = "date and time";
                        regExp = DATE_AND_TIME_FEEL_REGEXP;
                        break;
                    case DmnAjvSchemaFormat.DAYS_TIME_DURATION:
                    case DmnAjvSchemaFormat.YEARS_MONTHS_DURATION:
                        feelFunction = "duration";
                        regExp = DURATION_FEEL_REGEXP;
                        break;
                }
                const matches = constraint.match(regExp);
                if (matches) {
                    const minAllowed = matches[0].replace(`${feelFunction}("`, "").replace('")', "");
                    const minAllowedIncluded = constraint.startsWith("[");
                    const maxAllowed = matches[1].replace(`${feelFunction}("`, "").replace('")', "");
                    const maxAllowedIncluded = constraint.endsWith("]");
                    return { minAllowed, minAllowedIncluded, maxAllowed, maxAllowedIncluded };
                }
            }
            return {};
        };
        this.parseEnumerationFromConstraints = (constraint, type) => {
            let feelFunction;
            let regExp;
            switch (type) {
                case DmnAjvSchemaFormat.DATE:
                    feelFunction = "date";
                    regExp = DATE_ENUM_REGEXP;
                    break;
                case DmnAjvSchemaFormat.TIME:
                    feelFunction = "time";
                    regExp = TIME_ENUM_REGEXP;
                    break;
                case DmnAjvSchemaFormat.DATE_TIME:
                    feelFunction = "date and time";
                    regExp = DATE_AND_TIME_ENUM_REGEXP;
                    break;
                case DmnAjvSchemaFormat.DAYS_TIME_DURATION:
                case DmnAjvSchemaFormat.YEARS_MONTHS_DURATION:
                    feelFunction = "duration";
                    regExp = DURATION_ENUM_REGEXP;
                    break;
            }
            const matches = constraint.match(regExp);
            if (matches) {
                return [
                    ...matches.map((matchedString) => matchedString.trim().replace(`${feelFunction}("`, "").replace('"),', "").replace('")', "")),
                ];
            }
            return [];
        };
        this.ajv = new Ajv({
            allErrors: true,
            schemaId: "auto",
            useDefaults: true,
            removeAdditional: "all",
            verbose: true,
        });
        this.ajv.addMetaSchema(metaSchemaDraft04);
        this.ajv.addKeyword(X_DMN_TYPE_KEYWORD, {});
        this.ajv.addKeyword(X_DMN_ALLOWED_VALUES_KEYWORD, {
            compile: this.constraintCompiler(),
        });
        this.ajv.addKeyword(X_DMN_TYPE_CONSTRAINTS_KEYWORD, {
            compile: this.constraintCompiler(),
        });
        this.ajv.addKeyword(X_DMN_DESCRIPTIONS_KEYWORD, {});
        this.ajv.addKeyword(RECURSION_KEYWORD, {});
        this.ajv.addKeyword(RECURSION_REF_KEYWORD, {});
        this.ajv.addFormat(DAYS_AND_TIME_DURATION_FORMAT, {
            type: "string",
            validate: (data) => !!data.match(DAYS_AND_TIME_DURATION_REGEXP),
        });
        this.ajv.addFormat(YEARS_AND_MONTHS_DURATION_FORMAT, {
            type: "string",
            validate: (data) => !!data.match(YEARS_AND_MONTHS_DURATION_REGEXP),
        });
    }
    constraintCompiler() {
        return (schema, parentSchema, it) => {
            if (!parentSchema.format) {
                return (data) => true;
            }
            const { minAllowed, minAllowedIncluded, maxAllowed, maxAllowedIncluded } = this.parseRangeFromConstraints(schema !== null && schema !== void 0 ? schema : "", parentSchema.format);
            const enumeratedValues = this.parseEnumerationFromConstraints(schema !== null && schema !== void 0 ? schema : "", parentSchema.format);
            const isUnderTheMinBoundary = minAllowedIncluded ? (value, minBoundary) => value < minBoundary : (value, minBoundary) => value <= minBoundary;
            const isOverTheMaxBoundary = maxAllowedIncluded ? (value, maxBoundary) => value > maxBoundary : (value, maxBoundary) => value >= maxBoundary;
            return (rawData) => {
                var _a, _b, _c, _d;
                let data = rawData instanceof Date ? rawData.toISOString() : rawData;
                if (data.includes(".") && data.endsWith("Z")) {
                    data = data.substring(0, data.lastIndexOf("."));
                }
                if (minAllowed && maxAllowed) {
                    if (parentSchema.format === "time") {
                        if (isUnderTheMinBoundary(data, minAllowed) || isOverTheMaxBoundary(data, maxAllowed)) {
                            return false;
                        }
                    }
                    else if ((_a = parentSchema.format) === null || _a === void 0 ? void 0 : _a.includes("duration")) {
                        const actualDuration = duration(data).asMilliseconds();
                        const minDuration = duration(minAllowed).asMilliseconds();
                        const maxDuration = duration(maxAllowed).asMilliseconds();
                        if (isUnderTheMinBoundary(actualDuration, minDuration) ||
                            isOverTheMaxBoundary(actualDuration, maxDuration)) {
                            return false;
                        }
                    }
                    else if ((_b = parentSchema.format) === null || _b === void 0 ? void 0 : _b.includes("date")) {
                        const actualDate = new Date(data);
                        if (actualDate.toString() === "Invalid Date") {
                            return false;
                        }
                        const minAllowedDate = new Date(minAllowed);
                        if (minAllowedDate && isUnderTheMinBoundary(actualDate, minAllowedDate)) {
                            return false;
                        }
                        const maxAllowedDate = new Date(maxAllowed);
                        if (maxAllowedDate && isOverTheMaxBoundary(actualDate, maxAllowedDate)) {
                            return false;
                        }
                    }
                }
                else if (enumeratedValues) {
                    if (parentSchema.format === "time") {
                        return enumeratedValues.includes(data);
                    }
                    else if ((_c = parentSchema.format) === null || _c === void 0 ? void 0 : _c.includes("duration")) {
                        const actualDuration = duration(data).asMilliseconds();
                        return enumeratedValues.some((value) => {
                            const enumeratedDurationValue = duration(value).asMilliseconds();
                            return enumeratedDurationValue === actualDuration;
                        });
                    }
                    else if ((_d = parentSchema.format) === null || _d === void 0 ? void 0 : _d.includes("date")) {
                        const actualDate = new Date(data);
                        if (actualDate.toString() === "Invalid Date") {
                            return false;
                        }
                        return enumeratedValues.some((value) => {
                            const enumeratedDateValue = new Date(value);
                            return !(enumeratedDateValue < actualDate) && !(enumeratedDateValue > actualDate);
                        });
                    }
                }
                return true;
            };
        };
    }
    getAjv() {
        return this.ajv;
    }
}
//# sourceMappingURL=ajv.js.map