/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.server.metrics.group;

import org.apache.fluss.metrics.CharacterFilter;
import org.apache.fluss.metrics.groups.AbstractMetricGroup;
import org.apache.fluss.metrics.registry.MetricRegistry;

import javax.annotation.Nullable;

import java.util.Map;

import static org.apache.fluss.metrics.utils.MetricGroupUtils.makeScope;

/** Metrics for the table buckets with table as parent group. */
public class BucketMetricGroup extends AbstractMetricGroup {
    // will be null if the bucket doesn't belong to a partition
    private final @Nullable String partitionName;
    private final int bucket;

    public BucketMetricGroup(
            MetricRegistry registry,
            @Nullable String partitionName,
            int bucket,
            TableMetricGroup parent) {
        super(registry, makeScope(parent, String.valueOf(bucket)), parent);
        this.partitionName = partitionName;
        this.bucket = bucket;
    }

    @Override
    protected void putVariables(Map<String, String> variables) {
        if (partitionName != null) {
            variables.put("partition", partitionName);
        } else {
            // value of empty string indicates non-partitioned tables
            variables.put("partition", "");
        }
        variables.put("bucket", String.valueOf(bucket));
    }

    @Override
    protected String getGroupName(CharacterFilter filter) {
        return "bucket";
    }

    public TableMetricGroup getTableMetricGroup() {
        return (TableMetricGroup) parent;
    }
}
