// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.plans.commands;

import org.apache.doris.catalog.Env;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.ErrorCode;
import org.apache.doris.common.ErrorReport;
import org.apache.doris.mysql.privilege.PrivPredicate;
import org.apache.doris.nereids.trees.plans.PlanType;
import org.apache.doris.nereids.trees.plans.visitor.PlanVisitor;
import org.apache.doris.qe.ConnectContext;
import org.apache.doris.qe.StmtExecutor;

import java.util.HashMap;
import java.util.Map;

/**
 * alter workload policy command
 */
public class AlterWorkloadPolicyCommand extends AlterCommand {
    private final String workloadPolicyName;
    private final Map<String, String> properties;

    /**
     * constructor
     */
    public AlterWorkloadPolicyCommand(String workloadPolicyName, Map<String, String> inputProperties) {
        super(PlanType.ALTER_WORKLOAD_POLICY_COMMAND);
        this.workloadPolicyName = workloadPolicyName;
        this.properties = new HashMap<>();
        for (String key : inputProperties.keySet()) {
            properties.put(key.toLowerCase(), inputProperties.get(key));
        }
    }

    @Override
    public void doRun(ConnectContext ctx, StmtExecutor executor) throws Exception {
        // check auth
        if (!Env.getCurrentEnv().getAccessManager().checkGlobalPriv(ConnectContext.get(), PrivPredicate.ADMIN)) {
            ErrorReport.reportAnalysisException(ErrorCode.ERR_SPECIFIC_ACCESS_DENIED_ERROR, "ADMIN");
        }

        if (properties == null || properties.isEmpty()) {
            throw new AnalysisException("properties can't be null when alter workload schedule policy");
        }
        Env.getCurrentEnv().getWorkloadSchedPolicyMgr().alterWorkloadSchedPolicy(workloadPolicyName, properties);
    }

    @Override
    public <R, C> R accept(PlanVisitor<R, C> visitor, C context) {
        return visitor.visitAlterWorkloadPolicyCommand(this, context);
    }
}
